//////////////////////////////////////////////////////////////////////////////////
// Project Name: [ CDX Class Library - CDX.lib ]
// Source File:  [ CDX Main header file ]
// Author:       [ Bil Simser - bsimser@home.com ]
// Revision:     [ 1.99a ]
//////////////////////////////////////////////////////////////////////////////////
#ifndef CDX_H
#define CDX_H

#pragma warning (disable : 4244)

// if this is defined then we only want DX3 interfaces
#ifdef NTDX3

// we _only_ want DirectX 3.0 interfaces
#define DIRECTDRAW_VERSION 0x0300
#define DIRECTINPUT_VERSION 0x0300
#define DIRECTSOUND_VERSION 0x0300

#endif

// CDX Currently support DirectX 6
#define CDX_DDVER		0x0600
#define CDX_DIVER		0x0600
#define CDX_DSVER		0x0600

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <stdio.h>
#include <ddraw.h>
#include <dinput.h>
#include <dsound.h>
#include <dplay.h>
#include <dplobby.h>
#include <vfw.h>

//<JJH>
// CDX memory types
#define CDXMEM_VIDEOONLY		0
#define CDXMEM_SYSTEMONLY		1
#define CDXMEM_VIDTHENSYS		2

// CDXSprite BLT types
#define CDXBLT_BLK				0
#define CDXBLT_BLKALPHA			1
#define CDXBLT_BLKALPHAFAST		2
#define CDXBLT_BLKSCALED		3
#define CDXBLT_BLKSHADOW		4
#define CDXBLT_BLKSHADOWFAST	5
#define CDXBLT_BLKROTATED		6
#define CDXBLT_BLKROTOZOOM		7
#define CDXBLT_BLKHFLIP			8
#define CDXBLT_BLKVFLIP			9

#define CDXBLT_TRANS			100
#define CDXBLT_TRANSALPHA		101
#define CDXBLT_TRANSALPHAFAST	102
#define CDXBLT_TRANSSCALED		103
#define CDXBLT_TRANSSHADOW		104
#define CDXBLT_TRANSSHADOWFAST	105
#define CDXBLT_TRANSROTATED		106
#define CDXBLT_TRANSROTOZOOM	107
#define CDXBLT_TRANSHFLIP		108
#define CDXBLT_TRANSVFLIP		109

#define CDXBLT_TRANSALPHALUT	110
#define CDXBLT_TRANSALPHAORIG	111

typedef struct
{
    RGBQUAD depth;		// How many bits of precision per channel
    RGBQUAD Position;	// At what bit position does the color start
	UINT bpp;
} RGBFORMAT;
//<JJH>

//////////////////////////////////////////////////////////////////////////////////
// Globals
//////////////////////////////////////////////////////////////////////////////////
extern "C"
{
	// CDX Functions
	extern void Clip(int *DestX, int *DestY, RECT *SrcRect, RECT *DestRect);

	// DDUTILS Functions
#if DIRECTDRAW_VERSION >= CDX_DDVER
	extern IDirectDrawPalette* DDLoadPalette(IDirectDraw4 *pdd, LPCSTR szBitmap);
	extern IDirectDrawSurface4* DDLoadBitmap(IDirectDraw4 *pdd, LPCSTR szBitmap, int dx, int dy, BOOL memoryType);
	extern IDirectDrawSurface4* DDLoadSizeBitmap(IDirectDraw4 *pdd, LPCSTR szBitmap, int *dx, int *dy, BOOL memoryType);
	IDirectDrawSurface4 *DDCreateSurface(IDirectDraw4 *pdd, DWORD width, DWORD height, DDSURFACEDESC2 *ddsd);
	extern HRESULT             DDReLoadBitmap(IDirectDrawSurface4 *pdds, LPCSTR szBitmap);
	extern HRESULT             DDCopyBitmap(IDirectDrawSurface4 *pdds, HBITMAP hbm, int x, int y, int dx, int dy);
	extern DWORD               DDColorMatch(IDirectDrawSurface4 *pdds, COLORREF rgb);
	extern HRESULT             DDSetColorKey(IDirectDrawSurface4 *pdds, COLORREF rgb);
//<JJH>
	extern BOOL __cdecl        IsMMX(void);
    extern BOOL                GetRGBFormat(IDirectDrawSurface4* Surface, RGBFORMAT* rgb);
//<JJH>
#else
	extern IDirectDrawPalette* DDLoadPalette(IDirectDraw *pdd, LPCSTR szBitmap);
	extern IDirectDrawSurface* DDLoadBitmap(IDirectDraw *pdd, LPCSTR szBitmap, int dx, int dy, BOOL memoryType);
	extern IDirectDrawSurface* DDLoadSizeBitmap(IDirectDraw *pdd, LPCSTR szBitmap, int *dx, int *dy, BOOL memoryType);
	IDirectDrawSurface *DDCreateSurface(IDirectDraw *pdd, DWORD width, DWORD height, DDSURFACEDESC *ddsd);
	extern HRESULT             DDReLoadBitmap(IDirectDrawSurface *pdds, LPCSTR szBitmap);
	extern HRESULT             DDCopyBitmap(IDirectDrawSurface *pdds, HBITMAP hbm, int x, int y, int dx, int dy);
	extern DWORD               DDColorMatch(IDirectDrawSurface *pdds, COLORREF rgb);
	extern HRESULT             DDSetColorKey(IDirectDrawSurface *pdds, COLORREF rgb);
//<JJH>
	extern BOOL __cdecl        IsMMX(void);
    extern BOOL                GetRGBFormat(IDirectDrawSurface* Surface, RGBFORMAT* rgb);
//<JJH>
#endif
	// DirectX Errors
	extern void DDError(HRESULT hErr, void* hWnd, LPCSTR szFile, int iLine);
	extern void DSError(HRESULT hErr, void* hWnd);
}

class CDXSurface;

//////////////////////////////////////////////////////////////////////////////////
// CDXScreen Class
//////////////////////////////////////////////////////////////////////////////////
class CDXScreen
{
public:
	CDXScreen();
	CDXScreen(void *hWnd, DWORD Width, DWORD Height, DWORD BPP, BOOL bVGA = FALSE);
	CDXScreen(void *hWnd, int Width, int Height);
	~CDXScreen();

	BOOL CreateFullScreen(void *hWnd, DWORD Width, DWORD Height, DWORD BPP, BOOL bVGA = FALSE);
	BOOL CreateWindowed(void *hWnd, int Width, int Height);
	BOOL LoadBitmap(const char* szFilename);
	BOOL LoadPalette(const char* szFilename);
	void Fill(DWORD FillColor);
	void Restore(void);
	HRESULT Flip(void);

	void SetColor(int col, int r, int g, int b);
	void GetColor(int col, int *r, int *g, int *b);
	void SetPalette(int Start, int Count, LPPALETTEENTRY lpPE);
	void GetPalette(int Start, int Count, LPPALETTEENTRY lpPE);
	void FillPalette(int r, int g, int b);
	void GreyScale(void);
	void FadeIn(int delay, LPPALETTEENTRY lpPE);
	void FadeOut(int delay);
	void FadeTo(int r, int g, int b, int delay);

	int GetWidth(void) { return m_dwPixelWidth; }
	int GetHeight(void) { return m_dwPixelHeight; }
	int GetBPP(void) { return m_BPP; }
#if DIRECTDRAW_VERSION >= CDX_DDVER
	LPDIRECTDRAW4 GetDD(void) { return m_lpDD; }
#else
	LPDIRECTDRAW GetDD(void) { return m_lpDD; }
#endif
	CDXSurface* GetFront(void) { return m_lpDDSFront; }
	CDXSurface* GetBack(void) { return m_lpDDSBack; }
	LPDIRECTDRAWPALETTE GetPalette(void) { return m_lpDDPalette; }

public:
#if DIRECTDRAW_VERSION >= CDX_DDVER
	LPDIRECTDRAW4 m_lpDD;
	LPDIRECTDRAWSURFACE4 m_ZBuffer;
#else
	LPDIRECTDRAW m_lpDD;
	LPDIRECTDRAWSURFACE m_ZBuffer;
#endif
	CDXSurface* m_lpDDSFront;
	CDXSurface* m_lpDDSBack;
	LPDIRECTDRAWPALETTE m_lpDDPalette;
	LPDIRECTDRAWCLIPPER m_lpClipper;

	DWORD m_dwPixelWidth;
	DWORD m_dwPixelHeight;
	DWORD m_BPP;
	void* m_hWnd;

	BOOL m_bFullScreen;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Class
//////////////////////////////////////////////////////////////////////////////////
class CDXSurface
{
public:
	CDXSurface();
//<JJH>
	CDXSurface(CDXScreen *pScreen, int Width, int Height, BOOL memoryType);
	CDXSurface(CDXScreen *pScreen, const char *szFilename, BOOL memoryType);
//<JJH>
	~CDXSurface();

//<JJH>
	BOOL Create(CDXScreen *pScreen, int Width, int Height, BOOL memoryType);
	BOOL Create(CDXScreen *pScreen, const char *szFilename, BOOL memoryType);
//<JJH>

	void SetDest(int t, int l, int b, int r);
 	void SetSrc(int t, int l, int b, int r);

//<JJH>
	void SetColorKey(DWORD col);
//<JJH>
	void SetColorKey(void);

	void Restore(void);
	void Fill(DWORD FillColor);

	void ChangeFont(const char* FontName, int Width, int Height, int Attributes = FW_NORMAL);
	void SetFont(void);
	HRESULT Lock(void);
	HRESULT UnLock(void);
	void GetDC();
	void ReleaseDC();
	void TextXY(int X, int Y, COLORREF Col, LPCTSTR pString);

//<JJH>
	void PutPixel(int X, int Y, DWORD Col);
	DWORD GetPixel(int X, int Y);
	void Rect(int X1,int Y1,int X2,int Y2,DWORD Col);
	void FillRect(int X1, int Y1, int X2, int Y2, DWORD Col);
	void Line(int X1, int Y1, int X2, int Y2, DWORD Col);
	void VLine(int Y1, int Y2, int X, DWORD Col);
	void HLine(int X1, int X2, int Y, DWORD Col);
	void Circle(int X, int Y, int Radius, DWORD Col);
	void FillCircle(int X, int Y, int Radius, DWORD Col);
//<JJH>

	virtual HRESULT Draw(CDXSurface* lpDDS);
	virtual HRESULT DrawFast(int X, int Y, CDXSurface* lpDDS);
	virtual HRESULT DrawTrans(int X, int Y, CDXSurface* lpDDS);
	virtual HRESULT DrawClipped(int X, int Y, CDXSurface* lpDDS, LPRECT ClipRect);
	virtual HRESULT DrawWindowed(CDXSurface* lpDDS);
	virtual HRESULT DrawScaled(int X, int Y, float Factor, CDXSurface* lpDDS);
	virtual HRESULT DrawScaled(int X, int Y, int Width, int Height, CDXSurface* lpDDS);
	virtual HRESULT DrawHFlip(int X, int Y, CDXSurface* lpDDS);
	virtual HRESULT DrawVFlip(int X, int Y, CDXSurface* lpDDS);
	virtual void DrawRotated(int X, int Y, double Angle, CDXSurface* lpDDS);

//<JJH>
	BOOL ValidateBlt(CDXSurface* lpCDXS, LONG *lDestX, LONG *lDestY, RECT *srcRect);
	BOOL ClipRect(RECT *Rect);
	void GetClipRect(RECT *clipRect);
	void SetClipRect(RECT *clipRect);

	HRESULT CDXBLT_Blk(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_Trans(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_BlkHFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_TransHFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_BlkVFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_TransVFlip(CDXSurface* lpCDXS, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_BlkAlphaFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_TransAlphaFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_BlkAlpha(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD ALPHA);
	HRESULT CDXBLT_TransAlpha(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD ALPHA);
	HRESULT CDXBLT_BlkShadow(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD SHADOW);
	HRESULT CDXBLT_TransShadow(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect, WORD SHADOW);
	HRESULT CDXBLT_BlkShadowFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect);
	HRESULT CDXBLT_TransShadowFast(CDXSurface* dest, LONG lDestX, LONG lDestY, RECT srcRect);
//<JJH>

public:
	int m_PixelWidth;
	int m_PixelHeight;
	const char* m_pFilename;

	HDC m_DC;

//<JJH>
	DWORD m_ColorKey;
//<JJH>

	RECT SrcRect;
	RECT DestRect;
	HFONT m_Font;
#if DIRECTDRAW_VERSION >= CDX_DDVER
	DDSURFACEDESC2 m_DDSD;
	LPDIRECTDRAWSURFACE4 m_lpDDS;
#else
	DDSURFACEDESC m_DDSD;
	LPDIRECTDRAWSURFACE m_lpDDS;
#endif
	CDXScreen *Screen;

//<JJH>
	BOOL m_MMX;
	RECT m_ClipRect;
	RGBFORMAT m_RGB;
//<JJH>
	char *m_FontName;
	BOOL m_FontLoaded;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXLayer Class
//////////////////////////////////////////////////////////////////////////////////
class CDXLayer : public CDXSurface
{
public:
//<JJH>
	CDXLayer(CDXScreen *pScreen, char *szFilename, BOOL memoryType);
//	CDXLayer(CDXScreen *pScreen, char *szFilename);
//<JJH>
	void ScrollUp(int Offset);
	void ScrollDown(int Offset);
	void ScrollLeft(int Offset);
	void ScrollRight(int Offset);
	void MoveTo(int XOffset, int YOffset);
	virtual HRESULT Draw(CDXSurface* lpDDS);
	virtual HRESULT Draw(int X, int Y, CDXSurface* lpDDS);
	virtual HRESULT DrawTrans(int X, int Y, CDXSurface* lpDDS);

public:
	int m_XOffset;
	int m_YOffset;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXTile Class
//////////////////////////////////////////////////////////////////////////////////
class CDXTile : public CDXSurface
{
public:
	CDXTile();
//<JJH>
	CDXTile(CDXScreen *pScreen, const char* szFilename, int w, int h, int num, BOOL memoryType);
	BOOL Create(CDXScreen *pScreen, const char* szFilename, int w, int h, int num, BOOL memoryType);
//	CDXTile(CDXScreen *pScreen, const char* szFilename, int w, int h, int num);
//	BOOL Create(CDXScreen *pScreen, const char* szFilename, int w, int h, int num);
//<JJH>

	BOOL CreateFromTLE(CDXScreen *pScreen, const char* szFilename);

public:
	int m_BlockWidth;
	int m_BlockHeight;
	int m_BlockNum;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXMapCell Class
//////////////////////////////////////////////////////////////////////////////////
class CDXMapCell
{
public:
	CDXMapCell();
	~CDXMapCell();
	void Save(FILE *fp);
	void Load(FILE *fp);
	virtual int GetTile();
	virtual void SetTile(int Tile);
private:
	int TileNum;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXMap Class
//////////////////////////////////////////////////////////////////////////////////
class CDXMap
{
public:
	CDXMap();
	CDXMap(CDXTile *pTiles, CDXScreen *pScreen);
	~CDXMap();

	void Create(int Width, int Height, int Fill);
	BOOL Load(const char *szFilename);
	BOOL Save(const char *szFilename);
	void Clear(void);
	void Fill(int TileNum);
	void BltTile(CDXSurface* lpDDS, int xdest, int ydest, int w, int h, int xoff, int yoff);
	void BltTileTrans(CDXSurface* lpDDS, int xdest, int ydest, int w, int h, int xoff, int yoff);
	void Draw(CDXSurface* lpDDS);
	void DrawTrans(CDXSurface* lpDDS);
	void DrawClipped(CDXSurface* lpDDS, LPRECT ClipRect);
	void MoveTo(int PosX, int PosY);
	void ScrollUp(int Offset);
	void ScrollDown(int Offset);
	void ScrollLeft(int Offset);
	void ScrollRight(int Offset);
	void WrapScrollUp(int Offset);
	void WrapScrollDown(int Offset);
	void WrapScrollLeft(int Offset);
	void WrapScrollRight(int Offset);
	void ScreenTileSize(int Width, int Height);
	int  GetTile(int MapX, int MapY);
	CDXMapCell *GetCell(int MapX, int MapY);
	void SetTile(int MapX, int MapY, int Tile);
	void LoadTiles(CDXTile *pTiles);
	virtual void InitData(int iSIZE, int iFill);

public:
	int m_PosX, m_PosY;
	int m_Width, m_Height;
	int m_TileWidth, m_TileHeight;
	int m_PixelWidth, m_PixelHeight;
	int SCREEN_TW, SCREEN_TH;
	int SCREEN_W, SCREEN_H;
	int SIZE;
	CDXMapCell *DATA;
	CDXScreen *Screen;
	CDXTile *Tiles;
};


//////////////////////////////////////////////////////////////////////////////////
// CDXIsoMap Class
//////////////////////////////////////////////////////////////////////////////////
class CDXIsoMap : public CDXMap
{
public:
//<WPS>
	CDXIsoMap(CDXTile *pTiles, CDXScreen *pScreen):CDXMap(pTiles, pScreen) {};
//<WPS>
    virtual void ScrollUp(int);
    virtual void ScrollDown(int);
    virtual void ScrollLeft(int);
    virtual void ScrollRight(int);
    virtual void BltTileTrans(CDXSurface*, int, int, int, int, int, int, int, int);
    virtual void DrawTrans(CDXSurface*);
	POINT ProcessClicks(int, int);
};

//////////////////////////////////////////////////////////////////////////////////
// CDXSprite Class
//////////////////////////////////////////////////////////////////////////////////
class CDXSprite
{
public:
	CDXSprite(void) {}
	CDXSprite(CDXTile* pTile);

//<JJH>
	CDXSprite(CDXScreen *pScreen, const char* Filename, int w, int h, int num, BOOL memoryType);
//	CDXSprite(CDXScreen *pScreen, const char* Filename, int w, int h, int num);
//<JJH>
	~CDXSprite(void);

	BOOL Create(CDXTile* pTile);
//<JJH>
	BOOL Create(CDXScreen *pScreen, const char* Filename, int w, int h, int num, BOOL memoryType);
//	BOOL Create(CDXScreen *pScreen, const char* Filename, int w, int h, int num);
//<JJH>

	void SetPos(int pX, int pY) { m_PosX = pX; m_PosY = pY; }
	void SetVel(int vX, int vY) { m_VelX = vX; m_VelY = vY; }
	void SetFrame(int Frame) { m_Frame = Frame; }
	void SetDelay(int Delay) { m_Delay = Delay; }
	void SetState(int State) { m_State = State; }
	void SetType(int Type) { m_Type = Type; }

	void Lock(void) { m_Tile->Lock(); }
	void UnLock(void) { m_Tile->UnLock(); }
	
//<JJH>
	void SetShadowOffset(LONG dx, LONG dy);
	void SetShadowValue(WORD Shade) { m_ShadowValue = Shade; }
	void SetAlphaValue(WORD Shade) { m_AlphaValue = Shade; }
	void SetColorKey(DWORD Col) { m_Tile->SetColorKey(Col); }
	void SetColorKey(void) { m_Tile->SetColorKey(); }
	//void ColorKey(int Col) { m_Tile->ColorKey(Col); }
//<JJH>

//<JJH>
	//void SelectFrame(void);
//<JJH>
	
	BOOL SpriteHit(CDXSprite* pSprite);
	BOOL TileHit(CDXMap* pMap, int Tile);

//<JJH>
	void Draw(CDXSurface* lpCDXS, DWORD ScrnWorldX, DWORD ScrnWorldY, WORD BltType);
	HRESULT BlkBlt(CDXSurface* lpDDSDest, RECT SrcRect, RECT DestRect);
	HRESULT TransBlt(CDXSurface* lpDDS, RECT SrcRect, RECT DestRect);
	HRESULT TransHFlip(CDXSurface* lpCDXSDest, RECT SrcRect, RECT DestRect);
	HRESULT BlkHFlip(CDXSurface* lpCDXSDest, RECT SrcRect, RECT DestRect);
	HRESULT TransVFlip(CDXSurface* lpCDXSDest, RECT SrcRect, RECT DestRect);
	HRESULT BlkVFlip(CDXSurface* lpCDXSDest, RECT SrcRect, RECT DestRect);
	HRESULT BlkAlpha(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT BlkAlphaFast(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT BlkShadow(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT BlkShadowFast(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT TransAlpha(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT TransAlphaFast(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT TransShadow(CDXSurface* dest, RECT srcRect, RECT destRect);
	HRESULT TransShadowFast(CDXSurface* dest, RECT srcRect, RECT destRect);

	//HRESULT Draw(CDXSurface* lpDDS);
	//HRESULT DrawFast(CDXSurface* lpDDS);
	//HRESULT DrawTrans(CDXSurface* lpDDS);
	//HRESULT DrawClipped(CDXSurface* lpDDS, LPRECT ClipRect);
	//HRESULT DrawWindowed(CDXSurface* lpDDS);
	//HRESULT DrawScaled(float Factor, CDXSurface* lpDDS);
	//void DrawRotated(float Angle, CDXSurface* lpDDS);
	//void DrawHFlip(CDXSurface* lpDDS);
	//void DrawVFlip(CDXSurface* lpDDS);
//<JJH>

public:
	int m_PosX, m_PosY, m_PosZ;
	int m_VelX, m_VelY;
	int m_Frame;
	int m_Delay;
	int m_State;
	int m_Type;
	int m_Angle;
	BOOL m_Flipped;

//<JJH>
	LONG m_ShadowOffsetX;
	LONG m_ShadowOffsetY;
	WORD m_ShadowValue;
	WORD m_AlphaValue;
	BOOL m_TileCreated;
//<JJH>

	CDXTile* m_Tile;
	CDXSprite* m_Next;
	CDXSprite* m_Prev;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXSpriteList Class
//////////////////////////////////////////////////////////////////////////////////
class CDXSpriteList
{
public:
	CDXSpriteList(void);
	~CDXSpriteList(void);

	CDXSprite *AddSprite(CDXTile* pTile, int Type=0, int PosX=0, int PosY=0,
		int VelX=0, int VelY=0, int State=0, int Frame=0, int Delay=0);
	void DelSprite(CDXSprite* pNode);

	CDXSprite* Next(void) { return m_List.m_Next; }
	CDXSprite* Prev(void) { return m_List.m_Prev; }
	CDXSprite* List(void) { return &m_List; }

//<JJH>
	void Draw(CDXSurface* lpCDXS, DWORD ScrnWorldX, DWORD ScrnWorldY, WORD BltType);

	//void Draw(CDXSurface* lpDDS);
	//void DrawFast(CDXSurface* lpDDS);
	//void DrawTrans(CDXSurface* lpDDS);
	//void DrawClipped(CDXSurface* lpDDS, LPRECT ClipRect);
	//void DrawWindowed(CDXSurface* lpDDS);
	//void DrawScaled(float Factor, CDXSurface* lpDDS);
	//void DrawRotated(float Angle, CDXSurface* lpDDS);
	//void DrawHFlip(CDXSurface* lpDDS);
	//void DrawVFlip(CDXSurface* lpDDS);
//<JJH>

public:
	CDXSprite m_List;
	int m_nSprites;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXSound Class
//////////////////////////////////////////////////////////////////////////////////
class CDXSound
{
public:
	CDXSound(void);
	~CDXSound(void);

	BOOL Create(void *hWnd);
	BOOL GetCaps(void);

public:
	LPDIRECTSOUND m_lpDS;
	DSCAPS m_DSCaps;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXSoundBuffer Class
//////////////////////////////////////////////////////////////////////////////////
class CDXSoundBuffer
{
public:
	CDXSoundBuffer(void);
	~CDXSoundBuffer(void);

	BOOL Load(CDXSound* pDS, const char* Filename, int Num = 1);
	BOOL Play(int Pan = 0, DWORD dwFlags = 0);
	BOOL Stop(void);
	void SetVolume(LONG Volume);

	BOOL LoadFromFile(void);
	BOOL CreateSoundBuffer(DWORD dwBufSize, DWORD dwFreq, DWORD dwBitsPerSample,
												 DWORD dwBlkAlign, BOOL bStereo);
	BOOL ReadData(FILE* fp, DWORD dwSize, DWORD dwPos);
	LPDIRECTSOUNDBUFFER GetFreeBuffer(void);

public:
	int m_nBuffers;
	int m_Current;
	const char* m_Filename;
	CDXSound* m_pDS;
	LPDIRECTSOUNDBUFFER m_lpDSB[1];
};

//////////////////////////////////////////////////////////////////////////////////
// CDXMusic Class
//////////////////////////////////////////////////////////////////////////////////
class CDXMusic
{
public:
	CDXMusic(void *hWnd);

	BOOL Play(const char *Filename);
	BOOL Stop(void);
	BOOL Pause(void);
	BOOL Resume(void);
	BOOL Restart(void);

public:
	HWND m_hWnd;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXInput Class
//////////////////////////////////////////////////////////////////////////////////
class CDXInput
{
public:
	CDXInput(void);
	~CDXInput(void);

	BOOL Create(void *hInst, void *hWnd);
	void ReAcquire(void);
	void UnAcquire(void);
	void Update(void);

	void SetActiveDevices(BOOL bMouse, BOOL bKeyboard, BOOL bJoystick);
	void SetMouseAbs(void);
	void SetJoystickAbs(void);
	void RunMouseControlPanel(void *hWnd = NULL);
	void RunJoystickControlPanel(void *hWnd = NULL);

public:
	LPDIRECTINPUT m_lpDI;
	LPDIRECTINPUTDEVICE m_lpDIDKeyboard;
	LPDIRECTINPUTDEVICE m_lpDIDMouse;
#if DIRECTDRAW_VERSION >= CDX_DDVER
	LPDIRECTINPUTDEVICE2 m_lpDIDJoystick;
#endif

	POINT Mouse;
	POINT Joystick;
	BOOL JoystickB1, JoystickB2;
	BOOL MouseLB, MouseMB, MouseRB;
	BYTE Keys[256];

	BOOL m_bMouse;
	BOOL m_bKeyboard;
	BOOL m_bJoystick;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXLinkSession Structure
//////////////////////////////////////////////////////////////////////////////////
struct CDXLinkSession
{
	char *m_Name;
	GUID  m_Guid;
	GUID  m_AppGuid;
	DWORD m_Flags;
	DWORD m_MaxPlayers;
	DWORD m_CurrentPlayers;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXLinkPlayer Structure
//////////////////////////////////////////////////////////////////////////////////
struct CDXLinkPlayer
{
	DPID m_ID;
	char *m_Name;
	DWORD m_Flags;
	HANDLE m_Event;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXLink Class
//////////////////////////////////////////////////////////////////////////////////
class CDXLink
{
public:
	CDXLink(void);
	~CDXLink(void);

	BOOL CreateIPX(void* hWnd);
	BOOL CreateTCPIP(void* hWnd, char* IPAddress);
	BOOL CreateModem(void* hWnd, char* PhoneNo);
	BOOL CreateSerial(void* hWnd);

	BOOL EnumSessions(CDXLinkSession* Sessions, GUID APP_GUID);
	BOOL HostSession(CDXLinkSession* Session);
	BOOL JoinSession(CDXLinkSession* Session);
	BOOL CloseSession(void);

	BOOL EnumPlayers(CDXLinkPlayer* Players, DWORD Flags = 0);
	BOOL CreatePlayer(CDXLinkPlayer* Player);
	BOOL DestroyPlayer(CDXLinkPlayer* Player);

	HRESULT Send(DPID idFrom, DPID idTo, DWORD dwFlags,
							 LPVOID lpData, DWORD dwDataSize);
	HRESULT Receive(LPDPID lpidFrom, LPDPID lpidTo,	DWORD dwFlags,
									LPVOID lpData, LPDWORD lpdwDataSize);

public:
	HRESULT rval;
	LPDIRECTPLAY3A m_DP;
	LPDIRECTPLAYLOBBY2A m_DPLobby;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXResource FILEENTRY Structure
//////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	long offset;
	char name[13];
} FILEENTRY, *LPFILEENTRY;

//////////////////////////////////////////////////////////////////////////////////
// CDXResource Class
//////////////////////////////////////////////////////////////////////////////////
class CDXResource
{
public:
	CDXResource(void);
	~CDXResource(void);

	BOOL Open(const char *Filename);
	void Close(void);
	char* GetFile(const char *Filename);

public:
	LPFILEENTRY m_pFE;
	int m_Infile;
	int m_Outfile;
	int m_nFiles;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXAvi Class
//////////////////////////////////////////////////////////////////////////////////
class CDXAvi : public CDXSurface
{
public:
	CDXAvi(void);
	~CDXAvi(void);

	BOOL Create(CDXScreen* pScreen, CDXSound* pSound, const char* Filename, BOOL bLoop = FALSE);
	BOOL Start(void);
	void Stop(void);
	void NextFrame(void);
	void ReadFrame(LONG Frame);
	BOOL GetSound(void);
	HRESULT DrawBuffer(void);

	virtual HRESULT Draw(CDXSurface* Dest);
	virtual HRESULT DrawFast(int X, int Y, CDXSurface* Dest);

public:
	BOOL m_bPlaying;
	BOOL m_bLoop;
	LONG m_Index;
	LONG m_nFrames;
	DWORD m_Fps;
	int m_TimerID;
	int m_TimeTick;

	CRITICAL_SECTION m_AccessBuffer;
	PAVISTREAM m_AviStream;
	AVISTREAMINFO m_StreamInfo;
	LPBYTE m_Input;
	LPBYTE m_Output;
	HIC  m_Decompressor;
	LONG m_Length;
	LONG m_LinePitch;
	LPBITMAPINFOHEADER m_InputFormat;
	LPBITMAPV4HEADER m_TargetFormat;

	PAVISTREAM m_AviSound;
	AVISTREAMINFO m_SoundInfo;
	WAVEFORMATEX* m_SoundFormat;
	LPDIRECTSOUNDBUFFER m_SoundBuffer;
	DWORD m_LoadPos;
	DWORD m_LoadSize;
	int m_SoundFramesAhead;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXMusicCd Class
//////////////////////////////////////////////////////////////////////////////////
class CDXMusicCd
{
public:
	CDXMusicCd(void);
	~CDXMusicCd(void);

	short	Read(void);
	short	GetNumberOfTracks(void)	{ return m_nNumberOfTracks; }
	void  SetNumberOfTracks(short nTracks) { m_nNumberOfTracks = nTracks; }
	short	GetTrackLength(short nTrack);
	void  SetTrackLength(short nTrack, short nNewLength);
	short	GetTotalLength(void);
	void  Play(short nTrack);
	void  Stop(void);

private:
	short	m_nNumberOfTracks;
	short	m_nTrackLength[100];
	MCI_STATUS_PARMS m_MCIStatus;
	MCI_OPEN_PARMS m_MCIOpen;
};

#define SHIFT 6		// change this to reflect the the size.
					// Ex. 64x64 tile equals 2^6. or a shift of 6
#define TILESIZE 64	// change this also to reflect tile size. 64x64.

//////////////////////////////////////////////////////////////////////////////////
// CDXAStar Class
//////////////////////////////////////////////////////////////////////////////////
class CDXAStar
{
private:
	struct NODE {     // node structure
		long f, h;
		int g, tmpg;
		int x, y;
		int NodeNum;
		NODE *Parent;
		NODE *Child[8]; // a node may have upto 8+(NULL) children.
		NODE *NextNode;  // for filing purposes
	};

	NODE *OPEN;    // the node list pointers
	NODE *CLOSED;
	NODE *PATH; // pointer to the best path

	struct STACK {      // the stack structure
		NODE *NodePtr;
		STACK *NextStackPtr;
	};

	STACK *Stack;
	
	BOOL isPath;   

    int ROWS, // tilemap data members, need to be initialisize
        COLS,         // with current map's width and height
        TOTAL_TILES; // to allocate memory for the
    int *TileMap;  // pointer to the A* own tilemap data array

public:
   
     // Modify only these 3 public member functions to support Your favorite Map
   	CDXAStar(CDXMap* pMap, int forbiddenTiles);
    ~CDXAStar();
   	void InitAstarTileMap(CDXMap* pMap, int forbiddenTiles);                

	// Must be called and be true
    // before getting the node entries. It frees the lists,
    // calls ::Findpath() and returns true if path is accessible
    BOOL NewPath(int sx, int sy, int dx, int dy);  

    BOOL ReachedGoal(void); // Call and check this function before using these 3 following
    void PathNextNode(void) { PATH=PATH->Parent; }
    int NodeGetX(void)      { return PATH->x; }
    int NodeGetY(void)      { return PATH->y; }

    // other usefull functions (do not change them they are used by the A* algorithm)
    int TileNum(int x, int y); // returns tilenum
	int FreeTile(int x, int y); // returns 1 = true if we can move on it

private:
    void BoundaryTiles(void); // sets 1's around the map area. To be called before init the A* map
    void FreeNodes(void);// frees lists memory

    // The A* Algorithm
    void FindPath(int sx, int sy, int dx, int dy);
	NODE *ReturnBestNode(void);
	void GenerateSuccessors(NODE *BestNode, int dx, int dy);
   	void GenerateSucc(NODE *BestNode,int x, int y, int dx, int dy);
	NODE *CheckOPEN(int tilenum);
	NODE *CheckCLOSED(int tilenum);
	void Insert(NODE *Successor);
	void PropagateDown(NODE *Old);
	void Push(NODE *Node);
	NODE *Pop(void);
};

//////////////////////////////////////////////////////////////////////////////////
// CDXStarField Class
//////////////////////////////////////////////////////////////////////////////////
#define MAXSTARS	10000
enum m_Mode { HORIZONTAL, VERTICAL, THREED };

class CDXStarFieldRecord
{
public:
	int m_X;
	int m_Y;
	int m_Z;
	int m_OldX;
	int m_OldY;
	int m_Color;
};

class CDXStarField
{
public:
	CDXStarField(CDXScreen*);
	~CDXStarField();

	void Create();
	void Update(BOOL bErase = TRUE);
	void CreateStars();
	void CalcPosition();
	void ClearStars();
	void SetNumberofStars(int);

public:
	CDXScreen* m_Screen;
	CDXStarFieldRecord m_Stars[MAXSTARS];
	
	int m_NumStars;
	int m_StarVelocity;
	int m_Width;
	int m_Height;
	int m_LWidth;
	int m_LHeight;
	int m_CenterX;
	int m_CenterY;
	int m_Left;
	int m_Right;
	int m_Top;
	int m_Bottom;
	
	m_Mode m_StarMode;

	BOOL m_Towards;
	BOOL m_Flashing;
	BOOL m_Colored;
};

#define MAX_MENU_ITEMS		20
#define MAX_STRING_LENGTH	80

class CDXMenu
{
public:
	CDXMenu();
	void Create(CDXSurface *pSurface, int Spacing);
	void SetTitle(const char *szTitle, COLORREF color);
	void SetWrap(BOOL bWrap);
	void AddItem(char *szString, COLORREF oncolor, COLORREF offcolor);
	void Draw(int X, int Y);
	void Up();
	void Down();
	void Home();
	void End();
	int Enter();

	CDXSurface *m_pSurface;
	
private:
	struct MenuItem {
		char m_szString[MAX_STRING_LENGTH];
		COLORREF m_OnColor;
		COLORREF m_OffColor;
	} m_MenuItems[MAX_MENU_ITEMS];

	int m_ItemCount;
	int m_Spacing;
	int m_CurrentItem;
	BOOL m_bWrap;

	struct Title {
		char m_szTitle[MAX_STRING_LENGTH];
		COLORREF m_color;
		BOOL m_bActive;
	} m_Title;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXEngine Class
//////////////////////////////////////////////////////////////////////////////////
class CDXEngine
{
public:
	CDXEngine();
	~CDXEngine();
//	BOOL Create(char *Title, DWORD Width, DWORD Height, DWORD BPP, BOOL Windowed = FALSE, HINSTANCE hInst);
	long PASCAL WinProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
	void Run(void);

public:
	CDXScreen *m_Screen;
};

//////////////////////////////////////////////////////////////////////////////////
// MACROS
//////////////////////////////////////////////////////////////////////////////////
#define RELEASE(x) if(x != NULL) { x->Release(); x = NULL; }
#define ODS(a) OutputDebugString(a)

#endif
