# ====================================================================
# The Apache Software License, Version 1.1
#
# Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
# reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in
#    the documentation and/or other materials provided with the
#    distribution.
#
# 3. The end-user documentation included with the redistribution,
#    if any, must include the following acknowledgment:
#       "This product includes software developed by the
#        Apache Software Foundation (http://www.apache.org/)."
#    Alternately, this acknowledgment may appear in the software itself,
#    if and wherever such third-party acknowledgments normally appear.
#
# 4. The names "Apache" and "Apache Software Foundation" must
#    not be used to endorse or promote products derived from this
#    software without prior written permission. For written
#    permission, please contact apache@apache.org.
#
# 5. Products derived from this software may not be called "Apache",
#    nor may "Apache" appear in their name, without prior written
#    permission of the Apache Software Foundation.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
# ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
# USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
# OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# ====================================================================
#
# This software consists of voluntary contributions made by many
# individuals on behalf of the Apache Software Foundation.  For more
# information on the Apache Software Foundation, please see
# <http://www.apache.org/>.
#

#for more functionality see the HTTPD::UserAdmin module:
# http://www.perl.com/CPAN/modules/by-module/HTTPD/HTTPD-Tools-x.xx.tar.gz
#
# usage: dbmmanage <DBMfile> <command> <user> <password> <groups> <comment>

package dbmmanage;
#                               -ldb    -lndbm    -lgdbm    -lsdbm
BEGIN { @AnyDBM_File::ISA = qw(SDBM_File) }
use strict;
use Fcntl;
use AnyDBM_File ();

sub usage {
    my $cmds = join "|", sort keys %dbmc::;
    die <<SYNTAX;
Usage: dbmmanage [enc] dbname command [username [pw [group[,group] [comment]]]]

    where enc is  -d for crypt encryption (default except on Win32, Netware)
                  -m for MD5 encryption (default on Win32, Netware)
                  -s for SHA1 encryption
                  -p for plaintext

    command is one of: $cmds

    pw of . for update command retains the old password
    pw of - (or blank) for update command prompts for the password

    groups or comment of . (or blank) for update command retains old values
    groups or comment of - for update command clears the existing value
    groups or comment of - for add and adduser commands is the empty value
SYNTAX
}

sub need_sha1_crypt {
    if (!eval ('require "Digest/SHA1.pm";')) {
        print STDERR <<SHAERR;
dbmmanage SHA1 passwords require the interface or the module Digest::SHA1
available from CPAN:
 
    http://www.cpan.org/modules/by-module/Digest/Digest-MD5-2.12.tar.gz
 
Please install Digest::SHA1 and try again, or use a different crypt option:

SHAERR
        usage();
    }
}

sub need_md5_crypt {
    if (!eval ('require "Crypt/PasswdMD5.pm";')) {
        print STDERR <<MD5ERR;
dbmmanage MD5 passwords require the module Crypt::PasswdMD5 available from CPAN
 
    http://www.cpan.org/modules/by-module/Crypt/Crypt-PasswdMD5-1.1.tar.gz
 
Please install Crypt::PasswdMD5 and try again, or use a different crypt option:

MD5ERR
        usage();
    }
}

# if your osname is in $newstyle_salt, then use new style salt (starts with '_' and contains
# four bytes of iteration count and four bytes of salt).  Otherwise, just use
# the traditional two-byte salt.
# see the man page on your system to decide if you have a newer crypt() lib.
# I believe that 4.4BSD derived systems do (at least BSD/OS 2.0 does).
# The new style crypt() allows up to 20 characters of the password to be
# significant rather than only 8.
#
my $newstyle_salt_platforms = join '|', qw{bsdos}; #others?
my $newstyle_salt = $^O =~ /(?:$newstyle_salt_platforms)/;

# Some platforms just can't crypt() for Apache
#
my $crypt_not_supported_platforms = join '|', qw{MSWin32 NetWare}; #others?
my $crypt_not_supported = $^O =~ /(?:$crypt_not_supported_platforms)/;

my $crypt_method = "crypt";

if ($crypt_not_supported) {
    $crypt_method = "md5";
}

# Some platforms won't jump through our favorite hoops
#
my $not_unix_platforms = join '|', qw{MSWin32 NetWare}; #others?
my $not_unix = $^O =~ /(?:$not_unix_platforms)/;

if ($crypt_not_supported) {
    $crypt_method = "md5";
}

if (@ARGV[0] eq "-d") {
    shift @ARGV;
    if ($crypt_not_supported) {
        print STDERR 
              "Warning: Apache/$^O does not support crypt()ed passwords!\n\n";
    }
    $crypt_method = "crypt";
}

if (@ARGV[0] eq "-m") {
    shift @ARGV;
    $crypt_method = "md5";
}

if (@ARGV[0] eq "-p") {
    shift @ARGV;
    if (!$crypt_not_supported) {
        print STDERR 
              "Warning: Apache/$^O does not support plaintext passwords!\n\n";
    }
    $crypt_method = "plain";
}

if (@ARGV[0] eq "-s") {
    shift @ARGV;
    need_sha1_crypt();
    $crypt_method = "sha1";
}

if ($crypt_method eq "md5") {
    need_md5_crypt();
}

my($file,$command,$key,$crypted_pwd,$groups,$comment) = @ARGV;

usage() unless $file and $command and defined &{$dbmc::{$command}};

# remove extension if any
my $chop = join '|', qw{db.? pag dir};
$file =~ s/\.($chop)$//;

my $is_update = $command eq "update";
my %DB = ();
my @range = ();
my($mode, $flags) = $command =~ 
    /^(?:view|check)$/ ? (0644, O_RDONLY) : (0644, O_RDWR|O_CREAT);

tie (%DB, "AnyDBM_File", $file, $flags, $mode) || die "Can't tie $file: $!";
dbmc->$command();
untie %DB;


my $x;
sub genseed {
    my $psf;
    if ($not_unix) {
	srand (time ^ $$ or time ^ ($$ + ($$ << 15)));
    }
    else {
        for (qw(-xlwwa -le)) { 
	    `ps $_ 2>/dev/null`;
            $psf = $_, last unless $?;
        }
        srand (time ^ $$ ^ unpack("%L*", `ps $psf | gzip -f`));
    }
    @range = (qw(. /), '0'..'9','a'..'z','A'..'Z');
    $x = int scalar @range;
}

sub randchar { 
    join '', map $range[rand $x], 1..shift||1;
}

sub saltpw_crypt {
    genseed() unless @range; 
    return $newstyle_salt ? 
	join '', "_", randchar, "a..", randchar(4) :
        randchar(2);
}

sub cryptpw_crypt {
    my ($pw, $salt) = @_;
    $salt = saltpw_crypt unless $salt;
    crypt $pw, $salt;
}

sub saltpw_md5 {
    genseed() unless @range; 
    randchar(8);
}

sub cryptpw_md5 {
    my($pw, $salt) = @_;
    $salt = saltpw_md5 unless $salt;
    Crypt::PasswdMD5::apache_md5_crypt($pw, $salt);
}

sub cryptpw_sha1 {
    my($pw, $salt) = @_;
    '{SHA}' . Digest::SHA1::sha1_base64($pw) . "=";
}

sub cryptpw {
    if ($crypt_method eq "md5") {
        return cryptpw_md5(@_);
    } elsif ($crypt_method eq "sha1") {
        return cryptpw_sha1(@_);
    } elsif ($crypt_method eq "crypt") {
        return cryptpw_crypt(@_);
    }
    @_[0]; # otherwise return plaintext
}

sub getpass {
    my $prompt = shift || "Enter password:";

    unless($not_unix) { 
	open STDIN, "/dev/tty" or warn "couldn't open /dev/tty $!\n";
	system "stty -echo;";
    }

    my($c,$pwd);
    print STDERR $prompt;
    while (($c = getc(STDIN)) ne '' and $c ne "\n" and $c ne "\r") {
	$pwd .= $c;
    }

    system "stty echo" unless $not_unix;
    print STDERR "\n";
    die "Can't use empty password!\n" unless length $pwd;
    return $pwd;
}

sub dbmc::update {
    die "Sorry, user `$key' doesn't exist!\n" unless $DB{$key};
    $crypted_pwd = (split /:/, $DB{$key}, 3)[0] if $crypted_pwd eq '.';
    $groups = (split /:/, $DB{$key}, 3)[1] if !$groups || $groups eq '.';
    $comment = (split /:/, $DB{$key}, 3)[2] if !$comment || $comment eq '.';
    if (!$crypted_pwd || $crypted_pwd eq '-') {
        dbmc->adduser;
    }
    else {
        dbmc->add;
    }
}

sub dbmc::add {
    die "Can't use empty password!\n" unless $crypted_pwd;
    unless($is_update) {
	die "Sorry, user `$key' already exists!\n" if $DB{$key};
    }
    $groups = '' if $groups eq '-';
    $comment = '' if $comment eq '-';
    $groups .= ":" . $comment if $comment;
    $crypted_pwd .= ":" . $groups if $groups;
    $DB{$key} = $crypted_pwd;
    my $action = $is_update ? "updated" : "added";
    print "User $key $action with password encrypted to $DB{$key} using $crypt_method\n";
}

sub dbmc::adduser {
    my $value = getpass "New password:";
    die "They don't match, sorry.\n" unless getpass("Re-type new password:") eq $value;
    $crypted_pwd = cryptpw $value;
    dbmc->add;
}

sub dbmc::delete {
    die "Sorry, user `$key' doesn't exist!\n" unless $DB{$key};
    delete $DB{$key}, print "`$key' deleted\n";
}

sub dbmc::view {
    print $key ? "$key:$DB{$key}\n" : map { "$_:$DB{$_}\n" if $DB{$_} } keys %DB;
}

sub dbmc::check {
    die "Sorry, user `$key' doesn't exist!\n" unless $DB{$key};
    my $chkpass = (split /:/, $DB{$key}, 3)[0];
    my $testpass = getpass();
    if (substr($chkpass, 0, 6) eq '$apr1$') {
        need_md5_crypt;
        $crypt_method = "md5";
    } elsif (substr($chkpass, 0, 5) eq '{SHA}') {
        need_sha1_crypt;
        $crypt_method = "sha1";
    } elsif (length($chkpass) == 13 && $chkpass ne $testpass) {
        $crypt_method = "crypt";
    } else {
        $crypt_method = "plain";
    }
    print $crypt_method . (cryptpw($testpass, $chkpass) eq $chkpass 
                           ? " password ok\n" : " password mismatch\n");
}

sub dbmc::import {
    while(defined($_ = <STDIN>) and chomp) {
	($key,$crypted_pwd,$groups,$comment) = split /:/, $_, 4;
	dbmc->add;
    }
}

