/*
 *  This file defines some Tcl Procedures that provide Object Oriented
 *  extensions to Tcl.  The routines herein provide a means for defining
 *  Classes of objects, and methods associated with those objects and classes.
 *  There is no explicit definition of class or instance variables;  instead,
 *  they are created during use.
 *
 *  These routines use the Array feature of Tcl to accomplish this.  A special
 *  array called __oClasses uses the names of defined classes as indices.  The
 *  value of a given class index is the list of superclasses of the class.
 *  There is also an array called __oObjects.  It's indices are the names of
 *  defined objects.  The value of each entry is the class of the corresponding
 *  object.
 *
 *  Objects themselves are global arrays.  The indices are the instance
 *  variables for the object.
 *
 *  Methods are stored by defining them as procedures.  A method <method> for
 *  class <class> with arguments <arguments> is defined as
 *
 *      proc <class>::<method> {<arguments>} {...}
 *
 *  When creating a new object of class <class>, if a method for that class
 *  exists named <class> (ie, a Tcl procedure named <class>::<class> exists),
 *  then it is called during object creation.
 *
 *  When destroying an object of class <class>, if a method for that class
 *  exists named ~<class> (ie, a Tcl procedure named <class>::~<class> exists),
 *  then it is called during object destruction.
 *
 *  Within all methods, a global variable named "oThis" is available.  It
 *  contains the name of the object being acted upon.  To reference this
 *  variable, place the Tcl command "global oThis" in your method definition.
 */

#ifndef _OOPS_TCL_
#define _OOPS_TCL_

#ifndef _MISC_TCL_
#ifndef NO_PROCS
#define NO_PROCS
#include <misc.Tpp>
#undef NO_PROCS
#else
#include <misc.Tpp>
#endif
#endif


/*
 *  oClassExists <class>
 *
 *  Returns a boolean value representing whether or not the given class
 *  is defined.
 */

proc oClassExists {className} {
    global __oClasses
    return [info exists __oClasses($className)]
}



/*
 *  oObjExists <object>
 *
 *  Returns a boolean value representing whether or not the given object
 *  exists.
 */

proc oObjExists {objectName} {
    global __oObjects
    return [info exists __oObjects($objectName)]
}



/*
 *  oVarExists <object> <instanceVariable>
 *
 *  Returns a boolean value representing whether or not the given variable name
 *  exists for the given object.
 */

proc oVarExists {objectName varName} {
    global $objectName
    return [info exists [format %s(%s) $objectName $varName]]
}



/*
 *  oType <object>
 *
 *  Returns the class name of the given object.
 */

proc oType {objectName} {
    global __oObjects
    if [oObjExists $objectName] {
    	return $__oObjects($objectName)
    } else {
        error [format {No such object as ^"%s^".} $objectName]
    }
}



/*
 *	oParentOf <classList> <className>
 *
 *	Returns a boolean value representing whether or not the given class
 *	is in the class list, or is a parent of any of the classes in the
 *  class list.
 *
 *	No parameter checking is done for speed.
 */

proc oParentOf {classList className} {
	global __oClasses
	foreach class $classList {
		if [string compare $class $className] {
			return 1
		}
		set parents $__oClasses($class)
		if {IsntEmpty($parents)} {
			if [oChildOf $parents $className] {
				return 1
			}
		}
	}
	return 0
}



/*
 *  oTypeOf <class> <object>
 *
 *  Returns a boolean value representing whether or not the given object
 *	is a type of the given class (i.e., is derived from that class,
 *	either directly or by inheritance).
 */

proc oTypeOf {className objectName} {
	if [oClassExists $className] {
    	return [oParentOf [oType $objectName] $className]
	} else {
		error [format {No such class as ^"%s^".} $className]
	}
}



/*
 *  oSet <object> <instanceVariable> <newValue>
 *
 *  Sets the given variable for the given object to the given value.
 */

proc oSet {objectName varName varValue} {
    global $objectName
    if [oObjExists $objectName] {
	    set [format %s(%s) $objectName $varName] $varValue
	    return $varValue
    } else {
        error [format {No such object as ^"%s^".} $objectName]
    }
}



/*
 *  oGet <object> <instanceVariable>
 *
 *  Gets the value of the given variable for the given object.
 */

proc oGet {objectName varName} {
    global $objectName
    if [oObjExists $objectName] {
	    return [set [format %s(%s) $objectName $varName]]
    } else {
        error [format {No such object as ^"%s^".} $objectName]
    }
}



/*
 *  oClass <className> <classDefinitions> [<superClassList>]
 *
 *  Defines an object class.  The className is the name of the class, the
 *  classDefinition is a list of methods, and if given, superClass is a list of
 *  parent classes.
 */

proc oClass {className classDefinition {superClass {}}} {
    global __oClasses
    if {IsEmpty($className)
        || (-1 != [string first :: $className])} {
        error [format {Invalid class name ^"%s^".} $className]
    }
    if [oClassExists $className] {
        error [format {Class ^"%s^" already defined.} $className]
    }
    foreach sClass $superClass {
        if {! [oClassExists $sClass]} {
            error [format {Superclass ^"%s^" not defined.} $sClass]
        }
    }
    try {
        set n [llength $classDefinition]
        for {set i 0} {$i < $n} {incr i 3} {
            set ip1 [expr {$i + 1}]
            set ip2 [expr {$i + 2}]
            set methodName [lindex $classDefinition $i]
            if {-1 != [string first :: $methodName]} {
                error [format {^"%s^" is an invalid method name.} $methodDame]
            }
            proc [format %s::%s $className $methodName] ^
                [lindex $classDefinition $ip1] ^
                [lindex $classDefinition $ip2]
        }
    } catch errorString {
        foreach methodName [info procs [format %s::* $className]] {
            rename $methodName {}
        }
        error $errorString
    }
    set __oClasses($className) $superClass
}



/*
 *  oMethod <class> <method>
 *
 *  Finds a method definition for a given class, and returns the name of the
 *  procedure to call to invoke that method.
 *
 *	No parameter checking is done for speed.
 */

proc oMethod {classNames methodName} {
    global __oClasses __oThisClass
    foreach class $classNames {
        set method $class::$methodName
        if {IsntEmpty([info procs $method])} {
            set __oThisClass $class
            return $method
        }
        set superClasses $__oClasses($class)
        if {IsntEmpty($superClasses)} {
            set method [oMethod $superClasses $methodName]
            if {IsntEmpty($method)} {
                return $method
            }
        }
    }
    return
}



/*
 *  oNew <object> <class> [<parameters>]
 *
 *  Creates a new instance of a class object, optionally initializing the list
 *  of instance varable names to the associated values.  If there is a creation
 *  method defined for this class, it is called with the parameter list given.
 *  If not, only a single parameter is supported.  It must be a list of
 *  instance variable names & values to be defined.  This does not call
 *  parents' constructors -- the child routine must do that itself.
 */

proc oNew {objectName className {args {}}} {
    global oThis __oObjects __oThisClass
    if [oObjExists $objectName] {
        error [foramt {Object ^"%s^" already exists.} $objectName]
    }
    if {! [oClassExists $className]} {
        error [format {Class ^"%s^" is not defined.} $className]
    }
    set method $className::$className
    if {IsEmpty([info procs $method])} {
        if {1 < [llength $args]} {
            error [format {Invalid object initialization for object ^"%s^".} $objectName]
        }
        set vars [lindex $args 0]
        set n [llength $vars]
        for {set i 0} {$i < $n} {incr i 2} {
            set var [lindex $vars $i]
            set val [lindex $vars [expr {$i + 1}]]
            /*  oSet $objectName $var $val -- Can't call 'cause __oObjects isn't set up yet.  */
            set [format %s(%s) $objectName $var] $val
        }
    } {
        set tmpThisClass $__oThisClass
        set tmpThis $oThis
        set __oThisClass $className
        set oThis $objectName
        set err [catch {eval [concat $method $args]} errorString]
        set oThis $tmpThis
        set __oThisClass $tmpThisClass
        if {$err} {
            error $errorString {} $err
        }
    }
    set __oObjects($objectName) $className
}



/*
 *  oDestroy <object> [<parameters>]
 *
 *  Frees up memory associated with an object.  If there is a destructor method
 *  defined for the class, it is called with any parameters given.
 */

proc oDestroy {objectName {args {}}} {
    global oThis __oObjects __oThisClass $objectName
    set class [oType $objectName]		/*	This will throw an exception if the object doesn't exist.  */
    set method $class::~$class
    if {IsntEmpty([info procs $method])} {
        set tmpThisClass $__oThisClass
        set __oThisClass $class
        set tmpThis $oThis
        set oThis $objectName
        set err [catch {eval [concat $method $args]} errorString]
        set oThis $tmpThis
        set __oThisClass $tmpThisClass
        if {$err} {
            error $errorString {} $err
        }
    }
    if {IsEmpty($args)} {
        unset __oObjects($objectName) $objectName
    } {
        error {Invalid arguments.}
    }
}



/*
 *  oSend <object> <method> [<parameters>]
 *
 *  Sends a message to a given object, using the given parameters.
 */

proc oSend {objectName methodName {args {}}} {
    global oThis __oThisClass
    set class [oType $objectName]		/*  This will throw an exception if the object doesn't exist.  */
    set tmpThisClass $__oThisClass
    set method [oMethod $class $methodName]
    if {IsEmpty($method)} {
        error [format {Method ^"%s^" is not defined for object ^"%s^".} $methodName $objectName]
    }
    set tmpThis $oThis
    set oThis $objectName
    set err [catch {eval [concat $method $args]} retval]
    set oThis $tmpThis
    set __oThisClass $tmpThisClass
    if {$err} {
        error $retval {} $err
    }
    return $retval
}



/*
 *  oInherited <method> [<parameters>]
 *
 *  Sends the given method to the superclass of the currently executing method.
 */

proc oInherited {methodName {args {}}} {
    global oThis __oClasses __oThisClass
    if [IsntEmpty(__oThisClass)] {
	    set superClass $__oClasses($__oThisClass)
	    set tmpThisClass $__oThisClass
	    set method [oMethod $superClass $methodName]
	   	if {IsEmpty($method)} {
	        error [format {Inherited method ^"%s^" is not defined for object ^"%s^".} $methodName $oThis]
	    }
	    set err [catch {eval [concat $method $args]} retval]
	    set __oThisClass $tmpThisClass
	    if {$err} {
	        error $retval {} $err
	    }
	    return $retval
    } else {
    	error "oInherited only defined within methods"
    }
}



global oThis __oClasses __oObjects __oThisClass
set oThis {}
set __oThisClass {}
#endif  /*  _OOPS_TCL_  */
