//**
//**  AQMAPI.H
//**
//**  Header file for AQM 2.3D API library.
//**


//**
//**  SET STRUCTURE PACKING TO 1 BYTE
//**
//**  Single-byte structure packing is used because it is the lowest
//**  common denominator between 16 and 32-bit compilers.
//**
//**  Data members are manually aligned on DWORD boundaries, through
//**  the use of padding fields.
//**

#pragma pack(1)


//**
//**  DATATYPE DEFINITIONS FOR CROSS-PLATFORM COMPATIBILITY
//**

// DWORD BOOLEAN TYPE
#ifndef DBOOL
#ifndef _16BIT_
#define DBOOL BOOL
#else
#define DBOOL DWORD
#endif
#endif


// HANDLE TO A PIPE
#ifndef HPIPE
#ifndef _16BIT_
#define HPIPE HANDLE
#else
#define HPIPE HFILE
#endif
#endif


//**
//**  BUFFER SIZE DEFINITIONS
//**
//**  Each buffer size has two definitions.  The first is the maximum
//**  length, or number of characters, for the data field, and the 
//**  second is the size of the buffer required for it, including the
//**  null terminator.
//**
//**  Each string buffer which does not have a size divisible by 4 must
//**  have a padding definitions which will round its size up to 4.
//**


// FILENAMES
#define  AQM_FNAME_LEN           (MAX_PATH) // WHICH IS 256
#define  AQM_FNAME_SIZE          (AQM_FNAME_LEN+1)
#define  AQM_FNAME_PAD           (AQM_FNAME_SIZE%4)


// QUEUE NAMES
#define  AQM_QNAME_LEN           (16)
#define  AQM_QNAME_SIZE          (AQM_QNAME_LEN+1)
#define  AQM_QNAME_PAD           (AQM_QNAME_SIZE%4)

// USERNAMES
#define  AQM_USER_LEN            (63)
#define  AQM_USER_SIZE           (AQM_USER_LEN+1)
#if AQM_USER_SIZE%4 != 0
#error AQM_USER_SIZE is not evenly divisible by 4.
#endif

// USER FULL NAME
#define  AQM_USER_FULL_NAME_LEN  (95)
#define  AQM_USER_FULL_NAME_SIZE (AQM_USER_FULL_NAME_LEN+1)
#if AQM_USER_FULL_NAME_SIZE%4 != 0
#error AQM_USER_FULL_NAME_SIZE is not evenly divisible by 4.
#endif


// DOMAIN NAMES
#define  AQM_DOMAIN_LEN          (63)
#define  AQM_DOMAIN_SIZE         (AQM_DOMAIN_LEN+1)
#if AQM_DOMAIN_SIZE%4 != 0
#error AQM_DOMAIN_SIZE is not evenly divisible by 4.
#endif


// PASSWORDS
#define  AQM_PASSWORD_LEN        (63)
#define  AQM_PASSWORD_SIZE       (AQM_PASSWORD_LEN+1)
#if AQM_PASSWORD_SIZE%4 != 0
#error AQM_PASSWORD_SIZE is not evenly divisible by 4.
#endif


// ACCOUNT NAMES, e.g. "DEV\TomD"
#define  AQM_ACCOUNT_NAME_LEN    (AQM_DOMAIN_LEN+1+AQM_USER_LEN)
#define  AQM_ACCOUNT_NAME_SIZE   (AQM_ACCOUNT_NAME_LEN+1)
#if AQM_ACCOUNT_NAME_SIZE%4 != 0
#error AQM_ACCOUNT_NAME_SIZE is not evenly divisible by 4.
#endif


// QUEUE DESCRIPTION TEXTS
#define  AQM_QDESCR_LEN          (63)
#define  AQM_QDESCR_SIZE         (AQM_QDESCR_LEN+1)
#if AQM_QDESCR_SIZE%4 != 0
#error AQM_QDESCR_SIZE is not evenly divisible by 4.
#endif


// QUEUE DESCRIPTION TEXTS
#define  AQM_STATUS_DETAIL_TEXT_LEN   (399)
#define  AQM_STATUS_DETAIL_TEXT_SIZE  (AQM_STATUS_DETAIL_TEXT_LEN+1)
#if AQM_STATUS_DETAIL_TEXT_SIZE%4 != 0
#error AQM_STATUS_DETAIL_TEXT_SIZE is not evenly divisible by 4.
#endif


// JOB NAMES
#define  AQM_JNAME_LEN           (15)
#define  AQM_JNAME_SIZE          (AQM_JNAME_LEN+1)
#if AQM_JNAME_SIZE%4 != 0
#error AQM_JNAME_SIZE is not evenly divisible by 4.
#endif



// JOB COMMAND-LINE PARAMETERS
#define  AQM_JPARAMS_LEN         (AQM_FNAME_LEN*2 + 1)
#define  AQM_JPARAMS_SIZE        (AQM_JPARAMS_LEN+1)
#define  AQM_JPARAMS_PAD         (AQM_JPARAMS_SIZE%4)



// COMPUTER NAMES
#define  AQM_COMPUTER_LEN        (MAX_COMPUTERNAME_LENGTH)
#define  AQM_COMPUTER_SIZE       (AQM_COMPUTER_LEN+1)
#define  AQM_COMPUTER_PAD        (AQM_COMPUTER_SIZE%4)




// SIDs  (SECURITY IDENTIFIERS)
// (leaving 4 bytes extra for safety margin - 72 as opposed to 68 bytes actually required)
#define  AQM_SID_SIZE  (sizeof(SID) + (sizeof(DWORD) * SID_MAX_SUB_AUTHORITIES))





//**
//**  OTHER CONSTANT DEFINITIONS
//**

// MAXIMUM NUMBER OF STARTUP JOBS
#define  AQM_MAX_STARTUP_JOBS        (4)




//**
//**  STRUCTURES AND DATATYPES
//**


typedef struct _AQM_CREATE_QUEUE_PARAMS	// AQM_CREATE_QUEUE_PARAMS
   {
   CHAR    Name [AQM_QNAME_SIZE];	// NAME OF QUEUE
   CHAR    Pad1 [AQM_QNAME_PAD];	// PADDING
   CHAR    Descr [AQM_QDESCR_SIZE];	// QUEUE DESCRIPTION
   DWORD   Priority;	// QUEUE PRIORITY (0 - 255)
   DWORD   PendJobsLimit;	// PENDING JOBS LIMIT
   DWORD   ExecJobsLimit;	// EXECUTING JOBS LIMIT
   DBOOL   Stopped;	// TRUE = QUEUE INITIALLY STOPPED
   DBOOL   Closed;	// TRUE = QUEUE INITIALLY CLOSED
   DBOOL   AutoOpenClose;	// TRUE = AUTO OPEN/CLOSE ENABLED
   DWORD   AutoOpenMinute;	// MINUTE OF DAY TO AUTOMATICALLY OPEN QUEUE
   DWORD   AutoCloseMinute;	// MINUTE OF DAY TO AUTOMATICALLY CLOSE QUEUE
   DBOOL   AutoStartStop;	// TRUE = AUTO START/STOP ENABLED
   DWORD   AutoStartMinute;	// MINUTE OF DAY TO AUTOMATICALLY START QUEUE
   DWORD   AutoStopMinute;	// MINUTE OF DAY TO AUTOMATICALLY STOP QUEUE
   DWORD   RetainMinutes;	// MINUTES TO RETAIN COMPLETED JOBS
   } AQM_CREATE_QUEUE_PARAMS;






typedef struct _AQM_SUBMIT_JOB_PARAMS	// AQM_SUBMIT_JOB_PARAMS
   {
   CHAR    Name [AQM_JNAME_SIZE];	// JOB NAME
   CHAR    File [AQM_FNAME_SIZE];	// JOB FILE
   CHAR    Pad1 [AQM_FNAME_PAD];	// PADDING
   CHAR    Params [AQM_JPARAMS_SIZE];	// JOB COMMAND-LINE PARAMETERS
   CHAR    Pad2 [AQM_JPARAMS_PAD];	// PADDING
   CHAR    LogFile [AQM_FNAME_SIZE];	// LOG FILE FILENAME
   CHAR    Pad3 [AQM_FNAME_PAD];	// PADDING
   DBOOL   LogNumber;	// TRUE = APPEND JOB NUMBER TO LOG FILENAME
   DBOOL   NotifyOnStart;	// TRUE = NOTIFY WHEN JOB STARTS
   DBOOL   NotifyOnCompl;	// TRUE = NOTIFY WHEN JOB COMPLETES
   CHAR    Notify [AQM_USER_SIZE];	// USER TO NOTIFY (OR EMPTY STRING FOR OWNER)
   DBOOL   PasswordPresent;	// TRUE = PASSWORD PRESENT
   CHAR    Password [AQM_PASSWORD_SIZE];	// PASSWORD
   FILETIME AfterTime;	// TIME AFTER WHICH JOB MAY START (OR 0,0 FOR NONE)
   DWORD   EveryMinutes;	// JOB'S "/EVERY" TIME (OR 0 FOR NONE)
   DBOOL   DetectInput;	// TRUE = ABORT JOB IF WAITING FOR INPUT
   DWORD   ExitCodeSource;	// EXIT CODE SOURCE (SEE AQM_ECS_...)
   CHAR    ExitCodeFile [AQM_FNAME_SIZE];	// ASCII TEXT CONDITION CODE FILE
   CHAR    Pad4 [AQM_FNAME_PAD];	// PADDING
   DBOOL   SaveExitCodeFile;	// TRUE = DON'T DELETE EXIT CODE FILE
   DBOOL   ExitCodeHandling;	// TRUE = EXIT CODE HANDLING ENABLED
   DWORD   FailExitCodeRelation;	// EXIT CODE RELATION TO DEFINE FAILURE
   DWORD   FailExitCode;	// EXIT CODE TO DEFINE FAILURE
   DBOOL   FailNotify;	// TRUE = NOTIFY USER IF JOB FAILS
   CHAR    FailNotifyUser [AQM_USER_SIZE];	// USER TO NOTIFY (OR EMPTY STRING FOR OWNER)
   DBOOL   FailReportEvent;	// TRUE = REPORT EVENT IF JOB FAILS
   CHAR    FailureJob [AQM_FNAME_SIZE];	// JOB FILE TO EXECUTE IF JOB FAILS (EMPTY FOR NONE)
   CHAR    Pad5 [AQM_FNAME_PAD];	// PADDING
   CHAR    SuccessJob [AQM_FNAME_SIZE];	// JOB FILE TO EXECUTE IF JOB SUCCEEDS (EMPTY FOR NONE)
   CHAR    Pad6 [AQM_FNAME_PAD];	// PADDING
   CHAR    Queue [AQM_QNAME_SIZE];	// QUEUE TO SUBMIT JOB TO
   CHAR    Pad7 [AQM_QNAME_PAD];	// PADDING
   CHAR    Username [AQM_USER_SIZE];	// USERNAME TO RUN JOB AS (ADDED IN 2.3A)
   CHAR    Domain [AQM_DOMAIN_SIZE];	// DOMAIN OF ACCOUNT TO RUN JOB AS (ADDED IN 2.3A)
   } AQM_SUBMIT_JOB_PARAMS;








//**
//**  SYSTEM ACCESS TYPES
//**


// CONTROL (START, PAUSE, MODIFY SETTINGS, ETC), NOT INCLUDING SHUTTING IT DOWN
#define   AQM_SYSTEM_CONTROL	0x0001L

// CREATE QUEUES IN THE SYSTEM
#define   AQM_SYSTEM_CREATE_QUEUE	0x0002L

// SET THE PERMISSIONS ON THE SYSTEM
#define   AQM_SYSTEM_SET_PERMISSIONS	(WRITE_DAC)

// ALL ACCESS
#define   AQM_SYSTEM_ALL_ACCESS	(AQM_SYSTEM_CONTROL \
	| AQM_SYSTEM_CREATE_QUEUE \
	| AQM_SYSTEM_SET_PERMISSIONS)




//**
//**  QUEUE ACCESS TYPES
//**


// CONTROL (START, STOP, OPEN, CLOSE, MODIFY PROPERTIES)
#define   AQM_QUEUE_CONTROL	0x0001L

// SUBMIT JOBS TO THE QUEUE
#define   AQM_QUEUE_SUBMIT	0x0002L

// DELETE A QUEUE
#define   AQM_QUEUE_DELETE	(DELETE)

// SET THE QUEUE'S PERMISSIONS
#define   AQM_QUEUE_SET_PERMISSIONS	(WRITE_DAC)

// ALL ACCESS
#define   AQM_QUEUE_ALL_ACCESS	(AQM_QUEUE_CONTROL \
	| AQM_QUEUE_SUBMIT \
	| AQM_QUEUE_SET_PERMISSIONS \
	| AQM_QUEUE_DELETE)



//**
//**  JOB ACCESS TYPES
//**

// CONTROL (RESCHEDULE, MODIFY PROPERTIES)
#define   AQM_JOB_CONTROL	(0x1L)

// ABORT THE JOB
#define   AQM_JOB_ABORT	(0x2L)

// PAUSE OR UNPAUSE THE JOB
#define   AQM_JOB_PAUSE	(0x4L)

// SET THE JOB'S PERMISSIONS
#define   AQM_JOB_SET_PERMISSIONS	(WRITE_DAC)

// CANCEL THE JOB
#define   AQM_JOB_CANCEL	(DELETE)

// ALL ACCESS
#define   AQM_JOB_ALL_ACCESS	(AQM_JOB_CONTROL \
	| AQM_JOB_ABORT \
	| AQM_JOB_PAUSE \
	| AQM_JOB_CANCEL \
	| AQM_JOB_SET_PERMISSIONS)





//**
//**  JOB STATUS CODES
//**

#define AQM_JS_UNDEFINED              (0)	// UNDEFINED (SHOULD NEVER BE SET)
#define AQM_JS_HOLDING                (1)	// JOB IS HOLDING
#define AQM_JS_PENDING                (2)	// JOB IS PENDING
#define AQM_JS_EXECUTING              (3)	// JOB IS EXECUTING
#define AQM_JS_FAILED_TO_START        (4)	// JOB FAILED TO START
#define AQM_JS_ABORTED                (5)	// JOB WAS ABORTED
#define AQM_JS_COMPLETED              (6)	// JOB COMPLETED
#define AQM_JS_EXECUTION_ERROR        (7)	// INTERNAL AQM ERROR OCCURED IN EXECUTION OF JOB



//**
//**  JOB ABORT REASON CODES
//**

#define AQM_AR_UNDEFINED              (0)	// UNDEFINED
#define AQM_AR_ABORTED_BY_USER        (1)	// JOB WAS ABORTED BY USER
#define AQM_AR_WAS_WAITING_FOR_INPUT  (2)	// JOB WAS WAITING FOR INPUT
#define AQM_AR_SERVICE_SHUTDOWN       (3)	// JOB WAS ABORTED DUE TO SERVICE SHUTDOWN




//**
//**  JOB COMPLETION STATUS CODES
//**

#define AQM_CS_UNDEFINED              (0)	// NORMAL (NO EXIT CODE HANDLING) OR UNDEFINED
#define AQM_CS_SUCCESS                (1)	// EXITED WITH EXIT CODE DEFINED AS SUCCESS
#define AQM_CS_FAILURE                (2)	// EXITED WITH EXIT CODE DEFINED AS FAILURE
#define AQM_CS_EXCEPTION              (3)	// EXITED DUE TO EXCEPTION IN PRIMARY PROCESS



//**
//**  JOB FAILED TO START REASON CODES
//**

#define AQM_FSR_UNDEFINED               (0)	// UNDEFINED
#define AQM_FSR_NO_LOGON_SERVERS        (1)	// NO LOGON SERVERS AVAILABLE
#define AQM_FSR_LOGON_FAILED            (2)	// LOGON FAILED
#define AQM_FSR_JOB_FILE_NOT_FOUND      (3)	// JOB FILE NOT FOUND
#define AQM_FSR_JOB_FILE_ACCESS_DENIED  (4)	// JOB FILE ACCESS DENIED
#define AQM_FSR_LOG_FILE_ACCESS_DENIED  (5)	// LOG FILE ACCESS DENIED





//**
//**  EXIT CODE SOURCE CODES
//**
//**  These values are used to specify the source for a job's condition
//**  code.  For example, a job's condition code may come from the
//**  exit code of its primary process, or from a file containing a
//**  text representation of a number.
//**
	// EXIT CODE COMES FROM...
#define AQM_ECS_PRIMARY_PROCESS      (0)	// EXIT CODE OF PRIMARY PROCESS OF JOB
#define AQM_ECS_TEXT_FILE            (1)	// TEXT FILE CONTAINING DECIMAL NUMBER



//**
//**  EXIT CODE RELATIONS TO DEFINE JOB FAILURE ENUMERATED TYPE
//**

#define AQM_ECR_GREATER_THAN         (0)	// JOB FAILS IF EXIT CODE IS GREATER THAN VALUE
#define AQM_ECR_LESS_THAN            (1)	// JOB FAILS IF EXIT CODE IS LESS THAN VALUE
#define AQM_ECR_EQUAL_TO             (2)	// JOB FAILS IF EXIT CODE IS EQUAL TO VALUE
#define AQM_ECR_NOT_EQUAL_TO         (3)	// JOB FAILS IF EXIT CODE IS NOT EQUAL TO VALUE







//**
//**  AQM_STARTUP_JOB STRUCTURE
//**
//**  This structure defines a startup job, which is a special job that is
//**  executed when the AQM service is started.
//**

typedef struct _AQM_STARTUP_JOB	// AQM_STARTUP_JOB STRUCTURE
   {
   CHAR    File [AQM_FNAME_SIZE];	// ... FILE TO EXECUTE
   CHAR    Pad1 [AQM_FNAME_PAD];	// ... PADDING
   DWORD   Cond;	// ... CONDITION FOR EXECUTING THE JOB
   DWORD   TimeLimit;	// ... TIME LIMIT FOR JOB (SECONDS)
   DBOOL   AbortIfTimedOut;	// ... TRUE = ABORT IF EXCEEDS STARTUP JOB TIME LIMIT
   } AQM_STARTUP_JOB;



//**
//**  STARTUP JOB CONDITIONS
//**
//**  Each startup job has a conditions value, which determines when
//**  it should execute.  A value of "Never" or an empty job filename
//**  indicate that the startup job entry is not in use.
//**

#define AQM_STARTUP_ALWAYS	(1)
#define AQM_STARTUP_SYSTEM_BOOT	(2)
#define AQM_STARTUP_NON_ORDERLY	(3)






//**
//**  AQM_JOB_INFO STRUCTURE
//**

typedef struct _AQM_JOB_INFO
   {
   DWORD   Status;	// JOB STATUS
   DWORD   Reason;	// STATUS REASON CODE
   DBOOL   StatusDetails;	// TRUE = JOB HAS STATUS DETAIL TEXT
	// STATUS DETAIL TEXT
   CHAR    StatusDetailText [AQM_STATUS_DETAIL_TEXT_SIZE];
   CHAR    Name [AQM_JNAME_SIZE];	// JOB NAME
   CHAR    File [AQM_FNAME_SIZE];	// FILESPEC OF JOB BATCH FILE
   CHAR    Pad1 [AQM_FNAME_PAD];	// PADDING
   CHAR    Params [AQM_JPARAMS_SIZE];	// JOB COMMAND-LINE PARAMETERS
   CHAR    Pad2 [AQM_JPARAMS_PAD];	// PADDING
   CHAR    Username [AQM_USER_SIZE];	// USERNAME
   CHAR    Domain [AQM_DOMAIN_SIZE];	// DOMAIN
   CHAR    FullName [AQM_USER_FULL_NAME_SIZE];// USER'S FULL NAME
   DBOOL   PasswordPresent;	// TRUE = PASSWORD PRESENT
   CHAR    Password [AQM_PASSWORD_SIZE];	// PASSWORD
   CHAR    LogFile [AQM_FNAME_SIZE];	// LOG FILE FILENAME
   CHAR    Pad3 [AQM_FNAME_PAD];	// PADDING
   DBOOL   LogNumber;	// TRUE = JOB NUMBER IS APPENDED TO LOG FILE
   DBOOL   NotifyOnStart;	// TRUE = NOTIFY WHEN JOB STARTS
   DBOOL   NotifyOnCompl;	// TRUE = NOTIFY WHEN JOB COMPLETES
   CHAR    Notify [AQM_USER_SIZE];	// USER TO NOTIFY
   FILETIME AfterTime;	// TIME AFTER WHICH JOB MAY BE STARTED
   DWORD   EveryMinutes;	// "/EVERY" TIME IN MINUTES
   DWORD   ResubmittedAs;	// JOB NUMBER OF RESUBMITTED JOB
   DBOOL   DetectInput;	// TRUE = ABORT JOB IF WAITING FOR INPUT
   FILETIME TimeSubmitted;	// TIME SUBMITTED
   FILETIME TimeStarted;	// TIME STARTED
   FILETIME TimeCompleted;	// TIME COMPLETED
   LARGE_INTEGER CpuTime;   	// CPU TIME CONSUMED BY JOB
   DWORD   ExitCode;	// JOB'S PROCESS EXIT CODE
   DWORD   JobNumber;	// JOB NUMBER
   DWORD   ExitCodeSource;	// EXIT CODE SOURCE (SEE AQM_ECS_...)
   CHAR    ExitCodeFile [AQM_FNAME_SIZE];	// ASCII TEXT CONDITION CODE FILE
   CHAR    Pad4 [AQM_FNAME_PAD];	// PADDING
   DBOOL   SaveExitCodeFile;	// TRUE = DON'T DELETE EXIT CODE FILE
   DBOOL   ExitCodeHandling;	// TRUE = EXIT CODE HANDLING ENABLED
   DWORD   FailExitCodeRelation;	// EXIT CODE RELATION TO DEFINE FAILURE
   DWORD   FailExitCode;	// EXIT CODE TO DEFINE FAILURE
   DBOOL   FailNotify;	// TRUE = NOTIFY USER IF JOB FAILS
   CHAR    FailNotifyUser [AQM_USER_SIZE];	// USER TO NOTIFY (OR EMPTY FOR OWNER)
   DBOOL   FailReportEvent;	// TRUE = REPORT EVENT IF JOB FAILS
   CHAR    FailureJob [AQM_FNAME_SIZE];	// JOB TO EXECUTE IF JOB FAILS (OR EMPTY FOR NONE)
   CHAR    Pad5 [AQM_FNAME_PAD];	// PADDING
   CHAR    SuccessJob [AQM_FNAME_SIZE];	// JOB TO EXECUTE IF JOB SUCCEEDS (OR EMPTY FOR NONE)
   CHAR    Pad6 [AQM_FNAME_PAD];	// PADDING
   CHAR    Queue [AQM_QNAME_SIZE];	// NAME OF QUEUE JOB IS IN
   CHAR    Pad7 [AQM_QNAME_PAD];	// PADDING
   DBOOL   Paused;	// TRUE = JOB IS PAUSED (EXECUTING ONLY)
	BYTE Expansion [1024];
   } AQM_JOB_INFO;




//**
//**  AQM_QUEUE_INFO STRUCTURE
//**

typedef struct _AQM_QUEUE_INFO
   {
   CHAR    Name [AQM_QNAME_SIZE];	// NAME OF QUEUE
   CHAR    Pad1 [AQM_QNAME_PAD];	// PADDING
   CHAR    Descr [AQM_QDESCR_SIZE];	// QUEUE DESCRIPTION
   DBOOL   Stopped;	// TRUE = QUEUE IS STOPPED
   DBOOL   Closed;	// TRUE = QUEUE IS CLOSED
   DWORD   Priority;	// PRIORITY WITHIN PRODUCT
   DWORD   PendJobsLimit;	// PENDING JOBS LIMIT
   DWORD   ExecJobsLimit;	// EXECUTING JOBS LIMIT
   DWORD   RetainMinutes;   	// AMOUNT OF TIME QUEUE RETAINS COMPLETED JOBS
   DBOOL   AutoOpenClose;	// TRUE=AUTO-OPEN/CLOSE ENABLED
   DWORD   AutoOpenMinute;	// MINUTE OF DAY TO AUTOMATICALLY OPEN QUEUE
   DWORD   AutoCloseMinute;	// MINUTE OF DAY TO AUTOMATICALLY CLOSE QUEUE
   DBOOL   AutoStartStop;	// TRUE=AUTO-START/STOP ENABLED
   DWORD   AutoStartMinute;	// MINUTE OF DAY TO AUTOMATICALLY START QUEUE
   DWORD   AutoStopMinute;	// MINUTE OF DAY TO AUTOMATICALLY STOP QUEUE
   unsigned long QueueQueueNumber;	// Queue number for queue (preserved across restarts)
   CHAR    Creator [AQM_ACCOUNT_NAME_SIZE];	// ACCOUNT NAME OF QUEUE'S CREATOR
   FILETIME TimeCreated;	// TIME CREATED
   DWORD ExecutingJobs;
   DWORD   JobsCompleted;	// TOTAL JOBS COMPLETED
   LARGE_INTEGER CpuTime;   	// COMPLETED JOBS CPU TIME
   BYTE Expansion [512];
   } AQM_QUEUE_INFO;

//**  AQM_SYSTEM_INFO STRUCTURE
//**
//**  This structure holds information about a system, which is a container
//**  object that holds all queues.  The information in this structure applies
//**  to the entire AQM server.
//**

typedef struct _AQM_SYSTEM_INFO
   {
	// ... FILENAME OF PRIMARY SETTINGS FILE
   CHAR     PrimarySettingsFile [AQM_FNAME_SIZE];
   CHAR     Pad1 [AQM_FNAME_PAD];	// PADDING

	// ... FILENAME OF BACKUP SETTINGS FILE
   CHAR     BackupSettingsFile [AQM_FNAME_SIZE];
   CHAR     Pad2 [AQM_FNAME_PAD];	// PADDING

   CHAR     LogFile [AQM_FNAME_SIZE];	// ... FILENAME OF LOG FILE
   CHAR     Pad3 [AQM_FNAME_PAD];	// PADDING
   DWORD    LogFileSizeLimit;	// ... LOG FILE SIZE LIMIT IN KBYTES
   DWORD    LogFileSeqNumber;	// ... NEXT LOG FILE SEQUENCE NUMBER

   DWORD    JobNumberWrapsAfter;	// ... JOB NUMBER WRAPS TO 1 AFTER THIS NUMBER

   CHAR     DefaultQueue [AQM_QNAME_SIZE];	// ... NAME OF DEFAULT QUEUE
   CHAR     Pad4 [AQM_QNAME_PAD];	// PADDING

   DWORD    PriorityQuantum;	// ... PRIORITY QUANTUM (MSEC)

   DWORD    SystemBootThreshold;	// ... SYSTEM BOOT THRESHOLD (SECONDS)

   DWORD    CheckpointSaveInterval;	// ... CHECKPOINT SAVE INTERVAL IN SECONDS

   DWORD    LastJobNumberUsed;	// ... LAST JOB NUMBER USED

   DWORD    TotalExecJobs;	// ... TOTAL EXECUTING JOBS
   DWORD    MaxExecJobs;	// ... MAX ALLOWED EXECUTING JOBS
   
   DWORD    TotalJobProcesses;	// ... TOTAL JOB PROCESSES
   DWORD    MaxJobProcesses;	// ... MAX ALLOWED JOB PROCESSES

   DBOOL    StartupComplete;	// ... TRUE = STARTUP COMPLETE (STARTUP JOBS DONE)

   DBOOL    StartupErrors;	// ... TRUE = THERE WERE STARTUP ERRORS

	// ... STARTUP JOBS
   AQM_STARTUP_JOB StartupJob [AQM_MAX_STARTUP_JOBS];

   } AQM_SYSTEM_INFO;






//**
//**  CONSTANT DEFINITIONS
//**

#define AQM_API_DEFAULT_TIMEOUT_MSEC (30000)



//**
//**
//**  FUNCTIONS PROTOTYPES
//**
//**  Parameters marked with a ">" character are input parameters.
//**  Parameters marked with a "*" character are output parameters passed by
//**  reference, which means that the actual value specified for the parameter
//**  must be a pointer to a variable which receives the value.  For example:
//**
//**         DWORD* rJobNumber  // * JOB NUMBER OF SUBMITTED JOB
//**
//**  In this example, the caller must pass the address of a DWORDvariable to
//**  function.  If the function returns with success status, then the variable
//**  will contain the job number of the submitted job.  Note that if the
//**  function fails, then the value of the variable is undefined.
//**


BOOL    AqmSetTimeout (	// SET TIMEOUT FOR AQM OPERATIONS
        DWORD   Timeout);	// > TIMEOUT IN MSEC

DWORD   AqmCreateQueue (	// CREATE A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        AQM_CREATE_QUEUE_PARAMS * pParams);	// > POINTER TO PARAMETERS

DWORD   AqmDeleteQueue (	// DELETE A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DBOOL   AbortExecutingJobs);	// > TRUE = ABORT ANY EXECUTING JOBS IN THE QUEUE

DWORD   AqmGetSystemInfo (	// GET SYSTEM INFO
        LPCSTR  ServerName,	// > SERVER NAME
        AQM_SYSTEM_INFO **rpSI);	// * POINTER TO SYSTEM INFO

DWORD	AqmGetSystemSD (LPCSTR ServerName, SECURITY_DESCRIPTOR** rpSD);

DWORD   AqmGetQueueInfo (	// GET QUEUE INFO
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        AQM_QUEUE_INFO **rpQI);	// * POINTER TO QUEUE INFO

DWORD   AqmGetJobInfo (	// GET JOB INFO
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber,	// > JOB NUMBER
        AQM_JOB_INFO **rpJI);	// * POINTER TO JOB INFO BUFFER

DWORD   AqmGetAllQueuesInfo (	// GET ALL QUEUES
        LPCSTR  ServerName,	// > SERVER NAME
        AQM_QUEUE_INFO **rpQueuesInfo,	// * POINTER TO QUEUES INFO BUFFER
        DWORD  *rNumQueues);	// * NUMBER OF QUEUES

DWORD   AqmGetQueueAndJobsInfo (	// GET QUEUE AND JOBS INFO
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        AQM_QUEUE_INFO **rpQI,	// * POINTER TO QUEUE INFO BUFFER
		CHAR* InfoFileName,
        DWORD  *rNumJobs);	// * NUMBER OF JOBS

DWORD   AqmGetExecutingJobsInfo (	// GET EXECUTING JOBS INFO
        LPCSTR  ServerName,	// > SERVER NAME
        CHAR* InfoFileName,
        DWORD  *rNumJobs);	// * NUMBER OF JOBS

DWORD   AqmSetLogSizeLimit (	// SET LOG SIZE LIMIT
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   LogSizeLimit);	// > LOG SIZE LIMIT (KB)

DWORD   AqmSetMaxExecJobs (	// SET MAX EXEC JOBS FOR SYSTEM
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   MaxExecJobs);	// > MAX EXECUTING JOBS

DWORD   AqmSetMaxJobProcesses (	// SET MAX JOB PROCESSES FOR SYSTEM
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   MaxJobProcesses);	// > MAX JOB PROCESSES

DWORD   AqmSetLogFile (	// SET LOG FILE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  LogFile);	// > LOG FILE

DWORD   AqmClearDefaultQueue (	// CLEAR DEFAULT QUEUE
        LPCSTR  ServerName);	// > SERVER NAME

DWORD   AqmSetDefaultQueue (	// SET DEFAULT QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName);	// > QUEUE NAME

DWORD   AqmSetStartupJob (	// SET STARTUP JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   Number,	// > JOB NUMBER (0-BASED)
        AQM_STARTUP_JOB * pJob);	// > POINTER TO STARTUP JOB INFO

DWORD   AqmClearStartupJob (	// CLEAR STARTUP JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   Number);	// > JOB NUMBER (0-BASED)

DWORD   AqmSubmitJob (	// SUBMIT A JOB
        LPCSTR  ServerName,	// > SERVER NAME
        AQM_SUBMIT_JOB_PARAMS * pParams,	// > POINTER TO PARAMETERS
        DWORD*  rJobNumber);	// * JOB NUMBER OF SUBMITTED JOB

DWORD   AqmRequeueJob (	// REQUEUE A JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber,	// > JOB NUMBER
        LPCSTR  QueueName);	// > NAME OF QUEUE TO REQUEUE JOB TO

DWORD   AqmCancelJob (	// CANCEL A JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber);	// > JOB NUMBER

DWORD   AqmAbortJob (	// ABORT AN EXECUTING JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber);	// > JOB NUMBER

DWORD   AqmPauseJob (	// PAUSE AN EXECUTING JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber);	// > JOB NUMBER

DWORD   AqmUnpauseJob (	// UNPAUSE A PAUSED EXECUTING JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber);	// > JOB NUMBER

DWORD   AqmSetQueueDescription (	// SET QUEUE DESCRIPTION
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        LPCSTR  Description );	// > POINTER TO DESCRIPTION STRING

DWORD   AqmSetQueuePriority (	// SET QUEUE PRIORITY
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DWORD   Priority );	// > QUEUE PRIORITY (0 - 255)

DWORD   AqmSetQueueMaxPend (	// SET QUEUE MAX PENDING JOBS VALUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DWORD   MaxPend );	// > MAX PENDING JOBS

DWORD   AqmSetQueueMaxExec (	// SET QUEUE MAX EXECUTING JOBS VALUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DWORD   MaxExec );	// > MAX EXECUTING JOBS

DWORD   AqmSetQueueRetainTime (	// SET QUEUE RETAIN JOBS TIME
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DWORD   RetainMinutes );	// > MINUTES TO RETAIN COMPLETED JOBS

DWORD   AqmSetQueueAutoStartStop (	// SET QUEUE AUTO START/STOP TIMES
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DWORD   AutoStartMinute,	// > AUTO START MINUTE (0 FOR 12:00 AM, 61 FOR 1:01 AM)
        DWORD   AutoStopMinute);	// > AUTO STOP MINUTE (0 FOR 12:00 AM, 61 FOR 1:01 AM)

DWORD   AqmStartQueue (	// START A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName );	// > QUEUE NAME

DWORD   AqmStopQueue (	// STOP A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName );	// > QUEUE NAME

DWORD   AqmSetQueueAutoOpenClose (	// SET QUEUE AUTO OPEN/CLOSE TIMES
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        DWORD   AutoOpenMinute,	// > AUTO OPEN MINUTE (0 FOR 12:00 AM, 61 FOR 1:01 AM)
        DWORD   AutoCloseMinute);	// > AUTO CLOSE MINUTE (0 FOR 12:00 AM, 61 FOR 1:01 AM)

DWORD   AqmOpenQueue (	// OPEN A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName );	// > QUEUE NAME

DWORD   AqmCloseQueue (	// CLOSE A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName );	// > QUEUE NAME

BOOL    AqmGetErrorText (	// GET MESSAGE TEXT FOR AN AQM ERROR CODE
        DWORD   ErrorCode,	// > ERROR CODE RETURNED BY AN AQM API FUNCTION
        LPSTR  *rpMessage );	// * POINTER TO MESSAGE (MUST BE FREED BY CALLER)

DWORD   AqmCheckSystemAccess (	// CHECK ACCESS TO A SYSTEM
        LPCSTR  ServerName,	// > SERVER NAME
        ACCESS_MASK AccessDesired,	// > ACCESS DESIRED
        BOOL *  rAccessGranted);	// * TRUE = ACCESS GRANTED

DWORD   AqmCheckQueueAccess (	// CHECK ACCESS TO A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  QueueName,	// > QUEUE NAME
        ACCESS_MASK AccessDesired,	// > ACCESS DESIRED
        BOOL *  rAccessGranted);	// * TRUE = ACCESS GRANTED

DWORD   AqmCheckJobAccess (	// CHECK ACCESS TO A JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber,	// > JOB NUMBER
        ACCESS_MASK AccessDesired,	// > ACCESS DESIRED
        BOOL *  rAccessGranted);	// * TRUE = ACCESS GRANTED

DWORD AqmDumpInternalQueues (void);

#ifndef _16BIT_
DWORD   AqmSetSystemSecurity (	// SET SECURITY DESCRIPTOR FOR SYSTEM
        LPCSTR  ServerName,	// > SERVER NAME
        PSECURITY_DESCRIPTOR pSD);	// > POINTER TO SELF-RELATIVE SECURITY DESCRIPTOR

DWORD   AqmSetQueueSecurity (	// SET SECURITY DESCRIPTOR FOR A QUEUE
        LPCSTR  ServerName,	// > SERVER NAME
        LPCSTR  Queue,	// > QUEUE NAME
        PSECURITY_DESCRIPTOR pSD);	// > POINTER TO SELF-RELATIVE SECURITY DESCRIPTOR

DWORD   AqmSetJobSecurity (	// SET SECURITY DESCRIPTOR FOR A JOB
        LPCSTR  ServerName,	// > SERVER NAME
        DWORD   JobNumber,	// > JOB NUMBER
        PSECURITY_DESCRIPTOR pSD);	// > POINTER TO SELF-RELATIVE SECURITY DESCRIPTOR
#endif //_16BIT_


//**
//**  REVERT TO DEFAULT PACKING
//**

#pragma pack()
