/*
 * Copyright Digital Products, Inc.. All Rights Reserved. 
 *
 * nsconfig
 *
 * This program reads configuration information from a file named 
 * nicprint.conf and writes the new configuration to a well known
 * port on the DPI unit. It then verifies success or failure of 
 * the write. 
 *
 */ 
/*
 *
 *
 * History:
 * 
 * 11/19/93   NLL   Added code necessary to read and send additional 
 *                  (serial port and protocol enabling) fields to 
 *                  Netsprint Unit.
 * 11/23/93   NLL   Updated valid_netmask to allow 255.B.C.D, rather
 *                     than just 255.255.C.D.
 * 12/16/94   JAC   Added default router to configuration parameters 
 *                  that are written.
 * 1/3/94     JAC   Added valid router routine to verify that the default
 *                  router is on the same subnet as the unit being
 *                  configured. 
 *                  Also fixed numerous bugs in error reporting (cleaned
 *                  up output, eliminated surprise "exits"). 
 * 1/15/94    JAC   more cleanup, this time fixed valid_addr to return a 
 *                  NULL IP address in case user wants no default gateway.
 *                  Also fixed a core dump in debug logging.
 *
 * 11/28/94   RMT   Modified Trailers field to be defaulted to none.
 *                  Deleted 1200 baud rate and added 57.6, 76.8, 115.2
 *                  Added Case Sensative check of banners.
 *  	            Corrected Sub Range Cheching for IP info.
 *		    Added DEC LAT off/on and POWERUP status page off/on.
 *		    parameters.
 */

#include <stdio.h>
#include "tcp_cnfg.h"
#include "p_stack.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/errno.h>
#include <netdb.h>
#include <ctype.h>


NSC_LEGAL_BAUD_RATE_TYPE nsc_legal_baud_rates [NSC_NUM_LEGAL_BAUD_RATES] =
{
        {2400L, BAUD_2400},
        {4800L, BAUD_4800},
        {9600L, BAUD_9600},
        {19200L, BAUD_19200},
        {38400L, BAUD_38400},
        {57600L, BAUD_57600},
        {76800L, BAUD_76800},
        {115200L, BAUD_115200}
};


NSC_DATA_BIT_CHOICE nsc_data_bit_choices [NSC_NUM_DATA_BIT_CHOICES] =
{
        {5, DATA_BITS_5},
        {6, DATA_BITS_6},
        {7, DATA_BITS_7},
        {8, DATA_BITS_8}
};

NSC_STOP_BIT_CHOICE nsc_stop_bit_choices [NSC_NUM_STOP_BIT_CHOICES] =
{
        {"1",   STOP_BITS_1},
        {"1.5", STOP_BITS_1_5},
        {"2",   STOP_BITS_2}
};



NSC_PARITY_CHOICE nsc_parity_choices [NUM_PARITY_CHOICES] =
{
        {"NONE", NO_PARITY},
        {"ODD", ODD_PARITY},
        {"EVEN", EVEN_PARITY}
};

#define SERV_TCP_PORT	10000


#define	filename "nicprint.conf" /* The configuration file to read */

/* This is the string that is printed if the command line options   */
/* are invalid or the user requests help.                           */
#define usage "\nusage: [-?] [-h] <internet address>\n"
#define VALIDOPTIONS "?vp:"

/* Prototypes */

void conf_error();
int is_in();
long valid_address();
unsigned long valid_netmask();
int valid_number();
int valid_snmp();
char* clean_up_SysDesc();
void send_configuration_details();



void nsc_get_serial_baud_rate ();
void nsc_get_serial_data_bits ();
void nsc_get_serial_stop_bits ();
void nsc_get_parity ();
void nsc_get_ready_busy ();
void nsc_get_xon_xoff ();

NSC_GET_SERIAL_ITEM_FUNCTION get_nsc_serl_item [NSC_NUM_SERL_ITEMS] = {

        nsc_get_serial_baud_rate,
        nsc_get_serial_data_bits,
        nsc_get_serial_stop_bits,
        nsc_get_parity,
        nsc_get_ready_busy,
        nsc_get_xon_xoff};


/* End of Prototypes */


/*
 *   Function:
 *
 *             void  nsc_upper_case (char *string);
 *
 *
 *   Description:
 *              
 *      This function converts a mixed case, NULL terminated character string 
 *      to upper case.
 *
 *
 *   Parameters:    string              string to be converted
 *
 *
 *   Return Values: none
 *
 */
void nsc_upper_case (string)
    char *string;
{
    int count;

    /* Convert to upper case */
    for (count=0;string[count]!='\0'; count++)
    {
        if ((string[count] >= 'a') && (string[count] <= 'z'))
                string[count] -= 'a'-'A';
    }
}


/*
 *   Function:
 *
 *             void  nsc_get_serial_baud_rate (char *string,
 *                                             NSC_SERL_CONFIG_TYPE *nsc_serl);
 *
 *
 *   Description:
 *              
 *      This function scans for one of several possible baudrates (reports 
 *      error if something else is there), then encodes this information
 *      into the baud_rate field of the nsc_serl structure.
 *
 *
 *   Parameters:    string              string to be scanned
 *                  nsc_serl            structure to be updated
 *
 *
 *   Return Values: none
 *
 */
void nsc_get_serial_baud_rate (string, nsc_serl)
char *string;
NSC_SERL_CONFIG_TYPE *nsc_serl;
{
    int count;
    long baud_rate;
    int match_found=0;


    if (sscanf (string, "%ld", &baud_rate) != 1)
    {
        conf_error("Illegal Baud Rate.\n Please specify: 2400, 4800, 9600, 19200, 38400, 57600, 76800 or 115200.");
        return;
    }

    for (count = 0; count <= NSC_NUM_LEGAL_BAUD_RATES - 1; count ++)
    {
        if (baud_rate == nsc_legal_baud_rates[count].baud_rate)
        {
            nsc_serl->baud_rate = 
                        nsc_legal_baud_rates[count].baud_code;
            match_found = 1;
            break;
        }
    }

    if (!match_found)
        conf_error("Illegal Baud Rate.\n Please specify: 2400, 4800, 9600, 19200, 38400, 57600, 76800 or 115200.");
}









/*
 *   Function:
 *
 *             void  nsc_get_serial_data_bits (char *string,
 *                                             NSC_SERL_CONFIG_TYPE *nsc_serl);
 *
 *
 *   Description:
 *              
 *      This function scans for one of several possible # of data bits
 *      (reports error if something else is there), then encodes this info
 *      into the data_bits field of the nsc_serl structure.
 *
 *
 *   Parameters:    string              string to be scanned
 *                  nsc_serl            structure to be updated
 *
 *
 *   Return Values: none
 *
 */
void nsc_get_serial_data_bits (string, nsc_serl)
char *string;
NSC_SERL_CONFIG_TYPE *nsc_serl;
{
    int count;
    int data_bits;
    int match_found=0;

    if (sscanf (string, "%d", &data_bits) != 1)
    {
        conf_error("Illegal Number of Data Bits.\n Please specify 5, 6, 7, or 8.");
        return;
    }

    for (count = 0; count <= NSC_NUM_DATA_BIT_CHOICES - 1; count ++)
    {
        if (data_bits == (int) nsc_data_bit_choices[count].num_bits)
        {                               
            nsc_serl->data_bits = nsc_data_bit_choices[count].num_bit_code;
            match_found = 1;
            break;
        }
    }

    if (!match_found)
        conf_error("Illegal Number of Data Bits.\n Please specify 5, 6, 7, or 8.");
}














/*
 *   Function:
 *
 *             void  nsc_get_serial_stop_bits (char *string,
 *                                             NSC_SERL_CONFIG_TYPE *nsc_serl);
 *
 *
 *   Description:
 *              
 *      This function scans for one of several possible # of stop bits
 *      (reports error if something else is there), then encodes this info
 *      into the stop_bits field of the nsc_serl structure.
 *
 *
 *   Parameters:    string              string to be scanned
 *                  nsc_serl            structure to be updated
 *
 *
 *   Return Values: none
 *
 */
void nsc_get_serial_stop_bits(string, nsc_serl)
char *string;
NSC_SERL_CONFIG_TYPE *nsc_serl;
{
    int count;
    int match_found=0;

    for (count = 0; count <= NSC_NUM_DATA_BIT_CHOICES - 1; count ++)
    {
        if (strcmp (string, 
                    nsc_stop_bit_choices[count].num_stop_bits_string)==0)
        {                               
            nsc_serl->stop_bits = 
                    nsc_stop_bit_choices[count].stop_bit_code;
            match_found = 1;
            break;
        }
    }

    if (!match_found)
        conf_error("Illegal Number of Stop Bits.\n Please specify 1, 1.5, or 2.");
}




















/*
 *   Function:
 *
 *             void  nsc_get_parity (char *string,
 *                                   NSC_SERL_CONFIG_TYPE *nsc_serl);
 *
 *
 *   Description:
 *              
 *      This function scans for one of several possible parity choices (Odd,
 *      even or None.  It reports error if something else is there.)
 *      It then encodes this info into the parity field of the 
 *      nsc_serl structure.
 *
 *
 *   Parameters:    string              string to be scanned
 *                  nsc_serl            structure to be updated
 *
 *
 *   Return Values: none
 *
 */
void nsc_get_parity(string, nsc_serl)
char *string;
NSC_SERL_CONFIG_TYPE *nsc_serl;
{
        int count;
        int match_found=0;

        nsc_upper_case (string);

        for (count = 0; count <= NUM_PARITY_CHOICES - 1; count ++)
        {
                if (strcmp (string, 
                            nsc_parity_choices[count].parity_string) == 0)
                {
                        match_found = TRUE;
                        nsc_serl->parity = 
                                nsc_parity_choices[count].parity_code;
                        break;
                }
        }


    if (!match_found)
        conf_error("Illegal Parity Specification.\n Please specify None, Odd or Even.");
}














/*
 *   Function:
 *
 *             void  nsc_get_ready_busy (char *string,
 *                                       NSC_SERL_CONFIG_TYPE *nsc_serl);
 *
 *
 *   Description:
 *              
 *      This function scans for one of several possible parity choices (Odd,
 *      even or None.  It reports error if something else is there.)
 *      It then encodes this info into the ready/busy field of the 
 *      nsc_serl structure.
 *
 *
 *   Parameters:    string              string to be scanned
 *                  nsc_serl            structure to be updated
 *
 *
 *   Return Values: none
 *
 */
void  nsc_get_ready_busy (string, nsc_serl)
char *string;
NSC_SERL_CONFIG_TYPE *nsc_serl;
{
        nsc_upper_case (string);

        if (strcmp (string, NSC_ENABLED_STRING) == 0)
                nsc_serl->ready_busy_enabled = TRUE;
        else if (strcmp (string, NSC_DISABLED_STRING) == 0)
                nsc_serl->ready_busy_enabled = FALSE;
        else
                conf_error("Illegal Ready/Busy Specification.\n Please specify Enabled or Disabled.");
        
}






















/*
 *   Function:
 *
 *             void  nsc_get_xon_xoff (char *string,
 *                                       NSC_SERL_CONFIG_TYPE *nsc_serl);
 *
 *
 *   Description:
 *              
 *      This function scans for one of several possible parity choices (Odd,
 *      even or None.  It reports error if something else is there.)
 *      It then encodes this info into the XOn/XOff field of the 
 *      nsc_serl structure.
 *
 *
 *   Parameters:    string              string to be scanned
 *                  nsc_serl            structure to be updated
 *
 *
 *   Return Values: none
 *
 */
void  nsc_get_xon_xoff(string, nsc_serl)
char *string;
NSC_SERL_CONFIG_TYPE *nsc_serl;
{
        nsc_upper_case (string);

        if (strcmp (string, NSC_ENABLED_STRING) == 0)
                nsc_serl->xon_xoff_enabled = TRUE;
        else if (strcmp (string, NSC_DISABLED_STRING) == 0)
                nsc_serl->xon_xoff_enabled = FALSE;
        else
                conf_error("Illegal Xon/Xoff Specification.\n Please specify Enabled or Disabled.");
        
}







int valid_router(router_addr, ip_addr, mask)
unsigned long router_addr;
unsigned long ip_addr;
unsigned long mask;

{
unsigned long	sub1, sub2;

#ifdef DEBUG
	printf("ip : %0x router ip : %0x mask : %0x\n", ip_addr, router_addr, mask );
#endif
	/* A router address of zero is OK, it means no default gateway */
	if (!router_addr)
		return(1);

	sub1 = router_addr & mask;
	sub2 = ip_addr & mask;

	return((sub1 == sub2));
}


main(argc, argv)
int    argc;
char **argv;
{
   FILE *in;
   int  count=0, c;
   int  status;
   long ret_addr;
   char *cp;

   char spare[80], data[80], dummy[80], token[6];
   tcp_cnfg_t conf_structure;
   char serial_in_or_out = NSC_SERIAL_IN;
   NSC_SERL_CONFIG_TYPE nsc_serl_struct;
   char nsc_serl_string[NSC_NUM_SERL_ITEMS][40];
     
   struct hostent *hostp;
   unsigned long ip_addr;
     
    
    
    
   protocol_stack_config_t protocols;

   protocols.protocol = PROTOCOL_NOVEL_NETWARE | 
                 PROTOCOL_APPLETALK |
                 PROTOCOL_DEC_LAT |
		 PROTOCOL_TCP_IP;


    nsc_serl_struct.baud_rate=          (long) BAUD_9600; 
    nsc_serl_struct.data_bits=          DATA_BITS_5;
    nsc_serl_struct.stop_bits=          STOP_BITS_2;
    nsc_serl_struct.parity=             NO_PARITY; 
    nsc_serl_struct.filler1=            0;
    nsc_serl_struct.ready_busy_enabled= TRUE; 
    nsc_serl_struct.xon_xoff_enabled=   TRUE;



   /* Open the configuration file. */ 
   if ((in=fopen(filename, "r")) == NULL)  conf_error("Unable to open file.");

   /******************************************/
   /* Parse the command line options.        */
   /******************************************/

   /* Exit if no command line option */
   if (argc < 2)
   {
	printf(usage);
	exit(0);
   }

   while ((c = getopt(argc, argv, VALIDOPTIONS)) != EOF)
   {
        switch(c)
        {
                case '?':
				printf(usage);
				exit(0);
                case 'h':
				printf(usage);
				exit(0);
                default:
				printf(usage);
				exit(0);
        }
   }
   if (hostp = gethostbyname(argv[1]))
   {
	ip_addr = *(long *)hostp->h_addr;
   }
   else
   {
	if ((ip_addr = inet_addr(argv[1])) == -1L)
	{
		conf_error("Invalid IP Address!");
		exit(1);
	}
   }

   /******************************************/
   /* Read in the information from the file. */
   /******************************************/

   /* This loops around to remove all the comments at the top of the file */
   while(!feof(in))
   {
	fgets(spare, sizeof(spare), in);
	if ((is_in(spare, '#') != 1))
	{
		sscanf( spare, "%s\t%s", dummy, data );
		break;
	}
	count++;
   }


   /******************************************/
   /* Now we start to read in the parameters */
   /* and check to see of they are in the    */
   /* correct format.                        */
   /******************************************/

   /* This reads in the Internet Address and checks it for validity.      */

   if ((status=(valid_address(data, &ret_addr))) <= 0)
			conf_error("\n\tReading in Internet Address from configuration file.\n\tEnsure that the parameter is in the format A.B.C.D where A, B, C, and D \n\tare all numerical values between and inclusive of 0-254."); 

   if (!ret_addr)
	conf_error("\n\tIP Address is NULL. Please enter a N.I.C. IP address in\n\tthe nicprint.conf file\n");

   conf_structure.ip_addr=ret_addr;

#ifdef DEBUG
	printf("%s\n", data);
#endif /* DEBUG */

/**************************************************************************/
/**************************************************************************/

   /* This reads in the IP Mask and checks it for validity.      */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s\n", spare, data); 
	if ((is_in(spare, '#') != 1))
	{

#ifdef DEBUG
	printf("valid_netmask returns : %0x\n", valid_netmask(data) );
#endif /* DEBUG */
  
		if ((conf_structure.subnet_mask=valid_netmask(data)) == 0) 
			conf_error("\n\tReading in Subnet Mask from configuration file.\n\tEnsure that the parameter is in the format 255.B.C.D where B, C, and D \n\tare all numerical values between and inclusive of 0-255.");
		break;
	}
   }

#ifdef DEBUG
	printf("%s\n", data);
	printf("Structure: %0x\n", conf_structure.subnet_mask);
#endif /* DEBUG */
  
 
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Forwarding requirement and checks it for validity.  */
   
   
   while (!feof(in))
   { 
	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			conf_structure.forw_bdcast=1;
			break;
		}
 		if ((data[0] == 'N') || (data[0] == 'n'))
		{
			conf_structure.forw_bdcast=0;
			break;
		}
		else
			conf_error("\n\tReading in Forwarding from configuration file.\n\tEnsure that the parameteris one of the following :\n\n\t\t\t\tY\tor\ty\n\t\t\tor\tN\tor\tn.\n"); 
	}
   }

#ifdef DEBUG
	printf("%c\n", data[0] );
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

	/* Setting default value for trailers as NONE. */

	conf_structure.trailers=0;

   /* This reads in the Trailers requirement and checks it for validity.  */

#ifdef NOT_DEFINED

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			conf_structure.trailers=1;
			break;
		}
		if ((data[0] == 'N') || (data[0] == 'n'))
		{
			conf_structure.trailers=0;
			break;
		}
		else
			conf_error("Reading in Trailers from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tY\tor\ty\n\t\t\tor\tN\tor\tn.\n"); 
	}
   }
#endif

#ifdef DEBUG
	printf("%c\n", data[0]);
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Base Port Number and checks it for validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((valid_number(data)) == 1)
		{
			conf_structure.base_port_number=atoi(data);
			break;
		}
		else
			conf_error("Reading in Base Port Number from configuration file.\nEnsure that the parameter contains only numerical [0-9] values in it's entry.\n"); 
	}
   }

#ifdef DEBUG
	printf("%s\n", data);
#endif /* DEBUG */

/**************************************************************************/
/**************************************************************************/

   /* This reads in the Keepalive requirement and checks it for validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			conf_structure.keepalive=1;
			break;
		}
		if ((data[0] == 'N') || (data[0] == 'n'))
		{
			conf_structure.keepalive=0;
			break;
		}
		else
			conf_error("Reading in Keepalive from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tY\tor\ty\n\t\t\tor\tN\tor\tn.\n"); 
	}
   }

#ifdef DEBUG
	printf("%c\n", data[0]);
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Max. Transmission Unit and checks it for validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((valid_number(data)) == 1)
		{
			conf_structure.mtu=atoi(data);
			break;
		}
		else
			conf_error("Reading in Max. Transmission Unit from configuration file.\nEnsure that the parameter contains only numerical [0-9] values in it's entry.\n"); 
	}
   }

#ifdef DEBUG
	printf("%s\n", data);
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Broadcast requirement and checks it for validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data);
	if ((is_in(spare, '#') != 1))
	{
		if (data[0]=='1')
		{
			conf_structure.broadcast = 1;
			break;
		}
		if (data[0]=='0')
		{
			conf_structure.broadcast = 0;
			break;
		}
		else
			conf_error("Reading in Broadcast from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\t1\n\t\t\tor\n\t\t\t\t0.\n"); 
	}
   }

			conf_structure.broadcast = 0;
#ifdef DEBUG
	printf("broadcast = %d\n", conf_structure.broadcast);
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

   /* This reads in the System Description and checks it for validity.  */
 
   while (!feof(in))
   { 
   	fscanf(in, "%s\t", spare);
   	fgets(data, sizeof(data), in); 
	if ((is_in(data, '#') != 1))
	{

#ifdef DEBUG
	printf("snmpid_desc returns : %s/%d\n", data, sizeof(data));
#endif /* DEBUG */
  
		if (sizeof(data) > 80) 
			conf_error("Reading in SNMP System Description from configuration file.\nEnsure that the parameter has less than 80 characters (including whitespace).\n"); 
	strcpy(conf_structure.sysDescr, data);
		break;
	}
   }

#ifdef DEBUG
	printf("System Description = %s\n", conf_structure.sysDescr);
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s\n", spare, data); 
	if ((is_in(spare, '#') != 1))
	{

#ifdef DEBUG
	printf("valid_snmpid returns : %d\n", valid_snmp(data) );
#endif /* DEBUG */
  
		if ((valid_snmp(data)) != 1) 
			conf_error("Reading in SNMP System ID from configuration file.\nEnsure that the parameter is in the format : \n\n\t\t\t\t\tXX.XX.XX.XX .\n"); 
		strcpy(conf_structure.sysObjectID, data);
		break;
	}
   }

#ifdef DEBUG
	printf("%s\n", data);
#endif /* DEBUG */
  
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Ethernet or 802.3  requirement and checks it for 
      validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			conf_structure.ether=1;
			break;
		}
 		if ((data[0] =='N') || (data[0] == 'n'))
		{
			conf_structure.ether=0;
			break;
		}
		else
			conf_error("Reading in Ethernet from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tY\tor\ty\n\t\t\tor\tN\tor\tn.\n"); 
                
	}
   }

#ifdef DEBUG
	printf("%c\n", data[0]);
#endif /* DEBUG */

/**************************************************************************/
/**************************************************************************/

   /* This reads in the default router IP Address and checks it for validity.      */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s\n", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
   		if ((status=(valid_address(data, &ret_addr))) < 0)
			conf_error("\n\tReading in default router IP Address from configuration file.\n\tEnsure that the parameter is in the format A.B.C.D where A, B, \n\tC, and D are all numerical values between and inclusive of 0-254."); 


		if (!valid_router(ret_addr, conf_structure.ip_addr, conf_structure.subnet_mask))
			conf_error("\n\tInvalid default router IP address. The router specified \n\tmust be on the same subnet as the N.I.C. .\n"); 

		conf_structure.default_router_ip=ret_addr;

#ifdef DEBUG
		printf("%s\n", data);
#endif /* DEBUG */

		break;
	}
   }


/**************************************************************************/
/**************************************************************************/

   /* This reads in the Serial Port In or Out information and checks for
      validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
                nsc_upper_case (data);
                if (strcmp(data, IN_STRING) == 0)
                {
                        serial_in_or_out = NSC_SERIAL_IN;
                        break;
                }
                else if (strcmp(data, OUT_STRING) == 0)
                {
                        serial_in_or_out = NSC_SERIAL_OUT;
                        break;
                }
                else
                {
			conf_error("Invalid serial port direction.  Please specify IN or OUT.");
                }
	}
   }

/**************************************************************************
 * 
 * The following loop is used to pull all the serial port information in the  
 * ASCII configuration file into the NSC_SERL_CONFIG_TYPE structure 
 * nsc_serl_struct, in preparation for its trip over TCP/IP, to the box
 * unit.
 *
 **************************************************************************/

   for (count=0; count<=NSC_NUM_SERL_ITEMS-1; count++)
   {
      while (!feof(in))
      { 
      	fscanf(in, "%s\t%s", spare, nsc_serl_string[count]); 

	if ((is_in(spare, '#') != 1))
	{

           get_nsc_serl_item[count] (nsc_serl_string[count], &nsc_serl_struct);
           break;

        }

      }

#ifdef DEBUG
	printf("%s\n", nsc_serl_string);
#endif /* DEBUG */
   }
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Novell Protocol  requirement and checks it for 
      validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			protocols.protocol |= PROTOCOL_NOVEL_NETWARE;
			break;
		}
 		if ((data[0] =='N') || (data[0] == 'n'))
		{
			protocols.protocol &= ~PROTOCOL_NOVEL_NETWARE;
			break;
		}
		else
			conf_error("Reading in Novell from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tY\tor\ty\n\t\tor");
	}
   }

#ifdef DEBUG
	printf("%c\n", data[0]);
#endif /* DEBUG */
/**************************************************************************/
/**************************************************************************/

   /* This reads in the Apple Protocol  requirement and checks it for 
      validity.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			protocols.protocol |= PROTOCOL_APPLETALK;
			break;
		}
 		if ((data[0] =='N') || (data[0] == 'n'))
		{
			protocols.protocol &= ~PROTOCOL_APPLETALK;
			break;
		}
		else
			conf_error("Reading in Apple from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tY\tor\ty\n\t\t\tor");
	}
   }

#ifdef DEBUG
	printf("%c\n", data[0]);
#endif /* DEBUG */
/**************************************************************************/
/**************************************************************************/

   /* This reads in the banner on/off indicator and checks it.  */

   while (!feof(in))
   {
   	fscanf(in, "%s\t%s", spare, token); 
	if ((is_in(spare, '#') != 1))
	nsc_upper_case(token);
	{
		if (!strcmp(token, "ON"))
		{
			conf_structure.banners=1;
			break;
		}
		else if (!strcmp(token, "OFF"))
		{
			conf_structure.banners=0;
			break;
		}
		else
			conf_error("Reading in banner on/off from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tON or OFF\t\n"); 
	}
   }

#ifdef DEBUG
	printf("banner : %s\n", token);
#endif /* DEBUG */

/**************************************************************************/
/**************************************************************************/

   /* This reads in the port pdl masks and checks them.  */

   count=0;
   strcpy(spare,"");
   strcpy(token,"");
   conf_structure.printer_pdl_mask[0] = 0;
   conf_structure.printer_pdl_mask[1] = 0;
   conf_structure.printer_pdl_mask[2] = 0;
   conf_structure.printer_pdl_mask[3] = 0;

   while ((!feof(in)) && (count < 4))
   {
     fgets(data, sizeof(data), in);

     sscanf(data, "%s\t", spare); 

     cp = data + strlen(spare) + 1;
     if ((is_in(spare, '#') != 1) && strlen(cp))
       {
	 while (*cp == ' ')
	   cp++;

	 while ((cp - data) < strlen(data))
	   {
	     status = sscanf(cp, "%s", token);

	     cp += strlen(token) + 1;

	     while (*cp == ' ')
	       cp++;
		nsc_upper_case(token);
	     if (!strcmp(token, "PCL"))
	       {
		 conf_structure.printer_pdl_mask[count] |= 1;
	       }
	     else if (!strcmp(token, "PS"))
	       {
		 conf_structure.printer_pdl_mask[count] |= 2;
	       }
	     else if (!strcmp(token, "ASCII"))
	       {
		 conf_structure.printer_pdl_mask[count] |= 4;
	       }
	     else if (!strcmp(token, "OTHER"))
	       {
		 conf_structure.printer_pdl_mask[count] |= 8;
	       }
	     else if (!strcmp(token, "UNUSED"))
	       {
		 conf_structure.printer_pdl_mask[count] = 0;
	       }
	     else
	       conf_error("Reading in pdl mask from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tON or OFF\t\n"); 

	   }

	 count++;
       }
   }

#ifdef DEBUG
	printf("masks : %d %d %d %d\n", conf_structure.printer_pdl_mask[0],
	       conf_structure.printer_pdl_mask[1], 
	       conf_structure.printer_pdl_mask[2],
	       conf_structure.printer_pdl_mask[3]);
#endif /* DEBUG */

/**************************************************************************/
/**************************************************************************/
/**************************************************************************/

   /* This reads in the LAT y/n indicator and checks it.  */

   while (!feof(in))
   { 
   	fscanf(in, "%s\t%s", spare, data); 
	if ((is_in(spare, '#') != 1))
	{
		if ((data[0] == 'Y') || (data[0] == 'y'))
		{
			protocols.protocol |= PROTOCOL_DEC_LAT;
			break;
		}
 		if ((data[0] =='N') || (data[0] == 'n'))
		{
			protocols.protocol &= ~PROTOCOL_DEC_LAT;
			break;
		}
		else
			conf_error("Reading in LAT from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tY\tor\ty\n\t\tor\n\n\t\t\t\tN\tor\tn\n\t\tor");
	}
   }

#ifdef DEBUG
	printf("%c\n", data[0]);
#endif /* DEBUG */

/**************************************************************************/
/**************************************************************************/

   /* This reads in the powerup status on/off indicator and checks it.  */

   while (!feof(in))
   {
   	fscanf(in, "%s\t%s", spare, token); 
	if ((is_in(spare, '#') != 1))
	nsc_upper_case(token);
	{
		if (!strcmp(token, "ON"))
		{
			conf_structure.pwrstatus=1;
			break;
		}
		else if (!strcmp(token, "OFF"))
		{
			conf_structure.pwrstatus=0;
			break;
		}
		else
			conf_error("Reading in powerup status on/off from configuration file.\nEnsure that the parameter\nis one of the following :\n\n\t\t\t\tON or OFF\t\n"); 
	}
   }

#ifdef DEBUG
	printf("pwrstatus : %s\n", token);
#endif /* DEBUG */


/**************************************************************************/

   /* Close the configuration file. */ 
   if ((fclose(in)) != 0)  conf_error("Unable to close file.");


   /***********************************/
   /* Send these details contained in */
   /* the structure "conf_structure"  */
   /* to the function                 */
   /* 'send_configuration_details'    */
   /* which will transfer it across   */
   /* the network.                    */
   /***********************************/

   send_configuration_details ( &conf_structure, serial_in_or_out,
                                &nsc_serl_struct, 
#ifdef DEBUG
                                nsc_serl_string,
#endif 
                                &protocols,
                                ip_addr, argv[1] );

} /* End of main() */

/*##########################################################################
  ###                                                                    ###
  ### Function Name: conf_error                                          ###
  ### Parameters   : message - This is a pointer to a string that will   ###
  ###                          be displayed upon error.                  ###
  ### Purpose      : This function is called once an error has occurred  ###
  ###                and it prints out a message indicatind what this    ###
  ###                error is before exiting the program.                ###
  ###                                                                    ###
  ##########################################################################*/
void conf_error( message )
char *message;
{

   printf("\nError configuring N.I.C. : \n\t%s\n", message);
   exit(1);
}

/*##########################################################################
  ###                                                                    ###
  ### Function Name: is_in                                               ###
  ### Parameters   : check_string - pointer to a string which is passed  ###
  ###                               through to be checked for the        ###
  ###                               occurrence of a character.           ###
  ###                                                                    ###
  ###                character    - pointer to the character that is to  ###
  ###                               to be searched for.                  ###
  ### Purpose      : This function will search 'check_string' for        ###
  ###                'character'. If it is found then the value '1' is   ###
  ###                returned, otherwise '0' is returned to indicate     ###
  ###                that it is not present.                             ###
  ###                                                                    ###
  ##########################################################################*/
int is_in( check_string, character)
char *check_string;
char  character;
{
   while(*check_string)
	if (*check_string == character)
		return 1;
	else
		check_string++;
		
	return 0;
}

/*##########################################################################
  ###                                                                    ###
  ### Function Name: valid_netmask.                                      ###
  ### Parameters   : netmask - a pointer to a string that is checked to  ###
  ###                          to see if it is a valid netmask.          ###
  ### Purpose      : This function takes in a string 'netmask' and       ###
  ###                determines if it is a valid netmask identifier.     ###
  ###                                                                    ###
  ##########################################################################*/

unsigned long valid_netmask( inetaddr ) 
char *inetaddr;
{
   char string_array1[4], string_array2[4], string_array3[4], string_array4[4],
        *string1=string_array1, *string2=string_array2,
        *string3=string_array3, *string4=string_array4;
   int  which_string=1;
   long inet_mask;

#ifdef DEBUG
	printf("valid netmask passed : %s.\n", inetaddr);
#endif /* DEBUG */

   memset(string_array1, 0, 4);
   memset(string_array2, 0, 4);
   memset(string_array3, 0, 4);
   memset(string_array4, 0, 4);
   do 
   {
#ifdef DEBUG
	printf("%c-", *inetaddr);
#endif /* DEBUG */

	switch (which_string)
	{
	  case 1: if (*inetaddr != '.') 
			*string1++ = *inetaddr;
		  else
			{
				*string1=0;
				which_string++;
			}
		  break;
	  case 2: if (*inetaddr != '.') 
			*string2++ = *inetaddr;
		  else
			{
				*string2=0;
				which_string++;
			}
		  break;
	  case 3: if (*inetaddr != '.') 
			*string3++ = *inetaddr;
		  else
			{
				*string3=0;
				which_string++;
			}
		  break;
	  case 4: if (*inetaddr != '.')
			 *string4++ = *inetaddr;
		else
			{
				*string4=0;
				which_string++;
			}
		  break;
	}
#ifdef DEBUG
	printf("Character = %c, String No. = %d\n", *inetaddr, which_string);
#endif /*DEBUG */
   } while (*inetaddr++);

#ifdef DEBUG
	printf("\nstring1: %s\n", string_array1);
	printf("string2: %s\n", string_array2);
	printf("string3: %s\n", string_array3);
	printf("string4: %s\n", string_array4);
#endif /* DEBUG */

   /**************************************************************/
   /* Check to see if each of the strings contain a valid number */
   /**************************************************************/

   if ((valid_number(string_array1))==0)
   	return 0;
   if ((valid_number(string_array2))==0)
   	return 0;
   if ((valid_number(string_array3))==0)
   	return 0;
   if ((valid_number(string_array4))==0)
   	return 0;

   /***************************************************************/
   /* Check to see if each of the numbers is in the correct range */
   /*          That is between 0 and 255 (inclusive)              */
   /***************************************************************/

   if ( (atoi(string_array1) == 255) )
   {
	inet_mask = atoi(string_array1) << 24;
   }
   else return (0);
   if ( (atoi(string_array2) >= 0) && (atoi(string_array2) <= 255) )
   {
	inet_mask |= atoi(string_array2) << 16;
   }
   else return (0);
   if ( (atoi(string_array3) >= 0) && (atoi(string_array3) <= 255) )
   {
	inet_mask |= atoi(string_array3) << 8;
   }
   else return (0);
   if ( (atoi(string_array4) >= 0) && (atoi(string_array4) <= 255) )
   {
	inet_mask |= atoi(string_array4);
   }
   else return(0);

#ifdef DEBUG
	printf("inet_mask : 0x%x\n", inet_mask);
#endif /* DEBUG */

   /***************************************************************/
   /* Pass the long value of the Internet Address back out.       */
   /***************************************************************/


   return (inet_mask);
}


/*##########################################################################
  ###                                                                    ###
  ### Function Name: valid_number                                        ###
  ### Parameters   : check_string - pointer to a string which is passed  ###
  ###                               through to be checked to see if it   ###
  ###                               is a valid number.                   ###
  ###                                                                    ###
  ### Purpose      : This function will check if each character in the   ###
  ###                string 'check_string' is a valid numerical value    ###
  ###                by using the function " isdigit. If it is a valid   ###
  ###                number then '1' is returned, else '0' is returned.  ###
  ###                                                                    ###
  ##########################################################################*/
int valid_number( check_string )
char *check_string;
{

#ifdef DEBUG
	printf("valid_number passed : %s\n", check_string);
#endif /* DEBUG */

   while(*check_string)
#ifdef DEBUG
   {
	printf("digit getting checked : %c\n", *check_string);
#endif /* DEBUG */

	if (isdigit(*check_string)==0)
		return 0;
	else
		check_string++;
#ifdef DEBUG
   }
#endif /* DEBUG */
		
	return 1;
}

/*##########################################################################
  ###                                                                    ###
  ### Function Name: valid_address.                                      ###
  ### Parameters   : inetaddr    - a string containing the internet that ###
  ###                              has to be checked for validity.       ###
  ###                ret_addr    - a long in which the converted string  ###
  ###                              will be returned.                     ###
  ### Purpose      : To ensure that the dotted notation format of the    ###
  ###                Internet address is indeed a correct one.           ###
  ###                                                                    ###
  ##########################################################################*/

long valid_address( inetaddr, ret_addr ) 
char *inetaddr;
int  *ret_addr;

{
   char string_array1[4], string_array2[4], string_array3[4], string_array4[4],
        *string1=string_array1, *string2=string_array2,
        *string3=string_array3, *string4=string_array4;
   int  which_string=1;

#ifdef DEBUG
	printf("valid address passed : %s.\n", inetaddr);
#endif /* DEBUG */

   memset(string_array1, 0, 4);
   memset(string_array2, 0, 4);
   memset(string_array3, 0, 4);
   memset(string_array4, 0, 4);
   do 
   {
#ifdef DEBUG
	printf("%c-", *inetaddr);
#endif /* DEBUG */

	switch (which_string)
	{
	  case 1: if (*inetaddr != '.') 
			*string1++ = *inetaddr;
		  else
			{
				*string1=0;
				which_string++;
			}
		  break;
	  case 2: if (*inetaddr != '.') 
			*string2++ = *inetaddr;
		  else
			{
				*string2=0;
				which_string++;
			}
		  break;
	  case 3: if (*inetaddr != '.') 
			*string3++ = *inetaddr;
		  else
			{
				*string3=0;
				which_string++;
			}
		  break;
	  case 4: if (*inetaddr != '.')
			 *string4++ = *inetaddr;
		else
			{
				*string4=0;
				which_string++;
			}
		  break;
	}
#ifdef DEBUG
	printf("Character = %c, String No. = %d\n", *inetaddr, which_string);
#endif /*DEBUG */
   } while (*inetaddr++);

#ifdef DEBUG
	printf("\nstring1: %s\n", string_array1);
	printf("string2: %s\n", string_array2);
	printf("string3: %s\n", string_array3);
	printf("string4: %s\n", string_array4);
#endif /* DEBUG */

   /**************************************************************/
   /* Check to see if each of the strings contain a valid number */
   /**************************************************************/

   if ((valid_number(string_array1))==0)
   	return(-1);
   if ((valid_number(string_array2))==0)
   	return(-1);
   if ((valid_number(string_array3))==0)
   	return(-1);
   if ((valid_number(string_array4))==0)
   	return(-1);

   /***************************************************************/
   /* Check to see if each of the numbers is in the correct range */
   /*          That is between 0 and 254 (inclusive)              */
   /***************************************************************/

   if ( (atoi(string_array1) >= 0) && (atoi(string_array1) < 255) )
   {
	*ret_addr = atoi(string_array1) << 24;
   }
   else return(-1);
   if ( (atoi(string_array2) >= 0) && (atoi(string_array2) < 255) )
   {
	*ret_addr |= atoi(string_array2) << 16;
   }
   else return(-1);
   if ( (atoi(string_array3) >= 0) && (atoi(string_array3) < 255) )
   {
	*ret_addr |= atoi(string_array3) << 8;
   }
   else return(-1);
   if ( (atoi(string_array4) >= 0) && (atoi(string_array4) < 255) )
   {
	*ret_addr |= atoi(string_array4);
   }
   else return(-1);

#ifdef DEBUG
	printf("ret_addr : 0x%x\n", *ret_addr);
#endif /* DEBUG */

   /***************************************************************/
   /* Pass the long value of the Internet Address back out.       */
   /***************************************************************/


   return (1);
}

/*##########################################################################
  ###                                                                    ###
  ### Function Name: clean_up_SysDesc                                    ###
  ### Parameters   : snmp_sys_decs - a string containing the System      ###
  ###                                Description and some unwanted info. ###
  ### Purpose      : The unwanted info has to be removed from the string ###
  ###                and just the System Description remaining.          ###
  ###                                                                    ###
  ##########################################################################*/
char* clean_up_SysDesc(snmp_sys_desc)
char *snmp_sys_desc;
{
  char *new_string;

  /* Step through the string one char a time until you come across */
  /* the tab character                                             */
  while (*snmp_sys_desc != '\t')
  {
#ifdef DEBUG
	printf("SNMP desc cleaning ....... %c\n", *snmp_sys_desc);
#endif
	*snmp_sys_desc++;
  }
  /* Step past the tab character and then transfer this string into */
  /* the new string. Return this string back out of the function.   */
  *snmp_sys_desc++;
  new_string = snmp_sys_desc;

  /* Overwrite newline character */
  while (*snmp_sys_desc != '\n')
  {
	snmp_sys_desc++;
  }

  *snmp_sys_desc = '\0';

#ifdef DEBUG
	printf("SNMP new string ....... %s\n", new_string);
#endif
  return(new_string);
}

/*##########################################################################
  ###                                                                    ###
  ### Function Name: valid_snmp.                                         ###
  ### Parameters   : snmpaddr    - a string containing the snmp address  ###
  ###                              that has to be checked for validity.  ###
  ### Purpose      : The SNMP System ID has to be converted from a       ###
  ###                "1.3.6." string into a straight number series.      ###
  ###                                                                    ###
  ##########################################################################*/
int valid_snmp( snmpaddr )
char *snmpaddr;
{

#ifdef DEBUG
	printf("\nSNMP checking : %s.\n", snmpaddr);
#endif /* DEBUG */
   if (sizeof(snmpaddr) > 33)
	return (-1);
   else
	return (1);
}

/*##########################################################################
  ###                                                                    ###
  ### Function Name: send_configuration_details.                         ###
  ###                                                                    ###
  ### Parameters   : tcp_details - a structure containing all the TCP    ###
  ###                               info to be sent across the network.  ###
  ###                serial_in_out- Indicates direction of serial port   ###
  ###                serial_details-a structure containing all the serial###
  ###                               info to be sent across the network.  ###
  ###                protocols     -indicates those protocols which are  ###
  ###                               active                               ###
  ###                connect_to -   the IP address of the NetSprint box. ###
  ###                remote_host -  name of the remote host.             ###
  ### Purpose      : To send across the configuration details to the     ###
  ###                NetSprint box, which will alter the NVRAM on the    ###
  ###                NetSprint box.                                      ###
  ###                                                                    ###
  ##########################################################################*/
void send_configuration_details( tcp_details, serial_in_out, serial_details,
#ifdef DEBUG
                                nsc_serl_string,
#endif 
                                protocols, connect_to, remote_host )
tcp_cnfg_t *tcp_details;
char serial_in_out;
NSC_SERL_CONFIG_TYPE *serial_details;
#ifdef DEBUG
char nsc_serl_string[NSC_NUM_SERL_ITEMS][40];
#endif 
protocol_stack_config_t *protocols;
unsigned long connect_to;
char *remote_host;
{
   int sockfd;
   char ack;
   extern int errno;
   struct sockaddr_in serv_addr;

#ifdef DEBUG
	printf("\n***********************************************************");
	printf("\n***\t\t\t\t\t\t\t***");
	printf("\n***\tIP Address:\t\t0x%x\t\t***", tcp_details->ip_addr);
	printf("\n***\tNetmask:\t\t0x%x\t\t***", tcp_details->subnet_mask);
	printf("\n***\tForwarding:\t\t%d\t\t\t***", tcp_details->forw_bdcast);
	printf("\n***\tTrailers:\t\t%d\t\t\t***", tcp_details->trailers);
	printf("\n***\tBase Port Number:\t%d\t\t\t***", tcp_details->base_port_number);
	printf("\n***\tKeepalive:\t\t%d\t\t\t***", tcp_details->keepalive);
	printf("\n***\tMax. Transmission Unit:\t%d\t\t\t***", tcp_details->mtu);
	printf("\n***\tBroadcast:\t\t%d\t\t\t***", tcp_details->broadcast);
	printf("\n***\tSystem Description:\t%s\t\t\t***", tcp_details->sysDescr);
	printf("\n***\tSystem Identification:\t%s\t\t\t***", tcp_details->sysObjectID);
	printf("\n***\tEthernet:\t\t%d\t\t\t***", tcp_details->ether);
	printf("\n***\t\t\t\t\t\t\t***");
	printf("\n***\t\t\t\t\t\t\t***");
        printf("\n***\tSerial Port Direction:\t%s\t\t\t***",(serial_in_out == NSC_SERIAL_IN) ? IN_STRING : OUT_STRING);
        printf("\n***\tSerial Port Baud Rate:\t%s\t\t\t***",nsc_serl_string[0]);
        printf("\n***\tSerial Port Data Bits:\t%s\t\t\t***",nsc_serl_string[1]);
        printf("\n***\tSerial Port Stop Bits:\t%s\t\t\t***",nsc_serl_string[2]);
        printf("\n***\tSerial Port Parity:\t%s\t\t\t***",nsc_serl_string[3]);
        printf("\n***\tSerial Port Ready/Busy:\t%s\t\t\t***",nsc_serl_string[4]);
        printf("\n***\tSerial Port XOn/Xoff:\t%s\t\t\t***",nsc_serl_string[5]);
        printf("\n***\tNovell-Protocol:\t%c\t\t\t***",protocols->protocol & PROTOCOL_NOVEL_NETWARE ?
                                                               'Y':'N');
        printf("\n***\tApple-Protocol:\t\t%c\t\t\t***",protocols->protocol & PROTOCOL_APPLETALK ?
                                                            'Y':'N');
	printf("\n***\t\t\t\t\t\t\t***");
	printf("\n***\t\t\t\t\t\t\t***");
	printf("\n***********************************************************\n");

#endif /* DEBUG */
   /******************************************************************/
   /***                                                            ***/
   /***  Fill in the "serv_addr Structure with the address of the  ***/
   /***  server that we want to connect with.                      ***/
   /***                                                            ***/
   /******************************************************************/

   memset((char *)&serv_addr, 0, sizeof(serv_addr));
   serv_addr.sin_family 	= AF_INET;
   serv_addr.sin_addr.s_addr 	= connect_to;
   serv_addr.sin_port 		= htons(SERV_TCP_PORT);

   /******************************************************************/
   /***                                                            ***/
   /***  Open a TCP socket ( of Internet Stream type )             ***/
   /***                                                            ***/
   /******************************************************************/

   if ( (sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0 )
	conf_error("\nclient: can't open stream socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Socket created with fd=%d\n", sockfd);
#endif /* DEBUG */

   /******************************************************************/
   /***                                                            ***/
   /***  Connect to the server.                                    ***/
   /***                                                            ***/
   /******************************************************************/

#ifdef DEBUG
        printf("Nsconfig: Connecting to %d\n", htonl(connect_to));
#endif /* DEBUG */

   if (connect(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0)
	conf_error("\nclient: can't connect with the server.\n");

   /******************************************************************/
   /***                                                            ***/
   /***  Send across the information.                              ***/
   /***                                                            ***/
   /******************************************************************/

   /* Write Internet Address to Server */
   tcp_details->ip_addr = htonl(tcp_details->ip_addr);
   if((write( sockfd, (char *)&tcp_details->ip_addr, sizeof(tcp_details->ip_addr))) < 0)
	conf_error("Nsconfig: Couldn't write Internet Address to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Internet Address=%x to socket.\n", tcp_details->ip_addr);
#endif /* DEBUG */

   /* Write Subnet Mask to Server */
   tcp_details->subnet_mask = htonl(tcp_details->subnet_mask);
   if((write( sockfd, (char *)&tcp_details->subnet_mask, sizeof(tcp_details->subnet_mask))) < 0)
	conf_error("Nsconfig: Couldn't write Subnet Mask to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Subnet Mask=%x to socket.\n", tcp_details->subnet_mask);
#endif /* DEBUG */

   /* Write Forward Broadcast to Server */
   if((write( sockfd, &tcp_details->forw_bdcast, sizeof(tcp_details->forw_bdcast))) < 0)
	conf_error("Nsconfig: Couldn't write Forward Broadcast to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Forward Broadcast=%d to socket.\n", tcp_details->forw_bdcast);
#endif /* DEBUG */

   /* Write Trailers to Server */
   if((write( sockfd, &tcp_details->trailers, sizeof(tcp_details->trailers))) < 0)
	conf_error("Nsconfig: Couldn't write Trailers to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Trailers=%d to socket.\n", tcp_details->trailers);
#endif /* DEBUG */

   /* Write Base Port Number to Server */
   tcp_details->base_port_number = htonl(tcp_details->base_port_number);
   if((write( sockfd, &tcp_details->base_port_number, sizeof(tcp_details->base_port_number))) < 0)
	conf_error("Nsconfig: Couldn't write Base Port Number to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Base Port Number=%d to socket.\n", tcp_details->base_port_number);
#endif /* DEBUG */

   /* Write Keepalive to Server */
   if((write( sockfd, &tcp_details->keepalive, sizeof(tcp_details->keepalive))) < 0)
	conf_error("Nsconfig: Couldn't write Keepalive to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Keepalive=%d to socket.\n", tcp_details->keepalive);
#endif /* DEBUG */

   /* Write Max. Transmission Unit to Server */
   tcp_details->mtu = htons(tcp_details->mtu);
   if((write( sockfd, &tcp_details->mtu, sizeof(tcp_details->mtu))) < 0)
	conf_error("Nsconfig: Couldn't write Max. Transmission Unit to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Max. Transmission Unit=%d to socket.\n", tcp_details->mtu);
#endif /* DEBUG */

   /* Write Broadcast to Server */
   if((write( sockfd, &tcp_details->broadcast, sizeof(tcp_details->broadcast))) < 0)
	conf_error("Nsconfig: Couldn't write Broadcast to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Broadcast=%d to socket.\n", tcp_details->broadcast);
#endif /* DEBUG */

   /* Write System Description to Server */
   if((write( sockfd, tcp_details->sysDescr, sizeof(tcp_details->sysDescr))) < 0)
	conf_error("Nsconfig: Couldn't write System Description to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing System Description=%s to socket.\n", tcp_details->sysDescr);
#endif /* DEBUG */

   /* Write System Object ID to Server */
   if((write( sockfd, tcp_details->sysObjectID, sizeof(tcp_details->sysObjectID))) < 0)
	conf_error("Nsconfig: Couldn't write System Object ID to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing System Object ID=%s to socket.\n", tcp_details->sysObjectID);
#endif /* DEBUG */

   /* Write Ethernet to Server */
   if((write( sockfd, &tcp_details->ether, sizeof(tcp_details->ether))) < 0)
	conf_error("Nsconfig: Couldn't write Ethernet to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Ethernet=%d to socket.\n", tcp_details->ether);
#endif /* DEBUG */

   /* Write default router Internet Address to Server */
   tcp_details->default_router_ip = htonl(tcp_details->default_router_ip);
   if((write( sockfd, (char *)&tcp_details->default_router_ip, sizeof(tcp_details->default_router_ip))) < 0)
	conf_error("Nsconfig: Couldn't write default router IP Address to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing default router IP Address=%x to socket.\n", tcp_details->default_router_ip);
#endif /* DEBUG */

   /* Write Direction of Serial Port to Server */
   if((write( sockfd, &serial_in_out, 
             sizeof(serial_in_out))) < 0)
	conf_error("Nsconfig: Couldn't write <Serial in or out> to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Serial In Or Out =%d to socket.\n", 
                serial_in_out);
#endif /* DEBUG */

   /* Byte serial fields */
   serial_details->baud_rate = htonl (serial_details->baud_rate);
   serial_details->ready_busy_enabled = 
                        htons (serial_details->ready_busy_enabled);
   serial_details->xon_xoff_enabled = 
                        htons (serial_details->xon_xoff_enabled);

   /* Write Serial Port Configuration */
   if(write(sockfd, serial_details, sizeof(*serial_details)) < 0)
	conf_error("Nsconfig: Couldn't write Serial Config Info to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Serial Config Info to socket.\n");
#endif /* DEBUG */                          








   /* Byte swap protocols */
   protocols->protocol = htonl (protocols->protocol);



   /* Write Protocols */
   if(write( sockfd, protocols, sizeof (*protocols)) < 0)
	conf_error("Nsconfig: Couldn't write Protocols Info to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing Protocols Info=0x%lx to socket.\n", 
               protocols->protocol);
#endif /* DEBUG */

   /******************************************************************/
   /***                                                            ***/
   /***  Listen for an Acknowledgement. (1==OK, 0==FAIL)           ***/
   /***  If OK, send the LPD banner on/off and printer capability  ***/
   /***  info.                                                     ***/
   /***                                                            ***/
   /******************************************************************/

   
   /* Read Acknowledgment from Server */
   if ((read( sockfd, &ack, sizeof(ack))) < 0)
	conf_error("Nsconfig: Couldn't read Acknowledgment from socket.\n");

   /* Notify the user of Success (ack = 1) or Failure (ack = 0). */
   if (ack == 0) /* Failure */
	conf_error("Nsconfig: Error on server end.\n");
   if (ack == 1) /* Success */
     {
       printf("\nNsconfig(%s) : Basic TCP/IP configuration succeeded.\n", 
               remote_host);

       printf("\nNsconfig: Sending LPD parameters...\n\n", 
               remote_host);

       if((write( sockfd, &ack, sizeof(ack))) < 0)
	 conf_error("Nsconfig: Couldn't write banner on/off to socket.\n");

     }
   else
	conf_error("Nsconfig: Error reading Acknowledgment from server.\n");

   /* Write banners to Server */
   if((write( sockfd, &tcp_details->banners, sizeof(tcp_details->banners))) < 0)
	conf_error("Nsconfig: Couldn't write banner on/off to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing banner on/off=%d to socket.\n", tcp_details->banners);
#endif /* DEBUG */

   /* Write port printer masks to Server */
   if((write( sockfd, &tcp_details->printer_pdl_mask[0], sizeof(tcp_details->printer_pdl_mask[0]))) < 0)
	conf_error("Nsconfig: Couldn't write port 1 pdl mask to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing port 1 pdl mask=%d to socket.\n", tcp_details->printer_pdl_mask[0]);
#endif /* DEBUG */

   /* Write port printer masks to Server */
   if((write( sockfd, &tcp_details->printer_pdl_mask[1], sizeof(tcp_details->printer_pdl_mask[1]))) < 0)
	conf_error("Nsconfig: Couldn't write port 2 pdl mask to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing port 2 pdl mask=%d to socket.\n", tcp_details->printer_pdl_mask[1]);
#endif /* DEBUG */

   /* Write port printer masks to Server */
   if((write( sockfd, &tcp_details->printer_pdl_mask[2], sizeof(tcp_details->printer_pdl_mask[2]))) < 0)
	conf_error("Nsconfig: Couldn't write port 3 pdl mask to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing port 3 pdl mask=%d to socket.\n", tcp_details->printer_pdl_mask[2]);
#endif /* DEBUG */

   /* Write port printer masks to Server */
   if((write( sockfd, &tcp_details->printer_pdl_mask[3], sizeof(tcp_details->printer_pdl_mask[3]))) < 0)
	conf_error("Nsconfig: Couldn't write port 4 pdl mask to socket.\n");

#ifdef DEBUG
        printf("Nsconfig: Writing port 4 pdl mask=%d to socket.\n", tcp_details->printer_pdl_mask[0]);
#endif /* DEBUG */

   /* Write PowerUP Status information to Server */
   if((write( sockfd, &tcp_details->pwrstatus, sizeof(tcp_details->pwrstatus))) < 0)
	conf_error("Nsconfig: Couldn't write PowerUP Status information to socket.\n", tcp_details->pwrstatus);

#ifdef DEBUG
        printf("Nsconfig: Writing PowerUP Status information %s  to socket.\n", tcp_details->pwrstatus);
#endif /* DEBUG */

   /******************************************************************/
   /***                                                            ***/
   /***  Listen for an Acknowledgement. (1==OK, 0==FAIL)           ***/
   /***  Close the connection down.                                ***/
   /***                                                            ***/
   /******************************************************************/

   
   /* Read Acknowledgment from Server */
   if ((read( sockfd, &ack, sizeof(ack))) < 0)
   {
	printf("Nsconfig: LPD is not supported by this Rom Release.\n");
	printf("\nNsconfig: Successfully reconfigured %s.\n", remote_host);
        close(sockfd);
        exit(0);
   }

   /* Notify the user of Success (ack = 1) or Failure (ack = 0). */
   if (ack == 0) /* Failure */
	conf_error("Nsconfig: Error on server end.\n");
   if (ack == 1) /* Success */
	printf("Nsconfig: Successfully reconfigured %s.\n", remote_host);
   else
	conf_error("Nsconfig: Error reading Acknowledgment from server.\n");

#ifdef DEBUG
        printf("Nsconfig: Reading Acknowledgement=%c from socket.\n", ack);
#endif /* DEBUG */

   close(sockfd);
}



