/*
 * Copyright Digital Products, Inc.. All Rights Reserved. 
 *
 *     Module Name: recognize.c
 *            Date: 13-Jan-94
 *          Author: John Carlton
 *
 *    Entry Points:        recognize( char *scan_buf ) 
 *
 * Description.
 * =======================================================================
 *
 *  This source file contains functions which perform emulation sensing
 *  of print jobs. It is a simplified version of DPI's emulation sensing
 *  code that is delivered to allow customers to port the DPI TCP/IP host
 *  resident NETPrint 100/200 software.
 *
 *  You may need DPI's fully functional emulation sensing to meet your 
 *  postscript and PCL printing requirements. Please contact DPI technical 
 *  support for assistance.
 *
 */ 


#include <stdio.h>

#include <string.h>
#include <ctype.h> 
#include "recognize.h"     /* defines constants and types used */

#define SIZEOF_XMIT_BUF 1024

int recognize(/* char *scan_buf */);

int make_uppercase (/* char *src_buf, int buf_len, char *dest_buf */);

int recognize(scan_buf)
char scan_buf[];

{
int	query;

    /********************************/
    /*   is it a PostScript job?    */
    /********************************/

        query = is_it_postscript (scan_buf);

        if (query == TRUE)
        {
#ifdef DEBUG
            fprintf(stderr," It is PostScript!\n");
#endif

            return(POSTSCRIPT_JOB);
        }

    /********************************/
    /*       is it a PCL job?       */
    /********************************/

        query = is_it_pcl (scan_buf);

        if (query == TRUE)
        {
#ifdef DEBUG
            fprintf(stderr," It is PCL!\n");
#endif

            return(PCL_JOB);
        }

    /********************************/
    /*     is it an ASCII job?      */
    /********************************/

        query = is_it_ascii (scan_buf);

        if (query == TRUE)
        {
#ifdef DEBUG
            fprintf(stderr," It is ASCII!\n");
#endif

            return(ASCII_JOB);
        }

	return(UNKNOWN_JOB);
}



/****************************************************************/
/*  Check scan_buf to see if an ASCII  job->                */
/*                                                              */
/*  To qualify as an ASCII job:                                 */
/*      - must not be PostScript                                */
/*      - must not be PCL                                       */
/*      - must NOT contain Escape (1bh)                         */
/*                                                              */
/* returns: TRUE  if YES ASCII                                  */
/*          FALSE if NOT ASCII    (it is OTHER by default)      */
/****************************************************************/

int is_it_ascii(scan_buf)
char scan_buf[];
{
    char *str_ptr;


    /***************************/
    /*    is Escape in data?   */
    /***************************/

    str_ptr = strchr (scan_buf, ESCAPE);

    if (str_ptr != 0)           /* found escape char */
    {
        return(FALSE);          /* indicate is not ascii */
    }

    return(TRUE);               /* indicate it is ascii */
}




/****************************************************************/
/*  check scan_buf to see if an HP PCL job->                    */
/*                                                              */
/* return:  TRUE if PCL                                         */
/*          FALSE if NOT PCL                                    */
/****************************************************************/

int is_it_pcl(scan_buf)
char scan_buf[];
{
    char c1, c2;

    c1 = scan_buf[0];
    c2 = scan_buf[1];

    if (c1 == 27 && (c2 == '&' || c2 == '*' || c2 == '(' ||
	c2 == ')' || c2 == '%' || c2 == 'Y' || c2 == 'Z' ||
        c2 == 'E'))
	return(TRUE);

    return (FALSE);
}

/*
 *
 *  Function: int is_it_postscript ()
 *
 *  Description:
 *
 *      check a block of UPPERCASE chars to determine if it is PostScript data.
 *      To be declared as PostScript data must start with %!PS-ADOBE
 *
 *  Parameters:
 *
 *
 *  Return Values:
 *
 *      TRUE     - data block is postscript
 *      FALSE    - data block cannot be identified as postscript
 *
 */

int is_it_postscript (scan_buf)
char scan_buf[];
{
    int query;

    /**********************************************************/
    /*     do quick check for PS-ADOBE at start of file       */
    /**********************************************************/

    query = contains_adobe(scan_buf);

    if (query == TRUE)
    {
        return(TRUE);        /* indicate is is ps */
    }
}

/*
 *
 *  Function: ERROR contains_string
 *
 *  Description:
 *
 *      test if scan_buf contains a string
 *
 *  Parameters:
 *
 *      *string to look for
 *      depth = how deep into scan_buf to look
 *
 *  Return Values:
 *
 *      TRUE     - string was found
 *      FALSE    - string not found
 *
 */

int contains_string (src_buf, string, depth)
char *src_buf ;
char *string;
int depth;

{
    char *str_ptr;
    char char_save;

    /***********************************************************/
    /* temporarily make "depth" byte of src_buf a zero to */
    /*  make it look like a "depth" byte string                */
    /***********************************************************/

    char_save = *(src_buf+depth-1);

    *(src_buf+depth-1) = 0;


    /**********************************************************/
    /*    search upper_buf for occurrences of string          */
    /**********************************************************/

    str_ptr = strstr (src_buf, string);       /* srch for string */

    /***********************************************************/
    /*  restore "depth" byte of src_buf to non-zero value */
    /***********************************************************/

    *(src_buf+depth-1) = char_save;


    /***********************/
    /*  was string found?  */
    /***********************/

    if (str_ptr != 0)       /* found it !  */
    {
#ifdef DEBUG
        fprintf(stderr, " Found string\n");
#endif

        return(TRUE);       /* indicate string was found */
    }

    return(FALSE);          /* string not found */
}


/*
 *
 *  Function: ERROR contains_adobe
 *
 *  Description:
 *
 *      check a first 50 bytes of scan_buf for  %!PS-ADOBE
 *      to determine if it is PostScript job
 *
 *  Parameters:
 *
 *      none
 *
 *  Return Values:
 *
 *      SUCCESS     - data block is postscript
 *      FAILURE     - data block cannot be identified as postscript
 *
 */

int contains_adobe (scan_buf)
char scan_buf[];
{
    char *ps_adobe = "%!PS-ADOBE";
    int query;

    /**********************************************************/
    /*    search 1st 50 chars of upper_buf for PS-ADOBE dsc   */
    /**********************************************************/

    query = contains_string (scan_buf, ps_adobe, 50);

    /***********************/
    /*  was string found?  */
    /***********************/

    if (query == TRUE)                      /* found it !  */
    {
#ifdef DEBUG
        fprintf (stderr, " Found PS-Adobe string\n");
#endif

        return(TRUE);                       /* indicate is is ps */
    }

    return(FALSE);                          /* not PS-ADOBE */
}


/*
 *
 *  Function: ERROR make_uppercase_buffer
 *
 *  Description:
 *
 *      copies and converts to UPPERCASE a block of print data
 *
 *
 *  Parameters:
 *
 *      pointer to block of chars to copy
 *
 *  Return Values:
 *
 *      SUCCESS     - data block copied and converted
 *      FAILURE     - data block cannot be copied or converted
 *
 */

int make_uppercase (src_buf, buf_len, dest_buf)
char *src_buf;
int buf_len;
char *dest_buf;

{
    char *char_loc;
    char *buf_ptr;
    int result;
    int count;
    int i;

    /************************************************/
    /*   1.  copy the block of chars to UPPER_buf   */
    /************************************************/

    if (buf_len > SIZEOF_XMIT_BUF)
    {
        buf_len = SIZEOF_XMIT_BUF;   /*   process a max of 2K chars    */
    }

    memcpy (dest_buf, src_buf, buf_len);

    /**********************************************************************/
    /*  2.  make buf into one big string, by converting all 0's to 0xED's */
    /**********************************************************************/

    buf_ptr = dest_buf;        /* init search ptr */

    count = buf_len;       /* init search length */

    while (buf_ptr != 0)
    {

        char_loc = memchr (buf_ptr, 0, count);    /* get ptr to 1st 0 */
                                                  /* returns 0 if not found */
        if (char_loc == 0)      /* ptr zero if not found */
        {
            buf_ptr = 0;        /* all done, get out */
        }
        else                    /* we found a zero */
        {
            *char_loc = 0xed;           /*  change 0 to non-zero  */

            buf_ptr = char_loc++;       /* set new start of buffer ptr */

            count =                     /* init remainder length to search */
             (buf_len - (buf_ptr - dest_buf));

        }
    }

    /*****************************************************************/
    /*  3.  make last char in buffer a zero (string terminator)      */
    /*****************************************************************/

    dest_buf [buf_len-1] = 0;

    /*******************************************************************/
    /*  4.  make all chars UPPERCASE, to make string searches easier   */
    /*******************************************************************/

    for (i=0; i < strlen(dest_buf); i++)
    {
        result = toupper(dest_buf[i]);

        dest_buf[i] = result;   /* copy uppercase char */
    }

    return (SUCCESS);

}
