/*
 * 
 * (c) Copyright 1991 OPEN SOFTWARE FOUNDATION, INC.
 * (c) Copyright 1991 HEWLETT-PACKARD COMPANY
 * (c) Copyright 1991 DIGITAL EQUIPMENT CORPORATION
 * To anyone who acknowledges that this file is provided "AS IS"
 * without any express or implied warranty:
 *                 permission to use, copy, modify, and distribute this
 * file for any purpose is hereby granted without fee, provided that
 * the above copyright notices and this notice appears in all source
 * code copies, and that none of the names of Open Software
 * Foundation, Inc., Hewlett-Packard Company, or Digital Equipment
 * Corporation be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission.  Neither Open Software Foundation, Inc., Hewlett-
 * Packard Company, nor Digital Equipment Corporation makes any
 * representations about the suitability of this software for any
 * purpose.
 * 
 */
/*
**
**  NAME:
**
**      pthread_signal.c
**
**  FACILITY:
**
**      pthreads
**
**  ABSTRACT:
**
**  Defines helper functions for dealing with signal in a pthreads environment.
** 
*/

#include <pthread.h>
#include <signal.h>
#include <pthread_sig.h>
#include <signal.h>

/* -------------------------------------------------------------------- */

static void *async_signal_handler
    _PTDSIG_PROTO_((
        void *dummy
    ));

static pthread_t thread_to_cancel;
static sigset_t async_sigset;

/* -------------------------------------------------------------------- */

/* 
 * A S Y N C _ S I G N A L _ H A N D L E R
 * 
 * This async signal handler is already running on the correct thread
 * stack.   ALL async signals map to a "cancel".  A cancel unwind happens
 * only at well defined points, so we can't RAISE the exception; just
 * post the cancel.
 */
static void *async_signal_handler(dummy)
void *dummy;
{
    /*
     * Wait for and handle asynchronous signals.
     */
    while (1)
    {
        sigwait(&async_sigset);
        pthread_cancel(thread_to_cancel);
    }
}


/*
 * P T H R E A D _ S I G N A L _ T O _ C A N C E L _ N P
 *
 * Async signal handling consists of creating an exception package helper
 * thread.  This helper thread will sigwait() on all async signals of
 * interest and will convert specific asynchronous signals (defined in the
 * async signal handler) to exceptions.  The helper thread then posts the
 * cancel to the thread that initialized the exception package for 
 * 'handling'.
 */
void pthread_signal_to_cancel_np(sigset, thread)
sigset_t *sigset;
pthread_t thread;
{
    pthread_t helper_thread;

    /* 
     * The helper thread will need the thread id of the first thread
     * to initialize the exception package.  The thread that initialized
     * the exception package will receive a cancel when an asynchronous
     * signal is received.
     */
    thread_to_cancel = thread;
    async_sigset = *sigset;

    /* 
     * Create a 'helper thread' to catch aynchronous signals.
     */
    pthread_create(&helper_thread, pthread_attr_default, 
                   /* (pthread_func_t) */ async_signal_handler, 0);

    /* 
     * The 'helper thread' will never be joined so toss any pthread package
     * internal record of the thread away to conserve resources.
     */
    pthread_detach(&helper_thread);
}
