/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included with this distribution in  *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.batik.css.value;

import org.apache.batik.css.CSSOMReadOnlyStyleDeclaration;
import org.apache.batik.css.CSSOMReadOnlyValue;
import org.apache.batik.css.HiddenChildElementSupport;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.ViewCSS;

/**
 * This class provides a relative value resolver for the 'font-weight' CSS
 * property.
 *
 * @author <a href="mailto:stephane@hillion.org">Stephane Hillion</a>
 * @version $Id: FontWeightResolver.java,v 1.7 2001/04/01 16:08:44 hillion Exp $
 */
public class FontWeightResolver implements RelativeValueResolver {

    /**
     * The application context.
     */
    protected CommonCSSContext context;
    
    /**
     * Creates a new FontWeightResolver object.
     * @param ctx The application context.
     */
    public FontWeightResolver(CommonCSSContext ctx) {
	context = ctx;
    }

    /**
     * Whether the handled property is inherited or not.
     */
    public boolean isInheritedProperty() {
        return true;
    }

    /**
     * Returns the name of the handled property.
     */
    public String getPropertyName() {
        return ValueConstants.CSS_FONT_WEIGHT_PROPERTY;
    }

    /**
     * Returns the default value for the handled property.
     */
    public CSSOMReadOnlyValue getDefaultValue() {
        return new CSSOMReadOnlyValue(ValueConstants.NORMAL_VALUE);
    }

    /**
     * Resolves the given value if relative, and puts it in the given table.
     * @param element The element to which this value applies.
     * @param pseudoElement The pseudo element if one.
     * @param view The view CSS of the current document.
     * @param styleDeclaration The computed style declaration.
     * @param value The cascaded value.
     * @param priority The priority of the cascaded value.
     * @param origin The origin of the cascaded value.
     */
    public void resolveValue(Element element,
                             String pseudoElement,
                             ViewCSS view,
                             CSSOMReadOnlyStyleDeclaration styleDeclaration,
                             CSSOMReadOnlyValue value,
                             String priority,
                             int origin) {
        ImmutableValue im = value.getImmutableValue();

        boolean b = im == ValueConstants.BOLDER_VALUE;
        if (b || im == ValueConstants.LIGHTER_VALUE) {
            Element p = HiddenChildElementSupport.getParentElement(element);
            CSSOMReadOnlyValue val;
            if (p == null) {
                val = new CSSOMReadOnlyValue
                    ((b)
                     ? createFontWeight(context.getBolderFontWeight(400))
                     : createFontWeight(context.getLighterFontWeight(400)));
            } else {
                CSSOMReadOnlyStyleDeclaration sd;
                sd = (CSSOMReadOnlyStyleDeclaration)view.getComputedStyle
                    (p, null);
                CSSOMReadOnlyValue prop;
                prop = (CSSOMReadOnlyValue)sd.getPropertyCSSValue(getPropertyName());
                im = prop.getImmutableValue();
                if (im.getPrimitiveType() == CSSPrimitiveValue.CSS_IDENT) {
                    if (im == ValueConstants.NORMAL_VALUE) {
                        val = new CSSOMReadOnlyValue
                            ((b)
                             ? createFontWeight(context.getBolderFontWeight(400))
                             : createFontWeight(context.getLighterFontWeight(400)));
                    } else {
                        val = new CSSOMReadOnlyValue
                            ((b)
                             ? createFontWeight(context.getBolderFontWeight(700))
                             : createFontWeight(context.getLighterFontWeight(700)));
                    }
                } else {
                    float f = im.getFloatValue(CSSPrimitiveValue.CSS_NUMBER);
                    val = new CSSOMReadOnlyValue
                        ((b)
                         ? createFontWeight(context.getBolderFontWeight(f))
                         : createFontWeight(context.getLighterFontWeight(f)));
                }
            }
            styleDeclaration.setPropertyCSSValue(getPropertyName(),
                                                 val,
                                                 priority,
                                                 origin);
        }
    }

    /**
     * Creates a font weight value.
     */
    protected ImmutableValue createFontWeight(float f) {
        switch ((int)f) {
        case 100: return ValueConstants.NUMBER_100;
        case 200: return ValueConstants.NUMBER_200;
        case 300: return ValueConstants.NUMBER_300;
        case 400: return ValueConstants.NUMBER_400;
        case 500: return ValueConstants.NUMBER_500;
        case 600: return ValueConstants.NUMBER_600;
        case 700: return ValueConstants.NUMBER_700;
        case 800: return ValueConstants.NUMBER_800;
        case 900: return ValueConstants.NUMBER_900;
        default:
            throw new InternalError();
        }
    }
}
