/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included with this distribution in  *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.batik.dom.events;

import org.w3c.dom.Node;
import org.w3c.dom.views.AbstractView;
import org.w3c.dom.events.EventTarget;
import org.w3c.dom.events.MouseEvent;

/**
 * The MouseEvent class provides specific contextual information
 * associated with Mouse events.
 *
 * @author <a href="mailto:stephane@hillion.org">Stephane Hillion</a>
 */
public class DOMMouseEvent extends DOMUIEvent implements MouseEvent {

    private int screenX; 
    private int screenY; 
    private int clientX; 
    private int clientY; 
    private boolean ctrlKey; 
    private boolean altKey; 
    private boolean shiftKey; 
    private boolean metaKey; 
    private short button; 
    private EventTarget relatedTarget;

    /**
     * DOM: <code>screenX</code> indicates the horizontal coordinate
     * at which the event occurred relative to the origin of the
     * screen coordinate system.  
     */
    public int getScreenX() {
	return screenX;
    }

    /**
     * DOM: <code>screenY</code> indicates the vertical coordinate at
     * which the event occurred relative to the origin of the screen
     * coordinate system.  
     */
    public int getScreenY() {
	return screenY;
    }

    /**
     * DOM: <code>clientX</code> indicates the horizontal coordinate
     * at which the event occurred relative to the DOM
     * implementation's client area.  
     */
    public int getClientX() {
	return clientX;
    }

    /**
     * DOM: <code>clientY</code> indicates the vertical coordinate at
     * which the event occurred relative to the DOM implementation's
     * client area.  
     */
    public int getClientY() {
	return clientY;
    }

    /**
     * DOM: <code>ctrlKey</code> indicates whether the 'ctrl' key was
     * depressed during the firing of the event.  
     */
    public boolean getCtrlKey() {
	return ctrlKey;
    }

    /**
     * DOM: <code>shiftKey</code> indicates whether the 'shift' key
     * was depressed during the firing of the event. 
     */
    public boolean getShiftKey() {
	return shiftKey;
    }

    /**
     * DOM: <code>altKey</code> indicates whether the 'alt' key was
     * depressed during the firing of the event.  On some platforms
     * this key may map to an alternative key name.  
     */
    public boolean getAltKey() {
	return altKey;
    }

    /**
     * DOM: <code>metaKey</code> indicates whether the 'meta' key was
     * depressed during the firing of the event.  On some platforms
     * this key may map to an alternative key name. 
     */
    public boolean getMetaKey() {
	return metaKey;
    }

    /**
     * DOM: During mouse events caused by the depression or release of
     * a mouse button, <code>button</code> is used to indicate which
     * mouse button changed state.  The values for <code>button</code>
     * range from zero to indicate the left button of the mouse, one
     * to indicate the middle button if present, and two to indicate
     * the right button.  For mice configured for left handed use in
     * which the button actions are reversed the values are instead
     * read from right to left.  
     */
    public short getButton() {
	return button;
    }

    /**
     * DOM: Used to identify a secondary <code>EventTarget</code> related
     * to a UI 
     * event.  Currently this attribute is used with the mouseover event to 
     * indicate the <code>EventTarget</code> which the pointing device exited 
     * and with the mouseout event to indicate the  <code>EventTarget</code> 
     * which the pointing device entered.
     */
    public EventTarget getRelatedTarget() {
	return relatedTarget;
    }

    /**
     * DOM: The <code>initMouseEvent</code> method is used to
     * initialize the value of a <code>MouseEvent</code> created
     * through the <code>DocumentEvent</code> interface.  This method
     * may only be called before the <code>MouseEvent</code> has been
     * dispatched via the <code>dispatchEvent</code> method, though it
     * may be called multiple times during that phase if necessary.
     * If called multiple times, the final invocation takes
     * precedence.
     *
     * @param typeArg Specifies the event type.
     * @param canBubbleArg Specifies whether or not the event can bubble.
     * @param cancelableArg Specifies whether or not the event's default  
     *   action can be prevented.
     * @param viewArg Specifies the <code>Event</code>'s 
     *   <code>AbstractView</code>.
     * @param detailArg Specifies the <code>Event</code>'s mouse click count.
     * @param screenXArg Specifies the <code>Event</code>'s screen x coordinate
     * @param screenYArg Specifies the <code>Event</code>'s screen y coordinate
     * @param clientXArg Specifies the <code>Event</code>'s client x coordinate
     * @param clientYArg Specifies the <code>Event</code>'s client y coordinate
     * @param ctrlKeyArg Specifies whether or not control key was depressed 
     *   during the <code>Event</code>.
     * @param altKeyArg Specifies whether or not alt key was depressed during 
     *   the  <code>Event</code>.
     * @param shiftKeyArg Specifies whether or not shift key was depressed 
     *   during the <code>Event</code>.
     * @param metaKeyArg Specifies whether or not meta key was depressed 
     *   during the  <code>Event</code>.
     * @param buttonArg Specifies the <code>Event</code>'s mouse button.
     * @param relatedTargetArg Specifies the <code>Event</code>'s related
     *   <code>EventTarget</code>. 
     */
    public void initMouseEvent(String typeArg, 
			       boolean canBubbleArg, 
			       boolean cancelableArg, 
			       AbstractView viewArg, 
			       int detailArg, 
			       int screenXArg, 
			       int screenYArg, 
			       int clientXArg, 
			       int clientYArg, 
			       boolean ctrlKeyArg, 
			       boolean altKeyArg, 
			       boolean shiftKeyArg, 
			       boolean metaKeyArg, 
			       short buttonArg, 
			       EventTarget relatedTargetArg) {
	initUIEvent(typeArg, canBubbleArg, cancelableArg, 
		    viewArg, (int) detailArg);
	this.screenX = screenXArg;
	this.screenY = screenYArg; 
	this.clientX = clientXArg; 
	this.clientY = clientYArg; 
	this.ctrlKey = ctrlKeyArg; 
	this.altKey = altKeyArg; 
	this.shiftKey = shiftKeyArg; 
	this.metaKey = metaKeyArg; 
	this.button = buttonArg;  
	this.relatedTarget = relatedTargetArg;
    }
}
