/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included with this distribution in  *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.batik.extension.svg;

import java.util.Collections;
import java.util.Iterator;
import java.util.Vector;

import org.apache.batik.bridge.BridgeContext;
import org.apache.batik.bridge.BridgeExtension;

/**
 * This is a Service interface for classes that want to extend the
 * functionality of the Bridge, to support new tags in the rendering tree.
 */
public class BatikBridgeExtension implements BridgeExtension {

    /**
     * Return the priority of this Extension.  Extensions are
     * registered from lowest to highest priority.  So if for some
     * reason you need to come before/after another existing extension
     * make sure your priority is lower/higher than theirs.  
     */
    public float getPriority() { return 1f; }

    /**
     * This should return the list of extensions implemented
     * by this BridgeExtension.
     * @return An iterator containing strings one for each implemented
     *         extension.
     */
    public Iterator getImplementedExtensions() {
        String [] extensions = {
            "http://xml.apache.org/batik/ext/poly/1.0" ,
            "http://xml.apache.org/batik/ext/star/1.0" ,
            "http://xml.apache.org/batik/ext/histogramNormalization/1.0" ,
            "http://xml.apache.org/batik/ext/solidColor/1.0" ,
            "http://xml.apache.org/batik/ext/colorSwitch/1.0" ,
            "http://xml.apache.org/batik/ext/multi-image/1.0" ,
        };
        Vector v = new Vector(extensions.length);
        for (int i=0; i<extensions.length; i++) {
            v.add(extensions[i]);
        }
        return Collections.unmodifiableList(v).iterator();
    }

    /**
     * This should return the individual or company name responsible
     * for the this implementation of the extension.
     */
    public String getAuthor() {
        return "Thomas DeWeese";
    }

    /**
     * This should contain a contact address (usually an e-mail address).
     */
    public String getContactAddress() {
        return "deweese@apache.org";
    }

    /**
     * This should return a URL where information can be obtained on
     * this extension.
     */
    public String getURL() {
        return "http://xml.apache.org/batik";
    }

    /**
     * Human readable description of the extension.
     * Perhaps that should be a resource for internationalization?
     * (although I suppose it could be done internally)
     */
    public String getDescription() {
        return "Example extension to standard SVG shape tags";
    }

    /**
     * This method should update the BridgeContext with support
     * for the tags in this extension.  In some rare cases it may
     * be necessary to replace existing tag handlers, although this
     * is discouraged.
     *
     * @param ctx The BridgeContext instance to be updated
     */
    public void registerTags(BridgeContext ctx) {
        ctx.putBridge(new BatikRegularPolygonElementBridge());
        ctx.putBridge(new BatikStarElementBridge());
        ctx.putBridge(new BatikHistogramNormalizationElementBridge());
        ctx.putBridge(new BatikMultiImageElementBridge());
        ctx.putBridge(new SolidColorBridge());
        ctx.putBridge(new ColorSwitchBridge());
    }
}
