/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included with this distribution in  *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.batik.gvt.text;

import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.BasicStroke;
import java.awt.Font;
import java.awt.font.TextAttribute;
import java.awt.font.FontRenderContext;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.GeneralPath;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.text.AttributedCharacterIterator;
import java.text.CharacterIterator;

import java.util.HashSet;
import java.util.Set;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

import org.apache.batik.gvt.TextNode;
import org.apache.batik.gvt.text.TextSpanLayout;
import org.apache.batik.gvt.text.GVTAttributedCharacterIterator;
import org.apache.batik.gvt.text.TextHit;
import org.apache.batik.gvt.font.GVTGlyphVector;
import org.apache.batik.gvt.font.MultiGlyphVector;
import org.apache.batik.gvt.font.AltGlyphHandler;
import org.apache.batik.gvt.font.GVTLineMetrics;
import org.apache.batik.gvt.font.GVTFont;
import org.apache.batik.gvt.font.GVTGlyphMetrics;
import org.apache.batik.gvt.font.AWTGVTFont;

/**
 * Implementation of TextSpanLayout which uses java.awt.font.GlyphVector.
 * @see org.apache.batik.gvt.text.TextSpanLayout
 *
 * @author <a href="bill.haneman@ireland.sun.com>Bill Haneman</a>
 * @version $Id: GlyphLayout.java,v 1.36 2002/03/07 22:07:44 deweese Exp $
 */
public class GlyphLayout implements TextSpanLayout {

    private GVTGlyphVector gv;
    private GVTFont font;
    private GVTLineMetrics metrics;
    private AttributedCharacterIterator aci;
    private FontRenderContext frc;
    private Point2D advance;
    private Point2D offset;
    private float   xScale=1;
    private float   yScale=1;
    private Point2D prevCharPosition;
    private TextPath textPath;
    private Point2D textPathAdvance;
    private int []  charMap;
    private boolean vertical, adjSpacing=true;

    // When layoutApplied is false it means that the glyph positions
    // are different from where they would be if you did
    // doExplicitGlyphLayout().
    private boolean layoutApplied = false;
    // When spacingApplied is false it means that xScale, yScale and
    // kerning/wordspacing stuff haven't been applied. This can
    // be rectified by calling adjustTextSpacing().  Note that when
    // spacing is actually used layoutApplied will be cleared it
    // is not garunteed that applying text spacing will cause it to
    // be cleared (it will only be cleared if the glyphs move).
    private boolean spacingApplied = false;
    // When pathApplied is false it means that the text has not been
    // layed out on the associated text path (if any).  If there is an
    // associated text path then this will clear both layoutApplied
    // and spacing applied but neither will be touched if no text path
    // is present.
    private boolean pathApplied    = false;


    
    public static final AttributedCharacterIterator.Attribute 
        TEXT_COMPOUND_DELIMITER 
        = GVTAttributedCharacterIterator.TextAttribute.TEXT_COMPOUND_DELIMITER;

    private static final AttributedCharacterIterator.Attribute X
        = GVTAttributedCharacterIterator.TextAttribute.X;

    private static final AttributedCharacterIterator.Attribute Y
        = GVTAttributedCharacterIterator.TextAttribute.Y;

    private static final AttributedCharacterIterator.Attribute DX
        = GVTAttributedCharacterIterator.TextAttribute.DX;

    private static final AttributedCharacterIterator.Attribute DY
        = GVTAttributedCharacterIterator.TextAttribute.DY;

    private static final AttributedCharacterIterator.Attribute ROTATION
        = GVTAttributedCharacterIterator.TextAttribute.ROTATION;

    private static final AttributedCharacterIterator.Attribute BASELINE_SHIFT
        = GVTAttributedCharacterIterator.TextAttribute.BASELINE_SHIFT;

    private static final AttributedCharacterIterator.Attribute WRITING_MODE
        = GVTAttributedCharacterIterator.TextAttribute.WRITING_MODE;

    private static final Integer WRITING_MODE_TTB
        = GVTAttributedCharacterIterator.TextAttribute.WRITING_MODE_TTB;

    static Set runAtts = new HashSet();

    static {
        runAtts.add(X);
        runAtts.add(Y);
        runAtts.add(DX);
        runAtts.add(DY);
        runAtts.add(ROTATION);
        runAtts.add(BASELINE_SHIFT);
    }

    static Set szAtts = new HashSet();

    static {
        szAtts.add(TextAttribute.SIZE);
    }



    /**
     * Creates the specified text layout using the
     * specified AttributedCharacterIterator and rendering context.
     *
     * @param aci the AttributedCharacterIterator whose text is to
     *  be laid out
     * @param charMap Indicates how chars in aci map to original
     *                text char array.
     * @param offset The offset position of this text layout
     * @param frc the FontRenderContext to use for generating glyphs.
     */
    public GlyphLayout(AttributedCharacterIterator aci,
                       int [] charMap,
                       Point2D offset,
                       FontRenderContext frc) {

        this.aci = aci;
        this.frc = frc;
        this.offset = offset;
        this.font = getFont();
        this.charMap = charMap;

        this.metrics = font.getLineMetrics
            (aci, aci.getBeginIndex(), aci.getEndIndex(), frc);

        // create the glyph vector
        this.gv = null;
        this.aci.first();
        this.vertical = (aci.getAttribute(WRITING_MODE) == WRITING_MODE_TTB);
        this.textPath =  (TextPath) aci.getAttribute
            (GVTAttributedCharacterIterator.TextAttribute.TEXTPATH);

        AltGlyphHandler altGlyphHandler 
            = (AltGlyphHandler)this.aci.getAttribute
            (GVTAttributedCharacterIterator.TextAttribute.ALT_GLYPH_HANDLER);
        if (altGlyphHandler != null) {
            // this must be an altGlyph text element, try and create
            // the alternate glyphs
            this.gv = altGlyphHandler.createGlyphVector
                (frc, this.font.getSize(), this.aci);
        }
        if (this.gv == null) {
            // either not an altGlyph or the altGlyphHandler failed to
            // create a glyph vector
            this.gv = font.createGlyphVector(frc, this.aci);
        }
    }


    GVTGlyphVector getGlyphVector() {
        return this.gv;
    }


    /**
     * Returns the current text position at the beginning
     * of glyph layout, before the application of explicit
     * glyph positioning attributes.
     */
    public Point2D getOffset() {
        return offset;
    }

    /**
     * Sets the scaling factor to use for string.  if ajdSpacing is
     * true then only the spacing between glyphs will be adjusted
     * otherwise the glyphs and the spaces between them will be
     * adjusted.  Only the scale factor in the progression direction
     * is used (x for horizontal text, y for vertical text
     * ).
     * @param xScale Scale factor to apply in X direction.
     * @param yScale Scale factor to apply in Y direction.
     * @param adjSpacing True if only spaces should be adjusted.  
     */
    public void setScale(float xScale, float yScale, boolean adjSpacing) {
        // Fix the off axis scale factor.
        if (vertical) xScale = 1;
        else          yScale = 1;

        if ((xScale != this.xScale) ||
            (yScale != this.yScale) ||
            (adjSpacing != this.adjSpacing)) {
            this.xScale = xScale;
            this.yScale = yScale;
            this.adjSpacing = adjSpacing;

            // We don't affect layoutApplied directly...
            // System.out.println("layoutApplied: " + layoutApplied);

            // However if we did path layout or spacing it's all junk now...
            spacingApplied = false;
            pathApplied = false;
        }
    }

    /**
     * Sets the text position used for the implicit origin
     * of glyph layout. Ignored if multiple explicit glyph
     * positioning attributes are present in ACI
     * (e.g. if the aci has multiple X or Y values).
     */
    public void setOffset(Point2D offset) {

        if ((offset.getX() != this.offset.getX()) ||
            (offset.getY() != this.offset.getY())) {
            if (layoutApplied) {
                // Already layed out need to shift glyph positions to
                // account for new offset.
                float dx = (float)(offset.getX()-this.offset.getX());
                float dy = (float)(offset.getY()-this.offset.getY());
                int numGlyphs = gv.getNumGlyphs();

                float [] gp = gv.getGlyphPositions(0, numGlyphs+1, null);
                for (int i=0; i<=numGlyphs; i++) {
                    gv.setGlyphPosition(i, new Point2D.Float(gp[2*i]+dx,
                                                             gp[2*i+1]+dy));
                }
            }

            // When not layed out (or after updating) just set the new
            // offset this will be factored in for any future layout
            // operations.
            this.offset = offset;

            // We don't affect layoutApplied or spacingApplied since
            // they both work off the offset value.

            // However if we did path layout it's all junk now...
            pathApplied = false;
        }
    }

    public GVTGlyphMetrics getGlyphMetrics(int glyphIndex) {
        return gv.getGlyphMetrics(glyphIndex);
    }

    /**
     * Returns true if the advance direction of this text is vertical.
     */
    public boolean isVertical() {
        return vertical;
    }

    /**
     * Returns true if this layout in on a text path.
     */
    public boolean isOnATextPath() {
        return (textPath != null);
    }


    /**
     * Returns the number of glyphs in this layout.
     */
    public int getGlyphCount() {
        return gv.getNumGlyphs();
    }


    /**
     * Returns the number of chars represented by the glyphs within the
     * specified range.
     *
     * @param startGlyphIndex The index of the first glyph in the range.
     * @param endGlyphIndex The index of the last glyph in the range.
     *
     * @return The number of chars.
     */
    public int getCharacterCount(int startGlyphIndex, int endGlyphIndex) {
        return gv.getCharacterCount(startGlyphIndex, endGlyphIndex);
    }

    /**
     * Returns true if the text direction in this layout is from left to right.
     */
    public boolean isLeftToRight() {
        aci.first();
        int bidiLevel = 
            ((Integer)aci.getAttribute
             (GVTAttributedCharacterIterator.TextAttribute.BIDI_LEVEL))
            .intValue();

        // Check if low bit is set if not then we are left to right
        // (even bidi level).
        return ((bidiLevel&0x01) == 0);
    }


    /**
     * This method makes certain that the layout has been 
     * completed at this point (much of the layout is done lazily).
     */
    private final void syncLayout() {
        if (!pathApplied) {
            // System.out.println("Doing Path Layout: " + this);
            doPathLayout();
        }
    }

    /**
     * Paints the text layout using the
     * specified Graphics2D and rendering context.
     * @param g2d the Graphics2D to use
     * @param context The current render context
     */
    public void draw(Graphics2D g2d) {
        syncLayout();
        gv.draw(g2d, aci);
    }

    /**
     * Returns the current text position at the completion
     * of glyph layout.
     */
    public Point2D getAdvance2D() {
        syncLayout();
        return advance;
    }


    /**
     * Returns the outline of the completed glyph layout.
     */
    public Shape getOutline() {
        syncLayout();

        return gv.getOutline();
    }

    /**
     * Returns the outline of the specified decorations on the glyphs,
     * @param decorationType an integer indicating the type(s) of decorations
     *     included in this shape.  May be the result of "OR-ing" several
     *     values together:
     * e.g. <tt>DECORATION_UNDERLINE | DECORATION_STRIKETHROUGH</tt>
     */
    public Shape getDecorationOutline(int decorationType) {
        syncLayout();

        Shape g = new GeneralPath();
        if ((decorationType & DECORATION_UNDERLINE) != 0) {
             ((GeneralPath) g).append(getUnderlineShape(), false);
        }
        if ((decorationType & DECORATION_STRIKETHROUGH) != 0) {
             ((GeneralPath) g).append(getStrikethroughShape(), false);
        }
        if ((decorationType & DECORATION_OVERLINE) != 0) {
             ((GeneralPath) g).append(getOverlineShape(), false);
        }
        return g;
    }

    /**
     * Returns the rectangular bounds of the completed glyph layout.
     */
    public Rectangle2D getBounds() {
        syncLayout();

        return gv.getVisualBounds();
    }

    /**
     * Returns the rectangular bounds of the completed glyph layout,
     * inclusive of "decoration" (underline, overline, etc.)
     */
    public Rectangle2D getDecoratedBounds() {
        return getBounds().createUnion
            (getDecorationOutline(DECORATION_ALL).getBounds2D());
    }

    /**
     * Returns the position to used when drawing a text run after this one.
     * It takes into account the text path layout if there is one.
     */
    public Point2D getTextPathAdvance() {
        syncLayout();
        if (textPath != null) {
            return textPathAdvance;
        } else {
            return getAdvance2D();
        }
    }


    /**
     * Returns the index of the glyph that has the specified char index.
     *
     * @param charIndex The original index of the character in the text node's
     * text string.
     * @return The index of the matching glyph in this layout's glyph vector,
     *         or -1 if a matching glyph could not be found.
     */
    public int getGlyphIndex(int charIndex) {
        int numGlyphs = getGlyphCount();
        int j=0;
        for (int i = 0; i < numGlyphs; i++) {
            int count = getCharacterCount(i, i);
            for (int n=0; n<count; n++) {
                int glyphCharIndex = charMap[j++];
                if (charIndex == glyphCharIndex)
                    return i;
                if (j >= charMap.length)
                    return -1;
            }
        }
        return -1;
    }

   /**
     * Returns a Shape which encloses the currently selected glyphs
     * as specified by the character indices.
     *
     * @param beginCharIndex the index of the first char in the
     * contiguous selection.
     * @param endCharIndex the index of the last char in the
     * contiguous selection.
     * @return The highlight shape or null if the spacified char range
     * does not overlap with the chars in this layout.  */
    public Shape getHighlightShape(int beginCharIndex, int endCharIndex) {
        syncLayout();

        if (beginCharIndex > endCharIndex) {
            int temp = beginCharIndex;
            beginCharIndex = endCharIndex;
            endCharIndex = temp;
        }
        GeneralPath shape = null;
        int start       = aci.getBeginIndex();
        int numGlyphs = getGlyphCount();

        boolean glyphOrientationAuto = isGlyphOrientationAuto();
        int glyphOrientationAngle = 90;
        if (!glyphOrientationAuto) {
            glyphOrientationAngle = getGlyphOrientationAngle();
        }
        Point2D.Float [] topPts = new Point2D.Float[2*numGlyphs];
        Point2D.Float [] botPts = new Point2D.Float[2*numGlyphs];

        int ptIdx = 0;

        int currentChar = start;
        for (int i = 0; i < numGlyphs; i++) {
            char ch = aci.setIndex(currentChar);
            int glyphCharIndex = charMap[currentChar-start];
            if ((glyphCharIndex >= beginCharIndex) &&
                (glyphCharIndex <= endCharIndex)) {
                Shape gbounds = gv.getGlyphLogicalBounds(i);
                if (gbounds != null) {
                    // We got something...
                    if (shape == null)
                        shape = new GeneralPath();

                    // We are pretty dumb here we assume that we always
                    // get back polygons with four sides to them if
                    // isn't met we are SOL.
                    float [] pts = new float[6];
                    int count = 0;
                    int type = -1;

                    PathIterator pi = gbounds.getPathIterator(null);
                    Point2D.Float firstPt = null;
                    if (isVertical()) {
                        if (glyphOrientationAuto) {
                            if (isLatinChar(ch))
                                glyphOrientationAngle = 90;
                            else
                                glyphOrientationAngle = 0;
                        }
                    }

                    while (!pi.isDone()) {
                        type = pi.currentSegment(pts);
                        if ((type == PathIterator.SEG_MOVETO) ||
                            (type == PathIterator.SEG_LINETO)) {
                            // LINETO or MOVETO
                            if (count > 4) break; // too many lines...
                            if (count == 4) {
                                // make sure we are just closing it..
                                if ((firstPt == null)     ||
                                    (firstPt.x != pts[0]) ||
                                    (firstPt.y != pts[1]))
                                    break;
                            } else {
                                Point2D.Float pt;
                                pt = new Point2D.Float(pts[0], pts[1]);
                                if (count == 0) firstPt = pt;
                                // Use sides of  rectangle...
                                switch (count) {
                                case 0: botPts[ptIdx]   = pt; break;
                                case 1: topPts[ptIdx]   = pt; break;
                                case 2: topPts[ptIdx+1] = pt; break;
                                case 3: botPts[ptIdx+1] = pt; break;
                                }
                            }
                        } else if (type == PathIterator.SEG_CLOSE) {
                                // Close in the wrong spot?
                            if ((count < 4) || (count > 5)) break;
                        } else {
                            // QUADTO or CUBETO
                            break;
                        }

                        count++;
                        pi.next();
                    }
                    if (pi.isDone()) {
                        // Sucessfully Expressed as a quadralateral...
                        if ((botPts[ptIdx]!=null) &&
                            ((topPts[ptIdx].x != topPts[ptIdx+1].x) ||
                             (topPts[ptIdx].y != topPts[ptIdx+1].y)))
                            // box isn't empty so use it's points...
                            ptIdx += 2;
                    } else {
                        // System.out.println("Type: " + type +
                        //                    " count: " + count);
                        // Wasn't a quadralateral so just add it don't try
                        // and merge it...
                        addPtsToPath(shape, topPts, botPts, ptIdx);
                        ptIdx = 0;
                        shape.append(gbounds, false);
                    }
                }
            }
            currentChar += getCharacterCount(i, i);
        }
        addPtsToPath(shape, topPts, botPts, ptIdx);

        return shape;
    }

    public static final float eps = 0.00001f;

    public static int makeConvexHull(Point2D.Float [] pts, int numPts) {
        // Sort the Pts in X...
        Point2D.Float tmp;
        // System.out.print("Sorting...");
        for (int i=1; i<numPts; i++) {
            // Simple bubble sort (numPts should be small so shouldn't
            // be too bad.).
            if ((pts[i].x < pts[i-1].x) ||
                ((pts[i].x == pts[i-1].x) && (pts[i].y < pts[i-1].y))) {
                tmp = pts[i];
                pts[i] = pts[i-1];
                pts[i-1] = tmp;
                i=0;
                continue;
            }
        }

        // System.out.println("Sorted");

        Point2D.Float pt0 = pts[0];
        Point2D.Float pt1 = pts[numPts-1];
        Point2D.Float dxdy = new Point2D.Float(pt1.x-pt0.x, pt1.y-pt0.y);
        float soln, c = dxdy.y*pt0.x-dxdy.x*pt0.y;

        Point2D.Float [] topList = new Point2D.Float[numPts];
        Point2D.Float [] botList = new Point2D.Float[numPts];
        botList[0] = topList[0] = pts[0];
        int nTopPts=1;
        int nBotPts=1;
        for (int i=1; i<numPts-1; i++) {
            Point2D.Float pt = pts[i];
            soln = dxdy.x*pt.y-dxdy.y*pt.x+c;
            if (soln < 0) {
                // Below line goes into bot pt list...
                while (nBotPts >= 2) {
                    pt0 = botList[nBotPts-2];
                    pt1 = botList[nBotPts-1];
                    float dx = pt1.x-pt0.x;
                    float dy = pt1.y-pt0.y;
                    float c0 = dy*pt0.x-dx*pt0.y;
                    soln = dx*pt.y-dy*pt.x+c0;
                    if (soln > eps) // Left turn add and we are done..
                        break;
                    if (soln > -eps) {
                        // On line take lowest Y of two and keep going
                        if (pt1.y < pt.y) pt = pt1;
                        nBotPts--;
                        break;
                    }
                    // right turn drop prev pt;
                    nBotPts--;
                }
                botList[nBotPts++] = pt;
            } else {
                // Above line goes into top pt list...
                while (nTopPts >= 2) {
                    pt0 = topList[nTopPts-2];
                    pt1 = topList[nTopPts-1];
                    float dx = pt1.x-pt0.x;
                    float dy = pt1.y-pt0.y;
                    float c0 = dy*pt0.x-dx*pt0.y;
                    soln = dx*pt.y-dy*pt.x+c0;
                    if (soln < -eps) // Right turn add and check next point.
                        break;
                    if (soln < eps) {
                        // On line take greatest Y of two and keep going
                        if (pt1.y > pt.y) pt = pt1;
                        nTopPts--;
                        break;
                    }
                    // left turn drop prev pt;
                    nTopPts--;
                }
                topList[nTopPts++] = pt;
            }
        }

        // Check last point in both sets...
        Point2D.Float pt = pts[numPts-1];
        while (nBotPts >= 2) {
            pt0 = botList[nBotPts-2];
            pt1 = botList[nBotPts-1];
            float dx = pt1.x-pt0.x;
            float dy = pt1.y-pt0.y;
            float c0 = dy*pt0.x-dx*pt0.y;
            soln = dx*pt.y-dy*pt.x+c0;
            if (soln > eps)
                // Left turn add and we are done..
                break;
            if (soln > -eps) {
                // On line take lowest Y of two and keep going
                if (pt1.y >= pt.y) nBotPts--;
                break;
            }
            // right turn drop prev pt;
            nBotPts--;
        }

        while (nTopPts >= 2) {
            pt0 = topList[nTopPts-2];
            pt1 = topList[nTopPts-1];
            float dx = pt1.x-pt0.x;
            float dy = pt1.y-pt0.y;
            float c0 = dy*pt0.x-dx*pt0.y;
            soln = dx*pt.y-dy*pt.x+c0;
            if (soln < -eps)
                // Right turn done...
                break;
            if (soln < eps) {
                // On line take lowest Y of two and keep going
                if (pt1.y <= pt.y) nTopPts--;
                break;
            }
            // left turn drop prev pt;
            nTopPts--;
        }

        int i=0;
        for (; i<nTopPts; i++)
            pts[i] = topList[i];

        // We always include the 'last' point as it is always on convex hull.
        pts[i++] = pts[numPts-1];

        // don't include botList[0] since it is the same as topList[0].
        for (int n=nBotPts-1; n>0; n--, i++)
            pts[i] = botList[n];

        // System.out.println("CHull has " + i + " pts");
        return i;
    }

    public static void addPtsToPath(GeneralPath shape,
                                     Point2D.Float [] topPts,
                                     Point2D.Float [] botPts,
                                     int numPts) {
        if (numPts < 2) return;
        if (numPts == 2) {
            shape.moveTo(topPts[0].x, topPts[0].y);
            shape.lineTo(topPts[1].x, topPts[1].y);
            shape.lineTo(botPts[1].x, botPts[1].y);
            shape.lineTo(botPts[0].x, botPts[0].y);
            shape.lineTo(topPts[0].x, topPts[0].y);
            return;
        }

        // Here we 'connect the dots' the best way we know how...
        // What I do is construct a convex hull between adjacent
        // character boxes, then I union that into the shape.  this
        // does a good job of bridging between adjacent characters,
        // but still closely tracking to text boxes.  The use of the
        // Area class is fairly heavy weight but it seems to keep up
        // in this instanace (probably because all the shapes are very
        // simple polygons).
        Point2D.Float [] boxes = new Point2D.Float[8];
        Point2D.Float [] chull = new Point2D.Float[8];
        boxes[4] = topPts[0];
        boxes[5] = topPts[1];
        boxes[6] = botPts[1];
        boxes[7] = botPts[0];
        Area []areas = new Area[numPts/2];
        int nAreas =0;
        for (int i=2; i<numPts; i+=2) {
            boxes[0] = boxes[4];
            boxes[1] = boxes[5];
            boxes[2] = boxes[6];
            boxes[3] = boxes[7];
            boxes[4] = topPts[i];
            boxes[5] = topPts[i+1];
            boxes[6] = botPts[i+1];
            boxes[7] = botPts[i];

            float delta,sz,dist;
            delta  = boxes[2].x-boxes[0].x;
            dist   = delta*delta;
            delta  = boxes[2].y-boxes[0].y;
            dist  += delta*delta;
            sz     = (float)Math.sqrt(dist);

            delta  = boxes[6].x-boxes[4].x;
            dist   = delta*delta;
            delta  = boxes[6].y-boxes[4].y;
            dist  += delta*delta;
            sz    += (float)Math.sqrt(dist);

            delta = ((boxes[0].x+boxes[1].x+boxes[2].x+boxes[3].x)-
                     (boxes[4].x+boxes[5].x+boxes[6].x+boxes[7].x))/4;
            dist = delta*delta;
            delta = ((boxes[0].y+boxes[1].y+boxes[2].y+boxes[3].y)-
                     (boxes[4].y+boxes[5].y+boxes[6].y+boxes[7].y))/4;
            dist += delta*delta;
            dist  = (float)Math.sqrt(dist);
            // Note here that dist is the distance between center
            // points, and sz is the sum of the length of the
            // diagonals of the letter boxes.  In normal cases one
            // would expect dist to be approximately equal to sz/2.
            // So here we merge if the two characters are within four
            // character widths of each other. If they are farther
            // apart than that chances are it's a 'line break' or
            // something similar where we will get better results
            // merging seperately, and anyways with this much space
            // between them the extra outline shouldn't hurt..
            GeneralPath gp = new GeneralPath();
            if (dist < 2*sz) {
                // Close enough to merge with previous char...
                System.arraycopy(boxes, 0, chull, 0, 8);
                int npts = makeConvexHull(chull, 8);
                gp.moveTo(chull[0].x, chull[0].y);
                for(int n=1; n<npts; n++)
                    gp.lineTo(chull[n].x, chull[n].y);
                gp.closePath();
            } else {
                // Merge all previous areas
                mergeAreas(shape, areas, nAreas);
                nAreas = 0; // Start fresh...

                // Then just add box (add the previous char box if first pts)
                if (i==2) {
                    gp.moveTo(boxes[0].x, boxes[0].y);
                    gp.lineTo(boxes[1].x, boxes[1].y);
                    gp.lineTo(boxes[2].x, boxes[2].y);
                    gp.lineTo(boxes[3].x, boxes[3].y);
                    gp.closePath();
                    shape.append(gp, false);
                    gp.reset();
                }
                gp.moveTo(boxes[4].x, boxes[4].y);
                gp.lineTo(boxes[5].x, boxes[5].y);
                gp.lineTo(boxes[6].x, boxes[6].y);
                gp.lineTo(boxes[7].x, boxes[7].y);
                gp.closePath();
            }
            areas[nAreas++] = new Area(gp);
        }

        mergeAreas(shape, areas, nAreas);
    }

    public static void mergeAreas(GeneralPath shape,
                                  Area []shapes, int nShapes) {
        // Merge areas hierarchically, this means that while there are
        // the same number of Area.add calls (n-1) the great majority
        // of them are very simple combinations.  This helps to speed
        // things up a tad...
        while (nShapes > 1) {
            int n=0;
            for (int i=1; i<nShapes;i+=2) {
                shapes[i-1].add(shapes[i]);
                shapes[n++] = shapes[i-1];
                shapes[i] = null;
            }

            // make sure we include the last one if odd.
            if ((nShapes&0x1) == 1)
                shapes[n-1].add(shapes[nShapes-1]);
            nShapes = nShapes/2;
        }
        if (nShapes == 1)
            shape.append(shapes[0], false);
    }

    /**
     * Perform hit testing for coordinate at x, y.
     *
     * @param x the x coordinate of the point to be tested.
     * @param y the y coordinate of the point to be tested.
     *
     * @return a TextHit object encapsulating the character index for
     *     successful hits and whether the hit is on the character
     *     leading edge.
     */
    public TextHit hitTestChar(float x, float y) {
        syncLayout();

        TextHit textHit = null;

        int currentChar = 0;
        for (int i = 0; i < gv.getNumGlyphs(); i++) {
            Shape gbounds = gv.getGlyphLogicalBounds(i);
            if (gbounds != null) {
                Rectangle2D gbounds2d = gbounds.getBounds2D();
                // System.out.println("Hit Test: [" + x + ", " + y + "] - " +
                //                    gbounds2d);
                if (gbounds.contains(x, y)) {
                    boolean isRightHalf =
                        (x > (gbounds2d.getX()+(gbounds2d.getWidth()/2d)));
                    boolean isLeadingEdge = !isRightHalf;
                    int charIndex = charMap[currentChar];
                    textHit = new TextHit(charIndex, isLeadingEdge);
                    return textHit;
                }
            }
            currentChar += getCharacterCount(i, i);
        }
        return textHit;
    }

//protected

    /**
     * Returns the GVTFont to use when rendering the specified
     * character iterator.  This should already be set as an attribute
     * on the aci.
     *
     * @param aci The character iterator to get the font attribute from.
     *
     * @return The GVTFont to use.  */
    protected GVTFont getFont() {
        aci.first();
        GVTFont gvtFont = (GVTFont)aci.getAttributes().get
            (GVTAttributedCharacterIterator.TextAttribute.GVT_FONT);

        if (gvtFont != null) 
            return gvtFont;

        // shouldn't get here
        return new AWTGVTFont(aci.getAttributes());
    }

    /**
     * Returns a shape describing the overline decoration for a given ACI.
     */
    protected Shape getOverlineShape() {
        double y = metrics.getOverlineOffset();
        float overlineThickness = metrics.getOverlineThickness();

        // need to move the overline a bit lower,
        // not sure if this is correct behaviour or not
        y += overlineThickness;

        // Not certain what should be done here...
        // aci.first();
        // Float dy = (Float) aci.getAttribute(DY);
        // if (dy != null)
        //     y += dy.floatValue();

        Stroke overlineStroke =
            new BasicStroke(overlineThickness);
        Rectangle2D logicalBounds = gv.getLogicalBounds();

        return overlineStroke.createStrokedShape(
                           new java.awt.geom.Line2D.Double(
                           logicalBounds.getMinX() + overlineThickness/2.0, offset.getY()+y,
                           logicalBounds.getMaxX() - overlineThickness/2.0, offset.getY()+y));
    }

    /**
     * Returns a shape describing the strikethrough line for a given ACI.
     */
    protected Shape getUnderlineShape() {

        double y = metrics.getUnderlineOffset();
        float underlineThickness = metrics.getUnderlineThickness();

        // need to move the underline a bit lower,
        // not sure if this is correct behaviour or not
        y += underlineThickness*1.5;

        BasicStroke underlineStroke =
            new BasicStroke(underlineThickness);

        // Not certain what should be done here...
        // aci.first();
        // Float dy = (Float) aci.getAttribute(DY);
        // if (dy != null)
        //     y += dy.floatValue();

        Rectangle2D logicalBounds = gv.getLogicalBounds();

        return underlineStroke.createStrokedShape(
                           new java.awt.geom.Line2D.Double(
                           logicalBounds.getMinX() + underlineThickness/2.0, offset.getY()+y,
                           logicalBounds.getMaxX() - underlineThickness/2.0, offset.getY()+y));
    }

    /**
     * Returns a shape describing the strikethrough line for a given ACI.
     */
    protected Shape getStrikethroughShape() {
        double y = metrics.getStrikethroughOffset();
        float strikethroughThickness = metrics.getStrikethroughThickness();

        Stroke strikethroughStroke =
            new BasicStroke(strikethroughThickness);

        // Not certain what should be done here...
        // aci.first();
        // Float dy = (Float) aci.getAttribute(DY);
        // if (dy != null)
        //     y += dy.floatValue();

        Rectangle2D logicalBounds = gv.getLogicalBounds();
        return strikethroughStroke.createStrokedShape(
                           new java.awt.geom.Line2D.Double(
                           logicalBounds.getMinX() + strikethroughThickness/2.0, offset.getY()+y,
                           logicalBounds.getMaxX() - strikethroughThickness/2.0, offset.getY()+y));
    }


    /**
     * Explicitly lays out each of the glyphs in the glyph
     * vector. This will handle any glyph position adjustments such as
     * dx, dy and baseline offsets.  It will also handle vertical
     * layouts.
     *
     * @param applyOffset Specifies whether or not to add the offset position
     * to each of the glyph positions.  */
    protected void doExplicitGlyphLayout() {

        this.gv.performDefaultLayout();

        float baselineAscent 
            = vertical ?
            (float) gv.getLogicalBounds().getWidth() :
            (metrics.getAscent() + Math.abs(metrics.getDescent()));

        int numGlyphs = gv.getNumGlyphs();
        // System.out.println("NumGlyphs: " + numGlyphs);

        float[] gp = gv.getGlyphPositions(0, numGlyphs+1, null);
        float verticalFirstOffset = 0f;

        boolean glyphOrientationAuto = isGlyphOrientationAuto();
        int glyphOrientationAngle = 0;
        if (!glyphOrientationAuto) {
            glyphOrientationAngle = getGlyphOrientationAngle();
        }
        int i=0, aciIndex = aci.getBeginIndex();
        char ch = aci.first();
        int runLimit = aciIndex;

        Float x=null, y=null, dx=null, dy=null, rotation=null;
        Object baseline=null;

        float init_x_pos = (float) offset.getX();
        float init_y_pos = (float) offset.getY();
        float curr_x_pos = init_x_pos;
        float curr_y_pos = init_y_pos;

        while (i < numGlyphs) {
            //System.out.println("limit: " + runLimit + ", " + aciIndex);
            if (aciIndex >= runLimit) {
                runLimit = aci.getRunLimit(runAtts);
                x        = (Float) aci.getAttribute(X);
                y        = (Float) aci.getAttribute(Y);
                dx       = (Float) aci.getAttribute(DX);
                dy       = (Float) aci.getAttribute(DY);
                rotation = (Float) aci.getAttribute(ROTATION);
                baseline = aci.getAttribute(BASELINE_SHIFT);
            }

            GVTGlyphMetrics gm = gv.getGlyphMetrics(i);

            if (i==0) {
                if (glyphOrientationAuto) {
                    if (isLatinChar(ch)) {
                        // it will be rotated 90
                        verticalFirstOffset = 0f;
                    } else {
                        // it won't be rotated
                        verticalFirstOffset = (float)gm.getBounds2D().getHeight();
                    }
                } else {
                    if (glyphOrientationAngle == 0) {
                        verticalFirstOffset = (float)gm.getBounds2D().getHeight();
                    } else {
                        verticalFirstOffset = 0f;
                    }
                }
            } else {
                if (glyphOrientationAuto && (verticalFirstOffset == 0f)
                    && !isLatinChar(ch)) {

                    verticalFirstOffset = (float)gm.getBounds2D().getHeight();
                }
            }

            // ox and oy are origin adjustments for each glyph,
            // computed on the basis of baseline-shifts, etc.
            float ox = 0f;
            float oy = 0f;
            float verticalGlyphRotation = 0f;
            float glyphRotation = 0f;


            if (ch != CharacterIterator.DONE) {
                if (vertical) {
                    if (glyphOrientationAuto) {
                        if (isLatinChar(ch)) {
                            // If character is Latin, then rotate by
                            // 90 degrees
                            verticalGlyphRotation = (float) (Math.PI / 2f);
                        } else {
                            verticalGlyphRotation = 0f;
                        }
                    } else {
                        verticalGlyphRotation = (float)Math.toRadians(glyphOrientationAngle);
                    }
                    if (textPath != null) {
                        // if vertical and on a path, any x's are ignored
                        x = null;
                    }
                } else {
                    if (textPath != null) {
                        // if horizontal and on a path, any y's are ignored
                        y = null;
                    }
                }

                // calculate the total rotation for this glyph
                if (rotation == null || rotation.isNaN()) {
                    glyphRotation = verticalGlyphRotation;
                } else {
                    glyphRotation = (rotation.floatValue() +
                                     verticalGlyphRotation);
                }

                if ((x != null) && !x.isNaN()) {
                    if (i != 0)  /// First x is factored into offset...
                        curr_x_pos = x.floatValue();
                } 
                if (dx != null && !dx.isNaN()) {
                    curr_x_pos += dx.floatValue();
                }

                if ((y != null) && !y.isNaN()) {
                    if (i != 0)
                        curr_y_pos = y.floatValue();
                } 
                if (dy != null && !dy.isNaN()) {
                    curr_y_pos += dy.floatValue();
                } else if (i > 0) {
                    curr_y_pos += gp[i*2 + 1]-gp[i*2 - 1];
                }

                float baselineAdjust = 0f;
                if (baseline != null) {
                    if (baseline instanceof Integer) {
                        if (baseline==TextAttribute.SUPERSCRIPT_SUPER) {
                            baselineAdjust = baselineAscent*0.5f;
                        } else if (baseline==TextAttribute.SUPERSCRIPT_SUB) {
                            baselineAdjust = -baselineAscent*0.5f;
                        }
                    } else if (baseline instanceof Float) {
                        baselineAdjust = ((Float) baseline).floatValue();
                    }
                    if (vertical) {
                        ox = baselineAdjust;
                    } else {
                        oy = -baselineAdjust;
                    }
                }

                if (vertical) {
                    // offset due to rotation of first character
                    oy += verticalFirstOffset;

                    if (glyphOrientationAuto) {
                        if (isLatinChar(ch)) {
                            ox += metrics.getStrikethroughOffset();
                        } else {
                            Rectangle2D glyphBounds = gv.getGlyphVisualBounds(i).getBounds2D();
                            ox -= (float)((glyphBounds.getMaxX() - gp[2*i]) - 
                                          glyphBounds.getWidth()/2);
                        }
                    } else {
                        // center the character if it's not auto orient
                        Rectangle2D glyphBounds = gv.getGlyphVisualBounds(i).getBounds2D();
                        if (glyphOrientationAngle == 0) {
                            ox -= (float)((glyphBounds.getMaxX() - gp[2*i]) - 
                                          glyphBounds.getWidth()/2);
                        } else if (glyphOrientationAngle == 180) {
                            ox += (float)((glyphBounds.getMaxX() - gp[2*i]) - 
                                          glyphBounds.getWidth()/2);
                        } else if (glyphOrientationAngle == 90) {
                            ox += metrics.getStrikethroughOffset();
                        } else { // 270
                            ox -= metrics.getStrikethroughOffset();
                        }
                    }
                }
            }

            // set the new glyph position
            gv.setGlyphPosition(i, new Point2D.Float(curr_x_pos+ox,curr_y_pos+oy));

            // calculte the position of the next glyph
            if (!ArabicTextHandler.arabicCharTransparent(ch)) {
                // only apply the advance if the current char is not transparen
                if (vertical) {
                    float advanceY = 0;
                    if (glyphOrientationAuto) {
                        if (isLatinChar(ch)) {
                            advanceY = gm.getHorizontalAdvance();
                        } else {
                            advanceY = gm.getVerticalAdvance();
                        }
                    } else {
                        if ((glyphOrientationAngle ==   0) ||
                            (glyphOrientationAngle == 180)) {
                            advanceY = gm.getVerticalAdvance();
                        } else if (glyphOrientationAngle == 90) {
                            advanceY = gm.getHorizontalAdvance();
                        } else { // 270
                            advanceY = gm.getHorizontalAdvance();
                            // need to translate so that the spacing
                            // between chars is correct
                            gv.setGlyphTransform(i, AffineTransform.getTranslateInstance(0, advanceY));
                        }
                    }
                    curr_y_pos += advanceY;
                } else {
                    curr_x_pos += gm.getHorizontalAdvance();
                }
            }

            // rotate the glyph
            if (glyphRotation != 0f) {
                AffineTransform glyphTransform = gv.getGlyphTransform(i);
                if (glyphTransform == null) {
                    glyphTransform = new AffineTransform();
                }
                glyphTransform.rotate(glyphRotation);
                gv.setGlyphTransform(i, glyphTransform);
            }

            aciIndex += gv.getCharacterCount(i,i);
            ch = aci.setIndex(aciIndex);
            i++;
        }
        // Update last glyph pos
        gv.setGlyphPosition(i, new Point2D.Float(curr_x_pos,curr_y_pos));

        offset  = new Point2D.Float(init_x_pos, init_y_pos);
        advance = new Point2D.Float(curr_x_pos - init_x_pos, 
                                    curr_y_pos - init_y_pos);

        layoutApplied  = true;
        spacingApplied = false;
        pathApplied    = false;
    }

    /**
     * Does any spacing adjustments that may have been specified.
     */
    protected void adjustTextSpacing() {

        if (spacingApplied) 
            // Nothing to do...
            return;

        if (!layoutApplied)
            // Must have clean layout to do spacing...
            doExplicitGlyphLayout();

        aci.first();
        Boolean customSpacing =  (Boolean) aci.getAttribute(
               GVTAttributedCharacterIterator.TextAttribute.CUSTOM_SPACING);
        if ((customSpacing != null) && customSpacing.booleanValue()) {
            advance = doSpacing
                ((Float) aci.getAttribute
                 (GVTAttributedCharacterIterator.TextAttribute.KERNING),
                 (Float) aci.getAttribute
                 (GVTAttributedCharacterIterator.TextAttribute.LETTER_SPACING),
                 (Float) aci.getAttribute
                 (GVTAttributedCharacterIterator.TextAttribute.WORD_SPACING));
            // Basic layout is now messed up...
            layoutApplied  = false;
        }

        // This will clear layoutApplied if it mucks with the current
        // character positions.
        applyStretchTransform(!adjSpacing);

        spacingApplied = true;
        pathApplied    = false;
    }

    /**
     * Performs any spacing adjustments required and returns the new advance
     * value.
     *
     * @param kern The kerning adjustment to apply to the space
     * between each char.
     * @param letterSpacing The amount of spacing required between each char.
     * @param wordSpacing The amount of spacing required between each word.  */
    protected Point2D doSpacing(Float kern,
                                Float letterSpacing,
                                Float wordSpacing) {
        boolean autoKern = true;
        boolean doWordSpacing = false;
        boolean doLetterSpacing = false;
        float kernVal = 0f;
        float letterSpacingVal = 0f;
        float wordSpacingVal = 0f;

        if ((kern instanceof Float) && (!kern.isNaN())) {
            kernVal = kern.floatValue();
            autoKern = false;
            //System.out.println("KERNING: "+kernVal);
        }
        if ((letterSpacing instanceof Float) && (!letterSpacing.isNaN())) {
            letterSpacingVal = letterSpacing.floatValue();
            doLetterSpacing = true;
            //System.out.println("LETTER-SPACING: "+letterSpacingVal);
        }
        if ((wordSpacing instanceof Float) && (!wordSpacing.isNaN())) {
            wordSpacingVal = wordSpacing.floatValue();
            doWordSpacing = true;
            //System.out.println("WORD_SPACING: "+wordSpacingVal);
        }

        int numGlyphs = gv.getNumGlyphs();

        float dx = 0f;
        float dy = 0f;
        Point2D newPositions[] = new Point2D[numGlyphs+1];
        Point2D prevPos = gv.getGlyphPosition(0);
        float x = (float) prevPos.getX();
        float y = (float) prevPos.getY();

        Point2D lastCharAdvance
            = new Point2D.Double(advance.getX() - (gv.getGlyphPosition(numGlyphs-1).getX() - x),
                                 advance.getY() - (gv.getGlyphPosition(numGlyphs-1).getY() - y));

        try {
            // do letter spacing first
            if ((numGlyphs > 1) && (doLetterSpacing || !autoKern)) {
                for (int i=1; i<=numGlyphs; ++i) {
                    Point2D gpos = gv.getGlyphPosition(i);
                    dx = (float)gpos.getX()-(float)prevPos.getX();
                    dy = (float)gpos.getY()-(float)prevPos.getY();
                    if (autoKern) {
                        if (vertical) dy += letterSpacingVal;
                        else dx += letterSpacingVal;
                    } else {
                        // apply explicit kerning adjustments,
                        // discarding any auto-kern dx values
                        if (vertical) {
                            dy = (float)
                            gv.getGlyphMetrics(i-1).getBounds2D().getHeight()+
                                kernVal + letterSpacingVal;
                        } else {
                            dx = (float)
                            gv.getGlyphMetrics(i-1).getBounds2D().getWidth()+
                                kernVal + letterSpacingVal;
                        }
                    }
                    x += dx;
                    y += dy;
                    newPositions[i] = new Point2D.Float(x, y);
                    prevPos = gpos;
                }

                for (int i=1; i<=numGlyphs; ++i) { // assign the new positions
                    if (newPositions[i] != null) {
                        gv.setGlyphPosition(i, newPositions[i]);
                    }
                }
            }

             // adjust the advance of the last character
            if (autoKern) {
                if (vertical) {
                    lastCharAdvance.setLocation(lastCharAdvance.getX(),
                            lastCharAdvance.getY() + letterSpacingVal);
                } else {
                    lastCharAdvance.setLocation(lastCharAdvance.getX()
                            + letterSpacingVal, lastCharAdvance.getY());
                }
            } else {
                if (vertical) {
                    lastCharAdvance.setLocation(lastCharAdvance.getX(),
                        gv.getGlyphMetrics(numGlyphs-2).getBounds2D().getHeight()+
                                kernVal + letterSpacingVal);
                } else {
                    lastCharAdvance.setLocation(
                        gv.getGlyphMetrics(numGlyphs-2).getBounds2D().getWidth()+
                                kernVal + letterSpacingVal, lastCharAdvance.getY());
                }
            }


            // now do word spacing
            dx = 0f;
            dy = 0f;
            prevPos = gv.getGlyphPosition(0);
            x = (float) prevPos.getX();
            y = (float) prevPos.getY();

            if ((numGlyphs > 1) && (doWordSpacing)) {
                for (int i = 1; i < numGlyphs; i++) {
                    Point2D gpos = gv.getGlyphPosition(i);
                    dx = (float)gpos.getX()-(float)prevPos.getX();
                    dy = (float)gpos.getY()-(float)prevPos.getY();
                    boolean inWS = false;
                    // while this is whitespace, increment
                    int beginWS = i;
                    int endWS = i;
                    GVTGlyphMetrics gm = gv.getGlyphMetrics(i);

                    // BUG: gm.isWhitespace() fails for latin SPACE glyph!
                    while ((gm.getBounds2D().getWidth()<0.01d) || gm.isWhitespace()) {
                        if (!inWS) inWS = true;
                        if (i == numGlyphs-1) {
                            // white space at the end
                            break;
                        }
                        ++i;
                        ++endWS;
                        gpos = gv.getGlyphPosition(i);
                        gm = gv.getGlyphMetrics(i);
                    }

                    if ( inWS ) {  // apply wordSpacing
                        int nWS = endWS-beginWS;
                        float px = (float) prevPos.getX();
                        float py = (float) prevPos.getY();
                        dx = (float) (gpos.getX() - px)/(nWS+1);
                        dy = (float) (gpos.getY() - py)/(nWS+1);
                        if (vertical) {
                            dy += (float) wordSpacing.floatValue()/(nWS+1);
                        } else {
                            dx += (float) wordSpacing.floatValue()/(nWS+1);
                        }
                        for (int j=beginWS; j<=endWS; ++j) {
                            x += dx;
                            y += dy;
                            newPositions[j] = new Point2D.Float(x, y);
                        }
                    } else {
                        dx = (float) (gpos.getX()-prevPos.getX());
                        dy = (float) (gpos.getY()-prevPos.getY());
                        x += dx;
                        y += dy;
                        newPositions[i] = new Point2D.Float(x, y);
                    }
                    prevPos = gpos;
                }
                Point2D gPos = gv.getGlyphPosition(numGlyphs);
                x += (float) (gPos.getX()-prevPos.getX());
                y += (float) (gPos.getY()-prevPos.getY());
                newPositions[numGlyphs] = new Point2D.Float(x, y);

                for (int i=1; i<=numGlyphs; ++i) { // assign the new positions
                    if (newPositions[i] != null) {
                        gv.setGlyphPosition(i, newPositions[i]);
                    }
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }

        // calculate the new advance
        double advX = gv.getGlyphPosition(numGlyphs-1).getX()
                     - gv.getGlyphPosition(0).getX();
        double advY = gv.getGlyphPosition(numGlyphs-1).getY()
                     - gv.getGlyphPosition(0).getY();
        Point2D newAdvance = new Point2D.Double(advX + lastCharAdvance.getX(),
                                                advY + lastCharAdvance.getY());
        return newAdvance;
    }

    /**
     * Stretches the text so that it becomes the specified length.
     *
     * @param stretchGlyphs if true xScale, yScale will be applied to
     *                      each glyphs transform.
     */
    protected void applyStretchTransform(boolean stretchGlyphs) {
        if ((xScale == 1) && (yScale==1)) 
            return;

        Rectangle2D bounds = gv.getVisualBounds();
        AffineTransform scaleAT = 
            AffineTransform.getScaleInstance(xScale, yScale);

        float initX   = (float) bounds.getX();
        float initY   = (float) bounds.getY();
        int numGlyphs = gv.getNumGlyphs();
        float [] gp   = gv.getGlyphPositions(0, numGlyphs+1, null);
        float dx = 0f;
        float dy = 0f;
        for (int i = 0; i <= numGlyphs; i++) {
            dx = gp[2*i]  -initX;
            dy = gp[2*i+1]-initY;
            gv.setGlyphPosition(i, new Point2D.Float(initX+dx*xScale,
                                                     initY+dy*yScale));

            if ((stretchGlyphs) && (i != numGlyphs)) {
                // stretch the glyph
                AffineTransform glyphTransform = gv.getGlyphTransform(i);
                if (glyphTransform != null) {
                    glyphTransform.preConcatenate(scaleAT);
                    gv.setGlyphTransform(i, glyphTransform);
                } else {
                    gv.setGlyphTransform (i, scaleAT);
                }
            }
        }

        advance = new Point2D.Float((float)(advance.getX()*xScale),
                                    (float)(advance.getY()*yScale));
        // Basic layout is now messed up...
        layoutApplied  = false;
    }

    /**
     * If this layout is on a text path, positions the characters
     * along the path.  
     */
    protected void doPathLayout() {
        if (pathApplied) 
            return;

        if (!spacingApplied)
            // This will layout the text if needed.
            adjustTextSpacing();

        // if doesn't have an attached text path, just return
        if (textPath == null) {
            // We applied the empty path (i.e. do nothing).
            pathApplied = true;
            return;
        }


        boolean horizontal = !isVertical();

        boolean glyphOrientationAuto = isGlyphOrientationAuto();
        int glyphOrientationAngle = 0;
        if (!glyphOrientationAuto) {
            glyphOrientationAngle = getGlyphOrientationAngle();
        }

        float pathLength = textPath.lengthOfPath();
        float startOffset = textPath.getStartOffset();
        int numGlyphs = gv.getNumGlyphs();

        // make sure all glyphs visible again, this maybe just a change in
        // offset so they may have been made invisible in a previous
        // pathLayout call
        for (int i = 0; i < numGlyphs; i++) {
            gv.setGlyphVisible(i, true);
        }

        // calculate the total length of the glyphs, this will become be
        // the length along the path that is used by the text
        float glyphsLength;
        if (horizontal) {
            glyphsLength = (float) gv.getLogicalBounds().getWidth();
        } else {
            glyphsLength = (float) gv.getLogicalBounds().getHeight();
        }

        // check that pathLength and glyphsLength are not 0
        if (pathLength == 0f || glyphsLength == 0f) {
            // We applied the empty path.
            pathApplied = true;
            textPathAdvance = advance;
            return;
        }

        // the current start point of the character on the path
        float currentPosition;
        if (horizontal) {
            currentPosition = (float)offset.getX() + startOffset;
        } else {
            currentPosition = (float)offset.getY() + startOffset;
        }

        // calculate the offset of the first glyph the offset will be
        // 0 if the glyph is on the path (ie. not adjusted by a dy or
        // dx)
        Point2D firstGlyphPosition = gv.getGlyphPosition(0);
        float glyphOffset = 0;   // offset perpendicular to path
        if (horizontal) {
            glyphOffset = (float)(firstGlyphPosition.getY());
        } else {
            glyphOffset = (float)(firstGlyphPosition.getX());
        }

        char ch = aci.first();
        int currentChar = aci.getBeginIndex();
        int lastGlyphDrawn = -1;
        float lastGlyphAdvance = 0;

        // iterate through the GlyphVector placing each glyph
        for (int i = 0; i < numGlyphs; i++) {

            Point2D currentGlyphPosition = gv.getGlyphPosition(i);

            // calculate the advance and offset for the next glyph, do it
            // now before we modify the current glyph position

            float glyphAdvance = 0;  // along path
            float nextGlyphOffset = 0;  // perpendicular to path eg dy or dx
            if (i < gv.getNumGlyphs()-1) {

                Point2D nextGlyphPosition = gv.getGlyphPosition(i+1);
                if (horizontal) {
                    glyphAdvance    = (float)(nextGlyphPosition.getX() -
                                              currentGlyphPosition.getX());
                    nextGlyphOffset = (float)(nextGlyphPosition.getY() -
                                              currentGlyphPosition.getY());
                } else {
                    glyphAdvance    = (float)(nextGlyphPosition.getY() -
                                              currentGlyphPosition.getY());
                    nextGlyphOffset = (float)(nextGlyphPosition.getX() -
                                              currentGlyphPosition.getX());
                }
            } else {
                // last glyph, use the glyph metrics
                GVTGlyphMetrics gm = gv.getGlyphMetrics(i);
                if (horizontal) {
                    glyphAdvance = gm.getHorizontalAdvance();
                } else {
                    if (glyphOrientationAuto) {
                        if (isLatinChar(ch)) {
                            glyphAdvance = gm.getHorizontalAdvance();
                        } else {
                            glyphAdvance = gm.getVerticalAdvance();
                        }
                    } else {
                        if ((glyphOrientationAngle == 0) ||
                            (glyphOrientationAngle == 180)) {
                            glyphAdvance = gm.getVerticalAdvance();
                        } else { // 90 || 270
                            glyphAdvance = gm.getHorizontalAdvance();
                        }
                    }
                }
            }

            // calculate the center line position for the glyph
            Rectangle2D glyphBounds = gv.getGlyphOutline(i).getBounds2D();
            float glyphWidth = (float) glyphBounds.getWidth();
            float glyphHeight = (float) glyphBounds.getHeight();

            float charMidPos;
            if (horizontal) {
                charMidPos = currentPosition + glyphWidth / 2f;
            } else {
                charMidPos = currentPosition + glyphHeight / 2f;
            }

            // Calculate the actual point to place the glyph around
            Point2D charMidPoint = textPath.pointAtLength(charMidPos);

            // Check if the glyph is actually on the path
            if (charMidPoint != null) {

                // Calculate the normal to the path (midline of glyph)
                float angle = textPath.angleAtLength(charMidPos);

                // Define the transform of the glyph
                AffineTransform glyphPathTransform = new AffineTransform();

                // rotate midline of glyph to be normal to path
                if (horizontal) {
                    glyphPathTransform.rotate(angle);
                } else {
                    glyphPathTransform.rotate(angle-(Math.PI/2));
                }

                // re-apply any offset eg from tspan, or spacing adjust
                if (horizontal) {
                    glyphPathTransform.translate(0, glyphOffset);
                } else {
                    glyphPathTransform.translate(glyphOffset, 0);
                }

                // translate glyph backwards so we rotate about the
                // center of the glyph
                if (horizontal) {
                    glyphPathTransform.translate(glyphWidth / -2f, 0f);
                } else {
                    if (glyphOrientationAuto) {
                        if (isLatinChar(ch)) {
                           glyphPathTransform.translate(0f, -glyphHeight/2f);
                        } else {
                            glyphPathTransform.translate(0f, glyphHeight/2f);
                        }
                    } else {
                        if (glyphOrientationAngle == 0 ) {
                            glyphPathTransform.translate(0f, glyphHeight/2f);
                        } else { // 90 || 180
                            glyphPathTransform.translate(0f, -glyphHeight/2f);
                        }
                    }
                }

                // set the new glyph position and transform
                AffineTransform glyphTransform = gv.getGlyphTransform(i);
                if (glyphTransform != null) {
                    glyphPathTransform.concatenate(glyphTransform);
                }

                gv.setGlyphTransform(i, glyphPathTransform);
                gv.setGlyphPosition(i, new Point2D.Double(charMidPoint.getX(),
                                                          charMidPoint.getY()));
                // keep track of the last glyph drawn to make calculating the
                // textPathAdvance value easier later
                lastGlyphDrawn = i;
                lastGlyphAdvance = glyphAdvance;

            } else {
                // not on path so don't render
                gv.setGlyphVisible(i, false);
            }
            currentPosition += glyphAdvance;
            glyphOffset += nextGlyphOffset;
            currentChar += gv.getCharacterCount(i,i);
            ch = aci.setIndex(aci.getBeginIndex() + i +
                              gv.getCharacterCount(i,i));
        }

        // store the position where a following glyph should be drawn,
        // note: this will only be used if the following text layout is not
        //       on a text path
        if (lastGlyphDrawn > -1) {
            Point2D lastGlyphPos = gv.getGlyphPosition(lastGlyphDrawn);
            if (horizontal) {
                textPathAdvance = new Point2D.Double
                    (lastGlyphPos.getX()+lastGlyphAdvance,
                     lastGlyphPos.getY());
            } else {
                textPathAdvance = new Point2D.Double
                    (lastGlyphPos.getX(),
                     lastGlyphPos.getY()+lastGlyphAdvance);
            }
        } else {
            textPathAdvance = new Point2D.Double(0,0);
        }

        // The default layout is junk now...
        layoutApplied  = false;
        // The spacing stuff is junk now.
        spacingApplied = false;
        pathApplied    = true;   
    }

    /**
     * Returns true if the specified character is within one of the Latin
     * unicode character blocks.
     *
     * @param c The char to test.
     *
     * @return True if c is latin.
     */
    protected boolean isLatinChar(char c) {

        Character.UnicodeBlock block = Character.UnicodeBlock.of(c);

        if (block == Character.UnicodeBlock.BASIC_LATIN ||
            block == Character.UnicodeBlock.LATIN_1_SUPPLEMENT ||
            block == Character.UnicodeBlock.LATIN_EXTENDED_ADDITIONAL ||
            block == Character.UnicodeBlock.LATIN_EXTENDED_A ||
            block == Character.UnicodeBlock.LATIN_EXTENDED_B) {
            return true;
        } else {
            return false;
        }

    }

    /**
     * Returns whether or not the vertical glyph orientation value is "auto".
     */
    protected boolean isGlyphOrientationAuto() {
        boolean glyphOrientationAuto = true;
        aci.first();
        if (aci.getAttribute(GVTAttributedCharacterIterator.TextAttribute.VERTICAL_ORIENTATION) != null) {
            glyphOrientationAuto = (aci.getAttribute(GVTAttributedCharacterIterator.TextAttribute.VERTICAL_ORIENTATION)
                                     == GVTAttributedCharacterIterator.TextAttribute.ORIENTATION_AUTO);
        }
        return glyphOrientationAuto;
    }

    /**
     * Returns the value of the vertical glyph orientation angle. This will be
     * one of 0, 90, 180 or 270.
     */
    protected int getGlyphOrientationAngle() {

        int glyphOrientationAngle = 0;

        aci.first();
        Float angle = (Float)aci.getAttribute(GVTAttributedCharacterIterator.
            TextAttribute.VERTICAL_ORIENTATION_ANGLE);
        if (angle != null) {
            glyphOrientationAngle = (int)angle.floatValue();
        }
        // if not one of 0, 90, 180 or 270, round to nearest value
        if (glyphOrientationAngle != 0 || glyphOrientationAngle != 90
            || glyphOrientationAngle != 180 || glyphOrientationAngle != 270) {
            while (glyphOrientationAngle < 0) {
                glyphOrientationAngle += 360;
            }
            while (glyphOrientationAngle >= 360) {
                glyphOrientationAngle -= 360;
            }
            if (glyphOrientationAngle <= 45 || glyphOrientationAngle > 315) {
                glyphOrientationAngle = 0;
            } else if (glyphOrientationAngle > 45 && glyphOrientationAngle <= 135) {
                glyphOrientationAngle = 90;
            } else if (glyphOrientationAngle > 135 && glyphOrientationAngle <= 225) {
                glyphOrientationAngle = 180;
            } else {
                glyphOrientationAngle = 270;
            }
        }
        return glyphOrientationAngle;
    }


    // Issues: 
    //   Should the font size of non-printing chars affect line spacing?
    //   Does line breaking get done before/after ligatures?
    //   What should be done if the next glyph does not fit in the
    //   flow rect (very narrow flow rect).
    //      Print the one char anyway.
    //      Go to the next flow rect.
    //   Should dy be considered for line offsets? (super scripts)
    //   Should p's & br's carry over from flow rect to flow rect if
    //   so how much????
    //   
    //   For Full justification:
    //       Streach glyphs to fill line? (attribute?)
    //       What to do with partial line (last line in 'p', 'line'
    //       element, or 'div' element), still full justify, just left 
    //       justify, attribute?
    //       What to do when only one glyph on line? left or center or stretch?
    //       For full to look good I think the line must be able to squeeze a
    //         bit as well as grow (pretty easy to add).
    //
    // This Only does horizontal languages.
    // text-decoration won't work on this text.
    // Soft hyphen isn't handled yet.
    /**
     * This will wrap the text associated with <tt>aci</tt> and
     * <tt>layouts</tt>.
     * @param aci The Attributed Charater Iterator for text to wrap.
     *            used exclusively to access font information.
     * @param layouts A List of GlyphLayout objects that are to
     *                be wrapped as a whole.
     * @param flowRects A List of Rectangle2D representing the regions
     *                  to flow text into.
     * @param brLocs A List of Integer.  Each integer indicates the
     *               the aci index of the char after the end of a
     *               br/line element.  Note that a particular value can 
     *               be repeated several times for sequences of br/line
     *               elements without chars between them.
     * @param pLocs A List of Integer.  Each integer indicates the
     *               the aci index of the char after the end of a
     *               p element.  Note that a particular value can be
     *               repeated several times for sequences of p
     *               elements without chars between them.
     * @param justification an integer in the range: 0-3. 0 is left, 
     *                      1 is center, 2 is right, and 
     *                      3 is fully justified.
     */
    public static void textWrapTextChunk(AttributedCharacterIterator aci,
                                         List layouts,
                                         List flowRects,
                                         List brLocs,
                                         List pLocs,
                                         int justification) {
        int aciIndex = aci.getBeginIndex();
        int aciEnd   = aci.getEndIndex();

        char ch = aci.first();
        
        Iterator iter = layouts.iterator();

        // Make a list of the GlyphVectors so we can construct a
        // multiGlyphVector that makes them all look like one big
        // glyphVector
        List gvs = new LinkedList();
        while (iter.hasNext()) {
            GlyphLayout gl = (GlyphLayout)iter.next();
            gvs.add(gl.getGlyphVector());
        }

        GVTGlyphVector gv = new MultiGlyphVector(gvs);

        int numGlyphs = gv.getNumGlyphs();
        float[] gp = gv.getGlyphPositions(0, numGlyphs+1, null);

        if (numGlyphs == 0) return;

        // Now pull the line break locations out into an array so
        // I can jump around a bit easier.
        int brIdx = 0;
        int [] brLoc = new int[brLocs.size()+1];
        iter = brLocs.iterator();
        while (iter.hasNext())
            brLoc[brIdx++] = ((Integer)iter.next()).intValue();
        brLoc[brIdx] = aciEnd+1;
        brIdx = 0;
            
        // Now pull the paragraph break locations out into an array so
        // I can jump around a bit easier.
        int pIdx = 0;
        int [] pLoc = new int[pLocs.size()+1];
        iter = pLocs.iterator();
        while (iter.hasNext())
            pLoc[pIdx++] = ((Integer)iter.next()).intValue();
        pLoc[pIdx] = aciEnd+1;
        pIdx = 0;
        

        // Get an iterator for the flow rects.
        Iterator flowRectsIter = flowRects.iterator();
        if (!flowRectsIter.hasNext()) {
            // No place to flow into, hide all glyphs.
            for (int i=0; i>numGlyphs; i++)
                gv.setGlyphVisible(i, false);
            return;
        }

        // Ok get the info for the first rectangle...
        Rectangle2D cRect = (Rectangle2D)flowRectsIter.next();
        float x0     = (float)cRect.getX();
        float y0     = (float)cRect.getY();
        float width  = (float)cRect.getWidth();
        float height = (float)cRect.getHeight();

        // Get the font size at the start of the string...
        float fontSize = 12;
        Float fsFloat = (Float)aci.getAttributes().get(TextAttribute.SIZE);
        if (fsFloat != null) {
            fontSize = fsFloat.floatValue();
        }
        // Figure out where the font size might change again...
        int runLimit  = aci.getRunLimit(TEXT_COMPOUND_DELIMITER);

        // This is our current max font size
        float maxFontSize = fontSize;
        // This is the font size of the last printing char.
        float chFontSize  = fontSize;

        // Information for backing up to word Break (used when
        // wrapping normal lines.  

        // Glyph index of last break char seen (-1 if no break char on line)
        int   wBreakIdx         = -1;
        // The ACI index of last break char seen.
        int   wBreakACI         = -1;
        // Glyph index of last 'printing' char before last space seen 
        // (needed to do visual justification on glyph bounds).
        int   wBreakChIdx       = -1;
        // The total advance for line including last non-space char.
        float wBreakAdv         =  0;
        // The total advance for line including spaces at end of line.
        float wBreakAdj         =  0;
        // The font size at last space seen.
        float wBreakFontSize    = fontSize;
        // The runLimit (for fonts) at last space seen.
        int   wBreakRunLimit    = runLimit;
        // The max font size since last space seen (printable chars only).  
        float wBreakMaxFontSize = maxFontSize;

        // Information for backing up to line start.
        // This is needed when we reach the end of a
        // line and realize it doesn't fit in the
        // current rect.
        // Glyph Index of first char on current line.
        int   lBreakIdx       = -1;
        // ACI Index of first char on current line.
        int   lBreakACI       = -1;
        // font size at start of line
        float lBreakFontSize  = fontSize;
        // runLimit (for font size) at start of line.
        int   lBreakRunLimit  = runLimit;
        // The index into the brLoc array at start of line
        int   lBreakBrIdx     = 0;
        // The index into the pLoc array at start of line
        int   lBreakPIdx      = 0;

        // Offset from top of current rect for baseline.
        float dy       = 0;
        // Printing char advance on line so far.
        float adv      = 0;
        // Advance of current char plus any leading non-printing chars.
        float chAdv    = 0;
        // GlyphIndex of last printing char seen 
        // (used for full justification on glyph bounds).
        int   lastChar = 0;
        // The descent from previous line we need to incorporate
        float prevDesc = 0;


        // Used to know if we are doing the first glyph after line start.
        int  lineStart     = 0;

        // Per-line information lists.  Used after line breaks have
        // been decided upon.
        List lineStarts    = new LinkedList(); // glyph index of line starts
        List lineLocs      = new LinkedList(); // Point2D of line start.
        List lineAdvs      = new LinkedList(); // Visual width of line.
        List lineAdjs      = new LinkedList(); // x Offset for line 
        List lineLastCharW = new LinkedList(); // used in full justification.
        List linePartial   = new LinkedList(); // should line be justified?

        lineStarts.add(new Integer(0));
        int i;
        boolean chIsSpace    = isSpace(ch);
        boolean partial      = false;
        float   nextLineMult = 1.0f;
        float   lineMult     = 1.0f;

        // Let's start looking at glyphs....
        for (i=0; i<numGlyphs; i++) {

            // Update font size for this char if needed.
            if (aciIndex >= runLimit) {
                runLimit = aci.getRunLimit(TEXT_COMPOUND_DELIMITER);
                fsFloat = (Float)aci.getAttributes().get(TextAttribute.SIZE);
                if (fsFloat != null) {
                    fontSize = fsFloat.floatValue();
                } else {
                    fontSize = 12;
                }
                // System.out.println("ACI: " + aciIndex + 
                //                    " runLimit: " + runLimit +
                //                    " FS: " + fontSize);
            }

            // Get advance for this glyph...
            float gmAdv = gp[2*i+2] - gp[2*i];

            // Add to the 'Char adv' which includes any non printing
            // chars before this char.
            chAdv += gmAdv;

            // System.out.println("Ch : '" + ch + "' Idx: " + aciIndex);
    
            // If it's a space remeber it as a breaking location but
            // don't check for line break yet (wait till non-space char).
            if (chIsSpace) {
                wBreakIdx      = i;
                wBreakChIdx    = lastChar;
                wBreakACI      = aciIndex;
                wBreakAdv      = adv;
                wBreakAdj      = adv+chAdv;
                wBreakFontSize = fontSize;
                wBreakRunLimit = runLimit;

                // New break loc so incorporate wBreakMaxFontSize
                if (wBreakMaxFontSize > maxFontSize)
                    maxFontSize = wBreakMaxFontSize;

                wBreakMaxFontSize = chFontSize;
            }

            if (!chIsSpace) {
                // Ok we are a printing char so include all the
                // advaces we have collected.
                adv += chAdv;

                // Remember the size of this char...
                chFontSize = fontSize;
            }

            boolean doBreak = false;

            // Don't break at first char in line 
            // (otherwise nothing will ever get drawn).
            if (!chIsSpace &&
                (lineStart != i) && 
                (adv > width)
                ) {

                // Better logic for wrap (allow it to squeeze upto 1/2 as
                // much as it would streatch if word wrapped).
                // (((wBreakIdx == -1) && (adv > width)) ||  // no break avail
                //  (adv-width > (width-wBreakAdv)*.5))



                if (wBreakIdx == -1) {
                    // Break in the middle of word.  Back out gmAdv
                    // since it will be used on the next line.
                    wBreakAdv   = adv-gmAdv;
                    wBreakAdj   = adv-gmAdv;
                    wBreakChIdx = lastChar;

                    i--; // reconsider letter (will get inc'd later).

                    // Include max font since last break char in max calc.
                    if (wBreakMaxFontSize > maxFontSize)
                        maxFontSize = wBreakMaxFontSize;
                } else {
                    // We have a break char to back up to.
                    // Reset state to just after breaking chars.
                    i          = wBreakIdx;
                    aciIndex   = wBreakACI;
                    fontSize   = wBreakFontSize;
                    chFontSize = wBreakFontSize;
                    runLimit   = wBreakRunLimit;

                    aciIndex += gv.getCharacterCount(i,i);
                    ch        = aci.setIndex(aciIndex);
                    chIsSpace = isSpace(ch);
                }

                nextLineMult = 1.0f;
                doBreak = true;
                partial = false;
            }

            // Track how many p's and br's we hit here.
            int pCount  = 0;
            int brCount = 0;
            // did we just pass a p or br?
            if ((aciIndex >= pLoc [pIdx]) ||
                (aciIndex >= brLoc[brIdx])) {

                // Count the P's here...
                while (aciIndex >= pLoc[pIdx]) {
                    pIdx++;
                    pCount++;
                    nextLineMult += 1.5;
                }

                // Count the br's here...
                while (aciIndex >= brLoc[brIdx]) {
                    brIdx++;
                    brCount++;
                    nextLineMult += 1.0;
                }

                if (doBreak) {
                    // If we were going to word wrap here anyway
                    // Don't double space...
                    nextLineMult -= 1.0f;
                } else {
                    // We need to specify the break
                    if (chIsSpace) {
                        wBreakAdv   = adv;
                        wBreakAdj   = adv+(chAdv-gmAdv);
                    } else {
                        // If a char with prior spaces then keep
                        // spaces on prev line and char on this line...
                        wBreakAdv   = adv-gmAdv;
                        wBreakAdj   = adv-gmAdv;
                    }

                    wBreakChIdx = lastChar;

                    i--; // reconsider letter.

                    // Include max font since break as max.
                    if (wBreakMaxFontSize > maxFontSize)
                        maxFontSize = wBreakMaxFontSize;
                }
                
                doBreak = true;
                partial = true;
            }


            if (doBreak) {
                // We will now attempt to break the line just
                // before the current char.

                // Note we are trying to figure out where the current
                // line is going to be placed (not the next line).  We
                // must wait until we have a potential line break so
                // we know how tall the line is.

                // Get the nomial line advance based on the
                // largest font we encountered on line...
                float ladv = (maxFontSize*.8f+prevDesc)*1.1f;

                // This is modified by any p's or br's we hit at
                // the end of the last line.
                dy += ladv*lineMult;

                // Remember the effect of p's br's at the end of this line.
                lineMult = nextLineMult;

                // Set ourself up for next line...
                nextLineMult = 0.0f;

                // System.out.println("Break L [" + lineStart + "," + i + 
                //                    "] Loc: " + (y0+dy) + 
                //                    " adv: " + wBreakAdv +
                //                    " adj: " + wBreakAdj);

                if ((dy + maxFontSize*.2f) <= height) {
                    // The line fits in the current flow rectangle.

                    // System.out.println("Fits: " + (dy + maxFontSize*.2f));


                    // Now remember info about start of next line.
                    // (needed so we can back up if that line doesn't
                    // fit in current rectangle).
                    lBreakIdx      = i+1;
                    lBreakACI      = aciIndex;
                    lBreakFontSize = fontSize;
                    lBreakRunLimit = runLimit;
                    lBreakPIdx     = pIdx;
                    lBreakBrIdx    = brIdx;

                    // Now we tweak line advance to account for
                    // visual bounds of last glyph.
                    Rectangle2D lastCharB = gv.getGlyphVisualBounds
                        (wBreakChIdx).getBounds2D();
                    Point2D     lastCharL = gv.getGlyphPosition
                        (wBreakChIdx);
                    float charW   = (float)
                        (lastCharB.getX()+lastCharB.getWidth()-
                         lastCharL.getX());
                    float charAdv = gp[2*wBreakChIdx+2]-gp[2*wBreakChIdx];
                    wBreakAdv -= charAdv-charW;

                    // Add all the info about the current line to lists.
                    lineLocs.add   (new Point2D.Float(x0, y0+dy));
                    lineAdvs.add   (new Float(wBreakAdv));
                    lineAdjs.add   (new Float(wBreakAdj));
                    lineLastCharW.add (new Float(charW));
                    linePartial.add(new Boolean(partial));

                    // Remember where next line starts.
                    lineStart = i+1;
                    lineStarts.add (new Integer(lineStart));

                    // Remember how far down this line goes into
                    // next line.
                    prevDesc        = maxFontSize*.2f;
                } else {
                    // The current line doesn't fit in the current
                    // flow rectangle so we need to go move line to
                    // the next flow rectangle.

                    // System.out.println("Doesn't Fit: " + 
                    //                    (dy + maxFontSize*.2f));


                    if (!flowRectsIter.hasNext())
                        break; // No flow rect stop layout here...

                    // Remember how wide this rectangle is...
                    float oldWidth = width;

                    // Get info for new flow rect.
                    cRect = (Rectangle2D)flowRectsIter.next();
                    x0     = (float)cRect.getX();
                    y0     = (float)cRect.getY();
                    width  = (float)cRect.getWidth();
                    height = (float)cRect.getHeight();

                    // New rect so no previous row to consider...
                    dy        = 0;
                    prevDesc  = 0;
                    lineMult  = 1.0f; // don't pile up lineMults from
                                      // previous flows?

                    if (cRect.getWidth() >= oldWidth) {
                        // new rect is same width or wider so
                        // we can reuse our work on this line.

                        // Just undo anything we added in
                        if (!chIsSpace)
                            adv -= chAdv;

                        chAdv -= gmAdv;
                        // Unadvance p's and br's for this char...
                        pIdx  -= pCount;
                        brIdx -= brCount;
                        continue;
                    }

                    // new rect is smaller so back up to line start
                    // and try with new flow rect.
                    i          = lBreakIdx-1; // loop will inc...
                    aciIndex   = lBreakACI;
                    fontSize   = lBreakFontSize;
                    chFontSize = lBreakFontSize;
                    runLimit   = lBreakRunLimit;
                    pIdx       = lBreakPIdx;
                    brIdx      = lBreakBrIdx;

                    ch       = aci.setIndex(aciIndex);
                    chIsSpace = isSpace(ch);
                }

                // Set fontSize max's to last printing char size.
                maxFontSize       = chFontSize;
                wBreakMaxFontSize = chFontSize;

                // New line so reset line advance info.
                adv=0;
                chAdv=0;
                wBreakIdx = -1;
                continue;
            }

            if (!chIsSpace) {
                // Don't include spaces in max font size calc.
                if (chFontSize > wBreakMaxFontSize) {
                    wBreakMaxFontSize = chFontSize;
                }

                // Remember this as last non-space char...
                lastChar = i;

                // Reset char advance (incorporated into adv above).
                chAdv = 0;
            }

            // Increment everything for ext char...
            aciIndex += gv.getCharacterCount(i,i);
            ch = aci.setIndex(aciIndex);
            chIsSpace = isSpace(ch);
        }

        // Include max font since break char in max.
        if (wBreakMaxFontSize > maxFontSize)
            maxFontSize = wBreakMaxFontSize;
        dy += (maxFontSize*.8f+prevDesc)*1.1f;
        if ((dy + .2f*maxFontSize) >= height) {
            // Last line of text didn't fit...
            i = lineStart;
        } else {
            Rectangle2D lastCharB = gv.getGlyphVisualBounds
                (lastChar).getBounds2D();
            Point2D     lastCharL = gv.getGlyphPosition(lastChar);
            float charW   = (float)
                (lastCharB.getX()+lastCharB.getWidth()-
                 lastCharL.getX());
            float charAdv = gp[2*lastChar+2]-gp[2*lastChar];

            lineStarts.add (new Integer(i));
            lineLocs.add   (new Point2D.Float(x0, y0+dy));
            lineAdvs.add   (new Float(adv-(charAdv-charW)));
            lineAdjs.add   (new Float(adv+chAdv));
            lineLastCharW.add (new Float(charW));
            linePartial.add(new Boolean(true));
        }

        // ignore any glyphs i+  (didn't fit in rects.)  
        for (int j=i; j<numGlyphs; j++) 
            gv.setGlyphVisible(j, false);

        // Limit ourselves to i glyphs...
        numGlyphs = i;


        Iterator lStartIter    =lineStarts.iterator();
        Iterator lLocIter      =lineLocs.iterator();
        Iterator lAdvIter      =lineAdvs.iterator();
        Iterator lAdjIter      =lineAdjs.iterator();
        Iterator lLastCharWIter=lineLastCharW.iterator();
        Iterator lPartIter      =linePartial.iterator();

        lineStart             = ((Integer)lStartIter.next()).intValue();
        Point2D.Float lineLoc = null;
        float         lineAdv = 0;
        float         lineAdj = 0;

        float xOrig=gp[0];
        float yOrig=gp[1];

        float xScale=1;
        float xAdj=0;
        float charW;

        // This loop goes through and puts glyphs where they belong
        // based on info collected in first trip through glyphVector...
        i =0;
        for (; i<numGlyphs; i++) {
            if (i == lineStart) {
                // Always comes through here on first char...

                // Update offset for new line based on last line length
                xOrig += lineAdj;

                // Get new values for everything...
                lineStart = ((Integer)lStartIter.next()).intValue();
                lineLoc   = (Point2D.Float)lLocIter.next();
                lineAdv   = (  (Float)lAdvIter.next())  .floatValue();
                lineAdj   = (  (Float)lAdjIter.next())  .floatValue();
                charW     = (  (Float)lLastCharWIter.next())  .floatValue();
                partial   = ((Boolean)lPartIter.next()) .booleanValue();

                xAdj = 0;
                xScale = 1;
                // Recalc justification info.
                switch (justification) {
                case 0: default: break;                  // Left
                case 1: xAdj = (width-lineAdv)/2; break; // Center
                case 2: xAdj =  width-lineAdv;    break; // Right
                case 3:                                  // Full
                    if ((!partial) && (lineStart != i+1)) {
                        // More than one char on line...
                        // Scale char spacing to fill line.
                        xScale = (width-charW)/(lineAdv-charW);
                    }
                    break;
                }
            }
            float x = lineLoc.x + (gp[2*i]  -xOrig)*xScale+xAdj;
            float y = lineLoc.y + (gp[2*i+1]-yOrig);
            gv.setGlyphPosition(i, new Point2D.Float(x, y));
        }

        float x = lineLoc.x + (gp[2*i]  -xOrig)*xScale+xAdj;
        float y = lineLoc.y + (gp[2*i+1]-yOrig);
        gv.setGlyphPosition(i, new Point2D.Float(x, y));
    }

    protected static boolean isSpace(char ch) {
        return ((ch == ' ') || (ch == '\t'));
    }
}
