/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import com.ibm.uddi.response.DispositionReport;

/**
 * Represents an UDDI defined error. This exception is thrown in cases
 * where the return value on the UDDI API cannot indicate an error condition.<P>
 * 
 * UDDIException usually contains a Disposition report that has detailed
 * information about the UDDI error as defined by the UDDI specification.
 * If the response is a SOAP fault, but does not contains a disposition
 * report, this exception may still be thrown.<P>
 * 
 * The DispositionReport class contains defines for the various error values
 * that can be returned.<P>
 * 
 * @author David Melgar
 */
public class UDDIException extends Exception {

   static final String UDDI_TAG = "Fault";

   String faultCode     = null;
   String faultString   = null;
   String faultActor    = null;
   String detail        = null;

   Element detailElement= null;
   DispositionReport dispositionReport = null;

   public UDDIException() {
   }

   /**
    * Constructer that parses the XML dom tree and extracts
    * useful attributes.
    * 
    * @param el     Root element of the tree within the SOAP body.
    */
   public UDDIException(Element el, boolean createDispositionReport) {
      if (isValidElement(el)) {
         // Extract useful attributes
         NodeList nl;
         Element tmp;
         nl = el.getElementsByTagName("faultcode");
         if (nl!=null && nl.getLength()>0) {
            tmp = (Element)nl.item(0);
            // I'm not sure this is correct. Need to test
            faultCode = getText(tmp);
         }
         nl = el.getElementsByTagName("faultstring");
         if (nl!=null && nl.getLength()>0) {
            tmp = (Element)nl.item(0);
            // I'm not sure this is correct. Need to test
            faultString = getText(tmp);
         }
         nl = el.getElementsByTagName("faultactor");
         if (nl!=null && nl.getLength()>0) {
            tmp = (Element)nl.item(0);
            // I'm not sure this is correct. Need to test
            faultActor = getText(tmp);
         }
         nl = el.getElementsByTagName("detail");
         // Try to create a disposition report
         if (nl!=null && nl.getLength()>0) {
            tmp = (Element)nl.item(0);
            detailElement = tmp;
            if (createDispositionReport) {
               try {
                  nl = el.getElementsByTagName(DispositionReport.UDDI_TAG);
                  if (nl!=null && nl.getLength()>0) {
                     tmp = (Element)nl.item(0);
                     dispositionReport = new DispositionReport(tmp);
                  }
               } catch (UDDIException e) {
                  // Ignore exception, we're handling it already
               }
            }
         }
      }
   }

   /**
    * Tests the passed in element to determine if the
    * element is a serialized version of this object.
    * 
    * @param el     Root element for this object
    */
   public static boolean isValidElement(Element el) {
      // This method can be in the base class even if this class sets the value
      // Handle soapFault if it returns a dom tree. Parse out values
      // Fault DOES show up as a "Fault" element within the body
      return UDDI_TAG.equals(el.getNodeName());
   }

   // Getters
   public String getFaultCode() {
      return faultCode;
   }

   public String getFaultString() {
      return faultString;
   }

   public String getFaultActor() {
      return faultActor;
   }

   public String getDetail() {
      return detail;
   }

   public Element getDetailElement() {
      return detailElement;
   }

   public DispositionReport getDispositionReport() {
      return dispositionReport;
   }

   /**
    * Utility function.
    * Returns text contained in child elements of the
    * passed in element
    * 
    * @param el     Element
    * @return 
    */
   protected String getText(Node el) {
      NodeList nl = el.getChildNodes();
      String result = "";
      for (int i = 0; i < nl.getLength(); i++) {
         if (nl.item(i).getNodeType()==Element.TEXT_NODE) {
            result += nl.item(i).getNodeValue();
         }
      }
      // Trim result to remove whitespace
      return result.trim();
   }

   /**
   * Provide exception message 
   * @return java.lang.String
   */
   public String toString() {
      if (dispositionReport != null) {
         return dispositionReport.getErrInfoText();
      }
      return getFaultString();
   }
}
