/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.client;

import com.ibm.uddi.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import java.net.URL;
import java.util.Arrays;
import java.util.Vector;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.apache.soap.Body;
import org.apache.soap.Envelope;
import org.apache.soap.SOAPException;
import org.apache.soap.messaging.Message;
import org.apache.soap.transport.SOAPTransport;
import org.w3c.dom.Element;
import com.ibm.uddi.datatype.business.BusinessEntity;
import com.ibm.uddi.datatype.service.BusinessService;
import com.ibm.uddi.datatype.tmodel.TModel;
import com.ibm.uddi.response.AuthToken;
import com.ibm.uddi.response.BindingDetail;
import com.ibm.uddi.response.BusinessDetail;
import com.ibm.uddi.response.BusinessDetailExt;
import com.ibm.uddi.response.BusinessList;
import com.ibm.uddi.response.DispositionReport;
import com.ibm.uddi.response.RegisteredInfo;
import com.ibm.uddi.response.ServiceDetail;
import com.ibm.uddi.response.ServiceList;
import com.ibm.uddi.response.TModelDetail;
import com.ibm.uddi.response.TModelList;
import com.ibm.uddi.util.AuthInfo;
import com.ibm.uddi.util.CategoryBag;
import com.ibm.uddi.util.DiscoveryURLs;
import com.ibm.uddi.util.FindQualifiers;
import com.ibm.uddi.util.IdentifierBag;
import com.ibm.uddi.util.TModelBag;
import com.ibm.uddi.util.UploadRegister;

/**
 * Represents a UDDI server and the actions that can be
 * invoked against it.<P>
 * 
 * The API is described in the UDDI API specification
 * available from <A HREF="http://www.uddi.org">
 * http://www.uddi.org </A>.
 * 
 * The UDDI API specification is required to understand
 * and utilize this API. This class attempts to closely
 * pattern the API document.<P>
 * 
 * @author David Melgar
 */
public class UDDIProxy {

   // Variables
   URL inquiryURL = null;
   URL publishURL = null;
   SOAPTransport transport = null;


   /**
    * Default constructor. 
   */
   public UDDIProxy() {
   }

   /**
    * Construct a UDDIProxy object.
    * 
    * @param inquiryURL URL to be used for inquiry requests.
    * @param publishURL URL to be used for publish requests.
    * @param transport  null indicates standard HTTP transport. Can pass in
    *                   a different transport to use.
    */
   public UDDIProxy(URL inquiryURL, URL publishURL, SOAPTransport transport) {
      this();
      this.inquiryURL = inquiryURL;
      this.publishURL = publishURL;
      this.transport = transport;
   }

   /**
    * Specifies the SOAP Transport to use.
    * If not set, defaults to the standard HTTP transport.
    * 
    * @param transport
    */
   public void setTransport(SOAPTransport transport) {
      this.transport = transport;
   }

   /**
    * Set the URL to be used for inquiry requests.
    * 
    * @param url
    * @exception java.net.MalformedURLException
    */
   public void setInquiryURL(String url) throws java.net.MalformedURLException {
      this.inquiryURL = new java.net.URL(url);
   }
   /**
    * Set the URL to be used for publish requests. If
    * HTTPS is specified as the procotol, it must be added
    * as a supported protocol. For Sun's SSL support, this can be done
    * with the following code fragment:
    * <PRE>
    *    System.setProperty("java.protocol.handler.pkgs",
    *                       "com.sun.net.ssl.internal.www.protocol");
    *    java.security.Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());
    * </PRE>
    * 
    * @param url
    * @exception java.net.MalformedURLException
    *                   Thrown if HTTPS is not registered as a valid URL protocol.
    */
   public void setPublishURL(String url) throws java.net.MalformedURLException {
      this.publishURL = new java.net.URL(url);
   }


   /**
    * The find_binding method returns a bindingDetail message that contains
    * a bindingTemplates structure with zero or more bindingTemplate structures
    * matching the criteria specified in the argument list.
    * 
    * @param findQualifiers
    *                   This collection of findQualifier elements can be used to alter the default
    *                   behavior of search functionality.
    * @param serviceKey Used to specify a particular instance of a businessService element
    *                   in the registered data.  Only bindings in the specific businessService
    *                   data identified by the serviceKey passed will be searched.
    * @param tmodelbag  This is a list of tModel uuid_key values that represent the technical
    *                   fingerprint to locate in a bindingTemplate structure contained within
    *                   the businessService instance specified by the serviceKey value.  If more
    *                   than one tModel key is specified in this structure, only bindingTemplate
    *                   information that exactly matches all of the tModel keys specified will
    *                   be returned (logical AND).  The order of the keys in the tModel bag is
    *                   not relevant.  All tModelKey values begin with a uuid URN qualifier
    *                   (e.g. "uuid:" followed by a known tModel UUID value.
    * @param maxRows    This optional integer value allows the requesting program to limit
    *                   the number of results returned.
    * @return This function returns a bindingDetail message on success.  In the event that no matches were located for the specified criteria, the bindingDetail structure returned in the response the will be empty (e.g. contain no bindingTemplate data.)
    *         In the even of a large number of matches, an Operator Site may truncate the result set.  If this occurs, the response message will contain the truncated attribute with the value of this attribute set to true.
    *         Searching using tModelBag will also return any bindingTemplate information that matches due to hostingRedirector references.  The resolved bindingTemplate structure will be returned, even if that bindingTemplate is owned by a different businessService structure.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BindingDetail find_binding(FindQualifiers findQualifiers,
                                     String serviceKey,
                                     TModelBag tmodelbag,
                                     int maxRows) 
          throws UDDIException, SOAPException {
      FindBinding request = new FindBinding();                
      request.setFindQualifiers(findQualifiers);
      request.setServiceKey(serviceKey);
      request.setTModelBag(tmodelbag);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new BindingDetail(send(request, true));
   }   

   /**
    * Find business matching specified criteria.
    * 
    * @param name    Partial business name to match. Leftmost match.
    * @param findQualifiers
    *                Optional findQualifiers. null indicates no find
    *                qualifiers.
    * @param maxRows Maximum number of results. 0 indicates no maximum.
    * @return This function returns a businessList on success.  In the event that no 
    *         matches were located for the specified criteria, a businessList
    *         structure with zero businessInfo structures is returned.
    * @exception UDDIException
    *                   Contains a DispositionReport that indicates the
    *                   error number.
    * @exception SOAPException
    *                   Thrown if non-UDDI related communication errors occur.
    */
   public BusinessList find_business(String name, FindQualifiers findQualifiers, int maxRows) 
          throws UDDIException, SOAPException {
      FindBusiness request = new FindBusiness();
      request.setName(name);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new BusinessList(send(request, true));
   }   
   /**
    * The find_business message returns a businessList message that matches
    * the conditions specified in the arguments.
    * 
    * @param bag     This is a list of business identifier references. The returned businessList
    *                contains businessInfo structures matching any of the identifiers passed
    *                (logical OR).
    * @param findQualifiers
    *                can be used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a businessList on success.  In the event that no 
    *         matches were located for the specified criteria, a businessList
    *         structure with zero businessInfo structures is returned.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessList find_business(IdentifierBag bag, FindQualifiers findQualifiers, int maxRows) 
          throws UDDIException, SOAPException {
      FindBusiness request = new FindBusiness();
      request.setIdentifierBag(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new BusinessList(send(request, true));
   }

   /**
    * The find_business message returns a businessList message that matches
    * the conditions specified in the arguments.
    * 
    * @param bag     This is a list of category references.  The returned businessList
    *                contains businessInfo structures matching all of the categories
    *                passed (logical AND).
    * @param findQualifiers
    *                can be used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a businessList on success.  In the event that no 
    *         matches were located for the specified criteria, a businessList
    *         structure with zero businessInfo structures is returned.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessList find_business(CategoryBag bag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindBusiness request = new FindBusiness();
      request.setCategoryBag(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new BusinessList(send(request, true));
   }   

   /**
    * The find_business message returns a businessList message that matches
    * the conditions specified in the arguments.
    * 
    * @param bag     The registered businessEntity data contains bindingTemplates that in turn
    *                contain specific tModel references.  The tModelBag argument lets you
    *                search for businesses that have bindings that are compatible with a
    *                specific tModel pattern.  The returned businessList contains businessInfo
    *                structures that match all of the tModel keys passed (logical AND).
    *                tModelKey values must be formatted as URN qualified UUID values
    *                (e.g. prefixed with "uuid:")
    * @param findQualifiers
    *                can be used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a businessList on success.  In the event that no 
    *         matches were located for the specified criteria, a businessList
    *         structure with zero businessInfo structures is returned.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessList find_business(TModelBag bag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindBusiness request = new FindBusiness();
      request.setTModelBag(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new BusinessList(send(request, true));
   }   

   /**
    * The find_business message returns a businessList message that matches
    * the conditions specified in the arguments.
    * 
    * @param bag     This is a list of URL's to be matched against the data associated
    *                with the discoveryURL's contents of registered businessEntity information.
    *                To search for URL without regard to useType attribute values, pass
    *                the useType component of the discoveryURL elements as empty attributes.
    *                If useType values are included, then the match will be made only on
    *                registered information that match both the useType and URL value.
    *                The returned businessList contains businessInfo structures matching
    *                any of the URL's passed (logical OR).
    * @param findQualifiers
    *                can be used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a businessList on success.  In the event that no 
    *         matches were located for the specified criteria, a businessList
    *         structure with zero businessInfo structures is returned.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessList find_business(DiscoveryURLs bag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindBusiness request = new FindBusiness();
      request.setDiscoveryURLs(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new BusinessList(send(request, true));
   }   

   /**
    * This function returns a serviceList on success.  In the event that no
    * matches were located for the specified criteria, the serviceList
    * structure returned will contain an empty businessServices structure.
    * 
    * @param businessKey
    *                used to specify a particular  BusinessEntity instance.
    * @param name    Represents a partial name.  Any businessService data contained in
    *                the specified businessEntity with a matching partial name value gets returned.
    * @param findQualifiers
    *                Used to alter the default behavior of search functionality.
    * @param maxRows Allows the requesting program to limit the number of results returned.
    *                A value of 0 indicates no limit.
    * @return This function returns a serviceList on success.  In the event that no 
    *         matches were located for the specified criteria, the serviceList
    *         structure returned will contain an empty businessServices structure.
    * @exception UDDIException
    * @exception SOAPException
    */
   public ServiceList find_service(String businessKey, String name, FindQualifiers findQualifiers, int maxRows) 
          throws UDDIException, SOAPException {
      FindService request = new FindService();
      request.setBusinessKey(businessKey);
      request.setName(name);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new ServiceList(send(request, true));
   }   
   /**
    * This function returns a serviceList on success.  In the event that no
    * matches were located for the specified criteria, the serviceList
    * structure returned will contain an empty businessServices structure.
    * 
    * @param businessKey
    *                This uuid_key is used to specify a particular  BusinessEntity instance.
    * @param bag     This is a list of category references.  The returned serviceList contains
    *                businessInfo structures matching all of the categories passed (logical AND).
    * @param findQualifiers
    *                used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a serviceList on success.  In the event that no
    *         matches were located for the specified criteria, the serviceList
    *         structure returned will contain an empty businessServices structure.
    * @exception UDDIException
    * @exception SOAPException
    */
   public ServiceList find_service(String businessKey, CategoryBag bag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindService request = new FindService();
      request.setBusinessKey(businessKey);
      request.setCategoryBag(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new ServiceList(send(request, true));
   }   
   /**
    * This function returns a serviceList on success.  In the event that no
    * matches were located for the specified criteria, the serviceList
    * structure returned will contain an empty businessServices structure.
    * 
    * @param businessKey
    *                This uuid_key is used to specify a particular  BusinessEntity instance.
    * @param bag
    * @param findQualifiers
    *                used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a serviceList on success.  In the event that no
    *         matches were located for the specified criteria, the serviceList
    *         structure returned will contain an empty businessServices structure.
    * @exception UDDIException
    * @exception SOAPException
    */
   public ServiceList find_service(String businessKey, TModelBag bag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindService request = new FindService();
      request.setBusinessKey(businessKey);
      request.setTModelBag(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new ServiceList(send(request, true));
   }   
   
   
   /**
    * This find_tModel message is for locating a list of tModel entries
    * that match a set of specific criteria. The response will be a list
    * of abbreviated information about tModels that match the criteria (tModelList).
    * 
    * @param name    This string value  represents a partial name.  The returned tModelList
    *                contains tModelInfo structures for businesses whose name matches the
    *                value passed (leftmost match).
    * @param findQualifiers
    *                used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a tModelList on success.  In the event that no 
    *         matches were located for the specified criteria, an empty tModelList
    *         object will be returned (e.g. will contain zero tModelInfo objects).
    *         This signifies zero matches
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelList find_tModel(String name, FindQualifiers findQualifiers, int maxRows) 
          throws UDDIException, SOAPException {
      FindTModel request = new FindTModel();
      request.setName(name);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new TModelList(send(request, true));
   }   
   /**
    * This find_tModel message is for locating a list of tModel entries
    * that match a set of specific criteria. The response will be a list
    * of abbreviated information about tModels that match the criteria (tModelList).
    * 
    * @param bag     This is a list of category references.  The returned tModelList contains
    *                tModelInfo structures matching all of the categories passed (logical AND).
    * @param findQualifiers
    *                used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a tModelList on success.  In the event that no 
    *         matches were located for the specified criteria, an empty tModelList
    *         object will be returned (e.g. will contain zero tModelInfo objects).
    *         This signifies zero matches
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelList find_tModel(CategoryBag bag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindTModel request = new FindTModel();
      request.setCategoryBag(bag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new TModelList(send(request, true));
   }   
   /**
    * This find_tModel message is for locating a list of tModel entries
    * that match a set of specific criteria. The response will be a list
    * of abbreviated information about tModels that match the criteria (tModelList).
    * 
    * @param identifierBag
    *                This is a list of business identifier references. The returned tModelList
    *                contains tModelInfo structures matching any of the identifiers
    *                passed (logical OR).
    * @param findQualifiers
    *                used to alter the default behavior of search functionality.
    * @param maxRows allows the requesting program to limit the number of results returned.
    * @return This function returns a tModelList on success.  In the event that no 
    *         matches were located for the specified criteria, an empty tModelList
    *         object will be returned (e.g. will contain zero tModelInfo objects).
    *         This signifies zero matches.
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelList find_tModel(IdentifierBag identifierBag, FindQualifiers findQualifiers, int maxRows)
          throws UDDIException, SOAPException {
      FindTModel request = new FindTModel();
      request.setIdentifierBag(identifierBag);
      request.setFindQualifiers(findQualifiers);
      if (maxRows!=0) request.setMaxRows(maxRows);
      return new TModelList(send(request, true));
   }   
   
   
   /**
    * The get_bindingDetail message is for requesting the run-time
    * bindingTemplate information location information for the purpose of
    * invoking a registered business API.
    * 
    * @param bindingKey uuid_key string that represent specific instance
    *                   of known bindingTemplate data.
    * @return This function returns a bindingDetail message on successful match
    * @exception UDDIException
    * @exception SOAPException
    */
   public BindingDetail get_bindingDetail(String bindingKey) 
          throws UDDIException, SOAPException {
      GetBindingDetail request = new GetBindingDetail();
      Vector keys = new Vector();
      keys.addElement(bindingKey);
      request.setBindingKeyStrings(keys);
      return new BindingDetail(send(request, true));
   }
   /**
    * The get_bindingDetail message is for requesting the run-time
    * bindingTemplate information location information for the purpose of
    * invoking a registered business API.
    * 
    * @param bindingKey Vector of uuid_key strings that represent specific instances
    *                   of known bindingTemplate data.
    * @return This function returns a bindingDetail message on successful match of one
    *         or more bindingKey values.  If multiple bindingKey values were passed, the
    *         results will be returned in the same order as the keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BindingDetail get_bindingDetail(Vector bindingKeyStrings) 
          throws UDDIException, SOAPException {
      GetBindingDetail request = new GetBindingDetail();
      request.setBindingKeyStrings(bindingKeyStrings);
      return new BindingDetail(send(request, true));
   }

   /**
    * The get_businessDetail message returns complete businessEntity information
    * for one or more specified businessEntitys
    * 
    * @param businessKey
    *               A uuid_key string that represents a specific instance of known
    *               businessEntity data.
    * @return This function returns a businessDetail object on successful match
    *         of one or more businessKey values.  If multiple businessKey values
    *         were passed, the results will be returned in the same order as the
    *         keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessDetail get_businessDetail(String businessKey) 
          throws UDDIException, SOAPException {
      GetBusinessDetail request = new GetBusinessDetail();
      Vector keys = new Vector();
      keys.addElement(businessKey);
      request.setBusinessKeyStrings(keys);
      return new BusinessDetail(send(request, true));
   }
   /**
    * The get_businessDetail message returns complete businessEntity information
    * for one or more specified businessEntitys
    * 
    * @param businessKeyStrings
    *               Vector of uuid_key strings that represent specific instances of known
    *               businessEntity data.
    * @return This function returns a businessDetail message on successful match
    *         of one or more businessKey values.  If multiple businessKey values
    *         were passed, the results will be returned in the same order as the
    *         keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessDetail get_businessDetail(Vector businessKeyStrings) 
          throws UDDIException, SOAPException {
      GetBusinessDetail request = new GetBusinessDetail();
      request.setBusinessKeyStrings(businessKeyStrings);
      return new BusinessDetail(send(request, true));
   }

   /**
    * The get_businessDetailExt message returns extended businessEntity
    * information for one or more specified businessEntitys.  This
    * message returns exactly the same information as the get_businessDetail
    * message, but may contain additional attributes if the source is
    * an external registry (not an Operator Site) that is compatible
    * with this API specification.
    * 
    * @param businessKey
    *               A uuid_key string that represents a specific instance of known
    *               businessEntity data.
    * @return This function returns a businessDetailExt message on successful match 
    *         of one or more businessKey values.  If multiple businessKey values
    *         were passed, the results will be returned in the same order as the
    *         keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessDetailExt get_businessDetailExt(String businessKey) 
          throws UDDIException, SOAPException {
      GetBusinessDetailExt request = new GetBusinessDetailExt();
      Vector keys = new Vector();
      keys.addElement(businessKey);
      return new BusinessDetailExt(send(request, true));
   }
   /**
    * The get_businessDetailExt message returns extended businessEntity
    * information for one or more specified businessEntitys.  This
    * message returns exactly the same information as the get_businessDetail
    * message, but may contain additional attributes if the source is
    * an external registry (not an Operator Site) that is compatible
    * with this API specification.
    * 
    * @param businessKeyStrings
    *               Vector of uuid_key strings that represent specific instances of known
    *               businessEntity data.
    * @return This function returns a businessDetailExt message on successful match 
    *         of one or more businessKey values.  If multiple businessKey values
    *         were passed, the results will be returned in the same order as the
    *         keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessDetailExt get_businessDetailExt(Vector businessKeyStrings) 
          throws UDDIException, SOAPException {
      GetBusinessDetailExt request = new GetBusinessDetailExt();
      request.setBusinessKeyStrings(businessKeyStrings);
      return new BusinessDetailExt(send(request, true));
   }

   /**
    * The get_serviceDetail message is used to request full information
    * about a known businessService structure.
    * 
    * @param serviceKey A uuid_key string that represents a specific instance of
    *                   known businessService data.
    * @return This function returns a serviceDetail message on successful match 
    *         of one or more serviceKey values.  If multiple serviceKey values
    *         were passed, the results will be returned in the same order as the
    *         keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public ServiceDetail get_serviceDetail(String serviceKey) 
          throws UDDIException, SOAPException {
      GetServiceDetail request = new GetServiceDetail();
      Vector keys = new Vector();
      keys.addElement(serviceKey);
      request.setServiceKeyStrings(keys);
      return new ServiceDetail(send(request, true));
   }

   /**
    * The get_serviceDetail message is used to request full information
    * about a known businessService structure.
    * 
    * @param serviceKeyStrings
    *               A vector of uuid_key strings that represent specific instances of
    *               known businessService data.
    * @return This function returns a serviceDetail message on successful match 
    *         of one or more serviceKey values.  If multiple serviceKey values
    *         were passed, the results will be returned in the same order as the
    *         keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public ServiceDetail get_serviceDetail(Vector serviceKeyStrings) 
          throws UDDIException, SOAPException {
      GetServiceDetail request = new GetServiceDetail();
      request.setServiceKeyStrings(serviceKeyStrings);
      return new ServiceDetail(send(request, true));
   }

   /**
    * The get_tModelDetail message is used to request full information
    * about a known tModel structure.
    * 
    * @param tModelKey A URN qualified uuid_key string that represent a specific
    *                  instance of known tModel data.  All tModelKey values begin with a
    *                  uuid URN qualifier (e.g. "uuid:" followed by a known tModel UUID value.)
    * @return This function returns a tModelDetail message on successful match
    *         of one or more tModelKey values.  If multiple tModelKey values
    *         were passed, the results will be returned in the same order as
    *         the keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelDetail get_tModelDetail(String tModelKey) 
          throws UDDIException, SOAPException {
      GetTModelDetail request = new GetTModelDetail();
      Vector keys = new Vector();
      keys.addElement(tModelKey);
      request.setTModelKeyStrings(keys);
      return new TModelDetail(send(request, true));
   }

   /**
    * The get_tModelDetail message is used to request full information
    * about a known tModel structure.
    * 
    * @param tModelKeyStrings
    *               A Vector of URN qualified uuid_key strings that represent specific
    *               instances of known tModel data.  All tModelKey values begin with a
    *               uuid URN qualifier (e.g. "uuid:" followed by a known tModel UUID value.)
    * @return This function returns a tModelDetail message on successful match
    *         of one or more tModelKey values.  If multiple tModelKey values
    *         were passed, the results will be returned in the same order as
    *         the keys passed.
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelDetail get_tModelDetail(Vector tModelKeyStrings) 
          throws UDDIException, SOAPException {
      GetTModelDetail request = new GetTModelDetail();
      request.setTModelKeyStrings(tModelKeyStrings);
      return new TModelDetail(send(request, true));
   }
   
   /**
    * The delete_binding message causes one or more bindingTemplate to be deleted.
    * 
    * @param authInfo   Contains an authentication token. Authentication tokens are obtained
    *                   using the get_authToken method.
    * @param bindingKey A uuid_key value that represents a specific instance of
    *                   known bindingTemplate data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator.
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_binding(String authInfo, String bindingKey) 
          throws UDDIException, SOAPException {
      DeleteBinding request = new DeleteBinding();
      request.setAuthInfo(authInfo);
      Vector keys = new Vector();
      keys.addElement(bindingKey);
      request.setBindingKeyStrings(keys);
      return new DispositionReport(send(request, false));
   }
   /**
    * The delete_binding message causes one or more bindingTemplate to be deleted.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param bindingKeyStrings
    *                 A vector of uuid_key strings that represents specific instances of
    *                 known bindingTemplate data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_binding(String authInfo, Vector bindingKeyStrings) 
          throws UDDIException, SOAPException {
      DeleteBinding request = new DeleteBinding();
      request.setAuthInfo(authInfo);
      request.setBindingKeyStrings(bindingKeyStrings);
      return new DispositionReport(send(request, false));
   }

   /**
    * The delete_business message is used to remove one or more
    * businessEntity structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param businessKey
    *                 Uuid_key string that represents specific instance of known
    *                 businessEntity data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_business(String authInfo, String businessKey) 
          throws UDDIException, SOAPException {
      DeleteBusiness request = new DeleteBusiness();
      request.setAuthInfo(authInfo);
      Vector keys = new Vector();
      keys.addElement(businessKey);
      request.setBusinessKeyStrings(keys);
      return new DispositionReport(send(request, false));
   }
   /**
    * The delete_business message is used to remove one or more 
    * businessEntity structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param businessKeyStrings
    *                 Vector of uuid_key strings that represent specific instances of known
    *                 businessEntity data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_business(String authInfo, Vector businessKeyStrings) 
          throws UDDIException, SOAPException {
      DeleteBusiness request = new DeleteBusiness();
      request.setAuthInfo(authInfo);
      request.setBusinessKeyStrings(businessKeyStrings);
      return new DispositionReport(send(request, false));
   }

   /**
    * The delete_service message is used to remove one or more 
    * businessService structures.
    * 
    * @param authInfo   Contains an authentication token. Authentication tokens are obtained
    *                   using the get_authToken method.
    * @param serviceKey uuid_key string that represents specific instance of known
    *                   businessService data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_service(String authInfo, String serviceKey) 
          throws UDDIException, SOAPException {
      DeleteService request = new DeleteService();
      request.setAuthInfo(authInfo);
      Vector keys = new Vector();
      keys.addElement(serviceKey);
      request.setServiceKeyStrings(keys);
      return new DispositionReport(send(request, false));
   }
   /**
    * The delete_service message is used to remove one or more 
    * businessService structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param serviceKeyStrings
    *                 Vector of uuid_key strings that represent specific instances of known
    *                 businessService data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_service(String authInfo, Vector serviceKeyStrings) 
          throws UDDIException, SOAPException {
      DeleteService request = new DeleteService();
      request.setAuthInfo(authInfo);
      request.setServiceKeyStrings(serviceKeyStrings);
      return new DispositionReport(send(request, false));
   }
   
   /**
    * The delete_tModel message is used to remove or retire one or more 
    * tModel structures.
    * 
    * @param authInfo  Contains an authentication token. Authentication tokens are obtained
    *                  using the get_authToken method.
    * @param tModelKey uuid_key string that represents specific instance of known
    *                  tModel data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_tModel(String authInfo, String tModelKey) 
          throws UDDIException, SOAPException {
      DeleteTModel request = new DeleteTModel();
      request.setAuthInfo(authInfo);
      Vector keys = new Vector();
      keys.addElement(tModelKey);
      request.setTModelKeyStrings(keys);
      return new DispositionReport(send(request, false));
   }
   /**
    * The delete_tModel message is used to remove or retire one or more 
    * tModel structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param tModelKeyStrings
    *                 Vector of uuid_key strings that represent specific instances of known
    *                 tModel data.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport delete_tModel(String authInfo, Vector tModelKeyStrings) 
          throws UDDIException, SOAPException {
      DeleteTModel request = new DeleteTModel();
      request.setAuthInfo(authInfo);
      request.setTModelKeyStrings(tModelKeyStrings);
      return new DispositionReport(send(request, false));
   }
   
   /**
    * The discard_authToken message is used to inform an Operator Site that the
    * authentication token can be discarded.  Subsequent calls that use the
    * same authToken may be rejected.  This message is optional for Operator
    * Sites that do not manage session state or that do not support the
    * get_authToken message.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @return Upon successful completion, a dispositionReport is returned with a 
    *         single success indicator.  Discarding an expired authToken will be
    *         processed and reported as a success condition.
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport discard_authToken(AuthInfo authInfo) 
          throws UDDIException, SOAPException {
      DiscardAuthToken request = new DiscardAuthToken();
      request.setAuthInfo(authInfo);
      return new DispositionReport(send(request, false));
   }
   
   /**
    * The get_authToken message is used to obtain an authentication token.  
    * Authentication tokens are opaque values that are required for all 
    * other publisher API calls.  This message is not required for Operator 
    * Sites that have an external mechanism defined for users to get an 
    * authentication token.  This API is provided for implementations that 
    * do not have some other method of obtaining an authentication token or 
    * certificate, or that choose to use userID and Password based authentication.
    * 
    * @param userid user that an individual authorized user was assigned by an Operator Site.
    *               Operator Sites will each provide a way for individuals to obtain a UserID
    *               and password that will be valid only at the given Operator Site.
    * @param cred   password or credential that is associated with the user.
    * @return This function returns an authToken object that contains a valid 
    *         authInfo object that can be used in subsequent calls to publisher
    *         API calls that require an authInfo value.
    * @exception UDDIException
    * @exception SOAPException
    */
   public AuthToken get_authToken(String userid, String cred) 
          throws UDDIException, SOAPException {
      GetAuthToken request = new GetAuthToken();
      request.setUserID(userid);
      request.setCred(cred);
      return new AuthToken(send(request, false));
   }


   /**
    * The get_registeredInfo message is used to get an abbreviated list
    * of all businessEntity keys and tModel keys that are controlled by
    * the individual associated the credentials passed.
    * 
    * @param authInfo Contains an authentication token.  Authentication tokens are obtained
    *                 using the get_authToken API call.
    * @return Upon successful completion, a registeredInfo object will be returned, 
    *         listing abbreviated business information in one or more businessInfo
    *         objects, and tModel information in one or more tModelInfo objects.
    *         This API is useful for determining the full extent of registered
    *         information controlled by a single user in a single call.
    * @exception UDDIException
    * @exception SOAPException
    */
   public RegisteredInfo get_registeredInfo(String authInfo) 
          throws UDDIException, SOAPException {
      GetRegisteredInfo request = new GetRegisteredInfo();
      request.setAuthInfo(authInfo);
      return new RegisteredInfo(send(request, false));
   }

   /**
    * The save_binding message is used to save or update a complete 
    * bindingTemplate structure.  This message can be used to add or 
    * update one or more bindingTemplate structures to one or more existing 
    * businessService structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param bindingTemplates
    *                 Vector of bindingTemplate objects.  The order in which these are
    *                 processed is not defined.  To save a new bindingTemplate, pass a
    *                 bindingTemplate object with an empty bindingKey attribute value.
    * @return This API returns a bindingDetail object containing the final results 
    *         of the call that reflects the newly registered information for the
    *         effected bindingTemplate objects.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BindingDetail save_binding(String authInfo, Vector bindingTemplates) 
          throws UDDIException, SOAPException {
      SaveBinding request = new SaveBinding();
      request.setAuthInfo(authInfo);
      request.setBindingTemplateVector(bindingTemplates);
      return new BindingDetail(send(request, false));
   }

   /**
    * The save_business message is used to save or update information about a
    * complete businessEntity structure.  This API has the broadest scope of
    * all of the save_x API calls in the publisher API, and can be used to make
    * sweeping changes to the published information for one or more
    * businessEntity structures controlled by an individual.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param businessEntities
    *                 Vector of businessEntity objects.  These objects can be obtained in advance
    *                 by using the get_businessDetail API call or by any other means.
    * @return This API returns a businessDetail message containing the final results
    *         of the call that reflects the new registered information for the
    *         businessEntity information provided.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessDetail save_business(String authInfo, Vector businessEntities) 
          throws UDDIException, SOAPException {
      SaveBusiness request = new SaveBusiness();
      request.setAuthInfo(authInfo);
      request.setBusinessEntityVector(businessEntities);
      return new BusinessDetail(send(request, false));
   }
   /**
    * The save_business message is used to save or update information about a
    * complete businessEntity structure.  This API has the broadest scope of
    * all of the save_x API calls in the publisher API, and can be used to make
    * sweeping changes to the published information for one or more
    * businessEntity structures controlled by an individual.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param uploadRegisters
    *                 Vector of UpLoadRegister objects which contain resolvable HTTP URL
    *                 addresses that each point to a single and valid businessEntity
    *                 or businessEntityExt structure.  This allows a registry to be updated
    *                 to reflect the contents of an XML document that is URL addressable.
    *                 The URL must return a pure XML document that only contains a
    *                 businessEntity structure as its top-level element, and be accessible
    *                 using the standard HTTP-GET protocol.
    * @return This API returns a businessDetail message containing the final results 
    *         of the call that reflects the new registered information for the
    *         businessEntity information provided.
    * @exception UDDIException
    * @exception SOAPException
    */
   public BusinessDetail save_business(String authInfo, UploadRegister[] uploadRegisters) 
          throws UDDIException, SOAPException {
      SaveBusiness request = new SaveBusiness();
      Vector keys = new Vector( Arrays.asList(uploadRegisters) );
      request.setAuthInfo(authInfo);
      request.setUploadRegisterVector(keys);
      return new BusinessDetail(send(request, false));
   }


   /**
    * The save_service message adds or updates one or more businessService 
    * structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param businessServices
    *                 Vector of businessService objects.  These objects can be obtained in
    *                 advance by using the get_serviceDetail API call or by any other means.
    * @return This API returns a serviceDetail object containing the final results 
    *         of the call that reflects the newly registered information for the
    *         effected businessService structures.
    * @exception UDDIException
    * @exception SOAPException
    */
   public ServiceDetail save_service(String authInfo, Vector businessServices) 
          throws UDDIException, SOAPException {
      SaveService request = new SaveService();
      request.setAuthInfo(authInfo);
      request.setBusinessServiceVector(businessServices);
      return new ServiceDetail(send(request, false));
   }

   /**
    * The save_tModel message adds or updates one or more tModel structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param tModels  Vector of complete tModel structures.  If adding a new tModel,
    *                 the tModelKey value should be passed as an empty element.
    * @return This API returns a tModelDetail message containing the final results 
    *         of the call that reflects the new registered information for the
    *         effected tModel structures
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelDetail save_tModel(String authInfo, Vector tModels) 
          throws UDDIException, SOAPException {
      SaveTModel request = new SaveTModel();
      request.setAuthInfo(authInfo);
      request.setTModelVector(tModels);
      return new TModelDetail(send(request, false));
   }

   /**
    * The save_tModel message adds or updates one or more tModel structures.
    * 
    * @param authInfo Contains an authentication token. Authentication tokens are obtained
    *                 using the get_authToken method.
    * @param uploadRegisters
    *                 Vector of UploadRegister objects that contain resolvable HTTP URL
    *                 addresses that each point to a single and valid tModel structure.
    *                 This allows a registry to be updated to reflect the contents of
    *                 an XML document that is URL addressable.  The URL must return a
    *                 pure XML document that only contains a tModel structure as its
    *                 top-level element, and be accessible using the standard HTTP-GET protocol.
    * @return This API returns a tModelDetail message containing the final results
    *         of the call that reflects the new registered information for the
    *         effected tModel structures
    * @exception UDDIException
    * @exception SOAPException
    */
   public TModelDetail save_tModel(String authInfo, UploadRegister[] uploadRegisters) 
          throws UDDIException, SOAPException {
      SaveTModel request = new SaveTModel();
      request.setAuthInfo(authInfo);
      Vector keys = new Vector( Arrays.asList(uploadRegisters) );
      request.setUploadRegisterVector(keys);
      return new TModelDetail(send(request, false));
   }

   /**
    * The validate_categorization service performs two functions.
    * It is used to verify that a specific category (keyValue) exists
    * within the given taxonomy.  The service also optionally restricts
    * the entities that may be classified within the category.
    * 
    * @param tModelKey The identifier of a registered tModel that is used as a namespace
    *                  qualifier that implies a specific taxonomy.
    * @param keyValueString
    *                  The category identifier  of the category within the identified taxonomy.
    * @param businessEntity
    *                  The businessEntity structure being categorized
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator.
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport validate_categorization(String tModelKey,
                                                    String keyValueString,
                                                    BusinessEntity businessEntity)
          throws UDDIException, SOAPException {
      ValidateCategorization request = new com.ibm.uddi.request.ValidateCategorization();
      request.setTModelKey(tModelKey);
      request.setKeyValue(keyValueString);
      request.setBusinessEntity(businessEntity);
      return new DispositionReport(send(request, true));
   }
   /**
    * The validate_categorization service performs two functions.
    * It is used to verify that a specific category (keyValue) exists
    * within the given taxonomy.  The service also optionally restricts
    * the entities that may be classified within the category.
    * 
    * @param tModelKey The identifier of a registered tModel that is used as a namespace
    *                  qualifier that implies a specific taxonomy.
    * @param keyValueString
    *                  The category identifier  of the category within the identified taxonomy.
    * @param businessService
    *                  The business service structure being categorized.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator.
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport validate_categorization(String tModelKey,
                                                    String keyValueString,
                                                    BusinessService businessService)
          throws UDDIException, SOAPException {
      ValidateCategorization request = new com.ibm.uddi.request.ValidateCategorization();
      request.setTModelKey(tModelKey);
      request.setKeyValue(keyValueString);
      request.setBusinessService(businessService);
      return new DispositionReport(send(request, true));
   }
   /**
    * The validate_categorization service performs two functions.
    * It is used to verify that a specific category (keyValue) exists
    * within the given taxonomy.  The service also optionally restricts
    * the entities that may be classified within the category.
    * 
    * @param tModelKey The identifier of a registered tModel that is used as a namespace
    *                  qualifier that implies a specific taxonomy.
    * @param keyValueString
    *                  The category identifier  of the category within the identified taxonomy.
    * @param tModel    The tModel structure being categorized.
    * @return Upon successful completion, a dispositionReport is returned with a
    *         single success indicator.
    * @exception UDDIException
    * @exception SOAPException
    */
   public DispositionReport validate_categorization(String tModelKey,
                                                    String keyValueString,
                                                    TModel tModel) 
          throws UDDIException, SOAPException {
      ValidateCategorization request = new com.ibm.uddi.request.ValidateCategorization();
      request.setTModelKey(tModelKey);
      request.setKeyValue(keyValueString);
      request.setTModel(tModel);
      return new DispositionReport(send(request, true));
   }

   /**
    * Sends a UDDIElement to either the inquiry or publish URL.
    * 
    * @param el
    * @param inquiry
    * @return An element representing a XML DOM tree containing the UDDI response.
    * @exception SOAPException
    */
   public Element send(UDDIElement el, boolean inquiry) throws SOAPException {
      Element base = null;
      try {
         DocumentBuilder docBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
         base = docBuilder.newDocument().createElement("tmp");
      } catch (Exception e) {
         e.printStackTrace();
      }

      el.saveToXML(base);
      return send((Element)base.getFirstChild(), inquiry);
   }

   /**
    * Sends an XML DOM tree indentified by the given element to either the
    * inquiry or publish URL.
    * 
    * @param el
    * @param inquiry
    * @return An element representing a XML DOM tree containing the UDDI response.
    * @exception SOAPException
    */
   public Element send(Element el, boolean inquiry) throws SOAPException {
      Envelope sendEnv = new Envelope();
      Body sendBody = new Body();

      Vector bodyEntry = new Vector();
      bodyEntry.add(el);
      sendBody.setBodyEntries(bodyEntry);

      sendEnv.setBody(sendBody);

      Message soapMessage = new Message();

      soapMessage.setSOAPTransport(transport);

      Element response = null;
      if (inquiry) {
         soapMessage.send(inquiryURL, "", sendEnv);
      } else {
         soapMessage.send(publishURL, "", sendEnv);
      }
      Envelope responseEnv = soapMessage.receiveEnvelope();

      Body responseBody = responseEnv.getBody();
      Element base = (Element)responseBody.getBodyEntries().firstElement();

      return base;
   }
}
