/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.request;
import java.util.Vector;
import org.w3c.dom.*;
import com.ibm.uddi.*;
import com.ibm.uddi.datatype.*;
import com.ibm.uddi.datatype.binding.*;
import com.ibm.uddi.datatype.business.*;
import com.ibm.uddi.datatype.service.*;
import com.ibm.uddi.datatype.tmodel.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import com.ibm.uddi.util.*;

/**
 * <p><b>General information:</b><p>
 *
 * This class represents an element within the UDDI version 1.0 schema.
 * This class contains the following types of methods:<ul>
 *
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an XML DOM element
 *       that is the appropriate element for this object.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>For sets of attributes, a get/setVector method is provided.
 *   <li>SaveToXML method. Serialized this class within a passed in element.
 *
 * </ul>
 * Typically this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 * 
 * <p><b>Element description:</b><p>
 * This message is used to delete information about a previously
 * registered bindingTemplate structure.
 * 
 * <p>
 *
 * @author David Melgar
 */
public class DeleteBinding extends UDDIElement {
   public static final String UDDI_TAG = "delete_binding";

   protected Element base = null;

   AuthInfo authInfo = null;
   // Vector of BindingKey objects
   Vector bindingKey = new Vector();

   /**
    * Default constructor.
    * Use of this constructor should be avoided. Use the required fields
    * constructor to provide validation. No validation of required
    * fields is performed when using the default constructor.
    *
    */
   public DeleteBinding() {
   }

   /**
    * Construct the object with required fields.
    *
    * @param authInfo	String
    * @param bindingKey	Vector of BindingKey Strings.
    */
   public DeleteBinding(String authInfo,
			Vector bindingKeyStrings) {
      this.authInfo = new AuthInfo( authInfo );
      Vector objects;
      objects = new Vector();
      for (int i = 0; i < bindingKeyStrings.size(); i++) {
         objects.addElement( new BindingKey((String)bindingKeyStrings.elementAt(i)) );
      }

      this.bindingKey = objects;
   }

   /**
    * Construct the object from a DOM tree. Used by 
    * UDDIProxy to construct object from received UDDI
    * message.
    * 
    * @param base   Element with name appropriate for this class.
    * @exception UDDIException
    *                   Thrown if DOM tree contains a SOAP fault or
    *                   disposition report indicating a UDDI error.
    */
   public DeleteBinding(Element base) throws UDDIException {
      // Check if its a fault. Throw exception if it is
      super(base);
      NodeList nl = null;
      nl = getChildElementsByTagName(base, AuthInfo.UDDI_TAG);
      if (nl.getLength() > 0) {
         authInfo = new AuthInfo((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, BindingKey.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         bindingKey.addElement(new BindingKey((Element)nl.item(i)));
      }
   }

   public void setAuthInfo(AuthInfo s) {
      authInfo = s;
   }
   public void setAuthInfo(String s) {
      authInfo = new AuthInfo();
      authInfo.setText(s);
   }

   /**
    * Set bindingKey vector
    *
    * @param s	Vector of <I>BindingKey</I> objects.
    */
   public void setBindingKeyVector(Vector s) {
      bindingKey = s;
   }

   /**
    * Set bindingKey
    *
    * @param s	Vector of <I>String</I> objects.
    */
   public void setBindingKeyStrings(Vector s) {
      bindingKey = new Vector();
      for (int i = 0; i < s.size(); i++) {
         bindingKey.addElement( new BindingKey((String)s.elementAt(i)) );
      }
   }

   public AuthInfo getAuthInfo() {
      return authInfo;
   }

   public String getAuthInfoString() {
      return authInfo.getText();
   }

   /**
    * Get bindingKey
    *
    * @return s	Vector of <I>BindingKey</I> objects.
    */
   public Vector getBindingKeyVector() {
      return bindingKey;
   }

   /**
    * Get bindingKey
    *
    * @return s	Vector of <I>String</I> objects.
    */
   public Vector getBindingKeyStrings() {
      Vector strings = new Vector();
      for (int i = 0; i < bindingKey.size(); i++) {
         strings.addElement( ((BindingKey)bindingKey.elementAt(i)).getText());
      }
      return strings;
   }

   /**
    * Save object to DOM tree. Used to serialize object
    * to a DOM tree, typically to send a UDDI message.
    * 
    * <BR>Used by UDDIProxy.
    * 
    * @param parent Object will serialize as a child element under the
    *               passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElement(UDDI_TAG);
      // Save attributes
      base.setAttribute("generic", UDDIElement.GENERIC);
      base.setAttribute("xmlns", UDDIElement.XMLNS);
      if (authInfo!=null) {
         authInfo.saveToXML(base);
      }
      for (int i=0; i < bindingKey.size(); i++) {
         ((BindingKey)(bindingKey.elementAt(i))).saveToXML(base);
      }
      parent.appendChild(base);
   }
}
