/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.request;
import java.util.Vector;
import org.w3c.dom.*;
import com.ibm.uddi.*;
import com.ibm.uddi.datatype.*;
import com.ibm.uddi.datatype.binding.*;
import com.ibm.uddi.datatype.business.*;
import com.ibm.uddi.datatype.service.*;
import com.ibm.uddi.datatype.tmodel.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import com.ibm.uddi.util.*;

/**
 * <p><b>General information:</b><p>
 *
 * This class represents an element within the UDDI version 1.0 schema.
 * This class contains the following types of methods:<ul>
 *
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an XML DOM element
 *       that is the appropriate element for this object.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>For sets of attributes, a get/setVector method is provided.
 *   <li>SaveToXML method. Serialized this class within a passed in element.
 *
 * </ul>
 * Typically this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 * 
 * <p><b>Element description:</b><p>
 * The extended businessDetail messages define an API that allows non-operator
 * nodes to provide extended information via a consistent API.  This message
 * is the request that will cause a businessDetailExt message to be returned.
 * 
 * <p>
 *
 * @author David Melgar
 */
public class GetBusinessDetailExt extends UDDIElement {
   public static final String UDDI_TAG = "get_businessDetailExt";

   protected Element base = null;

   // Vector of BusinessKey objects
   Vector businessKey = new Vector();

   /**
    * Default constructor.
    * Use of this constructor should be avoided. Use the required fields
    * constructor to provide validation. No validation of required
    * fields is performed when using the default constructor.
    *
    */
   public GetBusinessDetailExt() {
   }

   /**
    * Construct the object with required fields.
    *
    * @param businessKey	Vector of BusinessKey Strings.
    */
   public GetBusinessDetailExt(Vector businessKeyStrings) {
      Vector objects;
      objects = new Vector();
      for (int i = 0; i < businessKeyStrings.size(); i++) {
         objects.addElement( new BusinessKey((String)businessKeyStrings.elementAt(i)) );
      }

      this.businessKey = objects;
   }

   /**
    * Construct the object from a DOM tree. Used by 
    * UDDIProxy to construct object from received UDDI
    * message.
    * 
    * @param base   Element with name appropriate for this class.
    * @exception UDDIException
    *                   Thrown if DOM tree contains a SOAP fault or
    *                   disposition report indicating a UDDI error.
    */
   public GetBusinessDetailExt(Element base) throws UDDIException {
      // Check if its a fault. Throw exception if it is
      super(base);
      NodeList nl = null;
      nl = getChildElementsByTagName(base, BusinessKey.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         businessKey.addElement(new BusinessKey((Element)nl.item(i)));
      }
   }

   /**
    * Set businessKey vector
    *
    * @param s	Vector of <I>BusinessKey</I> objects.
    */
   public void setBusinessKeyVector(Vector s) {
      businessKey = s;
   }

   /**
    * Set businessKey
    *
    * @param s	Vector of <I>String</I> objects.
    */
   public void setBusinessKeyStrings(Vector s) {
      businessKey = new Vector();
      for (int i = 0; i < s.size(); i++) {
         businessKey.addElement( new BusinessKey((String)s.elementAt(i)) );
      }
   }

   /**
    * Get businessKey
    *
    * @return s	Vector of <I>BusinessKey</I> objects.
    */
   public Vector getBusinessKeyVector() {
      return businessKey;
   }

   /**
    * Get businessKey
    *
    * @return s	Vector of <I>String</I> objects.
    */
   public Vector getBusinessKeyStrings() {
      Vector strings = new Vector();
      for (int i = 0; i < businessKey.size(); i++) {
         strings.addElement( ((BusinessKey)businessKey.elementAt(i)).getText());
      }
      return strings;
   }

   /**
    * Save object to DOM tree. Used to serialize object
    * to a DOM tree, typically to send a UDDI message.
    * 
    * <BR>Used by UDDIProxy.
    * 
    * @param parent Object will serialize as a child element under the
    *               passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElement(UDDI_TAG);
      // Save attributes
      base.setAttribute("generic", UDDIElement.GENERIC);
      base.setAttribute("xmlns", UDDIElement.XMLNS);
      for (int i=0; i < businessKey.size(); i++) {
         ((BusinessKey)(businessKey.elementAt(i))).saveToXML(base);
      }
      parent.appendChild(base);
   }
}
