/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.request;
import java.util.Vector;
import org.w3c.dom.*;
import com.ibm.uddi.*;
import com.ibm.uddi.datatype.*;
import com.ibm.uddi.datatype.binding.*;
import com.ibm.uddi.datatype.business.*;
import com.ibm.uddi.datatype.service.*;
import com.ibm.uddi.datatype.tmodel.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import com.ibm.uddi.util.*;

/**
 * <p><b>General information:</b><p>
 *
 * This class represents an element within the UDDI version 1.0 schema.
 * This class contains the following types of methods:<ul>
 *
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an XML DOM element
 *       that is the appropriate element for this object.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>For sets of attributes, a get/setVector method is provided.
 *   <li>SaveToXML method. Serialized this class within a passed in element.
 *
 * </ul>
 * Typically this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 * 
 * <p><b>Element description:</b><p>
 * This message is used to save (add/update) information about
 * one or more bindingTemplate structures.
 * 
 * <p>
 *
 * @author David Melgar
 */
public class SaveBinding extends UDDIElement {
   public static final String UDDI_TAG = "save_binding";

   protected Element base = null;

   AuthInfo authInfo = null;
   // Vector of BindingTemplate objects
   Vector bindingTemplate = new Vector();

   /**
    * Default constructor.
    * Use of this constructor should be avoided. Use the required fields
    * constructor to provide validation. No validation of required
    * fields is performed when using the default constructor.
    *
    */
   public SaveBinding() {
   }

   /**
    * Construct the object with required fields.
    *
    * @param authInfo	String
    * @param bindingTemplate	Vector of BindingTemplate objects.
    */
   public SaveBinding(String authInfo,
			Vector bindingTemplate) {
      this.authInfo = new AuthInfo( authInfo );
      Vector objects;
      this.bindingTemplate = bindingTemplate;
   }

   /**
    * Construct the object from a DOM tree. Used by 
    * UDDIProxy to construct object from received UDDI
    * message.
    * 
    * @param base   Element with name appropriate for this class.
    * @exception UDDIException
    *                   Thrown if DOM tree contains a SOAP fault or
    *                   disposition report indicating a UDDI error.
    */
   public SaveBinding(Element base) throws UDDIException {
      // Check if its a fault. Throw exception if it is
      super(base);
      NodeList nl = null;
      nl = getChildElementsByTagName(base, AuthInfo.UDDI_TAG);
      if (nl.getLength() > 0) {
         authInfo = new AuthInfo((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, BindingTemplate.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         bindingTemplate.addElement(new BindingTemplate((Element)nl.item(i)));
      }
   }

   public void setAuthInfo(AuthInfo s) {
      authInfo = s;
   }
   public void setAuthInfo(String s) {
      authInfo = new AuthInfo();
      authInfo.setText(s);
   }

   /**
    * Set bindingTemplate vector
    *
    * @param s	Vector of <I>BindingTemplate</I> objects.
    */
   public void setBindingTemplateVector(Vector s) {
      bindingTemplate = s;
   }

   public AuthInfo getAuthInfo() {
      return authInfo;
   }

   public String getAuthInfoString() {
      return authInfo.getText();
   }

   /**
    * Get bindingTemplate
    *
    * @return s	Vector of <I>BindingTemplate</I> objects.
    */
   public Vector getBindingTemplateVector() {
      return bindingTemplate;
   }

   /**
    * Save object to DOM tree. Used to serialize object
    * to a DOM tree, typically to send a UDDI message.
    * 
    * <BR>Used by UDDIProxy.
    * 
    * @param parent Object will serialize as a child element under the
    *               passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElement(UDDI_TAG);
      // Save attributes
      base.setAttribute("generic", UDDIElement.GENERIC);
      base.setAttribute("xmlns", UDDIElement.XMLNS);
      if (authInfo!=null) {
         authInfo.saveToXML(base);
      }
      for (int i=0; i < bindingTemplate.size(); i++) {
         ((BindingTemplate)(bindingTemplate.elementAt(i))).saveToXML(base);
      }
      parent.appendChild(base);
   }
}
