/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.request;
import java.util.Vector;
import org.w3c.dom.*;
import com.ibm.uddi.*;
import com.ibm.uddi.datatype.*;
import com.ibm.uddi.datatype.binding.*;
import com.ibm.uddi.datatype.business.*;
import com.ibm.uddi.datatype.service.*;
import com.ibm.uddi.datatype.tmodel.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import com.ibm.uddi.util.*;

/**
 * <p><b>General information:</b><p>
 *
 * This class represents an element within the UDDI version 1.0 schema.
 * This class contains the following types of methods:<ul>
 *
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an XML DOM element
 *       that is the appropriate element for this object.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>For sets of attributes, a get/setVector method is provided.
 *   <li>SaveToXML method. Serialized this class within a passed in element.
 *
 * </ul>
 * Typically this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 * 
 * <p><b>Element description:</b><p>
 * This message is used to save (add/update) information describing
 * one or more businessEntity structures.  One of businessEntity or
 * uploadRegister is required.  Invalid if contains both or neither type.
 * 
 * <p>
 *
 * @author David Melgar
 */
public class SaveBusiness extends UDDIElement {
   public static final String UDDI_TAG = "save_business";

   protected Element base = null;

   AuthInfo authInfo = null;
   // Vector of BusinessEntity objects
   Vector businessEntity = new Vector();
   // Vector of UploadRegister objects
   Vector uploadRegister = new Vector();

   /**
    * Default constructor.
    * Use of this constructor should be avoided. Use the required fields
    * constructor to provide validation. No validation of required
    * fields is performed when using the default constructor.
    *
    */
   public SaveBusiness() {
   }

   /**
    * Construct the object with required fields.
    *
    * @param authInfo	String
    */
   public SaveBusiness(String authInfo) {
      this.authInfo = new AuthInfo( authInfo );
   }

   /**
    * Construct the object from a DOM tree. Used by 
    * UDDIProxy to construct object from received UDDI
    * message.
    * 
    * @param base   Element with name appropriate for this class.
    * @exception UDDIException
    *                   Thrown if DOM tree contains a SOAP fault or
    *                   disposition report indicating a UDDI error.
    */
   public SaveBusiness(Element base) throws UDDIException {
      // Check if its a fault. Throw exception if it is
      super(base);
      NodeList nl = null;
      nl = getChildElementsByTagName(base, AuthInfo.UDDI_TAG);
      if (nl.getLength() > 0) {
         authInfo = new AuthInfo((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, BusinessEntity.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         businessEntity.addElement(new BusinessEntity((Element)nl.item(i)));
      }
      nl = getChildElementsByTagName(base, UploadRegister.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         uploadRegister.addElement(new UploadRegister((Element)nl.item(i)));
      }
   }

   public void setAuthInfo(AuthInfo s) {
      authInfo = s;
   }
   public void setAuthInfo(String s) {
      authInfo = new AuthInfo();
      authInfo.setText(s);
   }

   /**
    * Set businessEntity vector
    *
    * @param s	Vector of <I>BusinessEntity</I> objects.
    */
   public void setBusinessEntityVector(Vector s) {
      businessEntity = s;
   }

   /**
    * Set uploadRegister vector
    *
    * @param s	Vector of <I>UploadRegister</I> objects.
    */
   public void setUploadRegisterVector(Vector s) {
      uploadRegister = s;
   }

   /**
    * Set uploadRegister
    *
    * @param s	Vector of <I>String</I> objects.
    */
   public void setUploadRegisterStrings(Vector s) {
      uploadRegister = new Vector();
      for (int i = 0; i < s.size(); i++) {
         uploadRegister.addElement( new UploadRegister((String)s.elementAt(i)) );
      }
   }

   public AuthInfo getAuthInfo() {
      return authInfo;
   }

   public String getAuthInfoString() {
      return authInfo.getText();
   }

   /**
    * Get businessEntity
    *
    * @return s	Vector of <I>BusinessEntity</I> objects.
    */
   public Vector getBusinessEntityVector() {
      return businessEntity;
   }

   /**
    * Get uploadRegister
    *
    * @return s	Vector of <I>UploadRegister</I> objects.
    */
   public Vector getUploadRegisterVector() {
      return uploadRegister;
   }

   /**
    * Get uploadRegister
    *
    * @return s	Vector of <I>String</I> objects.
    */
   public Vector getUploadRegisterStrings() {
      Vector strings = new Vector();
      for (int i = 0; i < uploadRegister.size(); i++) {
         strings.addElement( ((UploadRegister)uploadRegister.elementAt(i)).getText());
      }
      return strings;
   }

   /**
    * Save object to DOM tree. Used to serialize object
    * to a DOM tree, typically to send a UDDI message.
    * 
    * <BR>Used by UDDIProxy.
    * 
    * @param parent Object will serialize as a child element under the
    *               passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElement(UDDI_TAG);
      // Save attributes
      base.setAttribute("generic", UDDIElement.GENERIC);
      base.setAttribute("xmlns", UDDIElement.XMLNS);
      if (authInfo!=null) {
         authInfo.saveToXML(base);
      }
      for (int i=0; i < businessEntity.size(); i++) {
         ((BusinessEntity)(businessEntity.elementAt(i))).saveToXML(base);
      }
      for (int i=0; i < uploadRegister.size(); i++) {
         ((UploadRegister)(uploadRegister.elementAt(i))).saveToXML(base);
      }
      parent.appendChild(base);
   }
}
