/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.response;

import java.util.Vector;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import com.ibm.uddi.UDDIElement;
import com.ibm.uddi.UDDIException;


/**
 * Represents an UDDI defined error. This object is returned from certain
 * UDDI API calls. Other UDDI APIs return a data object if success and
 * generate a fault message if failure. The API returns the fault
 * by throwing a UDDIException.<P>
 * 
 * UDDIException usually contains a Disposition report that has detailed
 * information about the UDDI error as defined by the UDDI specification.
 * If the response is a SOAP fault, but does not contains a disposition
 * report, a UDDIException may be thrown without containing a 
 * DispositionReport.<P>
 * 
 * This class contains defines for the various error values
 * that can be returned.<P>
 * 
 * @author David Melgar
 */
public class DispositionReport extends UDDIElement {

   public static String UDDI_TAG = "dispositionReport";
   // XML attributes. Looks common to all message responses
   String genericAttrib  = null;
   String operatorAttrib = null;
   int    errno          = 0;
   String errCode        = null;
   String errInfoText    = null;
   
   /* Defines of possible error codes that can be returned */

   /**
    * signifies that the authentication token value passed in the authInfo argument 
    * is no longer valid because the token has expired.
    */
   public static final String E_authTokenExpired = "E_authTokenExpired";


   /**
    * signifies that the authentication token value passed in the authInfo argument
    * is either missing or is not valid.
    */
   public static final String E_authTokenRequired = "E_authTokenRequired";

   /**
    * signifies that the request cannot be satisfied because one or more uuid_key
    * values specified has previously been hidden or removed by the requester.
    * This specifically applies to the tModelKey values passed.
    */
   public static final String E_keyRetired = "E_keyRetired";

   /**
    * signifies that the request cannot be satisfied because one or more uuid_key
    * values specified is not a valid key value.  This will occur if a uuid_key
    * value is passed in a tModel that does not match with any known tModel key.
    */
   public static final String E_invalidKeyPassed = "E_invalidKeyPassed";

   /**
    * an error occurred with one of the uploadRegister URL values.
    */
   public static final String E_invalidURLPassed = "E_invalidURLPassed";

   /**
    * signifies that one or more of the uuid_key values passed refers to data
    * that is not controlled by the individual who is represented by the authentication token.
    */
   public static final String E_userMismatch = "E_userMismatch";

   /**
    * signifies that one or more of the uuid_key values passed refers to data
    * that is not controlled by the Operator Site that received the request for processing.
    */
   public static final String E_operatorMismatch = "E_operatorMismatch";

   /**
    * signifies that the given keyValue did not correspond to a category within
    * the taxonomy identified by a tModelKey value within one of the categoryBag elements provided.
    */
   public static final String E_invalidCategory = "E_invalidCategory";

   /**
    * Restrictions have been placed by the taxonomy provider on the types of
    * information that should be included at that location within a specific
    * taxonomy.  The validation routine chosen by the Operator Site has
    * rejected this tModel for at least one specified category.
    */
   public static final String E_categorizationNotAllowed = "E_categorizationNotAllowed";

   /**
    * signifies that user account limits have been exceeded.
    */
   public static final String E_accountLimitExceeded = "E_accountLimitExceeded";


   /**
    * Constructer that parses the XML dom tree and extracts
    * useful attributes.
    * 
    * @param el     Root element of the tree within the SOAP body.
    */
   public DispositionReport(Element el) throws UDDIException {
      if (UDDIException.isValidElement(el)) {
         UDDIException e = new UDDIException(el, false);
         // Is there a disposition report?
         NodeList nl = e.getDetailElement().getElementsByTagName(UDDI_TAG);
         if (nl.getLength()>0) {
            base = (Element)nl.item(0);
         } else {
            // Not a disposition report, some other error, throw it
            throw e;
         }
      } else {
         base = el;
      }

      if (isValidElement(base)) {
         // Extract useful attributes
         NodeList nl;
         Element tmp;
         // Extract attribute values
         genericAttrib = el.getAttribute("generic");
         operatorAttrib = el.getAttribute("operator");

         // Process embedded elements
         nl = el.getElementsByTagName("result");
         if (nl!=null && nl.getLength()>0) {
            tmp = (Element)nl.item(0);

            String errnoString = tmp.getAttribute("errno");
            Integer errnoInt = new Integer(errnoString);
            errno = errnoInt.intValue();

            nl = tmp.getElementsByTagName("errInfo");
            if (nl.getLength()>0) {
               tmp = (Element)nl.item(0);
               errCode = tmp.getAttribute("errCode");
               errInfoText = getText(tmp);
            }
         }
      }
   }

   // Getters
   public String getGeneric() {
      return this.genericAttrib;
   }

   public String getOperator() {
      return this.operatorAttrib;
   }

   public boolean success() {
      return (errno==0);
   }

   public int getErrno() {
      return errno;
   }

   public String getErrCode() {
      return errCode;
   }

   public String getErrInfoText() {
      return errInfoText;
   }

   /**
    * Tests the passed in element to determine if the
    * element is a serialized version of this object.
    * 
    * @param el     Root element for this object
    */
   public boolean isValidElement(Element el) {
      return el.getNodeName().equals(UDDI_TAG);
   }

   public void saveToXML(Element el) {
      // Not implemented
   }
}
