/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.request;
import java.util.Vector;
import org.w3c.dom.*;
import com.ibm.uddi.*;
import com.ibm.uddi.datatype.*;
import com.ibm.uddi.datatype.binding.*;
import com.ibm.uddi.datatype.business.*;
import com.ibm.uddi.datatype.service.*;
import com.ibm.uddi.datatype.tmodel.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import com.ibm.uddi.util.*;

/**
 * <p><b>General information:</b><p>
 *
 * This class represents an element within the UDDI version 1.0 schema.
 * This class contains the following types of methods:<ul>
 *
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an XML DOM element
 *       that is the appropriate element for this object.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>For sets of attributes, a get/setVector method is provided.
 *   <li>SaveToXML method. Serialized this class within a passed in element.
 *
 * </ul>
 * Typically this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 * 
 * <p><b>Element description:</b><p>
 * <p>
 *
 * @author David Melgar
 */
public class ValidateCategorization extends UDDIElement {
   public static final String UDDI_TAG = "validate_categorization";

   protected Element base = null;

   TModelKey tModelKey = null;
   KeyValue keyValue = null;
   BusinessEntity businessEntity = null;
   BusinessService businessService = null;
   TModel tModel = null;

   /**
    * Default constructor.
    * Use of this constructor should be avoided. Use the required fields
    * constructor to provide validation. No validation of required
    * fields is performed when using the default constructor.
    *
    */
   public ValidateCategorization() {
   }

   /**
    * Construct the object with required fields.
    *
    * @param tModelKey	String
    * @param keyValue	String
    */
   public ValidateCategorization(String tModelKey,
			String keyValue) {
      this.tModelKey = new TModelKey( tModelKey );
      this.keyValue = new KeyValue( keyValue );
   }

   /**
    * Construct the object from a DOM tree. Used by 
    * UDDIProxy to construct object from received UDDI
    * message.
    * 
    * @param base   Element with name appropriate for this class.
    * @exception UDDIException
    *                   Thrown if DOM tree contains a SOAP fault or
    *                   disposition report indicating a UDDI error.
    */
   public ValidateCategorization(Element base) throws UDDIException {
      // Check if its a fault. Throw exception if it is
      super(base);
      NodeList nl = null;
      nl = getChildElementsByTagName(base, TModelKey.UDDI_TAG);
      if (nl.getLength() > 0) {
         tModelKey = new TModelKey((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, KeyValue.UDDI_TAG);
      if (nl.getLength() > 0) {
         keyValue = new KeyValue((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, BusinessEntity.UDDI_TAG);
      if (nl.getLength() > 0) {
         businessEntity = new BusinessEntity((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, BusinessService.UDDI_TAG);
      if (nl.getLength() > 0) {
         businessService = new BusinessService((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, TModel.UDDI_TAG);
      if (nl.getLength() > 0) {
         tModel = new TModel((Element)nl.item(0));
      }
   }

   public void setTModelKey(TModelKey s) {
      tModelKey = s;
   }
   public void setTModelKey(String s) {
      tModelKey = new TModelKey();
      tModelKey.setText(s);
   }

   public void setKeyValue(KeyValue s) {
      keyValue = s;
   }
   public void setKeyValue(String s) {
      keyValue = new KeyValue();
      keyValue.setText(s);
   }

   public void setBusinessEntity(BusinessEntity s) {
      businessEntity = s;
   }

   public void setBusinessService(BusinessService s) {
      businessService = s;
   }

   public void setTModel(TModel s) {
      tModel = s;
   }

   public TModelKey getTModelKey() {
      return tModelKey;
   }

   public String getTModelKeyString() {
      return tModelKey.getText();
   }

   public KeyValue getKeyValue() {
      return keyValue;
   }

   public String getKeyValueString() {
      return keyValue.getText();
   }

   public BusinessEntity getBusinessEntity() {
      return businessEntity;
   }


   public BusinessService getBusinessService() {
      return businessService;
   }


   public TModel getTModel() {
      return tModel;
   }


   /**
    * Save object to DOM tree. Used to serialize object
    * to a DOM tree, typically to send a UDDI message.
    * 
    * <BR>Used by UDDIProxy.
    * 
    * @param parent Object will serialize as a child element under the
    *               passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElement(UDDI_TAG);
      // Save attributes
      base.setAttribute("generic", UDDIElement.GENERIC);
      base.setAttribute("xmlns", UDDIElement.XMLNS);
      if (tModelKey!=null) {
         tModelKey.saveToXML(base);
      }
      if (keyValue!=null) {
         keyValue.saveToXML(base);
      }
      if (businessEntity!=null) {
         businessEntity.saveToXML(base);
      }
      if (businessService!=null) {
         businessService.saveToXML(base);
      }
      if (tModel!=null) {
         tModel.saveToXML(base);
      }
      parent.appendChild(base);
   }
}
