/*
 * The source code contained herein is licensed under the IBM Public License   
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.         
 * 
 */

package com.ibm.uddi.util;
import java.util.Vector;
import org.w3c.dom.*;
import com.ibm.uddi.*;
import com.ibm.uddi.datatype.*;
import com.ibm.uddi.datatype.binding.*;
import com.ibm.uddi.datatype.business.*;
import com.ibm.uddi.datatype.service.*;
import com.ibm.uddi.datatype.tmodel.*;
import com.ibm.uddi.request.*;
import com.ibm.uddi.response.*;
import com.ibm.uddi.util.*;

/**
 * <p><b>General information:</b><p>
 *
 * This class represents an element within the UDDI version 1.0 schema.
 * This class contains the following types of methods:<ul>
 *
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an XML DOM element
 *       that is the appropriate element for this object.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>For sets of attributes, a get/setVector method is provided.
 *   <li>SaveToXML method. Serialized this class within a passed in element.
 *
 * </ul>
 * Typically this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 * 
 * <p><b>Element description:</b><p>
 * Service element.  Used in searching and categorization.
 * 
 * <p>
 *
 * @author David Melgar
 */
public class IdentifierBag extends UDDIElement {
   public static final String UDDI_TAG = "identifierBag";

   protected Element base = null;

   // Vector of KeyedReference objects
   Vector keyedReference = new Vector();

   /**
    * Default constructor.
    *
    */
   public IdentifierBag() {
   }

   /**
    * Construct the object from a DOM tree. Used by 
    * UDDIProxy to construct object from received UDDI
    * message.
    * 
    * @param base   Element with name appropriate for this class.
    * @exception UDDIException
    *                   Thrown if DOM tree contains a SOAP fault or
    *                   disposition report indicating a UDDI error.
    */
   public IdentifierBag(Element base) throws UDDIException {
      // Check if its a fault. Throw exception if it is
      super(base);
      NodeList nl = null;
      nl = getChildElementsByTagName(base, KeyedReference.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         keyedReference.addElement(new KeyedReference((Element)nl.item(i)));
      }
   }

   /**
    * Set keyedReference vector
    *
    * @param s	Vector of <I>KeyedReference</I> objects.
    */
   public void setKeyedReferenceVector(Vector s) {
      keyedReference = s;
   }

   /**
    * Get keyedReference
    *
    * @return s	Vector of <I>KeyedReference</I> objects.
    */
   public Vector getKeyedReferenceVector() {
      return keyedReference;
   }

   /**
    * Save object to DOM tree. Used to serialize object
    * to a DOM tree, typically to send a UDDI message.
    * 
    * <BR>Used by UDDIProxy.
    * 
    * @param parent Object will serialize as a child element under the
    *               passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElement(UDDI_TAG);
      // Save attributes
      for (int i=0; i < keyedReference.size(); i++) {
         ((KeyedReference)(keyedReference.elementAt(i))).saveToXML(base);
      }
      parent.appendChild(base);
   }
}
