/*
 * @(#)interrupt.h	1.10 95/11/29  
 *
 * Copyright (c) 1994 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

/*
 * Green threads interrupt dispatch interface
 *
 * NOTE: The Solaris-specific portion of the interrupt API is documented
 * 	 in interrupt_md.h
 */

#ifndef	_GREEN_INTERRUPT_H_
#define	_GREEN_INTERRUPT_H_

#include "bool.h"

/*
 * Type definitions.
 */

/*-
 * A function that handles interrupt dispatch requests is of type
 * intr_handler_t.  This type definition is mostly for convenience.
 * A declaration of a handler function, should look like this:
 *
 *	int myHandler(int interrupt, void *arg, void *info);
 *
 * An intr_handler_t is constrained:
 *
 *	- It runs on the exception stack.
 *	- It cannot yield.
 *	- It cannot allocate/free memory.
 *	- It can only call interrupt-safe routines.
 *	
 * An intr_handler_t returns:
 *
 *	1:	Interrupt handled; stop searching.
 *	0:	Interrupt not handled; keep searching.
 *
 * "arg" is set to the "handlerArg" specified in intrRegister().
 */
typedef int (*intr_handler_t)(int interrupt, void *arg, void *siginfo);


#define	N_INTERRUPTS	32

/*
 * Routines.
 */

/* Initialize the interrupt system */
extern void intrInit(void);

/* Add/Remove a handler for a particular interrupt */
extern void intrRegister(int interrupt, intr_handler_t handler,
			 void *handlerArg);
extern void intrUnregister(int interrupt, intr_handler_t handler,
			   void *handlerArg);

/* Dispatch an interrupt (called from the low-level handlers) */
extern void intrDispatch(int interrupt, void *siginfo);

/* Check whether we are running in the interrupt context. */
extern bool_t intrContext(void);


/*-
 * intrEnableDispatch(), intrDisableDispatch() --
 *	Enable/Disable target-specific support for dispatch a
 *	particular interrupt.  When an interrupt occurs, the
 *	target-dependant code will invoke intrDispatch().
 *
 *	intrEnableDispatch() is called when the first handler is
 *	registered.
 *
 *	intrDisableDispatch() is called when the last handler is
 *	unregistered.
 */
extern void intrEnableDispatch(int interrupt);
extern void intrDisableDispatch(int interrupt);

/*-
 * The target specific header file should define the following
 *
 * Constants
 *
 *	N_INTERRUPTS  -	The number of interrupt channels.  These
 *			are numbered from 0 to (N_INTERRUPTS - 1).
 *
 * Routines/Macros that control whether interrupts are enabled or
 * not.
 *
 *	void intrLock(void)	      -	Disable all interrupts.
 *	void intrUnlock(void)	      -	Enable all interrupts.
 *
 *		Note: intrLock()/intrUnlock() pairs can be nested.
 *
 *	bool_t intrIsLocked(void)     -	Return whether interrupts
 *					are disabled or not.
 *
 */

#include <interrupt_md.h>

/*
 * Target-specific routines.
 */

/*-
 * intrInitMD() --
 *	Initialize the machine-dependant interrupt software.
 *
 *	This routine should leave the all interrupts disabled as if
 *	one (1) intrLock() had been called.  At the end of the
 *	bootstrap, a single intrUnlock(), will be called to turn
 *	interrupts on.
 */
extern void intrInitMD(void);

/*-
 * intrEnableDispatch(), intrDisableDispatch() --
 *	Enable/Disable target-specific support for dispatch a
 *	particular interrupt.  When an interrupt occurs, the
 *	target-dependant code will invoke intrDispatch().
 *
 *	intrEnableDispatch() is called when the first handler is
 *	registered.
 *
 *	intrDisableDispatch() is called when the last handler is
 *	unregistered.
 */
extern void intrEnableDispatch(int interrupt);
extern void intrDisableDispatch(int interrupt);

#endif	/* !_GREEN_INTERRUPT_H_ */
