/*****************************************************************************

       Copyright  1995, 1996 Digital Equipment Corporation,
                       Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/* 
 *
 */
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/tty.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kd.h>
#include <linux/pci.h>
#include <linux/bios32.h>

#include <asm/io.h>
#include <asm/system.h>
#include <asm/segment.h>

#include "milo.h"
#include "video.h"

#include "tga.h"

static int tga_init(void);
static void tga_putchar(unsigned char c);

struct bootvideo tgavideo = {
  "TGA", 

  &tga_init, 
  &tga_putchar
  };

#define TGA_F_WIDTH 8
#define TGA_F_HEIGHT 16
#define TGA_F_HEIGHT_PADDED 18

static unsigned long video_mem_base;
static unsigned long video_mem_term;
static unsigned long video_tx_addr;

static unsigned long tga_fb_base;
static tga_ptr_t tga_regs_base;
static unsigned int tga_fb_width, tga_fb_height;

static void tga_init_video(void);
static void tga_clear_screen(void);
static int tga_blitc(unsigned int charattr, unsigned long addr);

#define TGA_WRITEW(v,a) tga_blitc(v,a)
#define MINW(port)              __readw((unsigned long)(port))
#define MOUTW(port, val)         {__writew((val), (unsigned long)(port)); mb();}

static void tga_set_cursor(int pos);

static int col;
static int row;
static int nrows;
static int ncols;

static void video_scrollscreen(void) 
{
    int j;
    unsigned int ote, value;
    unsigned int *dst, *src;


    ote = tga_fb_width / sizeof(int);

    for (row = 1; row < nrows; row++) {
	for (col = 0; col < ncols; col++) {
	    src = (unsigned int *) ((unsigned long) tga_fb_base
	      + (row * tga_fb_width * TGA_F_HEIGHT_PADDED)
	      + (col * TGA_F_WIDTH));
	    dst = (unsigned int *) ((unsigned long) tga_fb_base
	      + ((row - 1) * tga_fb_width * TGA_F_HEIGHT_PADDED)
	      + (col * TGA_F_WIDTH));

	    for (j = 0; j < TGA_F_HEIGHT_PADDED; j++) {
		value = readl(src);
		writel(value, dst);
		value = readl(src + 1);
		writel(value, (dst + 1));
		dst += ote;
		src += ote;
	    }
	}
    }

    /* Clear the last row by writing all spaces to it. */
    value = (unsigned short int) ' ';
    value = value | (0x07 << 8);

    for (j = 0; j < ncols; j++) {
	video_tx_addr =
	  video_mem_base + ((((nrows - 1) * 2) * ncols) + (j * 2));
	TGA_WRITEW(value, video_tx_addr);
    }

    /* Position cursor at bottom of cleared row */
    row = (nrows - 1);
    col = 0;
    tga_set_cursor((row * ncols) + col);

}


/*
 *-----------------------------------------------------------------------
 * tga_putchar --
 *      Put a character out to screen memory.
 *-----------------------------------------------------------------------
 */
static void tga_putchar(unsigned char c)
{
    unsigned short int outword;

    if (c == '\n') {

	/* just go onto the next row */
	row++;
	col = 0;
    }
      else if (c == '\r') {

	/* Go to beginning of current row */
	col = 0;
    }
      else if (c == '\b') {

	/* Backspace: go back one col unless we're already at beginning of
	 * row... */
	if (col > 0)
	     col--;
    }
      else {

	/* build the character */
	outword = (unsigned short int) c;
	outword = outword | (0x07 << 8);

	/* output it to the correct location */
	video_tx_addr = video_mem_base + ((row * 2) * ncols) + (col * 2);
#if 0
	printk("@tga_putchar: row = %d, column = %d\n", row, col);
	printk("@video_mem_base is @ 0x%p\n", video_mem_base);
	printk("@writing to mem_base + 0x%p\n",
	  video_tx_addr - video_mem_base);
#endif
	TGA_WRITEW(outword, video_tx_addr);

	/* move on where to write the next character */
	col++;
    }

    /* check for rows and cols overflowing */
    if (col > ncols - 1) {
	col = 0;
	 row++;
    }
    if (row > nrows - 1) {
	video_scrollscreen();
    }

    /* Put the cursor at the right spot... */
    tga_set_cursor((row * ncols) + col);
}

#define PCI_DEVICE_ID_DEC_TGA 0x0004

static int tga_init(void) 
{
    unsigned char pci_bus, pci_devfn;
    unsigned int mem_base;
    int status;
    tga_ptr_t tp;

    /* unsigned int temp; */

    /* first, find the TGA among the PCI devices... */
    status = pcibios_find_device(PCI_VENDOR_ID_DEC, PCI_DEVICE_ID_DEC_TGA, 
      0, &pci_bus, &pci_devfn);
    if (status == PCIBIOS_DEVICE_NOT_FOUND) {

	/* PANIC!!! */
	printk("tga_console_init: TGA not found!!! :-(\n");
	return FALSE;
    }

    /* read BASE_REG_0 for memory address */
    pcibios_read_config_dword(pci_bus, pci_devfn, 
      PCI_BASE_ADDRESS_0, &mem_base);
    mem_base &= ~15;
#ifdef DEBUG
    printk("tga_console_init: mem_base 0x%x\n", mem_base);
#endif					/* DEBUG */

    tga_regs_base = tp = 
      (tga_ptr_t)((unsigned long) mem_base + TGA_ASIC_OFFSET);
    tga_fb_base = ((unsigned long) mem_base + TGA_0_0_FB_OFFSET);

#ifdef DEBUG

    /* read some register contents and dump them */
    temp = readl(&tp->deep);
    printk("tga_console_init: DEEP register 0x%x\n", temp);

    temp = readl(&tp->mode);
    printk("tga_console_init: MODE register 0x%x\n", temp);

    temp = readl(&tp->horizontal_setup);
    printk("tga_console_init: HORIZ_SETUP register 0x%x\n", temp);
    printk("tga_console_init: fb_width %d\n", (temp & 0x1ff) << 2);

    temp = readl(&tp->vertical_setup);
    printk("tga_console_init: VERT_SETUP register 0x%x\n", temp);
    printk("tga_console_init: fb_height %d\n", temp & 0x7ff);

    temp = readl(&tp->base_address);
    printk("tga_console_init: BASE_ADDRESS register 0x%x\n", temp);
#endif					/* DEBUG */

    tga_init_video();
    tga_clear_screen();
    return TRUE;
}

unsigned char PLLbits[7] = {0x80, 0x04, 0x00, 0x24, 0x44, 0x80, 0xb8};

const unsigned char Palette[16][3] = {

  {0x00, 0x00, 0x00}, 			/* Black */
  {0xAA, 0x00, 0x00}, 			/* Red */
  {0x00, 0xAA, 0x00}, 			/* Green */
  {0xAA, 0xAA, 0x00}, 			/* Yellow */

  {0x00, 0x00, 0xAA}, 			/* Blue */
  {0xAA, 0x00, 0xAA}, 			/* Magenta */
  {0x00, 0xAA, 0xAA}, 			/* Cyan */
  {0xAA, 0xAA, 0xAA}, 			/* White */

  {0x00, 0x00, 0x00}, 			/* Hi Black */
  {0xFF, 0x00, 0x00}, 			/* Hi Red */
  {0x00, 0xFF, 0x00}, 			/* Hi Green */
  {0xFF, 0xFF, 0x00}, 			/* Hi Yellow */

  {0x00, 0x00, 0xFF}, 			/* Hi Blue */
  {0xFF, 0x00, 0xFF}, 			/* Hi Magenta */
  {0x00, 0xFF, 0xFF}, 			/* Hi Cyan */
  {0xFF, 0xFF, 0xFF}			/* Hi White */
  };

const unsigned long cursor_source[64] = {
  0x00000000000000ff, 0x00000000000000ff, 0x00000000000000ff,
  0x00000000000000ff, 
  0x00000000000000ff, 0x00000000000000ff, 0x00000000000000ff,
  0x00000000000000ff, 
  0x00000000000000ff, 0x00000000000000ff, 0x00000000000000ff,
  0x00000000000000ff, 
  0x00000000000000ff, 0x00000000000000ff, 0x00000000000000ff,
  0x00000000000000ff, 
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  };

#define MY_BT485_WRITE(r, v) \
	{								\
	  writel((r),&tga_regs_base->ramdac_setup);mb();		\
	  writel(((v)&0xff)|((r)<<8),&tga_regs_base->ramdac);mb();	\
	}

void tga_init_video(void) 
{
    int i, j, temp;
    unsigned char *cbp;
    tga_ptr_t tp = (tga_ptr_t) tga_regs_base;

    /* first, disable video timing */
    writel(0x03, &tp->video_valid);	/* SCANNING and BLANK */
    mb();

    /* write the DEEP register */
    while (readl(&tp->command_status) & 1)
					/* wait for not busy */
	continue;
    mb();
    writel(0x00014000, &tp->deep);
     mb();
    while (readl(&tp->command_status) & 1)
					/* wait for not busy */
	continue;
    mb();

    /* write some more registers */
    writel(0x0003, &tp->rop);
     mb();
    writel(0x00002000, &tp->mode);
     mb();
    writel(0x00000000, &tp->base_address);
     mb();

    /* write the PLL for 640x480 @ 60Hz */
    for (i = 0; i <= 6; i++) {
	for (j = 0; j <= 7; j++) {
	    temp = (PLLbits[i] >> (7 - j)) & 1;
	    if (i == 6 && j == 7)
		temp |= 2;
	    writel(temp, &tp->clock);
	    mb();
	}
    }

    /* write some more registers */
    writel(0xffffffff, &tp->planemask);
     mb();
    writel(0xffffffff, &tp->pixelmask);
     mb();
    writel(0x12345678, &tp->bm_color_0);
     mb();
    writel(0x12345678, &tp->bm_color_1);
     mb();

    /* init video timeing regs for 640x480 @ 60 Hz */
    writel(0x018608a0, &tp->horizontal_setup);
     mb();
    writel(0x084251e0, &tp->vertical_setup);
     mb();

    /* init BT485 RAMDAC registers */
    MY_BT485_WRITE(TGA_RAMDAC_485_CMD_0, 0xa2);
    MY_BT485_WRITE(TGA_RAMDAC_485_ADDR_PAL_WRITE, 0x01);
    MY_BT485_WRITE(TGA_RAMDAC_485_CMD_3, 0x14);
					/* cursor 64x64 */
    MY_BT485_WRITE(TGA_RAMDAC_485_CMD_1, 0x40);
    MY_BT485_WRITE(TGA_RAMDAC_485_CMD_2, 0x22);
					/* WIN cursor type */
    MY_BT485_WRITE(TGA_RAMDAC_485_PIXEL_MASK, 0xff);

    /* fill palette registers */
    MY_BT485_WRITE(TGA_RAMDAC_485_ADDR_PAL_WRITE, 0x00);
    writel(TGA_RAMDAC_485_DATA_PAL, &tp->ramdac_setup);
     mb();

    for (i = 0; i < 16; i++) {
	writel(Palette[i][0] | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
	writel(Palette[i][1] | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
	writel(Palette[i][2] | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
    }
    for (i = 0; i < 240 * 3; i += 4) {
	writel(0x55 | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
	writel(0x00 | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
	writel(0x00 | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
	writel(0x00 | (TGA_RAMDAC_485_DATA_PAL << 8), &tp->ramdac);
	 mb();
    }

    /* initialize RAMDAC cursor colors */
    MY_BT485_WRITE(TGA_RAMDAC_485_ADDR_CUR_WRITE, 0);

    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0xaa);
					/* overscan WHITE */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0xaa);
					/* overscan WHITE */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0xaa);
					/* overscan WHITE */

    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 1 BLACK */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 1 BLACK */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 1 BLACK */

    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 2 BLACK */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 2 BLACK */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 2 BLACK */

    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 3 BLACK */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 3 BLACK */
    MY_BT485_WRITE(TGA_RAMDAC_485_DATA_CUR, 0x00);
					/* color 3 BLACK */

    /* initialize RAMDAC cursor RAM */
    MY_BT485_WRITE(TGA_RAMDAC_485_ADDR_PAL_WRITE, 0x00);
    cbp = (unsigned char *) cursor_source;
    for (i = 0; i < 512; i++) {
	MY_BT485_WRITE(TGA_RAMDAC_485_CUR_RAM, *cbp++);
    }
    for (i = 0; i < 512; i++) {
	MY_BT485_WRITE(TGA_RAMDAC_485_CUR_RAM, 0xff);
    }

    /* finally, enable video scan & cursor (and pray for the monitor... :-) */
    writel(0x05, &tp->video_valid);
     mb();				/* SCANNING and CURSOR */

    /* oh, and set the globals describing the resolution... :-) */
    tga_fb_width = 640;
    tga_fb_height = 480;

}

static void tga_clear_screen()
{
    register int i, j, stride, nlongs;
    register unsigned int *dst;

    dst = (unsigned int *) ((unsigned long) tga_fb_base + 4096);
    stride = tga_fb_width * 8 / 32;
    nlongs = (tga_fb_width - (tga_fb_width % 8)) * 8 / 32;
    for (i = 0; i < tga_fb_height; i++, dst += stride) {
	for (j = 0; j < nlongs; j++) {
	    writel(0, (dst + j));
	}
    }
    memset((void *) video_mem_base, 0, (video_mem_term - video_mem_base));

    nrows = 25;
    ncols = 80;
    row = col = 0;


}
/*
static void tga_hide_cursor()
{
	unsigned long flags;
	save_flags(flags); cli();

	MY_BT485_WRITE(TGA_RAMDAC_485_CMD_2, 0x20);

	restore_flags(flags);
}
*/
static void tga_set_cursor(int pos)
{
    unsigned long flags;
    unsigned int xt, yt, row, col;

    col = pos % 80;
    row = (pos - col) / 80;

    xt = col * TGA_F_WIDTH + 64;
    yt = row * TGA_F_HEIGHT_PADDED + 64;

    save_flags(flags);
     cli();

    /* make sure it's enabled */
    MY_BT485_WRITE(TGA_RAMDAC_485_CMD_2, 0x22);
					/* WIN cursor type */

    MY_BT485_WRITE(TGA_RAMDAC_485_CUR_LOW_X, xt);
     mb();
    MY_BT485_WRITE(TGA_RAMDAC_485_CUR_HIGH_X, (xt >> 8));
     mb();
    MY_BT485_WRITE(TGA_RAMDAC_485_CUR_LOW_Y, yt);
     mb();
    MY_BT485_WRITE(TGA_RAMDAC_485_CUR_HIGH_Y, (yt >> 8));
     mb();

    restore_flags(flags);
}

/*
 * tga_blitc
 *
 * Displays an ASCII/DEC MTS character at a specified character cell position.
 * Used by the console.
 */
static unsigned int
  fontmask_bits[16] = {
  0x00000000, 
  0xff000000, 
  0x00ff0000, 
  0xffff0000, 
  0x0000ff00, 
  0xff00ff00, 
  0x00ffff00, 
  0xffffff00, 
  0x000000ff, 
  0xff0000ff, 
  0x00ff00ff, 
  0xffff00ff, 
  0x0000ffff, 
  0xff00ffff, 
  0x00ffffff, 
  0xffffffff
  };

/*
 * font table of displayable characters.
 */
char tga_builtin_font[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 0x99, 0x81, 0x81, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7e, 0xff, 0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0xff, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff,
  0xff, 0xff, 0xff, 0xff, 
  0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32, 0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0x78,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 0x30, 0x70, 0xf0, 0xe0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6,
  0xc0, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7, 0x3c, 0xdb, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8, 0xf0, 0xe0, 0xc0, 0x80,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0xfe, 0x3e, 0x1e, 0x0e, 0x06, 0x02,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x66,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x1b, 0x1b,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38, 0x0c, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0xfe, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c, 0x38, 0x38, 0x10, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c,
  0x00, 0x00, 0x00, 0x00, 
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06, 0x06, 0x86, 0xc6, 0x7c,
  0x18, 0x18, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 0x30, 0x60, 0xc6, 0x86,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x18, 0x0c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18, 0x30,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18,
  0x30, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0xc6, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06, 0x06, 0x06, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x0c, 0x0c, 0x1e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06, 0x06, 0x06, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x06, 0x0c, 0x78,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x06,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xde, 0xde, 0xde, 0xdc, 0xc0, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0x66, 0xfc,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0x66, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x6c, 0xf8,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x62, 0x66, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0x60, 0xf0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xde, 0xc6, 0xc6, 0x66, 0x3a,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x1e, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0xcc, 0x78,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0x66, 0xe6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xff, 0xdb, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0x60, 0x60, 0xf0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c,
  0x0c, 0x0e, 0x00, 0x00, 
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0x66, 0x66, 0xe6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60, 0x38, 0x0c, 0x06, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0x66, 0x66,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18, 0x3c, 0x66, 0xc3, 0xc3,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30, 0x60, 0xc1, 0xc3, 0xff,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38, 0x1c, 0x0e, 0x06, 0x02,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xff, 0x00, 0x00, 
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x66, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0, 0xc0, 0xc0, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xf0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x7c,
  0x0c, 0xcc, 0x78, 0x00, 
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66, 0x66, 0x66, 0x66, 0xe6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
  0x66, 0x66, 0x3c, 0x00, 
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0xe6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7c,
  0x60, 0x60, 0xf0, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x7c,
  0x0c, 0x0c, 0x1e, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0x60, 0xf0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60, 0x38, 0x0c, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x30, 0x30, 0x36, 0x1c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0x66,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0x3c, 0x66, 0xc3,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e,
  0x06, 0x0c, 0xf8, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xcc, 0x18, 0x30, 0x60, 0xc6, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x18, 0x18, 0x70,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x0c,
  0x06, 0x7c, 0x00, 0x00, 
  0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x38, 0x6c, 0x38, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60, 0x66, 0x3c, 0x0c, 0x06,
  0x3c, 0x00, 0x00, 0x00, 
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c, 0x60, 0x60, 0x66, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b, 0x7e, 0xd8, 0xdc, 0x77,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x3e, 0x6c, 0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xcc, 0xce,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc6, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e,
  0x06, 0x0c, 0x78, 0x00, 
  0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0, 0xc0, 0xc3, 0x7e, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xe6, 0xfc,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66, 0x6f, 0x66, 0x66, 0x66, 0xf3,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xd8, 0x70, 0x00, 0x00, 
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
  0x00, 0x00, 0x00, 0x00, 
  0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60, 0xc0, 0xc6, 0xc6, 0x7c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x06, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 0x60, 0xce, 0x9b, 0x06,
  0x0c, 0x1f, 0x00, 0x00, 
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xce, 0x96, 0x3e,
  0x06, 0x06, 0x00, 0x00, 
  0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x3c, 0x3c, 0x3c, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36, 0x6c, 0xd8, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8, 0xd8, 0xd8, 0xdc, 0x76,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc, 0xc6, 0xc6, 0xc6, 0xcc,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30, 0x60, 0xc6, 0xfe,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0x70,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60,
  0xc0, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c, 0x38,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0x6c, 0xee,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66, 0x66, 0x66, 0x66, 0x3c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb, 0xdb, 0x7e, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60, 0xc0,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60, 0x60, 0x60, 0x30, 0x1c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x00, 0x7e,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0xd8, 0x70,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x6c, 0x3c, 0x1c,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00
  };

static int tga_blitc(unsigned int charattr, unsigned long addr)
{
    register int
      status;

    int
      i, j, 
      row, col, temp, 
      c, attr, 
      ote;

    unsigned int
      fg, bg, fgmask, bgmask, 
      *dst, value0, value1;

    char
      *f_row;

    /* unsigned char glyph_row; */

    /*  tga_save_regs( tgap, t ); */

    c = charattr & 0x00ff;
    attr = (charattr >> 8) & 0x00ff;
    fg = attr & 0x0f;
    bg = (attr >> 4) & 0x0f;

    i = (c & 0xff) << 4;		/* NOTE: assumption of 16 bytes per
					 * character bitmap */

    /* calculate (row,col) from addr and video_mem_base */
    temp = (addr - video_mem_base) >> 1;
    col = temp % 80;
    row = (temp - col) / 80;

#if 0
    printk("@tga_blitc: row = %d, column = %d\n", row, col);
#endif

#ifdef NOT_NOW
    if (tgap->bits_per_pixel == 32) {
	dst = (unsigned int *) ((vm_offset_t) tgap->fb
	  + 16384
	  + (row * dp->fb_width * sp->f_height * 4)
	  + (col * sp->f_width * 4));
	f_row = &q_font[i];
	value0 = TGA_TRUECOLOR_WID_MASK;
	value1 = TGA_TRUECOLOR_WID_MASK | 0xffffff;
	for (j = 0; j < sp->f_height; j++) {
	    glyph_row = (unsigned char) *f_row++;
	    for (i = 0; i < sp->f_width; i++) {
		TGA_WRITE((dst + i), (glyph_row & 0x01 ? value1 : value0));
		glyph_row >>= 1;
	    }
	    dst += dp->fb_width;
	}
	status = 0;
    }
      else
#endif					/* NOT_NOW */
	{
	    dst = (unsigned int *) ((unsigned long) tga_fb_base
					/*                    + 4096 */
	      +(row * tga_fb_width * TGA_F_HEIGHT_PADDED)
	      + (col * TGA_F_WIDTH));
	    ote = tga_fb_width / sizeof(int);
	    f_row = (unsigned char *) &tga_builtin_font[i];

	    fgmask = fg << 8 | fg;
	    fgmask |= fgmask << 16;
	    bgmask = bg << 8 | bg;
	    bgmask |= bgmask << 16;

#if 0
	    printk("@tga_blitc: tga_fb_base @ 0x%p, dst @ 0x%p\n", 
	      tga_fb_base, dst);
#endif

	    for (j = 0; j < TGA_F_HEIGHT_PADDED; j++) {
		if (j < TGA_F_HEIGHT) {
		    value0 = fontmask_bits[((*f_row) >> 4) & 0xf];
		    value0 = (value0 & fgmask) | (~value0 & bgmask);
		    writel(value0, dst);
		    value1 = fontmask_bits[(*f_row) & 0xf];
		    value1 = (value1 & fgmask) | (~value1 & bgmask);
		    writel(value1, (dst + 1));
		    f_row++;
		} else {
		    writel(bgmask, dst);
		    writel(bgmask, (dst + 1));
		}
		dst += ote;
	    }
	    status = 0;
	}
/*  tga_restore_regs( tgap, t ); */

    return (status);
}


