/****************************************************************************
 *									    *
 *			  COPYRIGHT (c) 1990 - 2000			    *
 *			   This Software Provided			    *
 *				     By					    *
 *			  Robin's Nest Software Inc.			    *
 *			       2 Paradise Lane  			    *
 *			       Hudson, NH 03051				    *
 *			       (603) 883-2355				    *
 *									    *
 * Permission to use, copy, modify, distribute and sell this software and   *
 * its documentation for any purpose and without fee is hereby granted,	    *
 * provided that the above copyright notice appear in all copies and that   *
 * both that copyright notice and this permission notice appear in the	    *
 * supporting documentation, and that the name of the author not be used    *
 * in advertising or publicity pertaining to distribution of the software   *
 * without specific, written prior permission.				    *
 *									    *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, 	    *
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN	    *
 * NO EVENT SHALL HE BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL   *
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR    *
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS  *
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF   *
 * THIS SOFTWARE.							    *
 *									    *
 ****************************************************************************/
/*
 * Module:	dtinfo.c
 * Author:	Robin T. Miller
 * Date:	September 8, 1993
 *
 * Description:
 *	Setup device and/or system information for 'dt' program.
 */
#include "dt.h"
#if !defined(_QNX_SOURCE)
#  if !defined(sun)
#    include <sys/ioctl.h>
#  endif /* !defined(sun) */
#endif /* !defined(_QNX_SOURCE) */
#include <sys/stat.h>

/*
 * Modification History:
 *
 * August 1, 1999 by Robin Miller.
 *	Do a better job in setup_device_defaults() setting up the
 * defaults for min and increment counts.  For disks, this value
 * needs to be modulo the block size, since most disk drivers do
 * not permit non-modulo requests (Tru64 UNIX CAM is exception).
 *
 * December 21, 1998 by Robin Miller.
 *	- for DUNIX, allocate mtget structure for tapes.
 *	- changes for Malloc(), which now clears memory.
 *
 * December 19, 1998 by Robin Miller.
 *	For Steel DUNIX, strip trailing spaces from device names.
 *
 * October 29, 1998 by Robin Miller.
 *	Implement a random I/O data limit, instead of using the normal
 * data limit variable (not good to dual purpose this value).
 *
 * April 29, 1998 by Robin Miller.
 *	Add support for an alternate device directory.
 *
 * October 31, 1993 by Robin Miller.
 *	Enhance device type setup and honor user specified device type.
 *
 * October 29, 1993 by Robin Miller.
 * 	Added more checks in setup_device_type() to determine detect
 *	selection of terminal devices (e.g., console device).
 *
 * September 15, 1993 by Robin Miller.
 *	Added stat() of file specified to check for FIFO's and set the
 *	device type appropriately so other checks could be done prior
 *	to opening the device (O_NONBLOCK needs set before open()).
 *
 */

/*
 * Forward References:
 */
static void setup_device_defaults(struct dtype *dtp);

/************************************************************************
 *									*
 * setup_device_type() - Setup the device type.				*
 *									*
 * Description:								*
 *	This function sets up the specific device type to be tested.	*
 * Since each operating system has a different method of identifying	*
 * devices, so this table lookup allows the user to specify the	device	*
 * type.  Of course, this could be used to force errors.		*
 *									*
 * Inputs:	str = Pointer to device type string.			*
 *									*
 * Return Value:							*
 *		Returns pointer to device type table entry or NULL.	*
 *									*
 ************************************************************************/
struct dtype dtype_table[] = {
	{ "audio",	DT_AUDIO	},
	{ "block",	DT_BLOCK	},
	{ "comm",	DT_COMM		},
	{ "disk",	DT_DISK		},
	{ "graphics",	DT_GRAPHICS	},
	{ "memory",	DT_MEMORY	},
	{ "mmap",	DT_MMAP		},
	{ "mouse",	DT_MOUSE	},
	{ "network",	DT_NETWORK	},
	{ "fifo",	DT_FIFO		},
	{ "pipe",	DT_PIPE		},
	{ "printer",	DT_PRINTER	},
	{ "processor",	DT_PROCESSOR	},
	{ "raw",	DT_RAW		},
	{ "socket",	DT_SOCKET	},
	{ "special",	DT_SPECIAL	},
	{ "streams",	DT_STREAMS	},
	{ "tape",	DT_TAPE		},
	{ "terminal",	DT_TERMINAL	},
	{ "unknown",	DT_UNKNOWN	}
};
int num_dtypes = sizeof(dtype_table) / sizeof(dtype_table[0]);

struct dtype *
setup_device_type (char *str)
{
    int i;
    struct dtype *dtp;

    for (dtp = dtype_table, i = 0; i < num_dtypes; i++, dtp++) {
	if (strcmp(str, dtp->dt_type) == 0) {
	    setup_device_defaults (dtp);
	    return (dtp);
	}
    }
    fprintf (stderr, "Device type '%s' is invalid, valid entrys are:\n", str);
    for (dtp = dtype_table, i = 0; i < num_dtypes; i++, dtp++) {
	if ( (i % 4) == 0) fprintf (stderr, "\n");
	fprintf (stderr, "    %-12s", dtp->dt_type);
    }
    fprintf (stderr, "\n");
    return ((struct dtype *) 0);
}

/************************************************************************
 *									*
 * setup_device_defaults() - Setup the device defaults.			*
 *									*
 * Description:								*
 *	This function sets up the specific device type defaults, for	*
 * test parameters which were not specified.				*
 *									*
 * Inputs:	dip = The device information pointer.			*
 *									*
 * Return Value:							*
 *		Returns pointer to device type table entry or NULL.	*
 *									*
 ************************************************************************/
static void
setup_device_defaults (struct dtype *dtp)
{
    if ( (dtp->dt_dtype == DT_BLOCK) ||
	 (dtp->dt_dtype == DT_DISK) ) {
      /* TODO: We really need the real device size! */
      if (!lbdata_size) lbdata_size = BLOCK_SIZE;
      if (max_size && !min_size) min_size = BLOCK_SIZE;
      if (min_size && !incr_count) incr_count = BLOCK_SIZE;
    } else {
      if (max_size && !min_size) min_size = 1;
      if (min_size && !incr_count) incr_count = 1;
    }
    return;
}

/************************************************************************
 *									*
 * system_device_info() - Get system device information (if possible).	*
 *									*
 * Description:								*
 *	This function attempts to obtain device information necessary	*
 * for device specific testing, by using system dependent syscalls.	*
 *									*
 * Inputs:	dip = The device information pointer.			*
 *									*
 * Return Value:							*
 *		None.							*
 *									*
 ************************************************************************/
#if defined(ultrix) || defined(DEC)

#include <sys/devio.h>
#include <sys/ioctl.h>
#if defined(DEC)
#  include <sys/ioctl_compat.h>
#  if defined(PTOS)
#    include <io/common/devgetinfo.h>
#  endif /* defined(PTOS) */
#endif /* defined(DEC) */

void
system_device_info (struct dinfo *dip)
{
	struct devget devget;

	/*
	 * This ioctl() is Digital (Tru64) Unix specific.
	 */
	(void) bzero ((char *) &devget, sizeof(devget));
	if (ioctl (dip->di_fd, DEVIOCGET, (char *) &devget) < 0) {
	    if (dip->di_dtype == NULL) {
		dip->di_dtype = setup_device_type("unknown");
	    }
	    return;
	}
	switch (devget.category) {

	    case DEV_TAPE:		/* Tape category. */
		dip->di_dtype = setup_device_type("tape");
#if defined(EEI)
		dip->di_mt = Malloc(sizeof(*dip->di_mt));
		if (eei_flag) clear_eei_status(dip->di_fd, TRUE);
		dip->di_eei_sleep = EEI_SLEEP;
		dip->di_eei_retries = EEI_RETRIES;
#endif /* defined(EEI) */
		break;

	    case DEV_DISK: {		/* Disk category. */
#if defined(PTOS)
		device_info_t devinfo;
		/*
		 * Attempt to obtain the disk information.
		 */
		bzero ((char *) &devinfo, sizeof(devinfo));
		if ( (ioctl (dip->di_fd, DEVGETINFO, (char *) &devinfo) == 0) &&
		     (devinfo.version == VERSION_1) ) { /* This sucks... */
		    v1_disk_dev_info_t *diskinfo;
		    /* and, this is incredibly ugly... */
		    diskinfo = &devinfo.v1.devinfo.disk;
		    dip->di_dsize = diskinfo->blocksz;
#if 0
		    dip->di_capacity = diskinfo->capacity;
#endif
		}
#else /* !defined(PTOS) */
		DEVGEOMST devgeom;
		/*
		 * Attempt to obtain the disk geometry.
		 *
		 * NOTE: DEVGETGEOM *fails* on read-only devices (shit!).
		 */
		bzero ((char *) &devgeom, sizeof(devgeom));
		if (ioctl (dip->di_fd, DEVGETGEOM, (char *) &devgeom) == 0) {
		    dip->di_dsize = devgeom.geom_info.sector_size;
#if 0
		    /* NOTE: dev_size is whole disk, not open partition. */
		    dip->di_capacity = devgeom.geom_info.dev_size;
#endif
		}
#endif /* defined(PTOS) */
		dip->di_dtype = setup_device_type("disk");
		/*
		 * TODO: Need to read disklabel to pickup partition sizes,
		 *	 and to check for mounted file systems. More work!
		 */
		break;
	    }
	    case DEV_TERMINAL:		/* Terminal category. */
		dip->di_dtype = setup_device_type("terminal");
		break;

	    case DEV_PRINTER:		/* Printer category. */
		dip->di_dtype = setup_device_type("printer");
		break;

	    case DEV_SPECIAL:		/* Special category. */
		dip->di_dtype = setup_device_type("special");
		break;

	    default:
		if (dip->di_dtype == NULL) {
		    dip->di_dtype = setup_device_type("unknown");
		}
		break;
	}
	if (strncmp (devget.device, DEV_UNKNOWN, DEV_SIZE) != 0) {
	    int i;
	    dip->di_device = Malloc(DEV_SIZE + 1);
	    (void) strncpy (dip->di_device, devget.device, DEV_SIZE);
	    /*
	     * In Steel, device names have trailing spaces. grrr!
	     */
	    for (i = (DEV_SIZE); i--; ) {
		if ( isspace(dip->di_device[i]) ) {
		    dip->di_device[i] = '\0';
		} else {
		    break;
		}
	    }
	}
	return;
}

#else /* !defined(ultrix) && !defined(__osf__) */

void
system_device_info (struct dinfo *dip)
{
	if (dip->di_dtype == NULL) {
	    dip->di_dtype = setup_device_type("unknown");
	}
	return;
}

#endif /* defined(ultrix) || defined(DEC) */

/************************************************************************
 *									*
 * setup_device_info() - Setup Initial Device Information.		*
 *									*
 * Description:								*
 *	This function allocates a device information entry, and does	*
 * the initial setup of certain information based on known options.	*
 * This function is meant to be called prior to opening the device so	*
 * test specific functions are known for initial processing.		*
 *									*
 * Inputs:	dname = The device name.				*
 *									*
 * Return Value:							*
 *		Returns pointer to device information entry.		*
 *									*
 ************************************************************************/
struct dinfo *
setup_device_info (char *dname, struct dtype *dtp)
{
	struct dinfo *dip;
	struct stat sb;

	dip = (struct dinfo *) Malloc (sizeof(*dip));
	dip->di_fd = NoFd;
	dip->di_dname = dname;
	dip->di_funcs = &generic_funcs;
#if defined(AIO)
	if (aio_flag) {
	    dip->di_funcs = &aio_funcs;
	}
#endif /* defined(AIO) */
#if defined(MMAP)
	if (mmap_flag) {
	    dip->di_funcs = &mmap_funcs;
	    dtp = setup_device_type("mmap");
	}
#endif /* defined(MMAP) */

	/*
	 * If user specified a device type, don't override it.
	 */
	if (dtp == NULL) {
	    /*
	     * Determine test functions based on device name.
	     */
	    if ( (EQL (dname, DEV_PREFIX, DEV_LEN)) ||
		 (EQL (dname, ADEV_PREFIX, ADEV_LEN)) ) {
		char *dentry = (dname + DEV_LEN);
		if ( (ttyport_flag == TRUE) ||
		     (EQL (dentry, TTY_NAME, TTY_LEN)) ||
		     (EQL (dentry, CONSOLE_NAME, CONSOLE_LEN)) ) {
		    dtp = setup_device_type("terminal");
		} else if ( (EQL (dentry, TAPE_NAME, sizeof(TAPE_NAME)-1)) ||
			    (EQL (dentry, NTAPE_NAME, sizeof(NTAPE_NAME)-1)) ) {
		    dtp = setup_device_type("tape");
		} else if ( (EQL (dentry, DISK_NAME, sizeof(DISK_NAME)-1)) ||
			    (EQL (dentry, RDISK_NAME, sizeof(RDISK_NAME)-1)) ) {
		    dtp = setup_device_type("disk");
		} else if ( (EQL (dentry, CDROM_NAME, sizeof(CDROM_NAME)-1)) ||
			    (EQL (dentry, RCDROM_NAME, sizeof(RCDROM_NAME)-1)) ) {
		    dtp = setup_device_type("disk");
		}
	    }
#if defined(FIFO)
	    if ( (dtp == NULL) &&
		 (stat (dname, &sb) == SUCCESS) ) {
		if ( S_ISFIFO(sb.st_mode) ) {
		    verify_flag = FALSE;
		    dip->di_funcs = &fifo_funcs;
		    dtp = setup_device_type("fifo");
		}
	    }
#endif /* defined(FIFO) */
	    if ( (dtp == NULL) &&
		 (strlen(dname) == 1) && (*dname == '-') ) {
	        dtp = setup_device_type("pipe");
	    }
	    if ( (dtp == NULL) &&
		 (stat (dname, &sb) == SUCCESS)) {
	        if ( S_ISBLK(sb.st_mode) ) {
		    dtp = setup_device_type("block");
		} else if ( S_ISBLK(sb.st_mode) ) {
		    dtp = setup_device_type("raw");
		}
	    }
	} /* if (dtp == NULL) */

	/*
	 * Do special setup for certain device types.
	 */
	if (dip->di_dtype = dtp) {
	    if (dtp->dt_dtype == DT_TERMINAL) {
		ttyport_flag = TRUE;		/* this should go away... */
		dip->di_funcs = &tty_funcs;
	    }
	}

	/*
	 * If random I/O was selected, and a data or record limit was
	 * not specified (i.e. runtime=n), then setup the file size.
	 * This is necessary to limit random I/O within file size.
	 *
	 * NOTE:  I/O functions normally expect EOF to stop tests.
	 */
	if ( (io_type == RANDOM_IO) && (rdata_limit == (large_t)0) ) {
	    if ( (stat (dname, &sb) == SUCCESS) && sb.st_size) {
		rdata_limit = (u_long) sb.st_size;
		if (data_limit == INFINITY) data_limit = rdata_limit;
	    }
	}
	return (dip);
}
