/****************************************************************************
 *									    *
 *			  COPYRIGHT (c) 1990 - 2000			    *
 *			   This Software Provided			    *
 *				     By					    *
 *			  Robin's Nest Software Inc.			    *
 *			       2 Paradise Lane  			    *
 *			       Hudson, NH 03051				    *
 *			       (603) 883-2355				    *
 *									    *
 * Permission to use, copy, modify, distribute and sell this software and   *
 * its documentation for any purpose and without fee is hereby granted,	    *
 * provided that the above copyright notice appear in all copies and that   *
 * both that copyright notice and this permission notice appear in the	    *
 * supporting documentation, and that the name of the author not be used    *
 * in advertising or publicity pertaining to distribution of the software   *
 * without specific, written prior permission.				    *
 *									    *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, 	    *
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN	    *
 * NO EVENT SHALL HE BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL   *
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR    *
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS  *
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF   *
 * THIS SOFTWARE.							    *
 *									    *
 ****************************************************************************/
/*
 * Module:	dtprocs.c
 * Author:	Robin T. Miller
 * Date:	August 7, 1993
 *
 * Description:
 *	Functions to handle multiple processes for 'dt' program.
 *
 * Modification History:
 *
 * May 3, 1999 by Robin Miller.
 *	Allocate more space for unique file names, since the size
 * of pid_t is now 31 bits in Steel.
 *
 * April 8, 1999 by Robin Miller.
 *	Merge in Jeff Detjen's changes for current process count.
 *
 * April 29, 1998 by Robin Miller.
 *	Add support for an alternate device directory.
 *
 * July 17, 1995 by Robin Miller.
 *	Apply a severity priority to child exit status.
 *	Added flag to ensure process abortion occurs only once.
 *
 * July 5, 1995 by Robin Miller.
 *	Properly check for child process exiting as result of a signal.
 *
 * March 28, 1995 by Robin Miller.
 *	Report specific error for "no processes started", and exit with
 *	error status if the system process limit has been exceeded.
 *
 * November 4, 1994 by Robin Miller.
 *	Don't set SIGCHLD signal to SIG_IGN (set to SIG_DFL) or else
 *	waitpid() won't detect any child processes (OSF R1.3 and QNX).
 *	[ Unfortunately, the POSIX standard states "The specification
 *	  of the effects of SIG_IGN on SIGCHLD as implementation defined
 *	  and permits, but does NOT require, the System V effect of
 *	  causing terminating children to be ignored by wait(). Yuck!!! ]
 */
#include "dt.h"
#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>

#define PROC_ALLOC (sizeof(pid_t) * 3)	/* Extra allocation for PID.	*/

/*
 * Structure to track multiple processes.
 */
struct dt_procs {
	pid_t	dt_pid;			/* The child process ID.	*/
	int	dt_status;		/* The child exit status.	*/
	bool	dt_active;		/* The process active flag.	*/
};

struct dt_procs *ptable;		/* Multiple 'dt' procs table.	*/
int num_procs = 0;			/* Number of procs to create.	*/
int cur_proc = 0;			/* Current count of processes.	*/
int procs_active = 0;			/* Number of active processes.	*/

/*
 * abort_procs - Abort processes started by the parent.
 *
 */
void
abort_procs(void)
{
	static int aborted_processes = FALSE;
	struct dt_procs *dtp;
	int procs;
	pid_t pid;

	if ((ptable == NULL) || aborted_processes)  return;
	/*
	 * Force all processes to terminate.
	 */
	for (dtp = ptable, procs=0; procs < num_procs; procs++, dtp++) {
	    if ((pid = dtp->dt_pid) == (pid_t) 0) break;
	    if (debug_flag) {
		Fprintf("Aborting child process %d via a SIGTERM (%d)...\n",
								pid, SIGTERM);
	    }
	    if (dtp->dt_active) (void) kill (pid, SIGTERM);
	}
	aborted_processes = TRUE;
}

void
await_procs(void)
{
	pid_t wpid;
	struct dt_procs *dtp;
	int procs, status;

	if (debug_flag) {
	    Fprintf ("Waiting for %d child processes to complete...\n", procs_active);
	}
	while (1) {
	    if ((wpid = waitpid ((pid_t) -1, &child_status, 0)) == FAILURE) {
		if (errno == ECHILD) {
		    if (procs_active) abort();	/* Programming error... */
		    break;			/* No more children... */
		} else if (errno == EINTR) {
		    abort_procs();
		    continue;
		} else {
		    report_error ("waitpid", FALSE);
		    exit (FATAL_ERROR);
		}
	    }
	    /*
	     * Examine the child process status.
	     */
	    if ( WIFSTOPPED(child_status) ) {
		Fprintf ("Child process %d, stopped by signal %d.\n",
					wpid, WSTOPSIG(child_status));
		continue; /* Maybe attached from debugger... */
	    } else if ( WIFSIGNALED(child_status) ) {
		status = WTERMSIG(child_status);
		Fprintf ("Child process %d, exiting because of signal %d\n",
							wpid, status);
	    } else { /* Process must be exiting... */
		status = WEXITSTATUS (child_status);
		if (debug_flag) {
		    Fprintf ("Child process %d, exiting with status %d\n",
							wpid, status);
		}
	    }
	    if ( (exit_status == SUCCESS) && (status != SUCCESS) ) {
		if ( (oncerr_action == ABORT) &&
		     (status != WARNING) && (status != END_OF_FILE) ) {
		    abort_procs();		/* Abort procs on error. */
		}
		/*
		 * Save the most sever error for parent exit status.
		 *
		 * Severity Priorities:	WARNING		(lowest)
		 *			END_OF_FILE
		 *			Signal Number
		 *			FATAL_ERROR	(highest)
		 */
		if ( ((exit_status == SUCCESS) || (status == FATAL_ERROR)) ||
		     ((exit_status == WARNING) && (status > WARNING))      ||
		     ((exit_status == END_OF_FILE) && (status > WARNING)) ) {
		    exit_status = status;	/* Set error code for exit. */
		}
	    }
	    /*
	     * House keeping... (mostly sanity check, not really necessary).
	     */
	    for (dtp = ptable, procs = 0; procs < num_procs; procs++, dtp++) {
		if (dtp->dt_pid == wpid) {
		    dtp->dt_active = FALSE;
		    dtp->dt_status = status;
		    procs_active--;
		}
	    }
	}
}

pid_t
start_procs(void)
{
	struct dt_procs *dtp;
	size_t psize = (num_procs * sizeof(*dtp));
	int procs;

	if ((ptable = (struct dt_procs *)malloc(psize)) == NULL) {
	    report_error ("No memory for proc table", FALSE);
	    exit (FATAL_ERROR);
	}
	bzero((char *)ptable, psize);
#if !defined(__MSDOS__)
	(void) signal (SIGCHLD, SIG_DFL);
#endif
	(void) signal (SIGHUP, terminate);
	(void) signal (SIGINT, terminate);
	(void) signal (SIGTERM, terminate);

	cur_proc = 1;			/* Set the initial process count. */
	for (dtp = ptable, procs = 0; procs < num_procs; procs++, dtp++) {
	    if ((child_pid = fork()) == (pid_t) -1) {
		if (errno == EAGAIN) {
		    if (procs_active == 0) {
			Fprintf (
	"ERROR: could NOT start any processes, please check your system...\n");
			exit (FATAL_ERROR);
		    } else {
			Fprintf (
	"WARNING: system imposed process limit reached, only %d procs started...\n",
								procs_active);
		    }
		} else {
		    report_error ("fork", FALSE);
		    abort_procs();
		}
		break;
	    } else if (child_pid) {	/* Parent process gets the PID. */
		cur_proc++;		/* Increment the process count. */
		dtp->dt_pid = child_pid;
		dtp->dt_active = TRUE;
		procs_active++;
		if (debug_flag) {
		    Fprintf ("Started process %d...\n", child_pid);
		}
	    } else {			/* Child process... */
		struct stat sb;
		int make_unique = FALSE;

		child_pid = 0;		/* Zero indicates child process. */
		if (!output_file) break;
		if ( (stat(output_file, &sb) == SUCCESS) &&
		     ((sb.st_mode & S_IFMT) == S_IFREG) ) {
		    make_unique = TRUE;
		} else if ( (NEL (output_file, DEV_PREFIX, DEV_LEN)) &&
			    (NEL (output_file, ADEV_PREFIX, ADEV_LEN)) ) {
		    make_unique = TRUE;
		}
		/*
		 * Construct unique file name for file system I/O.
		 */
		if (make_unique) {
		    char *bp;
		    bp = (char *)malloc(strlen(output_file) + PROC_ALLOC);
		    (void)sprintf(bp, "%s-%d", output_file, getpid());
		    output_dinfo->di_dname = output_file = bp;
		}
		break;			/* Child process, continue... */
	    }
	}
	return (child_pid);
}
