#!/bin/ksh
#
# Author:  Robin Miller
# Date:    July 8, 1994
#
# Description:
#	Korn shell wrapper for 'scu' program to allow commands
# to be sent to detached 'scu' through pipes.  This method makes
# 'scu' appear as though it's a Korn shell builtin command.
#
# Caveats/Notes:
#	The stderr stream is redirected to stdout stream.
#	Exit status of 'scu' is return value from functions.
#	'scu' does NOT pipe to pager (writing is to a pipe).
#
typeset -fx scu scuIntr scuGetPrompt scuPromptUser scuStartup scuSetStatus
typeset -i scuIntrFlag=0
typeset -x HISTFILE=${HISTFILE:-$HOME/.scu_history}
typeset -x HISTSIZE=${HISTSIZE:-100}
typeset -x VISUAL=${VISUAL:-"emacs"}
typeset -x SCU_PATH=${SCU_PATH:-`whence scu`}
typeset -x SCU_PATH=${SCU_PATH:-/sbin/scu}
typeset -i SCU_PID
typeset -x scuCmd
typeset -i CmdStatus CamStatus ScsiStatus
typeset -i SenseCode SenseKey SenseAscq

#readonly scu scuIntr scuGetPrompt
#
# Check for arrow keys being defined for editing.
#
whence __A > /dev/null ||
{
    # These first 4 allow emacs history editing to use the arrow keys
    alias __A="" \
	  __B="" \
	  __C="" \
	  __D=""
}

#
# Catch signals, and forward on to 'scu' process.
#
function scuIntr
{
	scuIntrFlag=1
	kill -INT $SCU_PID
}

#
# This function loops reading input from the 'scu' process
# until we read the prompt string.  This is important so we
# keep things in sync between 'scu' commands for the main loop.
#
function scuGetPrompt
{
	status=1
	while read -r -u1 -p
	do
	    case $REPLY in

		scu\>\ \?\ *)
			scuSetStatus $REPLY
			status=$?
			break;;
		*)
			print -r - "$REPLY"
			;;
	    esac
	done
	return $status
}

#
# This function is used to get input from the terminal to
# parse and/or send to the 'scu' process.  It _must_ be a
# simple function (as it is), so signals get delivered to
# our trap handler properly.  Basically signals are _not_
# delivered until a function returns (as opposed to async
# signal delivery in C programs.
#
function scuPromptUser
{
	read -s scuCmd?"$1"
	return $?
}

#	   $1 $2    $3         $4        $5        $6        $7       $8
# Expect: scu> ? CmdStatus CamStatus ScsiStatus SenseCode SenseKey SenseAscq
#
function scuSetStatus
{
	saved_IFS=$IFS
	IFS=" "
	set -- $*
	CmdStatus=$3
	CamStatus=$4
	ScsiStatus=$5
	SenseCode=$6
	SenseKey=$7
	SenseAscq=$8
	IFS=$saved_IFS
	return $CmdStatus
}

function scu
{
	trap 'scuIntr' HUP INT QUIT TERM
	print -p - "$*" || return $?
	IFS=''
	status=0
	scuIntrFlag=0
	while read -r -u1 -p
	do
	    case $REPLY in

		*:\ \?\ *)
			prompt=${REPLY%\?\ *}
			scuPromptUser $prompt
			[[ $scuIntrFlag -eq 0  ]] && print -p - "$scuCmd"
			;;

		scu\>\ \?\ *)
			scuSetStatus $REPLY
			status=$?
			break;;

		*)	#[[ $scuIntrFlag -eq 1 ]] && break
			print -r - "$REPLY"
			;;
	    esac
	    scuIntrFlag=0
	done
	IFS="$SavedIFS"
	scuIntrFlag=0
	trap - HUP INT QUIT TERM
	return $status
}

function scuStartup
{
	$SCU_PATH -p $* 2<&1 |&
	[[ $? -ne 0 ]] && return $?
	SCU_PID=$!
	scuGetPrompt
	return $?
}

#
# This is main()...
#
set +o nounset
scuStartup
#scuStartup $*
typeset -x SavedIFS="$IFS "
return $?
