/*
 * Copyright (c) 1994, 1995 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  Tcl interface to HTTP/CGI (Common Gateway Interface)
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA  94301
 */

static char *SccsId = "@(#)tcl_cgi.c	1.12	10/12/95";

#define TCL_CGI_NAME	"Tcl_CGI"
#define TCL_CGI_VERSION	"1.5"

#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>
#include <tcl.h>

#define	eq(s, t)    (strcmp(s, t) == 0)
#define	eqc(s, t)    (strcasecmp(s, t) == 0)

char *unix_error(int errno);
void unescape_url(char *url);
void plustospace(char *str);
void strip_val(char *p);

extern char *malloc(int nbytes);
extern char *getenv(char *name);

/*
 *----------------------------------------------------------------------
 *
 * Cgi_GetQuery
 *
 *	This procedure reads the environment and retrieves the query, as
 *	appropriate to the type of query method (GET or POST).
 *
 *	It takes the no arguments.
 *	The query is returned, still URL-encoded.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */

/*ARGSUSED*/
int Cgi_GetQuery(
	ClientData	notUsed,
	Tcl_Interp	*interp,
	int		argc,
	char		**argv)
{
    char	*ev;
    int		length;
    int		rv;
    static int	beenhere = 0;
    extern int	errno;
    int		timeout = 10;

    if (argc == 3 && eq(argv[1], "-timeout")) {
	rv = Tcl_GetInt(interp, argv[2], &timeout);
	if (rv != TCL_OK || timeout < 0) {
	    Tcl_AppendResult(interp, "Invalid timeout value: ", argv[2],
			    (char *) NULL);
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }

    if (argc != 1) {
	Tcl_AppendResult(interp, "Usage: ", argv[0],
			" ?-timeout seconds?", (char *) NULL);
	return TCL_ERROR;
    }

    if (beenhere) {
	Tcl_AppendResult(interp, "Can't call ", argv[0], " more than once", (char *) NULL);
	return TCL_ERROR;
    }
    beenhere++;

    ev = getenv("REQUEST_METHOD");
    if (!ev) {
	Tcl_AppendResult(interp, "REQUEST_METHOD not set", (char *) NULL);
	return TCL_ERROR;
    }

    if (eqc(ev, "GET")) {
	ev = getenv("QUERY_STRING");
	if (!ev) {
	    Tcl_AppendResult(interp, "QUERY_STRING not set", (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_AppendResult(interp, ev, (char *) NULL);
    }

    else if (eqc(ev, "POST")) {
	int		length, count, got;
	char		*buf;
	Tcl_DString	str;
	fd_set		fds, rfds;
	struct timeval	tv;
	int		nfds;

	ev = getenv("CONTENT_LENGTH");
	if (!ev) {
	    Tcl_AppendResult(interp, "CONTENT_LENGTH not set", (char *) NULL);
	    return TCL_ERROR;
	}
	/*  Get length  */
	rv = Tcl_GetInt(interp, ev, &length);
	if (rv != TCL_OK || length < 0) {
	    Tcl_AppendResult(interp, "Invalid CONTENT_LENGTH: ", ev,
			    (char *) NULL);
	    return TCL_ERROR;
	}
	got = 0;

	/*  Allocate buffer  */
	buf = malloc(length+1);
	if (buf == 0) {
	    Tcl_AppendResult(interp, "malloc failure (", ev, " bytes)",
			(char *) NULL);
	    return TCL_ERROR;
	}
	for (count=length ; count>0 ; count--)
	    buf[count] = '-';

	Tcl_DStringInit(&str);
	FD_ZERO(&fds);
	FD_SET(fileno(stdin), &fds);
	while (length > got) {
	    rfds = fds;
	    tv.tv_usec = 0; tv.tv_sec = timeout;
	    nfds = select(fileno(stdin)+1, &rfds, 0, 0, &tv);
	    /*  Error from the select  */
	    if (nfds < 0) {
		if (errno == EINTR)
		    continue;
		else {
		    Tcl_AppendResult(interp, "stdin/select: ",
				unix_error(errno), (char *) NULL);
		    free(buf);
		    Tcl_DStringFree(&str);
		    return TCL_ERROR;
		}
	    }
	    /*  Timed out.  */
	    if (nfds == 0)
		break;
	    /*  Readable descriptor.  */
	    if (FD_ISSET(fileno(stdin), &rfds)) {
		count = read(fileno(stdin), buf, length-got);
		if (count < 0) {
		    if (errno == EINTR)
			continue;
		    else {
			Tcl_AppendResult(interp, "stdin/read: ",
					unix_error(errno), (char *) NULL);
			free(buf);
			return TCL_ERROR;
		    }
		}
		Tcl_DStringAppend(&str, buf, count);
		got += count;
		continue;
	    }
	}
	if (got < length) {
	    char	buf1[20], buf2[20];
	    sprintf(buf1, "%d", length);
	    sprintf(buf2, "%d", got);
	    Tcl_AppendResult(interp, "Couldn't read HTTP query: Expected ",
		buf1, " bytes, but only got ", buf2, " bytes", (char *) NULL);
	    free(buf);
	    Tcl_DStringFree(&str);
	    return TCL_ERROR;
	}
	free(buf);
	Tcl_DStringResult(interp, &str);
    }

    else {
	Tcl_AppendResult(interp, "Unsupported REQUEST_METHOD \"",
		ev, "\"", (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}  /* Cgi_GetQuery */



/*
 *----------------------------------------------------------------------
 *
 * Cgi_ParseQuery
 *
 *	This procedure is invoked to process the cgi_parse_query command.
 *	It takes the following arguments:
 *		cgi_parse_query [-strip] [-list]
 *				<query> <array-name> [list-names...]
 *	<query> is the HTTP query, still URL-encoded.  Ignoring the syntax,
 *		it consists of a list of NAME=VALUE clauses, where the NAMEs
 *		are not necessarily unique.
 *	<array-name> is the name of a Tcl array to put the values into.
 *	<list-names> (optional) are NAMEs that might occur more than once in
 *		<query>, and deserve special treatment.
 *
 *	If the "-strip" option is specified, non-printing characters (less
 *	than 040 (space) or greater than 0176 (tilde)) are stripped out of
 *	the values.  They are not interpreted in any way.
 *
 *	Normally, each NAME=VALUE clause results in array(NAME) being set to
 *	VALUE.  If NAME occurs multiple times, earlier values are
 *	overwritten, and only the last value is retained.  If NAME is given
 *	as one of the <list-names>, each VALUE is appended to array(NAME) as
 *	a list element.
 *
 *	If the "-list" option is specified, all query components are
 *	accumulated as list elements.  Note that this means that single
 *	values must be extracted using "lindex".
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */

/*ARGSUSED*/
int Cgi_ParseQuery(
	ClientData	notUsed,
	Tcl_Interp	*interp,
	int		argc,
	char		**argv)
{
    char	*cname;
    char	*query;
    char	*array;
    char	*name, *value;
    char	*v2;
    int		do_strip = 0;
    int		do_list = 0;
    int		append;
    int		i;

    /*  Skip command name  */
    cname = argv[0];
    argc--; argv++;

    /*  Handle any options  */
    while (argc > 1 && argv[0][0] == '-') {
	if (eq(argv[0], "-strip")) {		/*  Strip non-printing chars */
	    do_strip++;
	    argc--; argv++;
	    continue;
	}
	if (eq(argv[0], "-list")) {		/*  All components are lists */
	    do_list++;
	    argc--; argv++;
	    continue;
	}
	Tcl_AppendResult(interp, cname, ": unknown option \"",
		argv[0], "\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (argc < 2) {
	Tcl_AppendResult(interp, "Usage: ", cname,
		" ?-strip? ?-list? query array ?list-names...?", (char *) NULL);
	return TCL_ERROR;
    }
    query = argv[0];
    argc--; argv++;
    array = argv[0];
    argc--; argv++;

    /*  argc/argv now contain the list-names.  */

    /*  Erase the array, which may not exist.  */
    Tcl_UnsetVar(interp, array, 0);

    while (*query) {
	/*  Skip forward to '='  */
	name = query;
	while (*query != '=') {
	    if (*query == '\0') {
		Tcl_AppendResult(interp, "Missing \"=\" in query", (char *) NULL);
		return TCL_ERROR;
	    }
	    query++;
	}
	*query++ = '\0';

	/*  Skip forward to '&' or end of string  */
	value = query;
	while (*query && *query != '&') {
	    query++;
	}
	/*  null-terminate if not at end of string  */
	if (*query)
	    *query++ = '\0';

	/*  Now have URL-encoded name and value.  */
	plustospace(name);	unescape_url(name);
	plustospace(value);	unescape_url(value);
	if (do_strip)
	    strip_val(value);
	if (do_list)
	    append = 1;
	else {
	    append = 0;
	    for (i=0 ; i<argc ; i++)
		if (eq(argv[i], name)) {
		    append = 1;
		    break;
		}
	}
	if (append)
	    v2 = Tcl_SetVar2(interp, array, name, value,
		TCL_APPEND_VALUE | TCL_LIST_ELEMENT | TCL_LEAVE_ERR_MSG);
	else
	    v2 = Tcl_SetVar2(interp, array, name, value, TCL_LEAVE_ERR_MSG);
	if (v2 == NULL)
	    return TCL_ERROR;
    }
    
    return TCL_OK;
}  /* Cgi_ParseQuery */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *
 *	Copied from httpd util.c
 */
void plustospace(char *str) {
    register int x;

    for(x=0;str[x];x++) if(str[x] == '+') str[x] = ' ';
}

char x2c(char *what) {
    register char digit;

    digit = ((what[0] >= 'A') ? ((what[0] & 0xdf) - 'A')+10 : (what[0] - '0'));
    digit *= 16;
    digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A')+10 : (what[1] - '0'));
    return(digit);
}

void unescape_url(char *url) {
    register int x,y;

    for(x=0,y=0;url[y];++x,++y) {
        if((url[x] = url[y]) == '%') {
            url[x] = x2c(&url[y+1]);
            y+=2;
        }
    }
    url[x] = '\0';
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


int Cgi_Init(Tcl_Interp *interp)
{
    extern char	*getenv();
    int		rv;

    Tcl_CreateCommand(interp, "cgi_parse_query", Cgi_ParseQuery,
			(ClientData) 0, (void (*)()) 0);
    Tcl_CreateCommand(interp, "cgi_get_query", Cgi_GetQuery,
			(ClientData) 0, (void (*)()) 0);

    /*
     *  This procedure call should be removed if your version of Tcl doesn't
     *  include the "package" facility.
     */
    Tcl_Provide(interp, TCL_CGI_NAME, TCL_CGI_VERSION);

    return TCL_OK;
}  /* Cgi_Init */



/*  Remove any non-printing characters from the string.  */
void strip_val(char *p)
{
    char	*d;
    for (d=p ; *p ; p++)
	if (isprint(*p) || isspace(*p))
	    *d++ = *p;
    *d = '\0';
}


char *unix_error(int errno)
{
    static char	buf[150];
    extern int	sys_nerr;
    extern char	*sys_errlist[];

    if (errno<0 || errno>sys_nerr) {
	sprintf(buf, "error %d", errno);
	return buf;
    }
    else
	return sys_errlist[errno];
}  /* unix_error */
