/*
 * Copyright (c) 1994 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  Routines to interface to PBM utility routines.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Ramsey Haddad, haddad@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 */

static char *SccsId = "@(#)pbm_interface.c	1.3	10/16/94";

#include <setjmp.h>
#include <stdarg.h>
#include "pbm_tcl.h"
#include "pnm.h"
#include "pbmfont.h"


/*
 *  One of the major services provided by this module is trapping PBM errors
 *  and passing them up through the usual Tcl error-handling mechanisms.
 *
 *  PBM routines call pm_error() to report errors and pm_message() to print
 *  messages.  We provide replacements for both of those routines, here.
 *
 *  Errors are trapped with the variable pbm_errorstat, which contains a Tcl
 *  return code.  This is set to TCL_OK before any PBM routine is called,
 *  and checked when the routine returns.
 */

static char	*unix_error(int errno);

static int	 	pbm_error_stat;
static Tcl_Interp	*pbm_error_interp;
static jmp_buf		pbm_error_env;

/*
 *  Setup to handle errors -- invoke before pm_error() might be called.
 */
#define PBM_ERROR_INIT(interp) {				\
	pbm_error_interp = interp;				\
	pbm_error_stat = TCL_OK;				\
	if (setjmp(pbm_error_env) != 0)				\
		goto pbm_error;					\
}

/*
 *  Tell the error status, reseting pbm_error_interp so it can't be
 *  accidently used again.
 */
#define PBM_ERROR_STATUS (pbm_error_interp=0, pbm_error_stat)


/*
 *  Print an error for PBM.
 *  In the original code, this would exit.
 *  Instead, we use setjmp/longjmp to make the procedure return
 *  to our wrapper routine.
 */
/*VARARGS1*/
void pm_error(char *fmt, ...)
{
    va_list args;
    char	buf[1024];

    /*  Format the error message.  */
    va_start(args, fmt);
    vsprintf(buf, fmt, args);
    va_end(args);

    /*  Give the message to Tcl.  */
    if (pbm_error_stat != TCL_OK) {		/*  second error  */
	Tcl_AppendResult(pbm_error_interp, "\n", buf, (char *) NULL); 
    }
    else {
	Tcl_ResetResult(pbm_error_interp);
	Tcl_AppendResult(pbm_error_interp, buf, (char *) NULL); 
	pbm_error_stat = TCL_ERROR;
    }
    longjmp(pbm_error_env, 1);
    /*NOTREACHED*/
}




/*
 *  Get a color from a color name or other specification.
 */
int Pbm_GetColor(
	Tcl_Interp	*interp,
	char		*txt,
	pixel		*c,
	pixval		maxval)
{
    PBM_ERROR_INIT(interp);
    *c = ppm_parsecolor(txt, maxval);

pbm_error:
    return PBM_ERROR_STATUS;
}  /* Pbm_GetColor */



/*
 *  Allocate an arrary of the given size.
 */
int Pbm_AllocArray(
	Tcl_Interp	*interp,
	int		cols,
	int		rows,
	pixel		***pixels)
{
    PBM_ERROR_INIT(interp);
    *pixels = ppm_allocarray(cols, rows);

pbm_error:
    return PBM_ERROR_STATUS;
}  /* Pbm_AllocArray */




/*
 *  Return an appropriate string representation of the given errno.
 */
extern int errno;
static char *unix_error(int errno)
{
    static char	buf[150];
    extern int	sys_nerr;
    extern char	*sys_errlist[];

    if (errno<0 || errno>sys_nerr) {
	sprintf(buf, "error %d", errno);
	return buf;
    }
    else
	return sys_errlist[errno];
}  /* unix_error */



void	tcl_ReadPNM(
	FILE	*f,
	int	*dx,
	int	*dy,
	pixel	***bits,
	pixval	*maxval)
{
    int		format;
    *bits = pnm_readpnm(f, dx, dy, maxval, &format);
    return;
}  /* tcl_ReadPNM */


void	tcl_WritePNM(
	FILE	*f,
	int	dx,
	int	dy,
	pixel	**bits,
	pixval	maxval)
{
    pnm_writepnm(f, bits, dx, dy, maxval, PPM_TYPE, 0);
    return;
}  /* tcl_WritePNM */




/*
 *  Read an image file, by peeking at the magic number, then calling the
 *  appropriate routine to fill in the blanks.
 */
int Pbm_ReadImage(
	Tcl_Interp	*interp,
	char		*filename,
	pbm_pict	*p)
{
    FILE	*f;
    int		count;
    char	magic[4];	/*  Magic number of the file  */
    extern int	errno;

    f = fopen(filename, "r");
    if (f == 0) {
	Tcl_AppendResult(interp, filename, ": ", unix_error(errno),
			(char *) NULL);
	return TCL_ERROR;
    }
    count = fread(magic, 1, 4, f);
    if (0 == count && ferror(f)) {
	Tcl_AppendResult(interp, filename, ": ", unix_error(errno),
			(char *) NULL);
	fclose(f);
	return TCL_ERROR;
    }
    if (count < 4) {
	Tcl_AppendResult(interp, filename, ": unexpected end-of-file",
			(char *) NULL);
	fclose(f);
	return TCL_ERROR;
    }

    rewind(f);

    PBM_ERROR_INIT(interp);
    if (strncmp(magic, "GIF", 3) == 0) {		/*  GIF  */
	int	trans = 0;
	tcl_ReadGIF(f, &p->pp_x, &p->pp_y, &p->pp_bits,
			&p->pp_trans_color, &trans);
	p->pp_max_val = 255;
	if (trans)
	    p->pp_is_transparent = 1;
    }
    else if (magic[0] == 'P' &&				/*  PNM  */
	     (magic[1] == '1' || magic[1] == '4' || 	/*  PBM  */
	      magic[1] == '2' || magic[1] == '5' ||	/*  PGM  */
	      magic[1] == '3' || magic[1] == '6')) {	/*  PPM  */
	tcl_ReadPNM(f, &p->pp_x, &p->pp_y, &p->pp_bits, &p->pp_max_val);

	/* tcl_ReadPNM returns bw and gray formats with only blue set.
	   fix it here. */
	if (magic[1] == '1' || magic[1] == '4' || 	/*  PBM  */
	    magic[1] == '2' || magic[1] == '5') {	/*  PGM  */
	    int row, col;
	    for (row = 0; row < p->pp_y; row++)
	      for (col = 0; col < p->pp_x; col++) {
		  PPM_PUTR(p->pp_bits[row][col],
			   PPM_GETB(p->pp_bits[row][col])); 
		  PPM_PUTG(p->pp_bits[row][col],
			   PPM_GETB(p->pp_bits[row][col])); 
	      }
	}

	p->pp_is_transparent = 0;
    }
    else {
	Tcl_AppendResult(interp, filename, ": ", "unrecognized file format",
		    (char *) NULL);
	fclose(f);
	return TCL_ERROR;
    }

pbm_error:
    fclose(f);
    return PBM_ERROR_STATUS;
}  /* Pbm_ReadImage */



/*
 *  Write a pict to an output file.
 *	filename	- file to write
 *	format		- FILETYPE_GIF or FILETYPE_PPM
 *	interlace	- if should use interlace (GIF only)
 */
int Pbm_WriteImage(
	Tcl_Interp	*interp,
	char		*filename,
	pbm_pict	*p,
	int		format,
	int		interlace)
{
    FILE	*f;

    f = fopen(filename, "w");
    if (f == 0) {
	Tcl_AppendResult(interp, filename, ": ", unix_error(errno),
			(char *) NULL);
	return TCL_ERROR;
    }

    PBM_ERROR_INIT(interp);
    switch (format) {
    case FILETYPE_GIF:
	tcl_WriteGIF(f, p->pp_x, p->pp_y, p->pp_bits, p->pp_max_val,
			p->pp_trans_color, p->pp_is_transparent, interlace);
	break;
    case FILETYPE_PPM:
	tcl_WritePNM(f, p->pp_x, p->pp_y, p->pp_bits, p->pp_max_val);
	break;
    default:
	Tcl_AppendResult(interp, "unknown format", (char *) NULL);
	pbm_error_stat = TCL_ERROR;
    }

pbm_error:
    fclose(f);
    return PBM_ERROR_STATUS;
}  /* Pbm_WriteImage */



/*
 *  Load a built-in, BDF or pbm-style font.
 */
int Pbm_GetFont(
	Tcl_Interp	*interp,
	char		*name,
	struct font	**font)
{
    PBM_ERROR_INIT(interp);
    if (0 == strcmp(name, "fixed") || 0 == strcmp(name, "bdf"))
	*font = pbm_defaultfont(name);
    else
	*font = pbm_loadfont(name);

pbm_error:
    return PBM_ERROR_STATUS;
}  /* Pbm_GetFont */



