/*
 * Copyright (c) 1994 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  Support for PBM manipulations from Tcl.
 *  Operations on an existing picture.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Ramsey Haddad, haddad@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 */

static char *SccsId = "@(#)pbm_operations.c	1.6	11/3/95";

#include "pbm_tcl.h"
extern Tcl_HashTable	Pbm_PictHash;


/*
 *	PASTE <pict> ?-anchor position? x y
 *	[0]   [1]    [2]...
 *	anchor position:
 *		-n	-ne	-e	-se
 *		-s	-sw	-w	-nw
 *		-center
 */
int Pbm_PictPaste(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    char		*cname = argv[0];
    Tk_Anchor		anchor;
    int			is_anchor = 0;
    Tcl_HashEntry	*entry = 0;
    int			x, y, x_start, y_start, col, row;
    pbm_pict		*p_paste = 0;
    char		*stencil_color = 0;
    char		*option, *param;
    int			rv = TCL_OK;
    pixel		stencil;
    int			new_max_val;

    argv++; argc--;
    if (argc < 3) goto usage;

    /* find pict we are pasting */
    entry = Tcl_FindHashEntry(&Pbm_PictHash, argv[0]);
    if (!entry)	{
	Tcl_AppendResult(interp, "pict\t",
			 argv[0], " does not exist\n", (char *) NULL);
	goto abort;
    }
    p_paste = (pbm_pict *) Tcl_GetHashValue(entry);
    if (p == p_paste) {
	Tcl_AppendResult(interp, "You may not paste a pict onto itself.\n",
		 "\"duplicate\" it first and paste one copy onto the other.\n",
			 (char *) NULL);
	goto abort;
    }
    argv++; argc--;

    for ( ; argc>0 && argv[0][0] == '-' ; argv++,argc--) {
	option = argv[0];
	param = 0;

	if (eq(option, "-anchor")) {
	    test_arg();
	    rv = Tk_GetAnchor(interp, param, &anchor);
	    if (rv != TCL_OK) return rv;
	    is_anchor = 1;
	    continue;
	}
	if (eq(option, "-stencil")) {
	    test_arg();
	    stencil_color = param;
	    continue;
	}
	break; /* possibly negative integer for x */
    }

    /* get x and y */
    if (argc != 2) goto usage;
    if (Tcl_GetInt(interp, argv[0], &x_start) != TCL_OK ||
	Tcl_GetInt(interp, argv[1], &y_start) != TCL_OK) {
	return TCL_ERROR;
    }

    if (!is_anchor) {
	if (argv[0][0] == '-') {
	    if (argv[1][0] == '-')
	      anchor = TK_ANCHOR_SE;
	    else
	      anchor = TK_ANCHOR_NE;
	}
	else {
	    if (argv[1][0] == '-')
	      anchor = TK_ANCHOR_SW;
	    else
	      anchor = TK_ANCHOR_NW;
	}
    }
    if (x_start < 0 || argv[0][0] == '-') x_start += p->pp_x;
    if (y_start < 0 || argv[1][0] == '-') y_start += p->pp_y;
    if (anchor == TK_ANCHOR_NE || anchor == TK_ANCHOR_E ||
	anchor == TK_ANCHOR_SE) x_start -= p_paste->pp_x;
    if (anchor == TK_ANCHOR_SE || anchor == TK_ANCHOR_S ||
	anchor == TK_ANCHOR_SW) y_start -= p_paste->pp_y;
    if (anchor == TK_ANCHOR_N || anchor == TK_ANCHOR_S ||
	anchor == TK_ANCHOR_CENTER) x_start -= p_paste->pp_x/2;
    if (anchor == TK_ANCHOR_W || anchor == TK_ANCHOR_E ||
	anchor == TK_ANCHOR_CENTER) y_start -= p_paste->pp_y/2;

    /* max_val should be larger of two max_vals */
    new_max_val = max(p->pp_max_val, p_paste->pp_max_val);
    if (new_max_val > p->pp_max_val) {
	for (row = 0; row < p->pp_y; row++) {
	    for (col = 0; col < p->pp_x; col++) {
	      PPM_DEPTH(p->pp_bits[row][col],p->pp_bits[row][col],
			p->pp_max_val,new_max_val);
	  }
	}
	p->pp_max_val = new_max_val;
    }
      
    if (stencil_color) {
	rv = Pbm_GetColor(interp, stencil_color, &stencil, p->pp_max_val);
	if (rv != TCL_OK)
	    goto abort;
    }

    for (y = y_start, row = 0; row < p_paste->pp_y; y++, row++) {
	if (y < 0 || y >= p->pp_y) continue;
	for (x = x_start, col = 0; col < p_paste->pp_x; x++, col++) {
	    if (x < 0 || x >= p->pp_x) continue;
	    if (p_paste->pp_is_transparent &&
 		PPM_EQUAL(p_paste->pp_trans_color,p_paste->pp_bits[row][col]))
	      continue;
	    if (stencil_color)
	      PPM_DEPTH(p->pp_bits[y][x], stencil,
			p->pp_max_val,p->pp_max_val);
	    else
	      PPM_DEPTH(p->pp_bits[y][x], p_paste->pp_bits[row][col],
			p_paste->pp_max_val,p->pp_max_val);
	}
    }

    if (PBM_PICT_IS_TEMP(p_paste))
	Tcl_DeleteCommand(interp, p_paste->pp_name);
    return TCL_OK;

missing_param:
    rv = TCL_ERROR;
    Tcl_AppendResult(interp, "Missing parameter for \"", option,
		"\"", (char *) NULL);
    goto abort;

usage:
    Tcl_AppendResult(interp, "Usage:\t", cname,
		     " pict ?-anchor anchor? ?-stencil color? x y\n",
		     (char *) NULL);
    goto abort;

abort:
    return TCL_ERROR;
}  /* Pbm_PictPaste */


/*
 *	DUPLICATE newname ?options? 
 *	[0]       [1]     [2]
 */
int Pbm_PictDuplicate(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    char	*cname = argv[0];
    pbm_pict	*new;
    char	*name = argv[1];
    int		row, col;

    if (argc < 2) {
	Tcl_AppendResult(interp, "Usage:\t", cname,
			 " name \n\t",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if (Pbm_PictCreate1(interp, name, p->pp_x, p->pp_y, &new) != TCL_OK)
	return TCL_ERROR;

    /*  Copy the bits.  */
    for (row = 0; row < p->pp_y; row++) {
	for (col = 0; col < p->pp_x; col++) {
	    new->pp_bits[row][col] = p->pp_bits[row][col];
	}
    }
    new->pp_max_val = p->pp_max_val;
    new->pp_trans_color = p->pp_trans_color;
    new->pp_is_transparent = p->pp_is_transparent;
    Tcl_AppendResult(interp, name, (char *)NULL);
    return TCL_OK;
}  /* Pbm_PictDuplicate */



/*
 *	CUT   newname    x y dx dy
 *	[0]   [1]        [2]
 */
int Pbm_PictCut(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    char		*cname = argv[0];
    pbm_pict	*new;
    char	*name = argv[1];
    int row, col, row_n, col_n;
    int x, y, dx, dy;

    if (argc < 6) {
	Tcl_AppendResult(interp, "Usage:\t", cname,
			 " name x y dx dy\n\t",
			 (char *) NULL);
	return TCL_ERROR;
    }

    if (Tcl_GetInt(interp, argv[2], &x) != TCL_OK ||
	Tcl_GetInt(interp, argv[3], &y) != TCL_OK ||
	Tcl_GetInt(interp, argv[4], &dx) != TCL_OK ||
	Tcl_GetInt(interp, argv[5], &dy) != TCL_OK) {
	return TCL_ERROR;
    }
    /* negative x and y values are relative to right and bottom edges,
       respectively */
    if (x < 0 || argv[2][0] == '-') x += p->pp_x - dx;
    if (y < 0 || argv[3][0] == '-') y += p->pp_y - dy;

    if (x+dx > p->pp_x || y+dy > p->pp_y) {
	Tcl_AppendResult(interp, "Cut may not exceed bounds of pict.\n",
			 "Check size of pict and range of cut.\n",
			 (char *) NULL);
	return TCL_ERROR;
    }

    if (Pbm_PictCreate1(interp, name, dx, dy, &new) != TCL_OK)
	return TCL_ERROR;

    /*  Copy the bits.  */
    for (row = y, row_n = 0; row_n < dy; row++, row_n++)
      for (col = x, col_n = 0; col_n < dx; col++, col_n++)
	new->pp_bits[row_n][col_n] = p->pp_bits[row][col];

    new->pp_max_val = p->pp_max_val;
    new->pp_trans_color = p->pp_trans_color;
    new->pp_is_transparent = p->pp_is_transparent;

    Tcl_AppendResult(interp, name, (char *)NULL);
    return TCL_OK;
}  /* Pbm_PictCut */



/*
 *	CROP
 *	[0]
 */
int Pbm_PictCrop(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    char	*cname = argv[0];
    pixel	bg;
    int		top, bottom, left, right;
    int		row, col;

    int x, y, dx, dy;

    if (argc != 1) {
	Tcl_AppendResult(interp, "Usage:\t", cname, (char *) NULL);
	return TCL_ERROR;
    }

    /*  Nothing to crop if can't find background color.  */
    if (! Pbm_FindBackground(p, &bg))
	goto done;

    /*  Check left edge  */
    for (col=0 ; col < p->pp_x ; col++)
	for (row=0 ; row<p->pp_y ; row++)
	    if (!PPM_EQUAL(bg, p->pp_bits[row][col]))
		goto done_left;
    /*  Got here if all the same color.  */
    goto done;

done_left:    
    left = col;

    /*  Check right edge  */
    for (col=p->pp_x-1 ; col >= 0 ; col--)
	for (row=0 ; row<p->pp_y ; row++)
	    if (!PPM_EQUAL(bg, p->pp_bits[row][col]))
		goto done_right;
done_right:
    right = col;

    /*  Check top edge  */
    for (row=0 ; row < p->pp_y ; row++)
	for (col=p->pp_x-1 ; col >= 0 ; col--)
	    if (!PPM_EQUAL(bg, p->pp_bits[row][col]))
		goto done_top;
done_top:
    top = row;

    /*  Check bottom edge  */
    for (row=p->pp_y-1 ; row >= 0 ; row--)
	for (col=p->pp_x-1 ; col >= 0 ; col--)
	    if (!PPM_EQUAL(bg, p->pp_bits[row][col]))
		goto done_bottom;
done_bottom:
    bottom = row;

    for (row=top ; row<=bottom ; row++)
	for (col=left ; col<=right ; col++)
	    p->pp_bits[row-top][col-left] = p->pp_bits[row][col];

    p->pp_x = 1 + right - left;
    p->pp_y = 1 + bottom - top;

done:
    sprintf(interp->result, "%d %d", p->pp_x, p->pp_y);
    return TCL_OK;
}  /* Pbm_PictCut */



/*
 *	OUTPUT args...
 *	[0]    [1]        [2]
 */
int Pbm_PictOutput(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    char	*cname = argv[0];
    char	*fname;
    int		format = 0;
    int		interlace = 0;
    int		len;
    char	*option, *param;

    argv++;  argc--;
    for ( ; argc>0 && argv[0][0] == '-' ; argv++,argc--) {
	option = argv[0];
	param = 0;
	if (eq(option, "-ppm")) {
	    format = FILETYPE_PPM;
	    continue;
	}
	if (eq(option, "-gif")) {
	    format = FILETYPE_GIF;
	    continue;
	}
	if (eq(option, "-interlace")) {
	    interlace = 1;
	    continue;
	}
    }
    if (argc != 1) {
	Tcl_AppendResult(interp, "Usage:\t",
		cname, " ?-gif? ?-pbm? ?-interlace? filename", (char *) NULL);
	return TCL_ERROR;
    }
    fname = argv[0];

    /*  Try to deduce format from filename.  */
    if (format == 0 && (len = strlen(fname)) > 4) {
	if (0 == strcasecmp(&fname[len-4], ".gif")) format = FILETYPE_GIF;
	else if (0 == strcasecmp(&fname[len-4], ".ppm")) format = FILETYPE_PPM;
    }
    if (format == 0) {
	Tcl_AppendResult(interp, "couldn't determine type of output file",
		(char *) NULL);
	return TCL_ERROR;
    }

    return Pbm_WriteImage(interp, fname, p, format, interlace);
}  /* Pbm_PictOutput */


int dx[8] = {0, 0, 1, -1, 1, 1, -1, -1};
int dy[8] = {1, -1, 0, 0, 1, -1, 1, -1};
/*
 *	POUR ?-color color? ?-pattern pict? ?-anchor position?
 *	[0]   [1]    [2]...
 *           ?-bleed bleedcolor? x y
 *	anchor position:
 *		-n	-ne	-e	-se
 *		-s	-sw	-w	-nw
 *		-center
 */
int Pbm_PictPour(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    char		*cname = argv[0];
    char		*pour_color = 0;
    pixel		pour_pixel;
    Tcl_HashEntry	*pour_pict = 0;
    pbm_pict		*p_pour = 0;
    char		*bleed_color = 0;
    pixel		bleed_pixel;
    Tk_Anchor		anchor;
    int			is_anchor = 0;
    int			rv = TCL_OK;
    int			x, y, x_pour_origin, y_pour_origin, col, row;
    int			new_max_val;
    char		*option, *param;
    pixel		old_pixel;
    queue_element	*q_start, *q_stop;
    int			*touched = 0;
    int			d;
    int			neighbors = 4;

    argv++; argc--;
    if (argc < 2) goto usage;

    for ( ; argc>0 && argv[0][0] == '-' ; argv++,argc--) {
	option = argv[0];
	param = 0;

	if (eq(option, "-8neighbors")) {
	    neighbors = 8;
	    continue;
	}
	if (eq(option, "-color")) {
	    test_arg();
	    pour_color = param;
	    continue;
	}
	if (eq(option, "-pattern")) {
	    test_arg();
	    /* find pict we are pouring */
	    pour_pict = Tcl_FindHashEntry(&Pbm_PictHash, param);
	    if (!pour_pict)	{
		Tcl_AppendResult(interp, "pict\t",
				 param, " does not exist\n", (char *) NULL);
		goto abort;
	    }
	    p_pour = (pbm_pict *) Tcl_GetHashValue(pour_pict);
	    if (p == p_pour) {
		Tcl_AppendResult(interp,
				 "You may not pour a pict onto itself.\n",
		"\"duplicate\" it first and pour one copy onto the other.\n",
				 (char *) NULL);
		goto abort;
	    }
	    continue;
	}
	if (eq(option, "-anchor")) {
	    test_arg();
	    rv = Tk_GetAnchor(interp, param, &anchor);
	    if (rv != TCL_OK) return rv;
	    is_anchor = 1;
	    continue;
	}
	if (eq(option, "-bleed")) {
	    test_arg();
	    bleed_color = param;
	    continue;
	}
	break; /* possibly negative integer for x */
    }

    /* get x and y */
    if (argc != 2) goto usage;
    if (Tcl_GetInt(interp, argv[0], &x) != TCL_OK ||
	Tcl_GetInt(interp, argv[1], &y) != TCL_OK) {
	return TCL_ERROR;
    }
    if (!is_anchor) {
	if (argv[0][0] == '-') {
	    if (argv[1][0] == '-')
	      anchor = TK_ANCHOR_SE;
	    else
	      anchor = TK_ANCHOR_NE;
	}
	else {
	    if (argv[1][0] == '-')
	      anchor = TK_ANCHOR_SW;
	    else
	      anchor = TK_ANCHOR_NW;
	}
    }
    if (x < 0 || argv[0][0] == '-') x += p->pp_x;
    if (y < 0 || argv[1][0] == '-') y += p->pp_y;
    if (p_pour) {
	x_pour_origin = x;
	y_pour_origin = y;
	if (anchor == TK_ANCHOR_N || anchor == TK_ANCHOR_S ||
	    anchor == TK_ANCHOR_CENTER) x_pour_origin += p_pour->pp_x/2;
	if (anchor == TK_ANCHOR_W || anchor == TK_ANCHOR_E ||
	    anchor == TK_ANCHOR_CENTER) y_pour_origin += p_pour->pp_y/2;
	x_pour_origin = (x_pour_origin % p_pour->pp_x) - p_pour->pp_x;
	y_pour_origin = (y_pour_origin % p_pour->pp_y) - p_pour->pp_y;
    }

    /* max_val should be larger of two max_vals */
    if (p_pour) new_max_val = max(p->pp_max_val, p_pour->pp_max_val);
    else new_max_val = p->pp_max_val;
    if (new_max_val > p->pp_max_val) {
	for (row = 0; row < p->pp_y; row++) {
	    for (col = 0; col < p->pp_x; col++) {
	      PPM_DEPTH(p->pp_bits[row][col],p->pp_bits[row][col],
			p->pp_max_val,new_max_val);
	  }
	}
	p->pp_max_val = new_max_val;
    }
      
    if (pour_color) {
	rv = Pbm_GetColor(interp, pour_color, &pour_pixel, p->pp_max_val);
	if (rv != TCL_OK)
	    goto abort;
    } else if (!p_pour) {
	rv = Pbm_GetColor(interp, BLACK, &pour_pixel, p->pp_max_val);
	if (rv != TCL_OK)
	    goto abort;
    }
    if (bleed_color) {
	rv = Pbm_GetColor(interp, bleed_color, &bleed_pixel, p->pp_max_val);
	if (rv != TCL_OK)
	    goto abort;
    }

    p->n_pours++;
    if (!p->touched)
      p->touched = (int*)calloc(p->pp_x*p->pp_y,sizeof(int));
    if (!p->queue)
      p->queue = (queue_element*)malloc(p->pp_x*p->pp_y*sizeof(queue_element));
    old_pixel = p->pp_bits[y][x];
    q_start = p->queue;
    q_stop = p->queue;
    q_stop->x = x;
    q_stop->y = y;

    for (;q_start <= q_stop; q_start++) {
	for (d = 0; d < neighbors; d++) {
	    x = q_start->x + dx[d];
	    y = q_start->y + dy[d];
	    if (x < 0 || x >= p->pp_x) continue;
	    if (y < 0 || y >= p->pp_y) continue;
	    touched = p->touched + y*p->pp_x+x;
	    if (*touched == p->n_pours) continue;
	    *touched = p->n_pours;
	    if (p_pour) {
		PPM_DEPTH(pour_pixel,
			  p_pour->pp_bits[(y-y_pour_origin)%p_pour->pp_y]
			  [(x-x_pour_origin)%p_pour->pp_x],
			  p_pour->pp_max_val,p->pp_max_val);
		if (p_pour->pp_is_transparent &&
		    PPM_EQUAL(pour_pixel,p_pour->pp_trans_color)) {
		    pour_pixel = p->pp_bits[y][x];
		}

	    }
	    if (!PPM_EQUAL(p->pp_bits[y][x],old_pixel)) {
		if (bleed_color && PPM_EQUAL(p->pp_bits[y][x],bleed_pixel)) {
		    if (!p_pour || !p_pour->pp_is_transparent ||
			  !PPM_EQUAL(pour_pixel, p_pour->pp_trans_color))
		      p->pp_bits[y][x] = pour_pixel;
		}
		continue;
	    } else {
		p->pp_bits[y][x] = pour_pixel;
		q_stop++;
		q_stop->x = x;
		q_stop->y = y;
	    }
	}
    }

    if (p_pour && PBM_PICT_IS_TEMP(p_pour))
	Tcl_DeleteCommand(interp, p_pour->pp_name);
    return TCL_OK;

missing_param:
    rv = TCL_ERROR;
    Tcl_AppendResult(interp, "Missing parameter for \"", option,
		"\"", (char *) NULL);
    goto abort;

usage:
    Tcl_AppendResult(interp, "Usage:\t", cname,
		     " ?-color pourcolor? ?-pattern pict? ?-anchor position?\n\t?-bleed bleedcolor? ?-8neighbors? x y\n",
		     (char *) NULL);
    goto abort;

abort:
    return TCL_ERROR;
}  /* Pbm_PictPour */
