/*
 * Copyright (c) 1994 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  Support for PBM manipulations from Tcl.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Ramsey Haddad, haddad@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 */

static char *SccsId = "@(#)pbm_tcl.c	1.8	11/3/95";

#include "pbm_tcl.h"


/*  We use a Tcl hash table to store these.  */
typedef struct {
	char		*pe_name;
	pbm_pict	*pe_session;
} pbm_entry;

Tcl_HashTable	Pbm_PictHash;


/*  Forward declarations.  */
void	Pbm_PictDelete(pbm_pict *p);
void	Pbm_PictCleanup(pbm_pict *p);
int	Pbm_PictCreate(ClientData unused, Tcl_Interp *interp, int argc, char **argv);
int	Pbm_TextCreate(ClientData unused, Tcl_Interp *interp, int argc, char **argv);
int	Pbm_PictList(ClientData unused, Tcl_Interp *interp, int argc, char **argv);





/*
 *  Initialize data structures, generate commands.
 */
/*ARGSUSED*/
Pbm_Init(
	Tcl_Interp	*interp)
{
    Tcl_InitHashTable(&Pbm_PictHash, TCL_STRING_KEYS);
    Tcl_CreateCommand(interp, CMD_NEW, Pbm_PictCreate,
			(ClientData) 0, (void (*)()) 0);
    Tcl_CreateCommand(interp, CMD_LIST, Pbm_PictList,
			(ClientData) 0, (void (*)()) 0);
    Pbm_TextInit(interp);

    /*  Delete the following line if Tcl_Provide comes up undefined.  */
    Tcl_Provide(interp, MODULE_NAME, PBM_VERSION);
    return TCL_OK;
}  /* Pbm_Init */



/*
 *  Create a pict
 *	pbm_new	name  ?options? dx   dy
 *	pbm_new	name  ?options? filename
 *	[0]     [1]   [2]...
 */
int Pbm_PictCreate(
	ClientData	unused,
	Tcl_Interp	*interp,
	int		argc,
	char		**argv)
{
    char		*cname = argv[0];
    char		*name = argv[1];
    char		*tcolor = 0;
    char		*fcolor = 0;
    int			maxval = 0;
    pbm_pict		*p = 0;
    int			rv = TCL_OK;
    char		*option, *param;
    int			new;

    if (argc < 3)
	goto usage;

    /*
     *  Process the options.
     */
    argv++;  argc--;
    argv++;  argc--;
    for ( ; argc>0 && argv[0][0] == '-' ; argv++,argc--) {
	int	tmpi;
	option = argv[0];
	param = 0;

	if (eq(option, "-transparent")) {
	    test_arg();
	    tcolor = param;
	    continue;
	}
	if (eq(option, "-fill")) {
	    test_arg();
	    fcolor = param;
	    continue;
	}
	if (eq(option, "-maxval")) {
	    test_arg();
	    rv = Tcl_GetInt(interp, param, &maxval);
	    if (rv != TCL_OK) return rv;
	    if (maxval < 1 || maxval > 0x3ff) {
	        Tcl_AppendResult(interp,
		    "invalid maxval ", param, " - must be between 1 and 1023", (char *) NULL);
	        goto abort;
	    }
	    continue;
	}
	Tcl_AppendResult(interp, "unknown option \"", option, "\"",
	    (char *) NULL);
	rv = TCL_ERROR;
	goto abort;
    }

    /*
     *  Last arg = filename
     *  Read from a file.
     */
    if (argc == 1) {
	if (Pbm_PictCreate1(interp, name, 0, 0, &p) != TCL_OK)
	    return TCL_ERROR;
	rv = Pbm_ReadImage(interp, argv[0], p);
	if (rv != TCL_OK)
	    goto abort;
    }

    /*
     *  Last args = dx dy
     *  create an empty picture.
     */
    else if (argc == 2) {
	int		dx, dy;
	rv = Tcl_GetInt(interp, argv[0], &dx);
	if (rv != TCL_OK)
	    goto abort;
	rv = Tcl_GetInt(interp, argv[1], &dy);
	if (rv != TCL_OK)
	    goto abort;
	if (dx < 1 || dy < 1) {
	    Tcl_AppendResult(interp, "Size ", argv[0], "x", argv[1],
		" is too small", (char *) NULL);
	    rv = TCL_ERROR;
	    goto abort;
	}
	if (Pbm_PictCreate1(interp, name, dx, dy, &p) != TCL_OK)
	    goto abort;
    }

    else
	goto usage;

    /*
     *  Set up maxval.
     */
    if (argc == 1) {				/*  Read image from file.  */
	if (maxval > 0) {
	    Tcl_AppendResult(interp, "can't specify maxval when reading from a file",
		(char *) NULL);
	    rv = TCL_ERROR;
	    goto abort;
	}
    }
    else {
	if (maxval > 0)
	    p->pp_max_val = maxval;
	/*  Use default  */
    }

    /*
     *  Now that we know the maxval, set up transparency.
     */
    if (tcolor && !p->pp_is_transparent) {
	if (0 == strcmp(tcolor, "auto")) {
	    if (! Pbm_FindBackground(p, &p->pp_trans_color)) {
		Tcl_AppendResult(interp,
			"couldn't find background color for transparency",
			(char *) NULL);
		rv = TCL_ERROR;
		goto abort;
	    }
	}
	else {
	    rv = Pbm_GetColor(interp, tcolor, &p->pp_trans_color, p->pp_max_val);
	    if (rv != TCL_OK)
		goto abort;
	}
	p->pp_is_transparent = 1;
    }

    /*
     * Now that we know transparency, we can perform fill.
     */
    if (argc != 1) {
	pixel	fill_color;
	int	row, col;
	if (fcolor) {
	    rv = Pbm_GetColor(interp, fcolor, &fill_color, p->pp_max_val);
	    if (rv != TCL_OK)
	      goto abort;
	}
	else if (p->pp_is_transparent) {
	    fill_color = p->pp_trans_color;
	}
	else {
	    rv = Pbm_GetColor(interp, WHITE, &fill_color, p->pp_max_val);
	    if (rv != TCL_OK)
	      goto abort;
	}
	for (row = 0; row < p->pp_y; row++)
	  for (col = 0; col < p->pp_x; col++)
	    p->pp_bits[row][col] = fill_color;
    }

    /* 
     * Initialize pouring data structures
     */

    Tcl_AppendResult(interp, name, (char *)NULL);
    return TCL_OK;

missing_param:
    rv = TCL_ERROR;
    Tcl_AppendResult(interp, "Missing parameter for \"", option,
		"\"", (char *) NULL);
    goto abort;

usage:
    rv = TCL_ERROR;
    Tcl_AppendResult(interp, "Usage:\t",
cname, " name ?-transparent color? filename\n\t",
cname, " name ?-transparent color? ?-fill color? ?-maxval I? width height", (char *) NULL);
    goto abort;

    /*
     *  Free up all allocated things.
     *  If rv is set, return that value, otherwise TCL_ERROR.
     */
alloc_error:
abort:
    if (p) {
	Pbm_PictDelete(p);
    }

    return rv;
}  /* Pbm_PictCreate */




/*
 *  Free everything associated with a pict.
 *  This is called after the hash table entry has been deleted.
 */
void Pbm_PictCleanup(
	pbm_pict	*p)
{
    if (p->pp_bits)
	ppm_freearray(p->pp_bits, p->pp_y);
    if (p->pp_name)
	free(p->pp_name);
    if (p->queue)
        free(p->queue);
    if (p->touched)
        free(p->touched);
    free(p);
    return;
}  /* Pbm_PictCleanup */



/*
 *  Delete a picture.  This is called either as a result of the
 *  "PICT delete" command, or deleting the Tcl command associated with
 *  the picture.  It does the following things:
 *	Removes the hash table entry for the session.
 *	Call Pbm_PictCleanup() to free the structure.
 */
void Pbm_PictDelete(
	pbm_pict	*p)
{
    Tcl_HashEntry	*entry;

    entry = Tcl_FindHashEntry(&Pbm_PictHash, p->pp_name);
    if (entry)	Tcl_DeleteHashEntry(entry);
    Pbm_PictCleanup(p);

    return;
}  /* Pbm_PictDelete */




/*
 *  List all of the picts as a list-of-lists:
 *	{name dx dy maxval transcolor}
 *
 *	list
 *	[0]
 */
/*ARGSUSED*/
Pbm_PictList(
	ClientData	unused,
	Tcl_Interp	*interp,
	int		argc,
	char		**argv)
{
    char		buf[50];
    pbm_pict		*p;
    Tcl_HashEntry	*entry;
    Tcl_HashSearch	ptr;
    Tcl_DString		str;

    if (argc != 1) {
	Tcl_AppendResult(interp, "Usage: ", CMD_LIST, (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_DStringInit(&str);
    entry = Tcl_FirstHashEntry(&Pbm_PictHash, &ptr);
    while (entry) {
	p = (pbm_pict *) Tcl_GetHashValue(entry);
	Tcl_DStringAppend(&str, "{", 1);
	Tcl_DStringAppendElement(&str, p->pp_name);
	if (p->pp_is_transparent) {
	    char	*fmt;
	    u_int	mask;
	    if (p->pp_max_val <= (mask=0xf))
		fmt = " -transparent #%x%x%x";
	    else if (p->pp_max_val <= (mask=0xff))
		fmt = " -transparent #%02x%02x%02x";
	    else if (p->pp_max_val <= (mask=0xfff))
		fmt = " -transparent #%03x%03x%03x";
	    else {
		mask = 0xffff;
		fmt = " -transparent #%04x%04x%04x";
	    }
	    sprintf(buf, fmt, mask & PPM_GETR(p->pp_trans_color),
			mask & PPM_GETG(p->pp_trans_color),
			mask & PPM_GETB(p->pp_trans_color));
	    Tcl_DStringAppend(&str, buf, -1);
	}
	sprintf(buf, " -maxval %d %d %d", p->pp_max_val, p->pp_x, p->pp_y);
	Tcl_DStringAppend(&str, buf, -1);
	Tcl_DStringAppend(&str, "}", 1);
	entry = Tcl_NextHashEntry(&ptr);
	if (entry)
	    Tcl_DStringAppend(&str, "\n", 1);
    }
    Tcl_DStringResult(interp, &str);
    return TCL_OK;
}  /* Pbm_PictList */





/*
 *  The handler for a session procedure.  Called with:
 *	SESSION subcommand args...
 *	[0]     [1]        [2]
 */
Pbm_PictCommand(
	pbm_pict	*p,
	Tcl_Interp	*interp,
	int		argc,
	char		*argv[])
{
    int		rv;

    if(argc < 2) {
	Tcl_AppendResult(interp, "missing session subcommand, one of:\n\
	PICT paste | pour | duplicate | cut | size | output | delete ?options...?",
	    (char *) NULL);
	return TCL_ERROR;
    }

    /*  These commands don't delete temporary picts.  */
    if (eq(argv[1], "paste"))
	return Pbm_PictPaste(p, interp, argc-1, argv+1);
    else if (eq(argv[1], "crop"))
	return Pbm_PictCrop(p, interp, argc-1, argv+1);

    /*  These commands delete temporary picts.  */
    else if (eq(argv[1], "pour"))
	return Pbm_PictPour(p, interp, argc-1, argv+1);
    else if (eq(argv[1], "duplicate"))
	rv = Pbm_PictDuplicate(p, interp, argc-1, argv+1);
    else if (eq(argv[1], "cut"))
	rv = Pbm_PictCut(p, interp, argc-1, argv+1);
    else if (eq(argv[1], "output")) {
	rv = Pbm_PictOutput(p, interp, argc-1, argv+1);
    }

    else if (eq(argv[1], "size")) {
	sprintf(interp->result, "%d %d", p->pp_x, p->pp_y);
	rv = TCL_OK;
    }

    else if (eq(argv[1], "delete")) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "Usage: PICT delete", (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_DeleteCommand(interp, argv[0]);
	return TCL_OK;
	/*  The DeleteProc will do the rest of the cleanup.  */
    }

    else {
	Tcl_AppendResult(interp, "unknown picture subcommand \"", argv[1], "\"",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*  We get here after a temporary-pict-deleting command has finished.  */
    if (rv == TCL_OK && PBM_PICT_IS_TEMP(p))
	Tcl_DeleteCommand(interp, argv[0]);

    return rv;
}  /* Pbm_PictCommand */
