/*
 * Copyright (c) 1994 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
** Portions of this file are:
**
** Copyright (C) 1991 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/


/*
 *  Support for PBM text creation.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Ramsey Haddad, haddad@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 */

static char *SccsId = "@(#)pbm_text.c	1.3	11/3/95";

#include "pbm_tcl.h"
#include "pbmfont.h"


/*  We use a Tcl hash table to store these.  */
typedef struct {
	char		*pf_name;	/*  Our name for the font	*/
	struct font	*pf_font;	/*  PBM font structure		*/
} pbm_font;

Tcl_HashTable	Pbm_FontHash;


/*  pbm_interface.c  */
extern int	Pbm_GetFont(Tcl_Interp *interp, char *name, struct font **font);

/*  Forward declarations.  */
int	Pbm_TextCreate(ClientData unused, Tcl_Interp *interp, int argc, char **argv);
int	Pbm_FontList(ClientData unused, Tcl_Interp *interp, int argc, char **argv);

int Pbm_FontMake(Tcl_Interp *interp, char *name, struct font **font);
void Pbm_TextSize(char **lines, int linecount, struct font *font,
		int *dx, int *dy, int *maxleftbearing);
void Pbm_TextCopy(pbm_pict *p, char **lines, int linecount, struct font *font,
		pixel color, int xmargin, int ymargin, int maxleftb);


/*
 *  Initialize data structures, generate commands.
 */
/*ARGSUSED*/
Pbm_TextInit(
	Tcl_Interp	*interp)
{
    Tcl_HashEntry	*entry;
    pbm_font		*pf;
    int			new;
    int			rv;

    Tcl_InitHashTable(&Pbm_FontHash, TCL_STRING_KEYS);
    Tcl_CreateCommand(interp, CMD_TEXT, Pbm_TextCreate,
			(ClientData) 0, (void (*)()) 0);
    Tcl_CreateCommand(interp, CMD_FONT_LIST, Pbm_FontList,
			(ClientData) 0, (void (*)()) 0);

    /*  Create the initial entries in the font table.  */
    entry = Tcl_CreateHashEntry(&Pbm_FontHash, "bdf", &new);
    pf = (pbm_font *) malloc(sizeof(pbm_font));
    Check_Alloc(pf, "Pbm_TextInit");
    StrClone("bdf", pf->pf_name, "Pbm_TextInit");
    rv = Pbm_GetFont(interp, "bdf", &pf->pf_font);
    if (rv != TCL_OK)
	return rv;
    Tcl_SetHashValue(entry, pf);

    entry = Tcl_CreateHashEntry(&Pbm_FontHash, "fixed", &new);
    pf = (pbm_font *) malloc(sizeof(pbm_font));
    Check_Alloc(pf, "Pbm_TextInit");
    StrClone("fixed", pf->pf_name, "Pbm_TextInit");
    pf->pf_font = 0;			/*  Filled in later, if referenced.  */
    Tcl_SetHashValue(entry, pf);
    
    return TCL_OK;

alloc_error:
    return TCL_ERROR;
}  /* Pbm_TextInit */




/*
 *  Create a pict with text in it
 *	pbm_text ?options? text
 *	[0]     [1]
 *  Options:
 *	-font fontname	default: "bdf"
 *	-name pictname	default: internally-generated
 *	-bg color	default: white (transparent)
 *	-fg color	default: black
 *	-margin pixels	default: 1 pixel
 */
int Pbm_TextCreate(
	ClientData	unused,
	Tcl_Interp	*interp,
	int		argc,
	char		**argv)
{
    char		*cname = argv[0];
    char		*pictname = 0;
    char		*fontname = "bdf";
    char		*fgname = BLACK;
    char		*bgname = WHITE;
    pixel		fgcolor;
    pixel		bgcolor;
    int			transparent = 1;
    static int		namecount = 0;
    char		namebuf[30];
    struct font		*font;
    pbm_pict		*p = 0;
    int			dx, dy;
    char		**lines;
    int			linecount;
    int			margin = 1;
    int			mlb;
    int			rv = TCL_OK;
    char		*option, *param;

    /*
     *  Process the options.
     */
    argv++;  argc--;
    for ( ; argc>0 && argv[0][0] == '-' ; argv++,argc--) {
	int	tmpi;
	option = argv[0];
	param = 0;

	if (eq(option, "-font")) {
	    test_arg();
	    fontname = param;
	    continue;
	}
	if (eq(option, "-name")) {
	    test_arg();
	    pictname = param;
	    continue;
	}
	if (eq(option, "-bg")) {
	    test_arg();
	    bgname = param;
	    transparent = 0;
	    continue;
	}
	if (eq(option, "-fg")) {
	    test_arg();
	    fgname = param;
	    continue;
	}
	if (eq(option, "-margin")) {
	    test_arg();
	    rv = Tcl_GetInt(interp, param, &margin);
	    if (rv != TCL_OK)
		goto abort;
	    continue;
	}
	Tcl_AppendResult(interp, "unknown option \"", option, "\"",
	    (char *) NULL);
	rv = TCL_ERROR;
	goto abort;
    }

    /*
     *  Last arg = text to create.
     */
    if (argc != 1)
	goto usage;

    /*
     *  Transform the text into lines, expanding tabs.
     */
    {
	Tcl_DString	str;
	char		*c;
	int		n = 0;	/*  character on line  */
	int		fill;
	Tcl_DStringInit(&str);
	for (c=argv[0] ; *c ; c++) {
	    if (*c == '\n') {
		Tcl_DStringAppend(&str, c, 1);
		n = 0;
		continue;
	    }
	    if (*c == '\t') {
		fill = 8 - (n%8);
		Tcl_DStringAppend(&str, "\\ \\ \\ \\ \\ \\ \\ \\ ", fill*2);
		n += fill;
		continue;
	    }
	    if (*c == ' ') {
		Tcl_DStringAppend(&str, "\\ ", 2);
		n++;
		continue;
	    }
	    if (isprint(*c)) {
		Tcl_DStringAppend(&str, c, 1);
		n++;
		continue;
	    }
	    /*  Non-printing -> space  */
	    Tcl_DStringAppend(&str, " ", 1);
	    n++;
	}
	rv = Tcl_SplitList(interp, Tcl_DStringValue(&str), &linecount, &lines);
	Tcl_DStringFree(&str);
	if (rv != TCL_OK)
	    goto abort;
    }

    /*  Create a name, if needed.  */
    if (pictname == 0) {
	sprintf(namebuf, "%%text%d", namecount);
	pictname = namebuf;
	namecount++;
    }

    rv = Pbm_FontMake(interp, fontname, &font);
    if (rv != TCL_OK)
	goto abort;

    /*  Compute size  */
    Pbm_TextSize(lines, linecount, font, &dx, &dy, &mlb);

    /*  Create the PICT  */
    if (Pbm_PictCreate1(interp, pictname, dx+(2*margin), dy+(2*margin), &p) != TCL_OK)
	return TCL_ERROR;

    /*
     *  Set up foreground and background colors
     */
    rv = Pbm_GetColor(interp, bgname, &bgcolor, p->pp_max_val);
    if (rv != TCL_OK)
	goto abort;
    rv = Pbm_GetColor(interp, fgname, &fgcolor, p->pp_max_val);
    if (rv != TCL_OK)
	goto abort;
    if (transparent) {
	p->pp_is_transparent = 1;
	p->pp_trans_color = bgcolor;
    }

    /*  Fill the PICT with the background color  */
    {
	int	row, col;
	for (row = 0; row < p->pp_y; row++)
	    for (col = 0; col < p->pp_x; col++)
		p->pp_bits[row][col] = bgcolor;
    }

    Pbm_TextCopy(p, lines, linecount, font, fgcolor, margin, margin, mlb);

    Tcl_AppendResult(interp, pictname, (char *)NULL);
    return TCL_OK;

missing_param:
    rv = TCL_ERROR;
    Tcl_AppendResult(interp, "Missing parameter for \"", option,
		"\"", (char *) NULL);
    goto abort;

usage:
    rv = TCL_ERROR;
    Tcl_AppendResult(interp, "Usage: ", cname,
		" ?-name pictname? ?-font fontname? ?-bg color? ?-fg color? ?-margin size? text",
		(char *) NULL);
    goto abort;

    /*
     *  Free up all allocated things.
     *  If rv is set, return that value, otherwise TCL_ERROR.
     */
alloc_error:
abort:
    if (p) {
	Pbm_PictDelete(p);
    }

    return rv;
}  /* Pbm_TextCreate */





/*
 *  Given a font name, return a font structure for it,
 *  either from one already computed, or by reading it in from the file.
 */
int Pbm_FontMake(
	Tcl_Interp	*interp,
	char		*name,
	struct font	**font)
{
    Tcl_HashEntry	*entry = 0;
    pbm_font		*pf = 0;
    int			new;
    int			rv;

    entry = Tcl_CreateHashEntry(&Pbm_FontHash, name, &new);
    if (!new) {
	pf = (pbm_font *) Tcl_GetHashValue(entry);
	/*  Hack to allow us to delay font loading (see Pbm_Init).  */
	if (pf->pf_font == 0) {
	    rv = Pbm_GetFont(interp, name, &pf->pf_font);
	    if (rv != TCL_OK)
		return rv;
	}
	*font = pf->pf_font;
	return TCL_OK;
    }

    pf = (pbm_font *) malloc(sizeof(pbm_font));
    Check_Alloc(pf, "Pbm_FontMake");
    rv = Pbm_GetFont(interp, name, &pf->pf_font);
    if (rv != TCL_OK) {
	Tcl_DeleteHashEntry(entry);
	free(pf);
	return rv;
    }
    Tcl_SetHashValue(entry, pf);
    StrClone(name, pf->pf_name, "Pbm_FontMake");
    *font = pf->pf_font;
    return TCL_OK;

alloc_error:
    if (entry) Tcl_DeleteHashEntry(entry);
    if (pf) free(pf);
    return TCL_ERROR;
}  /* Pbm_FontMake */




/*
 *  List all of the loaded fonts.
 *
 *	font_list [fontname]
 *	[0]       [1]
 */
/*ARGSUSED*/
int Pbm_FontList(
	ClientData	unused,
	Tcl_Interp	*interp,
	int		argc,
	char		**argv)
{
    char		buf[50];
    pbm_font		*f;
    Tcl_HashEntry	*entry;
    Tcl_HashSearch	ptr;
    Tcl_DString		str;

    if (argc == 2)
	return Pbm_FontShow(interp, argv[1]);

    if (argc != 1) {
	Tcl_AppendResult(interp, "Usage: ", CMD_FONT_LIST, (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_DStringInit(&str);
    entry = Tcl_FirstHashEntry(&Pbm_FontHash, &ptr);
    while (entry) {
	f = (pbm_font *) Tcl_GetHashValue(entry);
	Tcl_DStringAppendElement(&str, f->pf_name);
	entry = Tcl_NextHashEntry(&ptr);
    }
    Tcl_DStringResult(interp, &str);
    return TCL_OK;
}  /* Pbm_FontList */




int Pbm_FontShow(
	Tcl_Interp	*interp,
	char		*fname)
{
    int		rv;
    struct font	*f;
    int		c;
    char	buf[100];

    rv = Pbm_FontMake(interp, fname, &f);
    if (rv != TCL_OK)
	return rv;

    sprintf(buf, "name %s\nmaxwidth %d\nmaxheight %d",
		fname, f->maxwidth, f->maxheight);
    Tcl_AppendResult(interp, buf, (char *) NULL);
    Tcl_AppendResult(interp, "\nCHR WD HT  X  Y XAD", (char *) NULL);
    for (c=0 ; c<256 ; c++) {
	struct glyph	*g = f->glyph[c];
	if (!g)
	    continue;
	if (isprint(c))
	    sprintf(buf, "\n  %c", c);
	else
	    sprintf(buf, "\n%03o", c);
	Tcl_AppendResult(interp, buf, (char *) NULL);
	sprintf(buf, " %2d %2d %2d %2d  %2d",
		    g->width, g->height, g->x, g->y, g->xadd);
	Tcl_AppendResult(interp, buf, (char *) NULL);
    }

    return TCL_OK;
}  /* Pbm_FontShow */

/*
 *  Compute the horizontal and vertical size of the given text in the given
 *  font.  Also compute the maximum left bearing of the first character(s)
 *  on the line(s).  (This is used to pad the left edge.)
 *
 *  Copied from pbmtext.c
 */
void Pbm_TextSize(
	char		**lines,
	int		linecount,
	struct font	*font,
	int		*dx,
	int		*dy,
	int		*maxleftbearing)
{
    int		maxwidth, maxleftb;
    int		line;
    char	*cp;

    /* The total height is easy to figure out */
    *dy = linecount * font->maxheight;

    /* The total width is not so easy */
    maxwidth = 0;
    maxleftb = 0;
    for (line=0 ; line<linecount ; ++line) {
	int isfirst = 1;
	int x = 0;
	int bwid = 0;
	char lastch;

	for ( cp = lines[line]; *cp != '\0'; ++cp ) {
	    if (!font->glyph[*cp])
		continue;
	    if (isfirst) {
		isfirst = 0;
		if (font->glyph[*cp]->x < 0)
			x = -font->glyph[*cp]->x;
		    else
			bwid += font->glyph[*cp]->x;

		    bwid += x;
	    }
	    bwid += font->glyph[*cp]->xadd;
	    lastch = *cp;
	}
	if (!isfirst) {
	    bwid -= font->glyph[lastch]->xadd;
	    bwid += font->glyph[lastch]->width + font->glyph[lastch]->x;
	}

	if (bwid > maxwidth)
	    maxwidth = bwid;
	if (x > maxleftb)
	    maxleftb = x;
    }

    *dx = maxwidth + maxleftb;
    *maxleftbearing = maxleftb;

    return;
}  /* Pbm_TextSize */



/*
 *  Copied from pbmtext.c
 */
void Pbm_TextCopy(
	pbm_pict	*p,
	char		**lines,
	int		linecount,
	struct font	*font,
	pixel		color,
	int		xmargin,
	int		ymargin,
	int		maxleftb)
{
    int		line;
    char	*cp;
    int		row, col;
    struct glyph *glyph;

    for (line=0 ; line<linecount ; ++line) {
	row = ymargin + line * font->maxheight;
	col = xmargin + maxleftb;

	for (cp=lines[line] ; *cp != '\0' ; ++cp) {
	    int h, w, y;

	    if (!(glyph = font->glyph[*cp]))
		continue;

	    y = row + font->maxheight + font->y - glyph->height - glyph->y;

	    for (h = 0; h < glyph->height; h++) {
		for (w = 0; w < glyph->width; w++) {
		    if (glyph->bmap[h * glyph->width + w])
			p->pp_bits[y][w + col + glyph->x] = color;
		}
		y++;
	    }
	    col += glyph->xadd;
	}
    }

    return;
}  /* Pbm_TextCopy */
