/*
 * Copyright (c) 1994 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  Support for PBM manipulations from Tcl.
 *  Utility routines:  Get, etc.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Ramsey Haddad, haddad@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 */

static char *SccsId = "@(#)pbm_util.c	1.4	11/2/94";

#include "pbm_tcl.h"
extern Tcl_HashTable	Pbm_PictHash;

/*
 *  Create a new pbm_pict, inserting it in the hash table, etc.
 *  All fields are initialized to default values.  
 *  pp_bits is allocated if (dx, dy) is non-zero.
 */
int Pbm_PictCreate1(
	Tcl_Interp	*interp,
	char		*name,		/*  name for pict		*/
	int		dx,		/*  size - if zero, don't allocate */
	int		dy,		/*  pp_bits			*/
	pbm_pict	**pp)		/*  The result			*/
{
    Tcl_HashEntry	*entry = 0;
    int			rv = TCL_OK;
    int			new;
    pbm_pict		*p;

    *pp = 0;
    entry = Tcl_CreateHashEntry(&Pbm_PictHash, name, &new);
    if (!new) {
	Tcl_AppendResult(interp,
		"pict \"", name, "\" is already defined", (char *) NULL);
	return TCL_ERROR;
    }

    *pp = (pbm_pict *) malloc(sizeof(pbm_pict));
    p = *pp;
    Check_Alloc(p, "Pbm_PictCreate");
    bzero((char *) p, sizeof(pbm_pict));

    StrClone(name,p->pp_name,"Pbm_PictCreate1");
    p->pp_x = dx;
    p->pp_y = dy;
    p->pp_max_val = 255;
    p->pp_is_transparent = 0;
    p->n_pours = 0;
    p->queue = 0;
    p->touched = 0;

    if (dx > 0 && dy > 0) {
	rv = Pbm_AllocArray(interp, p->pp_x, p->pp_y, &p->pp_bits);
	if (rv != TCL_OK)
	    goto abort;
    }

    /*  Create the command, set the hash table entry, and we're done.  */
    Tcl_CreateCommand(interp, name, (Tcl_CmdProc *)Pbm_PictCommand,
		(ClientData) p, (Tcl_CmdDeleteProc *) Pbm_PictDelete);
    Tcl_SetHashValue(entry, p);
    return TCL_OK;

alloc_error:
abort:
    if (p) {
	Pbm_PictCleanup(p);
    }
    if (entry)	Tcl_DeleteHashEntry(entry);
    return rv;
}  /* Pbm_PictCreate1 */




/*
 *  Find the background color of a pict, by some crude guesses.
 *  Return true if a background color could be found, zero otherwise.
 *  If a transparency color is set, return that color.
 */
int Pbm_FindBackground(
	pbm_pict	*p,
	pixel		*color)		/*  returned color  */
{
    pixel	tmp;

    if (p->pp_is_transparent) {
	*color = p->pp_trans_color;
	return 1;
    }
    tmp = p->pp_bits[0][0];
    if (! PPM_EQUAL(tmp, p->pp_bits[p->pp_y-1][0]))		return 0;
    if (! PPM_EQUAL(tmp, p->pp_bits[0][p->pp_x-1]))		return 0;
    if (! PPM_EQUAL(tmp, p->pp_bits[p->pp_y-1][p->pp_x-1]))	return 0;

    *color = tmp;
    return 1;
}  /* Pbm_FindBackground */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *
 *  Copied from tkGet.c
 */


/*
 *--------------------------------------------------------------
 *
 * Tk_GetAnchor --
 *
 *	Given a string, return the corresponding Tk_Anchor.
 *
 * Results:
 *	The return value is a standard Tcl return result.  If
 *	TCL_OK is returned, then everything went well and the
 *	position is stored at *anchorPtr;  otherwise TCL_ERROR
 *	is returned and an error message is left in
 *	interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
Tk_GetAnchor(interp, string, anchorPtr)
    Tcl_Interp *interp;		/* Use this for error reporting. */
    char *string;		/* String describing a direction. */
    Tk_Anchor *anchorPtr;	/* Where to store Tk_Anchor corresponding
				 * to string. */
{
    switch (string[0]) {
	case 'n':
	    if (string[1] == 0) {
		*anchorPtr = TK_ANCHOR_N;
		return TCL_OK;
	    } else if ((string[1] == 'e') && (string[2] == 0)) {
		*anchorPtr = TK_ANCHOR_NE;
		return TCL_OK;
	    } else if ((string[1] == 'w') && (string[2] == 0)) {
		*anchorPtr = TK_ANCHOR_NW;
		return TCL_OK;
	    }
	    goto error;
	case 's':
	    if (string[1] == 0) {
		*anchorPtr = TK_ANCHOR_S;
		return TCL_OK;
	    } else if ((string[1] == 'e') && (string[2] == 0)) {
		*anchorPtr = TK_ANCHOR_SE;
		return TCL_OK;
	    } else if ((string[1] == 'w') && (string[2] == 0)) {
		*anchorPtr = TK_ANCHOR_SW;
		return TCL_OK;
	    } else {
		goto error;
	    }
	case 'e':
	    if (string[1] == 0) {
		*anchorPtr = TK_ANCHOR_E;
		return TCL_OK;
	    }
	    goto error;
	case 'w':
	    if (string[1] == 0) {
		*anchorPtr = TK_ANCHOR_W;
		return TCL_OK;
	    }
	    goto error;
	case 'c':
	    if (strncmp(string, "center", strlen(string)) == 0) {
		*anchorPtr = TK_ANCHOR_CENTER;
		return TCL_OK;
	    }
	    goto error;
    }

    error:
    Tcl_AppendResult(interp, "bad anchor position \"", string,
	    "\": must be n, ne, e, se, s, sw, w, nw, or center",
	    (char *) NULL);
    return TCL_ERROR;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
