/*
 * Copyright (c) 1995 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  SQL constants and localizations for Informix.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 *  This is an Embedded SQL C-language source file.  It must first be
 *  processed with esql, to produce a C-language source file that
 *  can be compiled with cc.
 */


static char *SccsId = "@(#)sql_informix.ec	2.1\t3/28/95";

EXEC SQL include sqlca;
EXEC SQL include sqlda;
EXEC SQL include datetime;

#include <stdio.h>
#include <malloc.h>
#include <tcl.h>
#include "tcl_sql.h"

struct sqlca_s sqlca;

/*  Database commands local to Informix  */
char	*sql_local_dbcmd_list = "\n\
	is_ansi\n\
	has_log";

/*  For helping compute lengths.  */
static dtime_t		sql_dtime;
static intrvl_t		sql_intrvl;


/*
 *  Perform database-specific initializations.
 *	Set the DBDATE environment variable to determine the date format.
 */
int Sql_X_Init(
    Tcl_Interp	*interp)
{
    if (0 == Tcl_SetVar2(interp, "env", "DBDATE", "Y4MD-",
		TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG))
	return TCL_ERROR;
    sql_dtime.dt_qual = 0;
    dtcurrent(&sql_dtime);
    
    return TCL_OK;
}  /* Sql_X_Init */




/*
 *  Open a connection to a database/server:
 *	Makes the connection.
 *	Makes it current.
 *	Sets s->dbname
 *
 *  The arguments can be either:
 *	dbspec
 *		- name of an informix database
 *  or
 *	-user <username> <authinfo> <dbspec>
 *
 *  If a single database is being opened, a handle is not permitted.
 */
int Sql_X_Connect(
    Tcl_Interp	*interp,
    session	*s,
    int		*single_session,
    int		argc,
    char	**argv)
{
EXEC SQL BEGIN DECLARE SECTION ;
    char	*dbname;
    char	*user = 0;
    char	*auth = 0;
    char	*handle;
EXEC SQL END DECLARE SECTION ;
    int		i;
    int		rv;

    if (argc >= 4 && eq(argv[0], "-user")) {
	user = argv[1];
	auth = argv[2];
	argc -= 3;
	argv += 3;
    }

    if (argc == 1) {
	dbname = argv[0];
    }
    else {
	Tcl_AppendResult(interp, "wrong # args to ", CMD_OPENDB, ".  Usage:\n",
		CMD_OPENDB, SQL_OPENDB_USAGE, (char *) NULL);
	return TCL_ERROR;
    }

    if (0 == getenv("INFORMIXDIR")) {
	Tcl_AppendResult(interp, "INFORMIXDIR environment variable not set",
		(char *) NULL);
	return TCL_ERROR;
    }

    handle = s->handle;

    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    if (user)
	EXEC SQL CONNECT TO :dbname AS :handle USER :user USING :auth WITH CONCURRENT TRANSACTION ;
    else
	EXEC SQL CONNECT TO :dbname AS :handle WITH CONCURRENT TRANSACTION ;

    if (sqlca.sqlwarn.sqlwarn2 == 'W')
	s->ansi_db = 1;
    if (sqlca.sqlwarn.sqlwarn1 == 'W')
	s->has_log = 1;

    StrClone(dbname, s->dbname, "Sql_Connect");
    return TCL_OK;

alloc_error:
    return TCL_ERROR;

fail:
    rv = Sql_X_HandleError(interp);
    return rv;
}  /* Sql_X_Connect */


/*
 *  Set the current connection to a database/server.
 */
int Sql_X_SetConnection(
    Tcl_Interp	*interp,
    session	*s)
{
EXEC SQL BEGIN DECLARE SECTION ;
    char	*handle;
EXEC SQL END DECLARE SECTION ;
    int		rv;

    handle = s->handle;

    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL SET CONNECTION :handle ;
    return TCL_OK;

fail:
    rv = Sql_X_HandleError(interp);
    return rv;
}  /* Sql_X_SetConnect */



/*
 *  Close the current connection to a database/server.
 */
int Sql_X_Disconnect(Tcl_Interp *interp)
{
    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL DISCONNECT CURRENT;
    return TCL_OK;

fail:
    return Sql_X_HandleError(interp);
}  /* Sql_X_Disconnect */



/*
 *  Retrieve vendor-specific database name and version.
 */
int Sql_X_DbVersion(
    Tcl_Interp	*interp,
    int		argc,
    char	**argv)
{
    char		version[40];
    int			i;
    int			rv;

    /*  Undocumented, but better than nothing.  */
    extern		char *_sqvers;

    if(argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
	    argv[0], (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_AppendElement(interp, DB_NAME, (char *) NULL);

    strcpyn(version, _sqvers, sizeof(version)-1);
    for (i=strlen(version)-1 ; i>=0 ; i--)
	if (!isspace(version[i]))
	    break;
    version[i+1] = '\0';
    Tcl_AppendElement(interp, version, (char *) NULL);

    return TCL_OK;
}  /* Sql_X_Version */



/*
 *  Handle an SQL error:
 *	Report the SQL error message as the result
 *	Add the SQL error code and message to the ErrorInfo variable.
 *	Return TCL_ERROR
 *
 *  This procedure cleans up the SQL error string and returns it as the
 *  error message.  Note that this cleanup is dependent upon the
 *  particular SQL implementation.
 */
int Sql_X_HandleError(Tcl_Interp *interp)
{
    char	*errnum[15];
EXEC SQL BEGIN DECLARE SECTION ;
    char	sql_error_str[1024];
    int		sql_error_len;
EXEC SQL END DECLARE SECTION ;
    int		len, i;

    sprintf(errnum, "%d", _SQL_ERROR_CODE);
    EXEC SQL WHENEVER SQLERROR CONTINUE;

    EXEC SQL GET DIAGNOSTICS EXCEPTION 1 :sql_error_str = MESSAGE_TEXT ;
    sql_error_len = strlen(sql_error_str);

    /*  Collapse \r\n to \n.  */
    for (i=len=0 ; i<sql_error_len ; i++) {
	if (sql_error_str[i] != '\r')
	    sql_error_str[len++] = sql_error_str[i];
    }
    sql_error_str[len] = '\0';

    /*  strip off trailing whitespace  */
    while (isspace(sql_error_str[len-1]))
	sql_error_str[--len] = '\0';

    Tcl_AppendResult(interp, "sql error ", errnum, ": \"",
		sql_error_str, "\"", (char *)0);
    Tcl_SetErrorCode(interp, "sql", errnum, sql_error_str, (char *)NULL);
    return TCL_ERROR;
}  /* Sql_X_HandleError */





/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *
 *  Cursor-related localizations.
 *
 *  These mostly handle the different types supported by the various
 *  database systems.
 */


/*
 *  Place for SQL to store null indicators.
 *  This array is only used at the time of a fetch, so it can
 *  be shared among all cursors.
 */
static short	sql_null_ind[MAXCOLUMNS];



/*
 *  Given the result of
 *	SQL PREPARE
 *	SQL DESCRIBE
 *  on a SELECT statement:
 *	determine the storage requirements and offsets
 *	allocate storage
 *	setup the SQLDA
 *  for the given cursor.
 */
int Sql_X_SetupDA(
    Tcl_Interp		*interp,
    struct cursor	*cur
) {
    struct sqlda_	*da = SQLDA_PTR(cur);
    struct sqlvar_	*var;
    int			newlen;
    int			offset;
    int			i;
    char		buf1[10];
    char		buf2[10];
    char		buf3[40];

    offset = 0;
    for (i=0 ; i<da->sqld ; i++) {
	var = &da->sqlvar[i];
	newlen = var->sqllen;

	switch (var->sqltype) {
	/*
	 *  Results in fixed-length character string:
	 *	We allocate an extra byte to null-terminate it
	 *	We tell the sqlda about the byte if SQLDA_FIXED_CHAR_WANTS_NULL
	 *	is set.
	 */
	case SQLCHAR:
	case SQLNCHAR:
	    var->sqltype = _SQL_FIXED_STR;
	    newlen = var->sqllen += 1;
	    break;

	/*  Use the fixed number of characters required by DATE  */
	case SQLDATE:
	    newlen = var->sqllen = _SQL_DATE_LEN + 1;
	    var->sqltype = _SQL_FIXED_STR;
	    break;

	/*  Use dttoasc() to format the current date.  Use that length  */
	case SQLDTIME:
	    /*  sql_dtime already set to current time  */
	    sql_dtime.dt_qual = var->sqllen;
	    dttoasc(&sql_dtime, buf3);
	    newlen = var->sqllen = strlen(buf3) + 1;
	    var->sqltype = _SQL_FIXED_STR;
	    break;

	/*  Use intoasc() to format a zero interval.  Use that length  */
	case SQLINTERVAL:
	    /*  sql_dtime already set to current time  */
	    sql_intrvl.in_qual = var->sqllen;
	    dtsub(&sql_dtime, &sql_dtime, &sql_intrvl);
	    intoasc(&sql_intrvl, buf3);
	    newlen = var->sqllen = strlen(buf3) + 1;
	    var->sqltype = _SQL_FIXED_STR;
	    break;

	/*
	 *  Results in variable-length character string:
	 *	We allocate an extra byte to null-terminate it
	 *	We tell the sqlda about the byte.
	 */
	case SQLVCHAR:
	case SQLNVCHAR:
	    var->sqltype = _SQL_NULL_TERM_STR;
	    newlen = var->sqllen += 1;		/*  space for null byte  */
	    break;

	/*  Results as "float"  */
	case SQLDECIMAL:
	case SQLMONEY:
	    newlen = var->sqllen = sizeof(double);
	    var->sqltype = _SQL_FLOAT8;
	case SQLFLOAT:
	case SQLSMFLOAT:
	    if ((var->sqltype == SQLSMFLOAT && newlen != sizeof(float)) ||
		(var->sqltype == SQLFLOAT && newlen != sizeof(double))) {
		sprintf(buf1, "%d", i+1);
		sprintf(buf2, "%d", newlen);
		Tcl_AppendResult(interp, "Inconsistent size ", buf2,
			" in result column ", buf1, (char *)0);
		return TCL_ERROR;
	    }
	    break;

	/*
	 *  C Data types specified for these types in the manual are
	 *  inconsistent with the lengths that are returned by DESCRIBE.
	 */
	case SQLSERIAL:
	    var->sqltype = SQLINT;
	    newlen = var->sqllen = 4;
	case SQLSMINT:
	case SQLINT:
	    if ((var->sqltype == SQLSMINT && newlen != 2) ||
		(var->sqltype == SQLINT && newlen != 4)) {
		sprintf(buf1, "%d", i+1);
		sprintf(buf2, "%d", newlen);
		Tcl_AppendResult(interp, "Inconsistent size ", buf2,
			" in result column ", buf1, (char *)0);
		return TCL_ERROR;
	    }
	    break;

	case SQLBYTES:
	    sprintf(buf1, "%d", i+1);
	    Tcl_AppendResult(interp, "Unimplemented data type BYTES",
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;

	case SQLTEXT:
	    sprintf(buf1, "%d", i+1);
	    Tcl_AppendResult(interp, "Unimplemented data type TEXT",
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;

	default:
	    sprintf(buf1, "%d", i+1);
	    sprintf(buf2, "%d", var->sqltype);
	    Tcl_AppendResult(interp, "Unknown data type ", buf2,
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;
	}

	offset = rtypalign(offset, var->sqltype);
	var->sqldata = (char *)offset;
	offset += newlen;
    }

    /*  Fill in pointers to result variables (which we allocate).  */
    cur->buffer = (char *) malloc(offset);
    CheckAlloc(cur->buffer, "Sql_X_SetupDA");
    bzero((char *) cur->buffer, offset);
    for (i=0 ; i<da->sqld ; i++) {
	var = &da->sqlvar[i];
	var->sqldata += (long int) cur->buffer;
	var->sqlind = &sql_null_ind[i];			/*  shared!  */
    }

    return TCL_OK;
alloc_error:
    return TCL_ERROR;
}  /* Sql_X_SetupDA */
