/*
 * Copyright (c) 1993,1994,1995 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  SQL constants and localizations for ingres.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 *  This is an Embedded SQL C-language source file.  It must first be
 *  processed with "esqlc", to produce a C-language source file that can be
 *  compiled with cc.
 */


static char *SccsId = "@(#)sql_ingres.sc	2.2\t4/17/95";

EXEC SQL include sqlca;
EXEC SQL include sqlda;

#include <stdio.h>
#include <malloc.h>
#include <tcl.h>
#include "tcl_sql.h"

/*  Database commands local to Ingres  */
char	*sql_local_dbcmd_list = "";


/*
 *  Perform database-specific initializations.
 *	None, for ingres.
 */
int Sql_X_Init(
    Tcl_Interp	*interp)
{
    return TCL_OK;
}  /* Sql_X_Init */




/*
 *  Open a connection to a database/server:
 *	Makes the connection.
 *	Makes it current.
 *	Sets s->dbname
 *
 *  The arguments are:
 *	dbspec
 *		- name of an ingres database
 */
int Sql_X_Connect(
    Tcl_Interp	*interp,
    session	*s,
    int		*single_session,
    int		argc,
    char	**argv)
{
EXEC SQL BEGIN DECLARE SECTION ;
    char	*dbname;
    int		id;
EXEC SQL END DECLARE SECTION ;

    if (argc != 1) {
	Tcl_AppendResult(interp, "wrong # args to ", CMD_OPENDB, ".  Usage:\n",
		CMD_OPENDB, SQL_OPENDB_USAGE, (char *) NULL);
	return TCL_ERROR;
    }

    if (0 == getenv("II_SYSTEM")) {
	Tcl_AppendResult(interp, "II_SYSTEM environment variable not set",
		(char *) NULL);
	return TCL_ERROR;
    }

    dbname = argv[0];
    id = s->session_id;

    /*
     *  Older versions of ingres didn't support multiple sessions.
     *  We first try to open with session ID, falling back to single session
     *  mode if that fails.
     */

    EXEC SQL WHENEVER SQLERROR CONTINUE ;
    if (! *single_session) {
	EXEC SQL CONNECT :dbname SESSION :id ;
    }
    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    if (sqlca.sqlcode < 0) {
	EXEC SQL CONNECT :dbname ;
	/*  Set single session only if CONNECT succeeds.  */ 
	*single_session = 1;
    }

    StrClone(dbname, s->dbname, "Sql_Connect");
    return TCL_OK;

alloc_error:
    return TCL_ERROR;

fail:
    return Sql_X_HandleError(interp);
}  /* Sql_X_Connect */


/*
 *  Set the current connection to a database/server.
 */
int Sql_X_SetConnection(
    Tcl_Interp	*interp,
    session	*s)
{
EXEC SQL BEGIN DECLARE SECTION ;
    int		id;
EXEC SQL END DECLARE SECTION ;

    id = s->session_id;

    EXEC SQL WHENEVER SQLERROR CONTINUE ;
    EXEC SQL SET_SQL (session = :id) ;
    return TCL_OK;

fail:
    return Sql_X_HandleError(interp);
}  /* Sql_X_SetConnection */



/*
 *  Close the current connection to a database/server.
 */
int Sql_X_Disconnect(Tcl_Interp *interp)
{
    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL DISCONNECT ;
    return TCL_OK;

fail:
    return Sql_X_HandleError(interp);
}  /* Sql_X_Disconnect */



/*
 *  Retrieve vendor-specific database name and version.
 */
int Sql_X_DbVersion(
    Tcl_Interp	*interp,
    int		argc,
    char	**argv)
{
EXEC SQL BEGIN DECLARE SECTION ;
    char		version[33];
EXEC SQL END DECLARE SECTION ;
    int			i;
    int			rv;

    if(argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
	    argv[0], (char *) NULL);
	return TCL_ERROR;
    }

    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL SELECT cap_value
	INTO :version
	FROM iidbcapabilities
	WHERE cap_capability = 'COMMON/SQL_LEVEL' ;

    Tcl_AppendElement(interp, DB_NAME, (char *) NULL);
    for (i=sizeof(version)-2 ; i>=0 ; i--)
	if (version[i] != ' ')
	    break;
    version[i+1] = '\0';
    Tcl_AppendElement(interp, version, (char *) NULL);
    return TCL_OK;

fail:
    rv = Sql_X_HandleError(interp);
    return rv;
}  /* Sql_X_DbVersion */



/*
 *  Handle an SQL error:
 *	Report the SQL error message as the result
 *	Add the SQL error code and message to the ErrorInfo variable.
 *	Return TCL_ERROR
 *
 *  This procedure cleans up the SQL error string and returns it as the
 *  error message.  Note that this cleanup is dependent upon the
 *  particular SQL implementation.
 */
int Sql_X_HandleError(Tcl_Interp *interp)
{
    char	*errnum[15];
EXEC SQL BEGIN DECLARE SECTION ;
    char	sql_error_str[256];
EXEC SQL END DECLARE SECTION ;
    int		len;
    sprintf(errnum, "%d", sqlca.sqlcode);
    EXEC SQL WHENEVER SQLERROR CONTINUE;
    EXEC SQL INQUIRE_SQL (:sql_error_str = ERRORTEXT) ;
    len = strlen(sql_error_str);
    /*  strip off trailing newline  */
    if (sql_error_str[len-1] == '\n')
	sql_error_str[--len] = '\0';
    /*  strip off trailing (timestamp)  */
    if (sql_error_str[len-1] == ')' && sql_error_str[len-26] == '(' &&
	sql_error_str[len-31] == '\n') {
	len -= 31;
	sql_error_str[len-1] = '\0';
    }
    Tcl_AppendResult(interp, "sql error ", errnum, ": \"",
		sql_error_str, "\"", (char *)0);
    Tcl_SetErrorCode(interp, "sql", errnum, sql_error_str, (char *)NULL);
    return TCL_ERROR;
}  /* Sql_X_HandleError */





/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *
 *  Cursor-related localizations.
 *
 *  These mostly handle the different types supported by the various
 *  database systems.
 */


/*
 *  Place for SQL to store null indicators.
 *  This array is only used at the time of a fetch, so it can
 *  be shared among all cursors.
 */
static short	sql_null_ind[MAXCOLUMNS];



/*
 *  Given the result of
 *	SQL PREPARE
 *	SQL DESCRIBE
 *  on a SELECT statement:
 *	determine the storage requirements and offsets
 *	allocate storage
 *	setup the SQLDA
 *  for the given cursor.
 */
int Sql_X_SetupDA(
    Tcl_Interp		*interp,
    struct cursor	*cur
) {
    struct sqlda_	*da = SQLDA_PTR(cur);
    struct sqlvar_	*var;
    int			newlen;
    int			offset;
    int			i;
    char		buf1[10];
    char		buf2[10];

    offset = 0;
    for (i=0 ; i<da->sqld ; i++) {
	var = &da->sqlvar[i];
	newlen = var->sqllen;

	/*  Eliminate nullable data types  */
	if (var->sqltype < 0)
	    var->sqltype = -var->sqltype;

	switch (var->sqltype) {

	/*
	 *  Results in variable-length character string:
	 *	We allocate an extra byte to null-terminate it
	 *	We tell the sqlda about the byte.
	 */
	case IISQ_DTE_TYPE:
	    newlen = var->sqllen = _SQL_DATE_LEN;
	case IISQ_CHA_TYPE:
	case IISQ_VCH_TYPE:
	    var->sqltype = _SQL_NULL_TERM_STR;
	    newlen = var->sqllen += 1;
	    offset = 2 * ((offset+1)/2);
	    break;

	/*  Results as "float"  */
	case IISQ_DEC_TYPE:
	    var->sqllen = sizeof(double);
	case IISQ_FLT_TYPE:
	case IISQ_MNY_TYPE:
	    var->sqltype = _SQL_FLOAT_GENERIC;
	    newlen = var->sqllen;
	    offset = newlen * ((offset + newlen - 1) / newlen);
	    break;

	case IISQ_INT_TYPE:
	    offset = newlen * ((offset + newlen - 1) / newlen);
	    break;

	default:
	    sprintf(buf1, "%d", i+1);
	    sprintf(buf2, "%d", var->sqltype);
	    Tcl_AppendResult(interp, "Unknown data type ", buf2,
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;
	}
	var->sqldata = (char *)offset;
	offset += newlen;
    }

    /*  Fill in pointers to result variables (which we allocate).  */
    cur->buffer = (char *) malloc(offset);
    CheckAlloc(cur->buffer, "Sql_X_SetupDA");
    bzero((char *) cur->buffer, offset);
    for (i=0 ; i<da->sqld ; i++) {
	var = &da->sqlvar[i];
	var->sqldata += (long int) cur->buffer;
	var->sqlind = &sql_null_ind[i];			/*  shared!  */
    }

    return TCL_OK;
alloc_error:
    return TCL_ERROR;
}  /* Sql_X_SetupDA */
