/*
 * Copyright (c) 1995 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 *  SQL constants and localizations for Dec/Oracle Rdb.
 *
 *	Author: Glenn Trewitt, trewitt@pa.dec.com
 *		Digital Equipment Corporation
 *		Network Systems Laboratory
 *		Palo Alto, CA
 *
 *  This is an Embedded SQL C-language source file.  It must first be
 *  processed with sqlpre -l cc, to produce a C-language source file that
 *  can be compiled with cc.
 */


static char *SccsId = "@(#)sql_rdb.sc	2.1\t3/28/95";

#define SQLVAR_ELEMENTS 128

EXEC SQL include sqlca;
EXEC SQL include sqlda;

#include <stdio.h>
#include <malloc.h>
#include <tcl.h>
#include "tcl_sql.h"

/*  Database commands local to Rdb  */
char	*sql_local_dbcmd_list = "";


/*
 *  Perform database-specific initializations.
 *	Set the DBDATE environment variable to determine the date format.
 */
int Sql_X_Init(
    Tcl_Interp	*interp)
{
    
    return TCL_OK;
}  /* Sql_X_Init */




/*
 *  Open a connection to a database/server:
 *	Makes the connection.
 *	Makes it current.
 *	Sets s->dbname
 *
 *  The arguments can be either:
 *	dbspec
 *		- "filename mydatabase.rdb"
 *  or
 *	{handle dbspec} {handle dbspec}...
 *		- {db1 "filename fdb1.rdb"} {db2 "filename fdb2.rdb"}
 *
 *  If a single database is being opened, a handle is not permitted.
 */
int Sql_X_Connect(
    Tcl_Interp	*interp,
    session	*s,
    int		*single_session,
    int		argc,
    char	**argv
) {
EXEC SQL BEGIN DECLARE SECTION ;
    char	*dbname;
    char	*handle;
EXEC SQL END DECLARE SECTION ;
    Tcl_DString	dbs;
    char	**db;		/*  Parsed {handle db-spec}	*/
    int		dbc;		/*  Should be 2.		*/
    int		i;
    int		rv;

    Tcl_DStringInit(&dbs);

    if (argc == 1) {
	if (eqc(argv[0], "DEFAULT")) {
	    Tcl_DStringAppend(&dbs, "DEFAULT", -1);
	}
	else {
	    Tcl_DStringAppend(&dbs, "ATTACH ", -1);
	    Tcl_DStringAppend(&dbs, argv[0], -1);
	}
    }

    else {
	for (i=0 ; i<argc ; i++) {
	    rv = Tcl_SplitList(interp, argv[i], &dbc, &db);
	    if (rv != TCL_OK) {
		Tcl_DStringFree(&dbs);
		return rv;
	    }
	    if (dbc != 2) {
		Tcl_AppendResult(interp, "Usage:\tsql_opendb handle db-spec\n\
\tsql_opendb handle {a1 dbspec1} {a2 dbspec2}...",
		    (char *) NULL);
		Tcl_DStringFree(&dbs);
		return TCL_ERROR;
	    }
	    if (i != 0)
		Tcl_DStringAppend(&dbs, ", ", -1);

	    Tcl_DStringAppend(&dbs, "ATTACH ALIAS ", -1);
	    Tcl_DStringAppend(&dbs, db[0], -1);
	    Tcl_DStringAppend(&dbs, " ", -1);
	    Tcl_DStringAppend(&dbs, db[1], -1);
	    free(db);
	}
    }

    handle = s->handle;
    dbname = Tcl_DStringValue(&dbs);
    StrClone(dbname, s->dbname, "Sql_Connect");

    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL CONNECT TO :dbname AS :handle ;
    EXEC SQL SET CONNECT :handle ;

    Tcl_DStringFree(&dbs);
    return TCL_OK;

alloc_error:
    Tcl_DStringFree(&dbs);
    return TCL_ERROR;

fail:
    Tcl_DStringFree(&dbs);
    rv = Sql_X_HandleError(interp);
    return rv;
}  /* Sql_X_Connect */


/*
 *  Set the current connection to a database/server.
 */
int Sql_X_SetConnection(
    Tcl_Interp	*interp,
    session	*s
) {
EXEC SQL BEGIN DECLARE SECTION ;
    char	*handle;
EXEC SQL END DECLARE SECTION ;
    int		rv;

    handle = s->handle;

    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL SET CONNECT :handle ;
    return TCL_OK;

fail:
    rv = Sql_X_HandleError(interp);
    return rv;
}  /* Sql_X_SetConnection */



/*
 *  Close the current connection to a database/server.
 */
int Sql_X_Disconnect(Tcl_Interp *interp)
{
    EXEC SQL WHENEVER SQLERROR GOTO fail ;
    EXEC SQL DISCONNECT CURRENT;
    return TCL_OK;

fail:
    return Sql_X_HandleError(interp);
}  /* Sql_X_Disconnect */



/*
 *  Retrieve vendor-specific database name and version.
 */
int Sql_X_DbVersion(
    Tcl_Interp	*interp,
    int		argc,
    char	**argv
) {
EXEC SQL BEGIN DECLARE SECTION ;
    char		version[33];
EXEC SQL END DECLARE SECTION ;
    int			i;
    int			rv;

    if(argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
	    argv[0], (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_AppendElement(interp, DB_NAME, (char *) NULL);

    EXEC SQL WHENEVER SQLERROR GOTO fail ;
/*
    EXEC SQL SELECT cap_value
	INTO :version
	FROM iidbcapabilities
	WHERE cap_capability = 'COMMON/SQL_LEVEL' ;

    for (i=sizeof(version)-2 ; i>=0 ; i--)
	if (version[i] != ' ')
	    break;
    version[i+1] = '\0';
*/
    strcpy(version, "Rdb version - TBD");
    Tcl_AppendElement(interp, version, (char *) NULL);
    return TCL_OK;

fail:
    rv = Sql_X_HandleError(interp);
    return rv;
}  /* Sql_X_DbVersion */



/*
 *  Handle an SQL error:
 *	Report the SQL error message as the result
 *	Add the SQL error code and message to the ErrorInfo variable.
 *	Return TCL_ERROR
 *
 *  This procedure cleans up the SQL error string and returns it as the
 *  error message.  Note that this cleanup is dependent upon the
 *  particular SQL implementation.
 */
int Sql_X_HandleError(Tcl_Interp *interp)
{
    char	*errnum[15];
EXEC SQL BEGIN DECLARE SECTION ;
    char	sql_error_str[1024];
    int		sql_error_len;
EXEC SQL END DECLARE SECTION ;
    int		len, i;

    sprintf(errnum, "%d", _SQL_ERROR_CODE);
    EXEC SQL WHENEVER SQLERROR CONTINUE;

    sql_get_error_text(sql_error_str, sizeof(sql_error_str), &sql_error_len);

    /*  Collapse \r\n to \n.  */
    for (i=len=0 ; i<sql_error_len ; i++) {
	if (sql_error_str[i] != '\r')
	    sql_error_str[len++] = sql_error_str[i];
    }
    sql_error_str[len] = '\0';

    /*  strip off trailing newline  */
    if (sql_error_str[len-1] == '\n')
	sql_error_str[--len] = '\0';

    Tcl_AppendResult(interp, "sql error ", errnum, ": \"",
		sql_error_str, "\"", (char *)0);
    Tcl_SetErrorCode(interp, "sql", errnum, sql_error_str, (char *)NULL);
    return TCL_ERROR;
}  /* Sql_X_HandleError */





/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *
 *  Cursor-related localizations.
 *
 *  These mostly handle the different types supported by the various
 *  database systems.
 */


/*
 *  Place for SQL to store null indicators.
 *  This array is only used at the time of a fetch, so it can
 *  be shared among all cursors.
 */
static short	sql_null_ind[MAXCOLUMNS];



/*
 *  Given the result of
 *	SQL PREPARE
 *	SQL DESCRIBE
 *  on a SELECT statement:
 *	determine the storage requirements and offsets
 *	allocate storage
 *	setup the SQLDA
 *  for the given cursor.
 */
int Sql_X_SetupDA(
    Tcl_Interp		*interp,
    struct cursor	*cur
) {
    struct sqlda_	*da = SQLDA_PTR(cur);
    struct sqlvar_	*var;
    int			newlen;
    int			offset;
    int			i;
    char		buf1[10];
    char		buf2[10];

    offset = 0;
    for (i=0 ; i<da->sqld ; i++) {
	var = &da->sqlvar[i];
	newlen = var->sqllen;

	if (var->sqltype < 0)
	    var->sqltype = -var->sqltype;

	switch (var->sqltype) {

	/*
	 *  Results in fixed-length character string:
	 *	We allocate an extra byte to null-terminate it
	 *	We DO NOT tell the sqlda about the byte.
	 */
	case SQLDA_DATE:
	    var->sqllen = _SQL_DATE_LEN;
	    newlen = var->sqllen;
	case SQLDA_CHAR:
	    var->sqltype = _SQL_FIXED_STR;
	    newlen += 1;
	    offset = 2 * ((offset+1)/2);
	    break;

	/*
	 *  Results in variable-length character string:
	 *	We allocate an extra byte to null-terminate it
	 *	We tell the sqlda about the byte.
	 */
	case SQLDA_ASCIZ:
	case SQLDA_VARCHAR:
	    var->sqltype = _SQL_NULL_TERM_STR;
	    var->sqllen += 1;		/*  space for null byte  */
	    newlen = var->sqllen;
	    offset = 2 * ((offset+1)/2);
	    break;

	/*  Results as "float"  */
	case SQLDA_DECIMAL:
	    var->sqllen = sizeof(double);
	case SQLDA_FLOAT:
	    var->sqltype = _SQL_FLOAT_GENERIC;
	    newlen = var->sqllen;
	    offset = newlen * ((offset + newlen - 1) / newlen);
	    break;

	case SQLDA_TINYINT:
	case SQLDA_SMALLINT:
	case SQLDA_INTEGER:
	case SQLDA_QUADWORD:
	    newlen = 0xff & var->sqllen;
	    if ((var->sqltype == SQLDA_TINYINT && newlen != 1) ||
		(var->sqltype == SQLDA_SMALLINT && newlen != 2) ||
		(var->sqltype == SQLDA_INTEGER && newlen != 4) ||
		(var->sqltype == SQLDA_QUADWORD && newlen != 8)) {
		sprintf(buf1, "%d", i+1);
		sprintf(buf2, "%d", var->sqllen);
		Tcl_AppendResult(interp, "Inconsistent size ", buf2,
			" in result column ", buf1, (char *)0);
		return TCL_ERROR;
	    }
	    if (SQL_INT_SCALE(var) > 40) {
		sprintf(buf1, "%d", i+1);
		sprintf(buf2, "%d", var->sqltype);
		Tcl_AppendResult(interp, "Unknown data type ", buf2,
			" in result column ", buf1, (char *)0);
		return TCL_ERROR;
	    }
	    offset = newlen * ((offset + newlen - 1) / newlen);
	    break;

	case SQLDA_SEGSTRING:
	    sprintf(buf1, "%d", i+1);
	    Tcl_AppendResult(interp, "Unimplemented data type SEGSTRING",
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;

	case SQLDA_VARBYTE:
	    sprintf(buf1, "%d", i+1);
	    Tcl_AppendResult(interp, "Unimplemented data type VARBYTE",
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;

	default:
	    sprintf(buf1, "%d", i+1);
	    sprintf(buf2, "%d", var->sqltype);
	    Tcl_AppendResult(interp, "Unknown data type ", buf2,
		    " in result column ", buf1, (char *)0);
	    return TCL_ERROR;
	}
	var->sqldata = (char *)offset;
	offset += newlen;
    }

    /*  Fill in pointers to result variables (which we allocate).  */
    cur->buffer = (char *) malloc(offset);
    CheckAlloc(cur->buffer, "Sql_X_SetupDA");
    bzero((char *) cur->buffer, offset);
    for (i=0 ; i<da->sqld ; i++) {
	var = &da->sqlvar[i];
	var->sqldata += (long int) cur->buffer;
	var->sqlind = &sql_null_ind[i];			/*  shared!  */
    }

    return TCL_OK;
alloc_error:
    return TCL_ERROR;
}  /* Sql_X_SetupDA */
