/* --------------------------------- init.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* startup code.
*/

#include "fly.h"


extern struct SysDriver NEAR SysDriver;
extern struct TmDriver NEAR TmDriver;

static int NEAR
get_option (char *e)
{
	char	t, *p;
	int	errs = 0, itemp;
	long	ltemp;

	while (e[0] == ' ')
		++e;
	if (e[0] == '\0')
		return (0);

	for (t = 0, p = e; *p; ++p) {
		if (isspace (*p)) {
			t = *p;
			*p = '\0';
			break;
		}
	}

	LogPrintf (" %s\n", e);
	switch (e[0]) {
	case 'b':
		if (e[1] >= '0' && e[1] <= '9')
			st.windows = (int)(e[1] - '0');
		else
			st.windows = WIN_FULL;
		break;
	case 'c':
		if (get_long (e+2, &ltemp) ||
		    set_rrggbb (e[1], (Ulong)ltemp)) {
			LogPrintf ("bad color %s\n", e);
			++errs;
		}
		break;
	case 'd':			/* devices */
		switch (e[1]) {
		case 'k':		/* keyboard */
			st.kbdname = xfree (st.kbdname);
			st.kbdname = xstrdup (e+2);
			break;
		case 'n':		/* netport */
			{
				struct netname	*p;

				if (!NEW (p)) {
					LogPrintf ("no mem for netname\n");
					++errs;
					break;
				}
				p->name = xstrdup (e+2);
				p->next = st.netnames;
				st.netnames = p;
			}
			break;
		case 'p':		/* pointer */
			st.ptrname = xfree (st.ptrname);
			st.ptrname = xstrdup (e+2);
			break;
		case 's':		/* sound */
			st.sndname = xfree (st.sndname);
			st.sndname = xstrdup (e+2);
			break;
		case 't':		/* timer */
			st.timeropts = xfree (st.timeropts);
			st.timeropts = xstrdup (e+2);
			break;
		case 'v':		/* video */
			st.grname = xfree (st.grname);
			st.grname = xstrdup (e+2);
			break;
		default:
			goto too_bad;
		}
		break;
	case 'D':
		st.dtype = xfree (st.dtype);
		st.dtype = xstrdup (e+1);
		break;
	case 'f':
		st.fname = xfree (st.fname);
		st.fname = xstrdup (e+1);
		break;
	case 'F':
		/* already processed */
		break;
	case 'h':
	case '?':
		welcome (2);
#ifdef NOSTDERR
#define SHOW(x)	LogPrintf (x)
#else
#define SHOW(x)	fprintf (stderr, x)
#endif
		SHOW ("usage: fly8 [options]\n");
		SHOW ("       see fly.ini for options\n");
#undef SHOW
		++errs;
		break;
	case 'H':
		st.homename = xfree (st.homename);
		st.homename = xstrdup (e+1);
		break;
	case 'I':
		/* already processed */
		break;
	case 'l':
		st.flags ^= SF_LANDSCAPE;
		break;
	case 'L':
		/* already processed */
		break;
	case 'm':
		st.grmname = xfree (st.grmname);
		st.grmname = xstrdup (e+1);
		break;
	case 'M':
		st.mname = xfree (st.mname);
		st.mname = xstrdup (e+1);
		break;
	case 'n':
		switch (e[1]) {
		case 'b':		/* max num of display list lines */
			if (get_long (e+2, &ltemp))
				goto bad_num;
			itemp = (int)(ltemp/BUFLEN*2+1);
			if (itemp < 20)
				goto bad_num;
			st.maxbuffers = (Ushort)itemp;
			break;
		case 'i':		/* plane dynamics minimal interval */
			if (get_int (e+2, &itemp) || itemp < 1)
				goto bad_num;
			st.misc[6] = (int)itemp;
			break;
		case 'm':		/* num of macros */
			if (get_int (e+2, &itemp) || itemp < 1)
				goto bad_num;
			st.nMacros = (Ushort)itemp;
			break;
		case 't':		/* num of secs till shutdown */
			if (get_long (e+2, &ltemp) || ltemp < 0)
				goto bad_num;
			st.ShutdownTime = ltemp*1000L;
			break;
		bad_num:
			LogPrintf ("bad number in '%s'\n", e);
			++errs;
			break;
		default:
			goto too_bad;
		}
		break;
	case 'N':
		st.nikname = xfree (st.nikname);
		if (e[1]) {
			st.nikname = xstrdup (e+1);
			if (strlen (st.nikname) >LNAME-1)   /* keep it short */
				st.nikname[LNAME-1] = '\0';
		}
		break;
	case 'P':
		st.ptype = xfree (st.ptype);
		st.ptype = xstrdup (e+1);
		break;
	case 'q':
		st.quiet = 0;
		break;
	case 'r':
		itemp = 1;
		if (e[itemp] == '0') {
			st.network &= ~NET_ON;
			++itemp;
		} else
			st.network |= NET_ON;
		if (e[itemp] == 'l') {
			++itemp;
			if (e[itemp] == '0') {
				st.network &= ~NET_NOBCAST;
				++itemp;
			} else
				st.network |= NET_NOBCAST;
		}
		break;
	case 's':
		if (get_int (e+1, &itemp)) {
			LogPrintf ("use -sPortAddr\n");
			++errs;
		}
		st.misc[7] = itemp;
		break;
	case 'T':
		st.teamname = xfree (st.teamname);
		if (e[1]) {
			st.teamname = xstrdup (e+1);
			if (strlen (st.teamname) >LNAME-1)  /* keep it short */
				st.teamname[LNAME-1] = '\0';
		}
		break;
	case 'v':
		st.flags ^= SF_VERBOSE;
		break;
	case 'V':
		st.vmdname = xfree (st.vmdname);
		st.vmdname = xstrdup (e+1);
		break;
	case 'X':
		st.navname = xfree (st.navname);
		st.navname = xstrdup (e+1);
		break;
	case 'Y':
		st.lndname = xfree (st.lndname);
		st.lndname = xstrdup (e+1);
		break;
	case 'z':
		st.flags ^= SF_BLANKER;
		if (e[1]) {
			if (get_int (e+1, &itemp) ||
				itemp < 0) {
				LogPrintf ("use -znn\n");
				++errs;
			}
			st.drones = itemp;
		} else
			st.drones = 0;
		break;
	default:
too_bad:
		LogPrintf ("unknown option %s\n", e);
		++errs;
		break;
	}
	if (t)
		*p = t;
	return (errs);
}

static int NEAR
get_argopts (char **a)
{
	int	errs = 0;
	char	*p;

	if (*a)
		LogPrintf ("command line args:\n");

	while (T(p = *a++)) {
		p = xstrdup (p+(p[0] == '-'));
		errs += get_option (p);
		p = xfree (p);
	}
	return (errs);
}

static int NEAR
get_envopts (void)
{
	char	*e, *p, *q;
	int	errs = 0;

	e = getenv ("FLY8");
	if (e) {
		LogPrintf ("env args:\n");
		e = xstrdup (e);
		for (p = e; p && *p; p = q) {
			while (*p == ' ')	/* find start */
				++p;
			if (*p == '\0')
				break;
			q = strchr (p, ';');	/* find end */
			if (q)
				*q++ = '\0';	/* end string */
			errs += get_option (p);
		}
		e = xfree (e);
	}
	return (errs);
}

static FILE * NEAR
find_ini (void)
{
	FILE	*ini;
	char	*e, *p, *q;

	if (!st.iname)
		st.iname = xstrdup (INIFILE);

	if (T(ini = fopen (st.iname, RTMODE)))
		return (ini);

	if (st.fdir) {
		Sys->BuildFileName (st.filename, st.fdir, st.iname, "");
		if (T(ini = fopen (st.filename, RTMODE))) {
			st.iname = xfree (st.iname);
			st.iname = xstrdup (st.filename);
			return (ini);
		}
	}

	if (T(e = getenv ("HOME"))) {
		Sys->BuildFileName (st.filename, e, st.iname, "");
		if (T(ini = fopen (st.filename, RTMODE))) {
			st.iname = xfree (st.iname);
			st.iname = xstrdup (st.filename);
			return (ini);
		}
	}

	if (T(e = getenv ("PATH"))) {
		e = xstrdup (e);
		for (p = e; p && *p; p = q) {
			q = strchr (p, PATHSEP);	/* find end */
			if (q)
				*q++ = '\0';	/* end string */
			Sys->BuildFileName (st.filename, p, st.iname, "");
			if (T(ini = fopen (st.filename, RTMODE))) {
				st.iname = xfree (st.iname);
				st.iname = xstrdup (st.filename);
				return (ini);
			}
		}
		e = xfree (e);
	}
	return (ini);
}

static int NEAR
get_iniopts (void)
{
	int	errs = 0, l;
	FILE	*ini;
	char	opt[256], *p;

	if (!(ini = find_ini ()))
		return (0);

	LogPrintf ("%s args:\n", st.iname);
	while (fgets (opt, sizeof (opt), ini)) {
		for (p = opt; isspace (*p); ++p)
			;
		if ('\n' == p[0] || '#' == p[0] || '\0' == p[0])
			continue;
		l = strlen(p);
		p[l-1] = '\0';		/* remove '\n' */
		errs += get_option (p);
	}
	if (ferror (ini)) {
		perror ("error reading init file");
		++errs;
	}

	fclose (ini);
	return (errs);
}

static OBJECT * NEAR
create_viewer (int type, int nzoom)
{
	OBJECT	*p;

	if (T(p = create_object (O_VIEWER, 1))) {
		p->misc[0] = type;
		save_viewport (p);
		zoom (p->viewport, nzoom);
	} else
		MsgEPrintf (-50, "no viewer %u", type);
	return (p);
}

extern void FAR
initialize (char *argv[])
{
	POINTER	*ptr;
	char	**pp, *p;
	int	gotit;

	st.object_id = 1000;			/* reserve 1-999 */
	st.paralax = 12;
	st.focus = VMAX;
	st.gap = 64;
	st.quiet = 1;
	st.gravity = (int) (C_G*VONE);
	st.windows = WIN_FULL;
	st.flags1 |= SF_USEG;
	st.info = 1;
	st.extview = HDT_RADAR;
	st.maxbuffers = 100;
	st.SkyLines = 50;			/* sky lines */
	st.network |= NET_AUTOACCEPT;
	st.misc[6] = 100;			/* min plane interval */
	st.nMacros = 256;
	Sys = &SysNone;
	Tm = &TmNone;
	Snd = &SndNone;
	Kbd = &KbdNone;

/* First we bring up the system and time drivers, these are essential.
*/

	if (F(Sys = &SysDriver) || Sys->Init ()) {
		Sys = &SysNone;
		LogPrintf ("system init failed\n");
		die ();
	}

	if (F(Tm = &TmDriver) || Tm->Init (0)) {
		Tm = &TmNone;
		LogPrintf ("timer init failed\n");
		die ();
	}
	st.big_bang = Tm->Milli ();

	Frandomize ();

	for (pp = argv+1; T(p = *pp); ++pp) {
		if ('-' == p[0])
			++p;
		if ('I' == p[0]) {
			st.iname = xfree (st.iname);
			st.iname = xstrdup (p+1);
		} else if ('F' == p[0]) {
			st.fdir = xfree (st.fdir);
			st.fdir = xstrdup (p+1);
		} else if ('L' == p[0]) {
			if ('.' == p[1]) {
				st.flags1 |= SF_FLUSHLOG;
				++p;
			}
			if ('\0' != p[1]) {
				st.lname = xfree (st.lname);
				st.lname = xstrdup (p+1);
			}
		}
	}

	if (log_init ())
		die ();

	if (memory_init ()) {
		LogPrintf ("no memory mgr\n");
		die ();
	}

	LogPrintf ("Fly8 start: %s\n", Tm->Ctime ());
	LogPrintf ("Program  %s\n", argv[0]);
	welcome (1);

	if (get_iniopts () || get_envopts () || get_argopts (argv+1))
		die ();

	if (st.timeropts && Tm->Init (st.timeropts)) {
		Tm = &TmNone;
		LogPrintf ("timer re-init failed\n");
		die ();
	}
	if (!st.nikname)
		st.nikname = xstrdup ("JohnDoe");
	if (!st.teamname)
		st.teamname = xstrdup ("[*]");

	Fsrand ((int)Tm->Milli ());		/* don't be boring */

	if (funcs_init ()) {
		LogPrintf ("funcs init failed\n");
		die ();
	}

	Kbd = keyboard_init (st.kbdname);
	if (!Kbd || Kbd->Init (st.kbdname)) {
		Kbd = &KbdNone;
		LogPrintf ("keyboard init failed\n");
		die ();
	}

/* In the next section 'Snd' is temporarily set to zero. Handle this
 * carefully since many programs use it blindly.
*/
	gotit = 0;
	if (F(Snd = sound_init (st.sndname)) || Snd->Init (st.sndname)) {
		Snd = &SndNone;		/* always safe */
		LogPrintf ("sound init failed\n");
	} else
		gotit = 1;
	if (!gotit && st.sndname && (F(Snd = sound_init (0)) || Snd->Init (0))) {
		Snd = &SndNone;		/* always safe */
		LogPrintf ("default sound init failed\n");
	} else
		gotit = 1;

	if (!gotit && (F(Snd = sound_init ("")) || Snd->Init (""))) {
		Snd = &SndNone;		/* always safe */
		LogPrintf ("'NoSound' sound init failed\n");
		die ();
	}
	st.sndname = xfree (st.sndname);
	st.sndname = xstrdup (Snd->name);

	if (!NEW (CS)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	if (!NEW (CW)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	if (!NEW (CP)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	if (0 == (Gr = devices_init (st.grname))) {
		LogPrintf ("devices init failed\n");
		die ();
	}

	CS->device = devices_select (st.grmname);
	if (CS->device == 0 || Gr->Init (CS->device, st.grname)) {
		if (st.grmname) {
			LogPrintf ("no device: %s\n", st.grname);
			LogPrintf ("trying default\n");
			devices_release ();
			CS->device = devices_select (NULL);
		} else
			CS->device = 0;
		if (CS->device == 0 || Gr->Init (CS->device, st.grname)) {
			devices_release ();
			LogPrintf ("no device\n");
			die ();
		}
	}

	CS->FgColor = st.white;
	CS->BgColor = st.black;
	CS->BoColor = st.lgray;
	CS->FontWidth = CS->device->FontWidth;
	CS->FontHeight = CS->device->FontHeight;

	st.tfg = st.white;
	st.tbg = st.black;
	st.wfg = st.green;
	st.cfg = st.red;
	st.hfg = st.hudlow;
	st.hfgi = st.hudhigh;

	set_palette ();

/* Now we are in graphics mode!
*/
	st.grmname = xfree (st.grmname);
	st.grmname = xstrdup (CS->device->name);

	set_screen (CS->device->sizex, CS->device->sizey);
	windows_set ();
	set_textwin ();
	set_viewport ();

	Gr->SetTextPos (2,1);

	font_set (0);
	st.StFontSize = 8;

	if (msg_init ()) {
		LogPrintf ("no messages\n");
		die ();
	}

	if (mac_init ()) {
		LogPrintf ("no macros");
		die ();
	}

	MsgPrintf (-100, "System   %s", Sys->name);
	MsgPrintf (-100, "Timer    %s", Tm->name);
	MsgPrintf (-100, "Graphics %s", Gr->name);
	MsgPrintf (-100, " vmode   %s", st.vmdname);
	MsgPrintf (-100, " mode    %s", st.grmname);
	MsgPrintf (-100, "Sound    %s", Snd->name);
	MsgPrintf (-100, "Keyboard %s", Kbd->name);

	Gr->SetVisual (0);
	Gr->SetActive (0);

	MsgPrintf (-100, "nBuffers %u", st.maxbuffers);

	if (pointers_init ()) {
		LogPrintf ("no pointers\n");
		die ();
	}

	if (bodies_init ()) {
		LogPrintf ("no bodies\n");
		die ();
	}

	if (land_init ()) {
		LogPrintf ("no land\n");
		die ();
	}

	if (nav_init ()) {
		LogPrintf ("no nav\n");
		die ();
	}
	if ((st.home = nav_find (st.homename)) < 0) {
		LogPrintf ("bad home name \"%s\"\n", st.homename);
		st.home = 0;
	}

/* We are initialized, now get us a plane. First find the pointer.
*/
	ptr = pointer_select (st.ptrname);
	if (!ptr || ptr->control->Init (ptr, st.ptrname)) {
		if (ptr)
			ptr = pointer_release (ptr);
		if (st.ptrname) {
			MsgEPrintf (-100, "no ptr: %s", st.ptrname);
			st.ptrname = xfree (st.ptrname);
		}
		MsgPrintf (-100, "trying default");
		ptr = pointer_select (NULL);
		if (!ptr  || ptr->control->Init (ptr, 0)) {
			if (ptr)
				ptr = pointer_release (ptr);
			LogPrintf ("no pointer\n");
			die ();
		}
		st.ptrname = xstrdup (ptr->name);
	}
	MsgPrintf (-100, "Pointer  %s", ptr->name);

	CO = COT = 0;

	st.options = st.ptype;
	if (!(CC = create_object (O_PLANE, 1))) {
		LogPrintf ("no plane %s\n", st.ptype);
		pointer_release (ptr);
		die ();
	}
	CC->pointer = ptr;
	ptr->l[9] = 100;		/* brakes */
	CC->flags |= F_CC | F_FRIEND;
	CC->color = st.lblue;
	CC->gpflags |= GPF_PILOT;
	place_plane (CC, st.home);
	CV = CC;

/* Establish the virtual viewers.
*/
	create_viewer (HDT_FRONT, 0);
	create_viewer (HDT_REAR, 0);
	create_viewer (HDT_MAP, 0);
	create_viewer (HDT_RADAR, 0);
	create_viewer (HDT_TARGET, 12);
	create_viewer (HDT_PAN, 12);
	create_viewer (HDT_GAZE, 0);
	create_viewer (HDT_CHASE, 0);
	create_viewer (HDT_FOLLOW, 0);

	create_viewer (HDT_RIGHT, 0);
	create_viewer (HDT_LEFT,  0);

/* Now get the net going.
*/
	remote_init ();

/* We made it, tell the world.
*/
	welcome (0);			/* welcome everybody */
	if (st.quiet)
		Snd->List (TnHello, 0);	/* sing a song... */

	Snd->List (TnEngine, 0);

	if (st.flags & SF_BLANKER) {
		EE(CC)->hud1 &= ~HUD_PANEL;
		EE(CC)->radar = R_ON | R_LOCK | (3*R_MODE);
		EE(CC)->flags |= PF_CHASE | PF_KILL;
		EE(CC)->weapon = WE_M61;
		clear_text ();
	} else {
		EE(CC)->hud1 |= HUD_PANEL;
	}

	double_buffer (st.flags1 ^ SF_DBUFFERING);

	st.flags |= SF_INITED;
	if (st.misc[7])
		Gr->Shutters (-1);		/* turn on */

	st.DroneTime = st.present;

	{
		static Ushort	keys[] = {KF_INIT};	/* startup macro */

		mac_interpret (keys, rangeof (keys));
	}
	log_flush (1);
}
