/* 
 *     XaoS, a fast portable realtime fractal zoomer 
 *                  Copyright (C) 1996 by
 *
 *      Jan Hubicka          (hubicka@paru.cas.cz)
 *      Thomas Marsh         (tmarsh@austin.ibm.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef _plan9_
#include <u.h>
#include <stdio.h>
#include <libc.h>
#else
#include "aconfig.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#include <signal.h>
#endif
#include "config.h"
#ifndef _plan9_
#include <assert.h>
#endif
#include "zoom.h"
#include "help.h"
#include "gif.h"
#include "autopilot.h"
#include "complex.h"
#include "ui.h"
#include "palette.h"
#include "plane.h"
#include "ui.h"
#include "param.h"
#include "version.h"
extern struct ui_driver *drivers[];
extern int ndrivers;
#define SIZE 1024

static void ui_do_fractal(void);
static void ui_init(zoom_context *, void (*)(), int (*)(), int, void (*)(int, int, char *), int);
static void ui_savefile(void);
static void ui_coloringmode(int);
static void ui_mandelbrot(int, int);
static void ui_incoloringmode(void);
static int ui_autopilot(void);
static void ui_speedup(void);
static void ui_help(int);
static void ui_slowdown(void);
static int ui_mouse(int, int, int, int);
static int ui_inverse(int);
static int keybuffer[SIZE];
static int begin = 0, end = 0;
static int incalculation, resized;
static int callresize = 0;
#define add_buffer(char) if(((end+1)%SIZE)!=begin) {keybuffer[end]=char;end=(end+1)%SIZE;}
char *interruptkeys = "z123456789rmfq";
static int defformula[10];
static int defautopilot = 0;
static int defiters = DEFAULT_MAX_ITER;
static float defspeed = 1;
static char *defdriver = NULL;
static int todriver = 0;
static int deflist;
static int printconfig;
const struct params global_params[] =
{
    {"-mandel", P_SWITCH, defformula, "Start with Mandelbrot set fractal(default)"},
    {"-mandel3", P_SWITCH, defformula + 1, "Start with Mandelbrot set power 3 fractal"},
    {"-mandel4", P_SWITCH, defformula + 2, "Start with Mandelbrot set power 4 fractal"},
    {"-mandel5", P_SWITCH, defformula + 3, "Start with Mandelbrot set power 5 fractal"},
    {"-mandel6", P_SWITCH, defformula + 4, "Start with Mandelbrot set power 6 fractal"},
    {"-octo", P_SWITCH, defformula + 5, "Start with Octal fractal"},
    {"-newton", P_SWITCH, defformula + 6, "Start with Newton fractal"},
  {"-barnsley", P_SWITCH, defformula + 7, "Start with Barnsley fractal"},
    {"-phoenix", P_SWITCH, defformula + 8, "Start with Phoenix fractal"},
    {"-autopilot", P_SWITCH, &defautopilot, "Enable autopilot"},
    {"-iter", P_NUMBER, &defiters, "Default number of iterations"},
    {"-speed", P_FLOAT, &defspeed, "Zooming speed (default is 1)"},
    {"-driver", P_STRING, &defdriver, "Select driver"},
    {"-list", P_SWITCH, &deflist, "List available drivers. Than exit"},
    {"-config", P_SWITCH, &printconfig, "Print configuration. Than exit"},
    {NULL, 0, NULL, NULL}
};
#define xstoc(x)        ((number_t) (context->s.nc + x * ((context->s.mc - context->s.nc) \
                                / (number_t) context->width)))
#define ystoc(y)        ((number_t) (context->s.ni + y * ((context->s.mi - context->s.ni) \
                                / (number_t) context->height)))

#define textwidth driver->textheight
#define FRAMETIME (1000000/FRAMERATE)
static int autopilot = 0;
static double maxstep = MAXSTEP, speedup = STEP;
static zoom_context *context;
static double mul;
static int drawingtime, tbreak = 0;
static struct ui_driver *driver;
static int ministatus = 0;
static char statustext[256];
#ifndef _plan9_
#ifdef HAVE_GETTIMEOFDAY
static struct timeval tv1, tv2;
static struct timezone tzp;
#endif
#endif

#define DRIVER 4
#define PLANE 3
#define OUTCOLOR 2
#define HELP 1
#define FORMULA 0
static int numbertype = FORMULA;
static int helppage;
static void ui_display(void)
{
    driver->display();
    numbertype = FORMULA;
    if (ministatus)
	driver->print(0, 0, statustext);
}

extern int dynsize;
static void ui_outofmem(void)
{
    printf("\n\nXaoS is out of memory.\n"
	   "The memory space dynamically allocated by XaoS is approx:\n\n"
	   " 2*(width+16)*height+%i*width+%i*height+%i*(bigger of width or height)+%i bytes\n\n"
	   "Plus memory additionaly allocated by drivers. For example X11 driver in\n"
	   "truecolor or hicolor modes allocated again (width+16)*height*bytes per pixel.\n"
	   "Curses driver uses 9*64KB table and mac driver allocates 40Kb for saving.\n"
	   "So take this calculation as \"lower bound\"\n",
	   sizeof(*context->xpos) + sizeof(realloc_t),
	   sizeof(*context->xpos) + sizeof(realloc_t),
	   9 * dynsize + sizeof(int) * 2
	   ,sizeof(realloc_t) * 2 + sizeof(*context));
}

#ifndef _plan9_
static int lookup_timer(void)
{
#ifdef HAVE_GETTIMEOFDAY
    int time;


    do {
	gettimeofday(&tv2, &tzp);
    }
    while (tv2.tv_usec > 999999);
    time = (1000000 * (tv2.tv_sec - tv1.tv_sec) + (tv2.tv_usec - tv1.tv_usec));
    return time;
#else
    return (FRAMETIME + 1);
#endif
}
static int get_timer(void)
{
    int time;


    time = lookup_timer();
#ifdef HAVE_GETTIMEOFDAY
    tv1 = tv2;
#endif
    if (time < 0)
	time = 0;
    return time;
}
#else
extern int lookup_timer(void);	/*plan9 have its own timming stuff */
extern int get_timer(void);
#endif
static int waitcount;
static int waitcount1;
#define WAITTIME  200000
#define WAITTIME1 2000000
#define CHECKPROCESSEVENTS(b,k) assert(!((k)&~15)&&!((b)&~(BUTTON1|BUTTON2|BUTTON3)))
static void ui_waitfunc()
{
    char str[80];
    int l, x, y, b, k;
#ifdef HAVE_GETTIMEOFDAY
    if ((l = lookup_timer()) > (waitcount + 1) * WAITTIME) {
	if (l > (waitcount1 + 1) * WAITTIME1) {
	    ui_display();
	    waitcount1++;
	}
	sprintf(str, "Pass %i %3.2f%%", context->pass, (float) (context->max ? context->pos * 100.0 / context->max : 100.0));
	driver->processevents(0, &x, &y, &b, &k);
	CHECKPROCESSEVENTS(b, k);
	if (k & 3)
	    context->interrupt = 1;
	driver->print(0, context->height - textwidth, str);
	waitcount++;
    }
#endif
}


static void ui_do_fractal(void)
{
    int i, y;
    if (tbreak)
	get_timer();
    for (i = begin; i != end; i = (i + 1) % SIZE) {
	for (y = 0; y < strlen(interruptkeys); y++)
	    if (interruptkeys[y] == tolower(keybuffer[i]))
		return;
    }
    incalculation = 1;
    set_view(context, &context->s);
    waitcount = lookup_timer() / WAITTIME + 2;
    waitcount1 = lookup_timer() / WAITTIME1 + 1;
    do_fractal(context);
    /*s = context->s; */
    ui_display();
#ifdef HAVE_GETTIMEOFDAY
    drawingtime = get_timer();
    {
	double times = (context->currentformula->v.mc - context->currentformula->v.nc) / (context->s.mc - context->s.nc);
	sprintf(statustext, "%s %.2f times %2.2f frames/sec %c %i %i %i %i            ", times < 1 ? "unzoomed" : "zoomed", times < 1 ? 1.0 / times : times, drawingtime ? 1000000.0 / drawingtime : 0.0, autopilot ? 'A' : ' ', context->coloringmode + 1, context->incoloringmode + 1, context->plane + 1, context->maxiter);
    }
    if (drawingtime > 200000)
	drawingtime = 200000;
    mul = (double) drawingtime / FRAMETIME;
#else
    mul = 1.0;
    {
	double times = (context->currentformula->v.mc - context->currentformula->v.nc) / (context->s.mc - context->s.nc);
	sprintf(statustext, "%s %.2f times %c %i %i %i            ", times < 1 ? "unzoomed" : "zoomed", times < 1 ? 1.0 / times : times, autopilot ? 'A' : ' ', context->coloringmode + 1, context->incoloringmode + 1, context->plane + 1);
    }
#endif
    numbertype = FORMULA;
    if (ministatus)
	driver->print(0, 0, statustext);
#ifdef DEBUG
    printf("framerate:%f\n", 1000000.0 / drawingtime);
#endif
    incalculation = 0;
}
static double step = 0;

static void ui_outcoloringmode_menu(void)
{
    int i;
    char s[80];
    driver->print(0, 0, "Please select outcoloring mode:    ");
    for (i = 0; i < OUTCOLORING; i++) {
	sprintf(s, " %i - %-30s", i + 1, outcolorname[i]);
	driver->print(0, (i + 1) * textwidth, s);
    }
    numbertype = OUTCOLOR;
}
static void ui_incoloringmode(void)
{
    context->incoloringmode++;
    context->incoloringmode %= INCOLORING;
    ui_message();
    init_tables(context);
    ui_do_fractal();
}

static void ui_coloringmode(int mode)
{
    context->coloringmode = mode;
    context->coloringmode %= OUTCOLORING;
    ui_message();
    init_tables(context);
    ui_do_fractal();
}

static void ui_savefile(void)
{
    char *s;
    char str[256];

    if (!context->dirty) {
	driver->print(0, 0, "Recalculating image...");
	init_tables(context);
    }
    ui_do_fractal();
    driver->print(0, 0, "Writing gif image..  ");
    s = writegif(context);
    sprintf(str, "File %s saved", s);
    driver->print(0, 0, str);
}


void ui_tbreak(void)
{
    tbreak = 2;
}

static int ui_inverse(int mode)
{
    /*recalculate(context, &context->s.mc, &context->s.mi);
       recalculate(context, &context->s.nc, &context->s.ni); */
    context->plane = mode;
    context->plane %= PLANES;
    /*recalculateback(context, &context->s.mc, &context->s.mi);
       recalculateback(context, &context->s.nc, &context->s.ni); */
    init_tables(context);
    /*context->s = s; */
    ui_message();
    ui_do_fractal();
    ui_tbreak();
    return (context->plane);
}
static void ui_driver_menu(void)
{
    int i;
    char s[80];
    if (ndrivers != 1) {
	driver->print(0, 0, "Please select driver:                        ");
	driver->print(0, textwidth, "Warning..Xaos Can crash during this operation");
	driver->print(0, 2 * textwidth, "Palette will be set to default               ");
	driver->print(0, 3 * textwidth, " 1 - abort action                            ");
	for (i = 0; i < ndrivers; i++) {
	    sprintf(s, " %i - %-40s", i + 2, drivers[i]->name);
	    driver->print(0, (i + 4) * textwidth, s);
	}
	numbertype = DRIVER;
    } else
	driver->print(0, 0, "Only one driver available-function disabled");
}
static void ui_plane_menu(void)
{
    int i;
    char s[80];
    driver->print(0, 0, "Please select display plane:       ");
    for (i = 0; i < PLANES; i++) {
	sprintf(s, " %i - %-30s", i + 1, planename[i]);
	driver->print(0, (i + 1) * textwidth, s);
    }
    numbertype = PLANE;
}

static void ui_mandelbrot(int mousex, int mousey)
{
    context->mandelbrot ^= 1;
    context->pre = xstoc(mousex), context->pim = ystoc(mousey);
    recalculate(context, &context->pre, &context->pim);
    init_tables(context);
    ui_message();
    ui_do_fractal();
    ui_tbreak();
}

static int ui_autopilot(void)
{
    clean_autopilot();
    return (autopilot ^= 1);
}

#ifdef _UNDEFINED_
static void ui_speedup(void)
{
    speedup *= SPEEDUP, maxstep *= SPEEDUP;
    assert(speedup > 0 && maxstep > 0);
}


static void ui_slowdown(void)
{
    speedup /= SPEEDUP, maxstep /= SPEEDUP;
    assert(speedup > 0 && maxstep > 0);
}

#endif

static void ui_help(int number)
{
    int nlines = context->height / textwidth - 7;
    int i, page;
    int npages = (UGLYTEXTSIZE + driver->helpsize + nlines - 1) / (nlines ? nlines : 1);
    char str[80];
    if (nlines <= 0)
	return;
    if (!number)
	helppage--;
    if (number == 1)
	helppage++;
    if (helppage < 0)
	helppage = 0;
    if (helppage >= npages)
	helppage = npages - 1;
    page = helppage;
    sprintf(str, "------> Ugly help <-----> %2i/%2i <------", page + 1, npages);
    driver->print(0, textwidth, "                                       ");
    driver->print(0, 0, str);
    for (i = 0; i < nlines; i++) {
	if (page * nlines + i < UGLYTEXTSIZE)
	    driver->print(0, (i + 2) * textwidth, helptext[page * nlines + i]);
	else {
	    if (page * nlines + i < UGLYTEXTSIZE + driver->helpsize)
		driver->print(0, (i + 2) * textwidth, driver->help[page * nlines + i - UGLYTEXTSIZE]);
	    else
		driver->print(0, (i + 2) * textwidth, "~                                      ");
	}
    }
    driver->print(0, (nlines + 2) * textwidth, "                                       ");
    driver->print(0, (nlines + 3) * textwidth, "Hyperuglytext browser (tm) version 1.0 ");
    driver->print(0, (nlines + 4) * textwidth, "Press '1' for previous and '2' for next");
    driver->print(0, (nlines + 5) * textwidth, "             space for exit            ");
    numbertype = HELP;
}


void ui_status(void)
{
    char str[80];

    sprintf(str, "Fractal name:%s", context->currentformula->name[!context->mandelbrot]);
    driver->print(0, textwidth, str);
    sprintf(str, "Fractal type:%s", context->mandelbrot ? "Mandelbrot" : "Julia");
    driver->print(0, 2 * textwidth, str);
    sprintf(str, "View:[%1.12f,%1.12f]", (double) context->s.nc, (double) context->s.ni);
    driver->print(0, 3 * textwidth, str);
    sprintf(str, "size:[%1.12f,%1.12f]", (double) (context->s.mc - context->s.nc), (double) (context->s.mi - context->s.ni));
    driver->print(0, 4 * textwidth, str);
    sprintf(str, "Screen size:%4i:%-4i", context->width, context->height);
    driver->print(0, 5 * textwidth, str);
    sprintf(str, "Iterations:%4i   Palette size:%i", context->maxiter, context->num_colors);
    driver->print(0, 6 * textwidth, str);
    sprintf(str, "Autopilot:%-4s    Plane:%s", autopilot ? "On" : "Off", planename[context->plane]);
    driver->print(0, 7 * textwidth, str);
    sprintf(str, "zoomspeed:%f", (float) maxstep * 1000);
    driver->print(0, 8 * textwidth, str);
    sprintf(str, "incoloring:%s outcoloring:%s", incolorname[context->incoloringmode], outcolorname[context->coloringmode]);
    driver->print(0, 9 * textwidth, str);
    if (context->mandelbrot)
	strcpy(str, "Parameter:none");
    else
	sprintf(str, "Parameter:[%f,%f]", (float) context->pre, (float) context->pim);
    driver->print(0, 10 * textwidth, str);
}

void ui_message(void)
{
    char s[80];

    sprintf(s, "Please wait while calculating %s", context->currentformula->name[!context->mandelbrot]);
    driver->print(0, 0, s);
    ui_status();
}

static void ui_changed()
{
    ui_tbreak();
    ui_message();
}

static int ui_mouse(int mousex, int mousey, int mousebuttons, int iterchange)
{
    int inmovement = 0, slowdown = 1;
    static double autopilot_counter = 0;
    static number_t oldx = 0, oldy = 0;
    static int dirty = 0;
    static int iterstep = 1;
    static int pressed;
    char str[80];

    CHECKPROCESSEVENTS(mousebuttons, iterchange);

    if (tbreak)
	mul = 1.0;
    if (iterchange == 2 && lookup_timer() > FRAMETIME) {
	get_timer();
	context->maxiter += iterstep;
	if (!(context->maxiter % 10))
	    iterstep = 10;
	dirty = 1;
	sprintf(str, "Iterations:%i ", context->maxiter);
	driver->print(0, 0, str);
	init_tables(context);
	return 1;
    }
    if (iterchange == 1 && lookup_timer() > FRAMETIME) {
	get_timer();
	context->maxiter -= iterstep;
	if (!(context->maxiter % 10))
	    iterstep = 10;
	dirty = 1;
	if (context->maxiter < 0) {
	    context->maxiter = 0;
	    return autopilot;
	} else {
	    sprintf(str, "Iterations:%i ", context->maxiter);
	    driver->print(0, 0, str);
	}
	init_tables(context);
	return 1;
    }
    if (!iterchange || iterchange == 3)
	iterstep = 1;
    if (iterchange & 3)
	return 1;
    if (dirty)
	ui_message(), ui_do_fractal(), ui_tbreak(), dirty = 0;
    if (autopilot && numbertype == FORMULA) {
	static int mousex1, mousey1, mousebuttons1;
	autopilot_counter += mul;
	while (autopilot_counter > 1) {
	    do_autopilot(context, &mousex1, &mousey1, &mousebuttons1, ui_changed);
	    CHECKPROCESSEVENTS(mousebuttons1, 0);
	    autopilot_counter--;
	}
	mousex = mousex1, mousey = mousey1, mousebuttons = mousebuttons1;
    }
    if (tbreak)
	mul = 1;
    if (mul == 0)
	mul = 0.001;
    if (mousebuttons > 0) {
	switch (mousebuttons) {
	case BUTTON1:		/* button 1 */
	    step -= speedup * 2 * mul, slowdown = 0;
	    inmovement = 1;
	    break;
	case BUTTON3:		/* button 3 */
	    step += speedup * 2 * mul, slowdown = 0;
	    inmovement = 1;
	    break;
	case BUTTON2:		/* button 2 */
	    {
		number_t x = xstoc(mousex), y = ystoc(mousey);
		if (pressed && (oldx != x || oldy != y)) {
		    context->s.nc -= x - oldx;
		    context->s.mc -= x - oldx;
		    context->s.ni -= y - oldy;
		    context->s.mi -= y - oldy;
		    if (!step) {
			ui_do_fractal(), ui_tbreak();
		    }
		}
		pressed = 1;
		oldx = xstoc(mousex), oldy = ystoc(mousey);
	    }
	    break;
	default:
	    break;
	}
    }
    if (!(mousebuttons & BUTTON2))
	pressed = 0;
    if (slowdown) {
	if (step > 0) {
	    if (step < speedup * mul)
		step = 0;
	    else
		step -= speedup * mul;
	} else if (step < 0) {
	    if (step > -speedup * mul)
		step = 0;
	    else
		step += speedup * mul;
	}
    }
    if (step > maxstep)
	step = maxstep;
    else if (step < -maxstep)
	step = -maxstep;
    if (step) {
	context->s.mc += (context->s.mc - xstoc(mousex)) * step * mul;
	context->s.nc += (context->s.nc - xstoc(mousex)) * step * mul;
	context->s.mi += (context->s.mi - ystoc(mousey)) * step * mul;
	context->s.ni += (context->s.ni - ystoc(mousey)) * step * mul;
	inmovement = 1;
	ui_do_fractal();
	if (tbreak)
	    tbreak--;
    }
    if (!inmovement)
	ui_tbreak();
    if (iterchange & 4 && (lookup_timer() > FRAMETIME || mousebuttons)) {
	double mul1 = (double) lookup_timer() / FRAMETIME;
	double su = 1 + (SPEEDUP - 1) * mul1;
#ifdef _plan9_
	sleep(FRAMETIME / 1000);
#endif
	if (su > 2 * SPEEDUP)
	    su = SPEEDUP;
	if (!inmovement)
	    get_timer();
	else
	    su = 1 + (SPEEDUP - 1) * mul;
	speedup *= su, maxstep *= su;
	sprintf(str, "speed:%2.2f ", (double) speedup * (1 / STEP));
	driver->print(0, 0, str);
    }
    if (iterchange & 8 && (lookup_timer() > FRAMETIME || mousebuttons)) {
	double mul1 = (double) lookup_timer() / FRAMETIME;
	double su = 1 + (SPEEDUP - 1) * mul1;
#ifdef _plan9_
	sleep(FRAMETIME / 1000);
#endif
	if (su > 2 * SPEEDUP)
	    su = SPEEDUP;
	if (!inmovement)
	    get_timer();
	else
	    su = 1 + (SPEEDUP - 1) * mul;
	speedup /= su, maxstep /= su;
	sprintf(str, "speed:%2.2f ", (double) speedup * (1 / STEP));
	driver->print(0, 0, str);
    }
    if (iterchange & 12)
	inmovement = 1;
    return (inmovement | (autopilot && numbertype == FORMULA));
}
void ui_call_resize(void)
{
    callresize = 1;
}
void ui_resize(void)
{
    int w, h, scanline;
    char *b1, *b2;
    if (incalculation) {
	resized = 1;
	return;
    }
    ui_tbreak();
    driver->getsize(&w, &h);
    assert(w > 0 && w < 65000 && h > 0 & h < 65000);
    if (w != context->width || h != context->height || driver->updateafterresize) {
	driver->free_buffers(context->vbuff, context->back);
	if (!(scanline = driver->alloc_buffers(&b1, &b2))) {
	    driver->uninit();
	    printf("Can not allocate buffers\n");
	    ui_outofmem();
	    exit(-1);
	}
	assert(scanline >= w);
	if (!resize_to(context, w, h, scanline, b1, b2)) {
	    driver->uninit();
	    printf("Can not allocate tables\n");
	    ui_outofmem();
	    exit(-1);
	}
	ui_message();
	get_timer();
	ui_do_fractal();
	ui_tbreak();
    }
    ui_display();
}
static void ui_driver(int d)
{
    int w, h, scanline;
    char *b1, *b2;
    struct ui_driver *driver1;
    if (d < 0)
	d = 0;
    if (d >= ndrivers)
	d = ndrivers - 1;
    driver1 = driver;
    driver->free_buffers(context->vbuff, context->back);
    driver->uninit();
    driver = drivers[d];
    if (!driver->init()) {
	driver = driver1;
	printf("Can not initialize new driver\n");
	if (!driver->init()) {
	    printf("Can not return back to previous driver\n");
	    exit(-1);
	}
    }
    context->switch_function = driver->flip_buffers;
    context->fullscreen = driver->fullscreen;
    driver->getsize(&w, &h);
    assert(w > 0 && w < 65000 && h > 0 & h < 65000);

    if (!(scanline = driver->alloc_buffers(&b1, &b2))) {
	driver->uninit();
	printf("Can not allocate buffers\n");
	ui_outofmem();
	exit(-1);
    }
    assert(scanline >= w);
    init_tables(context);
    resize_to(context, w, h, scanline, b1, b2);
    mkdefaultpalette(context, driver->set_color, driver->randomsize);
    ui_message();
    ui_do_fractal();
    ui_tbreak();
    ui_display();
}
static void processbuffer(void)
{
    int k;
    for (; begin != end % SIZE;) {
	if (resized)
	    resized = 0, ui_resize();
	k = keybuffer[begin];
	begin = (begin + 1) % SIZE;
	ui_key(k);
    }
}

int ui_key(int key)
{
    int sym;
    int lasttype = numbertype;
#ifdef DEBUG
    printf("key:%c\n", key);
#endif
    if (incalculation) {
	int i;
	if (tolower(key) == ' ')
	    ui_display();
	else if (tolower(key) == 'z')
	    context->interrupt = 1;
	else {
	    add_buffer(key);
	    for (i = 0; i < strlen(interruptkeys); i++) {
		if (tolower(key) == interruptkeys[i])
		    context->interrupt = 1;
	    }
	}
	return 1;
    }
    switch (sym = tolower(key)) {
    case 'v':
	ui_driver_menu();
	break;
    case '=':
	callresize = 1;
	break;
    case 'm':
	{
	    int mousex, mousey, b;
	    driver->getmouse(&mousex, &mousey, &b);
	    ui_mandelbrot(mousex, mousey);
	}
	break;
    case 'i':
	ui_plane_menu();
	break;
    case 'f':
	ui_incoloringmode();
	break;
    case 'c':
	ui_outcoloringmode_menu();
	break;
    case 'h':
	ui_help(5);
	break;
    case 's':
	ui_savefile();
	break;
    case '?':
    case '/':
	ui_status();
	break;
    case 'r':
	ui_message();
	init_tables(context);
	ui_tbreak();
	ui_do_fractal();
	break;
    case 'p':
	mkpalette(context, driver->set_color, driver->randomsize);
	if (driver->randomsize) {
	    ui_message();
	    init_tables(context);
	    ui_tbreak();
	    ui_do_fractal();
	} else if (driver->updateafterpalette) {
	    ui_tbreak();
	    ui_display();
	}
	break;
    case ' ':
	ui_display();
	break;
    case 'l':
	ministatus ^= 1;
	ui_display();
	break;
    case 'a':
	ui_autopilot();
	return (1);
	break;
    case 'q':
	driver->free_buffers(context->vbuff, context->back);
	driver->uninit();
	printf("Thank you for using XaoS\n");
	exit(0);
	break;
    default:
	{
	    int number;
	    if (sym >= '1' && sym <= '9') {
		number = sym - '1';
		switch (numbertype) {
		case FORMULA:
		    set_formula(context, number);
		    ui_message();
		    ui_do_fractal();
		    ui_tbreak();
		    break;
		case HELP:
		    ui_help(number);
		    break;
		case OUTCOLOR:
		    ui_coloringmode(number);
		    break;
		case PLANE:
		    ui_inverse(number);
		    break;
		case DRIVER:
		    if (number)
			todriver = number;
		    else
			ui_display();
		    return 2;
		    break;
		}
	    }
	}
	break;
    }
    if (numbertype != FORMULA && lasttype == FORMULA)
	step = 0;
    processbuffer();
    return 0;
}
#ifdef _EFENCE_
extern int EF_ALIGNMENT;
extern int EF_PROTECT_BELOW;
extern int EF_PROTECT_FREE;
#endif
static void main_loop(void)
{
    int inmovement = 1;
    int x, y, b, k;
    while (1) {
	processbuffer();
	driver->processevents(!inmovement, &x, &y, &b, &k);
	inmovement = ui_mouse(x, y, b, k);
	if (todriver)
	    ui_driver(todriver - 1), todriver = 0;
	if (callresize)
	    ui_resize(), callresize = 0;
    }
}

int main(int argc, char **argv)
{
    int width, height, scanline;
    int i, formula = 0;
    char *buffer1, *buffer2;
    params_parser(argc, argv);
#ifdef _EFENCE_
    EF_ALIGNMENT = 0;
    EF_PROTECT_BELOW = 1;
    /* EF_PROTECT_FREE=1; */
#endif
#ifdef DEBUG
    printf("Initialising driver\n");
#endif
#ifndef _plan9_
    signal(SIGFPE, SIG_IGN);
#endif
    if (printconfig) {
#define tostring(s) #s
	printf("XaoS configuration\n"
	       "Version:   %s\n"
	       "Type size: %i\n"
	       "Maxiter:   %i\n"
	       "MaxStep:   %f\n"
	       "integer size: %i\n"

#ifndef _plan9_
#ifdef HAVE_ALLOCA
	       "using alloca\n"
#endif
#ifdef HAVE_LONG_DOUBLE
	       "using long double\n"
#endif
#ifdef USE_NCURSES
	       "using ncurses\n"
#endif
#ifdef const
	       "const disabled\n"
#endif
#ifdef inline
	       "inline disabled\n"
#endif
#ifdef HAVE_GETTIMEOFDAY
	       "using gettimeofday\n"
#endif
#ifdef MITSHM
	       "using mitshm\n"
#endif
#ifdef HAVE_MOUSEMASK
	       "using ncurses mouse\n"
#endif
#ifdef DEBUG
	       "debug enabled\n"
#endif
#ifdef NDEBUG
	       "assertions disabled\n"
#endif
#endif
	       ,XaoS_VERSION, sizeof(FPOINT_TYPE), DEFAULT_MAX_ITER, MAXSTEP, sizeof(int));
    }
    if (deflist || printconfig) {
	printf("Available drivers:\n");
	for (i = 0; i < ndrivers; i++) {
	    printf("   %s\n", drivers[i]->name);
	}
	exit(0);
    }
    if (defdriver != NULL) {
	for (i = 0; i < ndrivers; i++) {
	    int y;
	    for (y = 0; tolower(drivers[i]->name[y]) == tolower(defdriver[y]) && drivers[i]->name[y] != 0; y++);
	    if (drivers[i]->name[y] == 0) {
		driver = drivers[i];
		if (driver->init())
		    break;
		else {
		    printf("Can not initialize %s driver\n", defdriver);
		    exit(-1);
		}
	    }
	}
	if (i == ndrivers) {
	    printf("Unknown driver %s\n", defdriver);
	    exit(-1);
	}
    } else {
	for (i = 0; i < ndrivers; i++) {
	    driver = drivers[i];
	    if (driver->init())
		break;
	}
	if (i == ndrivers) {
	    printf("Can not initialize driver\n");
	    exit(-1);
	}
    }
#ifdef DEBUG
    printf("Getting size\n");
#endif
    driver->getsize(&width, &height);
    if (!(scanline = driver->alloc_buffers(&buffer1, &buffer2))) {
	driver->uninit();
	printf("Can not alocate buffers\n");
	exit(-1);
    }
    for (i = 0; i < nformulas; i++)
	if (defformula[i])
	    formula = i;
    context = make_context(width, height, scanline, formula, driver->fullscreen, driver->flip_buffers, ui_waitfunc, buffer1, buffer2);
    if (!context) {
	driver->uninit();
	printf("Can not allocate context\n");
	ui_outofmem();
	exit(-1);
    }
    set_formula(context, formula);
    srand(7);
    mkdefaultpalette(context, driver->set_color, driver->randomsize);
    tbreak = 2;
    context->maxiter = defiters;
    ui_message();
    ui_do_fractal();
    if (defautopilot)
	ui_autopilot();
    if (defspeed <= 0 || defspeed >= 100)
	defspeed = 1.0;
    speedup = STEP * defspeed;
    main_loop();
    driver->free_buffers(context->vbuff, context->back);
    driver->uninit();
    printf("Thank you for choosing XaoS!\n");
    return (0);
}
zoom_context *
 ui_getcontext()
{
    return (context);
}
