############################################################################
#
# Name:
#	sysman_menu_branch_class.tcl
#
# Abstract:
#	Main class definitions for the SysMan Menu branchs. 
#
# Notes:
#
# Class Description:
#	Menu_branch is a subclass of Menu. A branch is a menu item
#	that can expand/collapse. A leaf is menu item that we 
#	can execute. Branch/leaves have several ploymorphic methods
#
#
############################################################################



Class Menu_branch -superclass "Menu"


############################################################################
#
# Methods for the Menu_branch object: 
#	- Execute
#	- Is_item_exapanded
#	- Toggle_item_state
#
############################################################################

#
# Procedure:
#	Menu_branch Execute - "execute" this menu branch object - 
#	we collapse or expand it. 
# Inputs:
# 	cursel_index - index of the selected menu item for this object
#	cli          - true when we are invoked from the command line, 
#		       In this case, the selected branch is already 
#		       expanded so don't accidentally contract it
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
Menu_branch instproc Execute { {cursel_index 0} { cli false} } {
    global SM_menu_main
    global SM_menu_def_root
    global SM_menu_tasks_dir
    global SM_menu_debug
    global SM_display_accelerators
    global env

    set cat ${SM_menu_main}.cat

    # When cli is true, we are being called from the command line
    # to display this branch and it's already set visible so 
    # don't toggle. Otherwise, the user has selected the branch in the
    # UI and we need to toggle
    if { $cli == "false" } {
	$self Toggle_item_state		;# Update our collapsed/expanded state
    }
    # And then update the displayed tree accordingly
    # by regenerating it to reflect the current expand/collapsed items
    # and then jamming the results into the list box. 
    set menu_items [$SM_menu_def_root Build_menu "" $SM_display_accelerators]
    set kl {}
    foreach item $menu_items {
	menu_win.menu_tree setListItem kl $item $item
    }
    menu_win.menu_tree setVal $kl


    menu_win.menu_tree setSelection $cursel_index
    menu_win.menu_tree setViewable selected



} ;# end instproc Execute

#
# Procedure:
#	Menu_branch Is_item_expanded - returns "yes" if this menu branch 
#	is expanded
#
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
Menu_branch instproc Is_item_expanded { } {
    $self instvar item_state

    return [expr {($item_state == "expanded") ? "yes" : "no"}]    

} ;# end Menu_branch Is_item_expanded

#
# Procedure:
#	Toggle_item_state - given a menu_branch  object, if it's been expanded
#       then set it's state to collapsed and vice versa.
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	Only menu items with children are candidates for expansion. 
#	Leaf nodes represent the actual tasks. 
#
#	It's possible for the menu to be started with a command line
#	option that specifies that a specific branch be expanded. 
#	in order not to special case this, we unconditionally
#	set the state for the current item and then walk up
#	the tree making sure to set the state for the parents as well. 
#
Menu_branch instproc Toggle_item_state { } {

    $self instvar item_state		;# Collapsed or expanded
    $self instvar _parent		;# Menu item's parent object

    # [pjw] hack. We should really subclass Menu_branch to 
    # create Menu_root for the top of the menu. The top
    # is always expanded. This is the only menu root behavior
    # difference right now so just special case it here
    if {[cequal $_parent ""] == 1 } {	;# If we are the root
	return				;# Then never set to collapsed
    }
    


    set item_state [expr {($item_state == "collapsed") ? \
				"expanded" : "collapsed"}]


    # If we are expanding this node, then make sure all the 
    # parents are also expanded. This is a noop for interactive
    # menu use. It's when we want the start the menu with a 
    # specific branch that this is needed
    if { [cequal $item_state expanded] == 1} {	
	# The root of the tree has no _parent. Loop till we hit it
	set parent [$self set _parent]
	while { [cequal $parent ""] != 1} {
	    $parent set item_state "expanded"
	    set parent [$parent set _parent]
	}
    }

}			;# end Menu_branch Toggle_item_state

#
# Procedure:
#	Get_tree_item_text - returns the display string for this menu branch
#	If a branch is expanded, then it as - next to it's name. It not, 
#	it has a +. 
# Inputs:
#	accel - true if the menu accelerator should be displayed
#		along with the menu text
#	show_all - true when the -list command line switch is used. 
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
Menu_branch instproc Get_tree_item_text { {accel {}} {show_all {}} } {

    # If we are displaying a fully expanded tree, then always prefix a "-"
    if { $show_all != {} } {		
	return "- [$self Get_item_text $accel]"
    } else {
	# Else we are building the tree for the GUI display

	if { [$self Is_item_expanded] == "yes" } {	;# If expanded
	    set prefix "- "				;# Then prepend a "-"
	} else {
	    set prefix "+ "				;# Else a "+"
	}
	return "$prefix[$self Get_item_text $accel]"

    }

}			;# end instproc Get_tree_item_text

#
# Procedure:
#	addSubTree - add the specified subtree as a new child. The 
#	default method in our superclass adds the new subtree to the 
#	end of the current hierarchy level. We need to additionally 
#	support adding to the beginning, after a specified item, and at
#	the end
#
# Inputs:
#	st	 - the name of the subtree (branch) to add 
#	position - where to add it. "first", "end", or name of
#		   menu object to add it after (the object must
#		   exist at this level of the hierarchy or the default
#		   of "end" will be used.
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#	Since menu customizations are sourced in from the SYSMAN_MENU_TASKS_DIR
#	there is no guaranteed order in which the menu items will be added. 
#	IOW, a particular customization can not guarantee that another
#	customization has already been loaded. For this reason, 
#	the "name" position parameter should only be used for items
#	that exist in pre-defined menu structure.  
#
Menu_branch instproc addSubtree {st {position "end"} } {
    global SM_menu_main


    # Sanity check that we are not adding children to a menu leaf!
    if { [cequal [$self info class] Menu_leaf] == 1} {
	# Display the error message: Error: Menu item <this item> cannot 
	# be added as a child of menu leaf <this leaf>. Menu leaves may not
	# have children
	set task_name [$self Get_item_text]
	SM_DisplayInfoMsg error $st $task_name
	return
    }

    # Now add the subtree at the appropriate location
    switch $position {

	first {				;# Add subtree at top of current level

	    $self instvar _kids		;# Get list of our children
	    set _kids [linsert $_kids 0 $st] ;# Prepend new subtree
	    $st set _parent $self	;# Parent for new subtree is us
	}
					
	end {				;# Add subtree to end of current level

	    $self next $st		;# This is the default behavior
					;# for our superclass so let it
					;# do the work
	}

	default	{			;# Add after the specified item
	    # Note that position is the name of the item to add 
	    # the given subtree after. 

	    $self instvar _kids		;# Get list of our children

	    set i [lsearch -exact $_kids $position] ;# Find item to add after
	    if { $i == -1 } {		;# Item not in list
		$self next $st		;# So just add to end of current level
	    } else {			;# Else insert right after it
		set _kids [linsert $_kids [expr {$i + 1}] $st]
		$st set _parent $self   ;# Parent for new subtree is us
	    }		
	    
	}
    } ;# end switch position

    
}			;# end instproc addSubTree

#
# Procedure:
#	Get_attributes - format a branch's attributes for displaying
#	a formatted menu hierarchy
#
# Inputs:
#	indent - string of spaces to indent with
# Outputs:
# 	None
# Returns:
#	formatted string of this attrs in the form:
#	   -menu_<attr1> value
#	   ...
#	   -menu_<attrN> value
# Notes:
#
Menu_branch instproc Get_attributes { indent } {

	
    set attrs ""
    set indent "$indent   "		;# Add more indentation

    if { [$self set accel] != "" } {	;# Accelerator
	append attrs "\n$indent -menu_accel [$self set accel]"
    }
    if { [$self set privs] != "" } {	;# Privs
	append attrs "\n$indent -menu_privs [$self set privs]"
    }
    if { [$self set leaf_default_privs] != "" } {	;# Privs
	append attrs \
      "\n$indent -menu_leaf_default_privs [$self set leaf_default_privs]"
    }
    if { [$self set catalog] != "" } {	;# Catalog
	append attrs "\n$indent -menu_catalog [$self set catalog]"
    }
    if { [$self set help_volume] != "" } {	;# Help vol. 
	append attrs "\n$indent -menu_help_volume [$self set help_volume]"
    }
    if { [$self set help_locid] != "" } {	;# Help locid
	append attrs "\n$indent -menu_help_locid [$self set help_locid]"
    }
	
    return $attrs
    
}			;# end instproc Get_attributes
