# 
# @DEC_COPYRIGHT@
#
# HISTORY
# $Log: sysman_menu_class.tcl,v $
# Revision 1.1.1.1  2003/01/23 18:04:47  ajay
# Initial submit to CVS.
#
#
# Revision 1.1.11.2  2000/11/07  21:39:55  Peter_Wolfe
# 	SysMan code drop for BL3 code freeze
#
# Revision 1.1.2.26  2000/11/02  06:20:39  Peter_Wolfe
# 	Don't emit manpage leaves as dop actions
#
# Revision 1.1.2.25  2000/07/31  16:43:16  Peter_Wolfe
# 	Fix mismatched else
#
# Revision 1.1.2.24  2000/07/28  19:02:54  Peter_Wolfe
# 	Fix for 79438. Find... was including the display strings
# 	for branches (it's supposed to deal with leaves
# 	exclusively)
#
# Revision 1.1.2.23  1999/07/20  19:24:00  Peter_Wolfe
# 	Qar 73293. There needs to be a default priv for the
# 	all branches so that if a devo forgets to add one
# 	the leaves will at least inherit something. Call build_actions
# 	with the default priv to fix this
# 	[1999/07/20  18:02:21  Peter_Wolfe]
#
# Revision 1.1.2.22  1999/03/10  03:05:24  Peter_Wolfe
# 	Fix List_menu for the show_all case. Broken
# 	when I added -list_attributes yuck.
# 	[1999/03/10  03:05:02  Peter_Wolfe]
# 
# Revision 1.1.2.21  1999/02/23  03:57:27  Peter_Wolfe
# 	Add support for -list_hierarchy
# 	[1999/02/23  03:56:43  Peter_Wolfe]
# 
# Revision 1.1.2.20  1999/02/01  18:29:15  Peter_Wolfe
# 	Fix frink'isms.
# 	[1999/02/01  18:28:52  Peter_Wolfe]
# 
# Revision 1.1.2.19  1998/12/04  20:18:33  Peter_Wolfe
# 	Add -menu_focus support
# 	[1998/12/04  20:16:17  Peter_Wolfe]
# 
# Revision 1.1.2.18  1998/09/08  15:52:35  Peter_Wolfe
# 	Make find task by keyword case insensitive
# 	[1998/09/05  02:32:51  Peter_Wolfe]
# 
# Revision 1.1.2.17  1998/08/13  04:37:47  Peter_Wolfe
# 	Rmeove debug puts
# 	Enhance Get_item_text to support pruning (can fetch
# 	_title_lbl strings)
# 	[1998/08/13  03:49:58  Peter_Wolfe]
# 
# 	Fix catalog bug where keyword code was still using suit catalogs.
# 	[1998/08/12  20:58:23  Peter_Wolfe]
# 
# 	Add initial pruning support (original -captive or -pruned idea)
# 	[1998/08/11  04:25:11  Peter_Wolfe]
# 
# Revision 1.1.2.16  1998/07/14  19:34:04  Fred_Cassirer
# 	Fixed bug in build_actions where privileges were being propagated to
# 	the first leaf the next branch.
# 	[1998/07/14  15:37:25  Fred_Cassirer]
# 
# Revision 1.1.2.15  1998/06/17  21:26:35  Fred_Cassirer
# 	Added "suit:" prefix to all suitlet action entries
# 	[1998/06/17  21:26:10  Fred_Cassirer]
# 
# Revision 1.1.2.14  1998/06/17  19:25:35  Fred_Cassirer
# 	 	** Merge Information **
# 		** Command used:	bmerge **
# 		** Ancestor revision:	1.1.2.12 **
# 		** Merge revision:	1.1.2.13 **
# 	 	** End **
# 	Bmerged
# 	[1998/06/17  19:23:33  Fred_Cassirer]
# 
# 	Added support for -action switch to generate dop action clause input
# 	Added support for "menu_privs" property in menu definitions
# 	Added support for "menu_leaf_default_privs" property in menu definitions
# 	Update to support dop actions
# 	[1998/06/15  21:14:00  Fred_Cassirer]
# 
# Revision 1.1.2.13  1998/06/05  20:03:47  Peter_Wolfe
# 	Change catalog usage from SUIT to generic
# 	[1998/05/28  20:20:58  Peter_Wolfe]
# 
# Revision 1.1.2.12  1998/02/27  06:44:42  Peter_Wolfe
# 	Add -menu_help_volume
# 	Add -menu_help_volume and locid
# 	[1998/02/27  06:42:55  Peter_Wolfe]
# 
# Revision 1.1.2.11  1998/01/07  21:26:45  Peter_Wolfe
# 	Fix accel's to be bracketed by [] instead of ()
# 	so they are no confused with TLA's after menu names. For
# 	example:
# 	Point to Point Protocol (PPP) [ppp]
# 	Fix keyword searching bugs
# 	[1998/01/07  21:05:10  Peter_Wolfe]
# 
# Revision 1.1.2.10  1998/01/05  17:41:26  Peter_Wolfe
# 	Delete empty branches from our tree (Prune_empty_branches).
# 	This avoids having to add lots of special case code for
# 	displaying/traversing the menu
# 	[1998/01/05  17:40:48  Peter_Wolfe]
# 
# Revision 1.1.2.9  1997/12/09  17:26:18  Peter_Wolfe
# 	Fix buglet where sysmenu -accel foo, where foo is a branch,
# 	needs to start up with the that branch expanded and selected in
# 	the main view.
# 	[1997/12/09  17:04:24  Peter_Wolfe]
# 
# Revision 1.1.2.8  1997/10/10  15:01:02  Peter_Wolfe
# 	Add Find_named_item List_item routines
# 	[1997/10/10  14:49:12  Peter_Wolfe]
# 
# Revision 1.1.2.7  1997/07/25  15:26:36  Peter_Wolfe
# 	Remove debug puts
# 	[1997/07/25  15:23:20  Peter_Wolfe]
# 
# Revision 1.1.2.6  1997/07/07  21:26:46  Peter_Wolfe
# 	Change to fetch item text/keywords from per-app message catalogs
# 	[1997/07/07  21:24:54  Peter_Wolfe]
# 
# Revision 1.1.2.5  1997/07/02  23:04:34  Peter_Wolfe
# 	Fix init to not init subclass instvars
# 	[1997/07/02  22:57:49  Peter_Wolfe]
# 
# Revision 1.1.2.4  1997/07/01  14:54:37  Peter_Wolfe
# 	Supress the display and meny branches with no leaves
# 	Fix bug where _parent was getting stepped on by class initialization
# 	Add Find_accelerator to lookuop the obj that has the specified accel.
# 	[1997/07/01  14:47:56  Peter_Wolfe]
# 
# Revision 1.1.2.3  1997/06/17  14:30:54  Peter_Wolfe
# 	Update for Display Accel's and Find Task By Keyword
# 	[1997/06/16  17:21:13  Peter_Wolfe]
# 
# Revision 1.1.2.2  1997/05/28  20:10:24  Peter_Wolfe
# 	Initial submit
# 	[1997/05/28  20:09:19  Peter_Wolfe]
# 
# $EndLog$
# 
# @(#)$RCSfile: sysman_menu_class.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:04:47 $
# 


############################################################################
#
# Name:
#	sysman_menu_class.tcl
#
# Abstract:
#	Main class definitions for the SysMan Menu
#
# Notes:
#
# Class Description:
#
#	The menu is a subclass of SUIT's tree widget. The SUIT tree is used to
#	maintain the widget hierarchy in the app and this basic hierarchy/tree
#	data structure is just the ticket for the menu itself. We subclass
#	that with Menu_branch and Menu_leaf.  A Menu_branch is a menu item
#	that can expand to display more (children) menu items. A Menu_leaf is
#	an executable task that the menu invokes.
#
############################################################################



Class Menu -superclass "_UIT_Tree"



############################################################################
#
# Methods for the Menu object: 
#	- init
#	- Show_hierarchy
#	- Build_menu
#	- Show_menu_level
#	- Get_item_text
#
############################################################################

#
# Procedure:
#	init - per-object instance initialization for menu objects. 
#	Each menu item is represented by an object
# Inputs:
#	contents - any children menu items of this current item
#	args     - parameters (switches) for this menu item
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	There is one nasty order-of-execution issue here. In Otcl, 
#	the "eval $self next $args" line causes the 
#	object's command line methods (Menu_leaf foo -menu_accel bar
#	where menu_accel is a method) to be invoked. That means that 
#	we have to initilize the object's instvars BEFORE the 
#	eval $self next $args line. This is different from 
#	C++ where you tend to init the superclass first and then 
#	override/init the subclass specific stuff later. 
#
#
Menu instproc init { contents args } {

    global SM_menu_parent	;# Holds the string with our parent's name
    global SM_menu_level	;# Holds current indentation level of this item

    set parent $SM_menu_parent	;# Save the current parent's obj. name locally
    set SM_menu_parent $self	;# Current parent is this node (save globally)

    # Init our instvars
    $self set accel  ""			;# Accelerator for this menu item
    $self set keywords {}		;# Concept keywords for this item
    $self set catalog	   ""		;# Name of the msg. cat for this item
    $self set help_volume  ""		;# Name of help volume for this suitlet
    $self set help_locid   ""		;# locid to jump to
    $self set privs   ""		;# privilege list required to run
    $self set leaf_default_privs   ""	;# default priv list for all children
    $self set focus   none		;# Default focus is local node

    $self set item_state "collapsed"	;# All tree nodes start collapsed

    eval $self next $args		;# Allow our superclass to init 

    
    if { $parent != "" } {	;# If we aren't the root of the tree
	$parent addSubtree $self  ;# then add this node into the tree
    }

    eval $contents		;# Create this node's children (resursively)
    

    set SM_menu_parent $parent	;# Restore the global parent to be this node


} ;# end Menu instproc init


#
# Procedure:
#	Show_hierarchy - walks the menu hierachy and displays an indented
#	list of menu items. 
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	This is for debugging only.
#
Menu instproc Show_hierarchy { {indent ""} } {
    global SM_menu_main

    # Use this next line only when the SUIT cat object is available
    puts "$indent [$self Get_item_text]"
    # Use this line when it's not (i.e. when we haven't init'd SUIT cat stuff)
    # This will just print out the object name instead of the obj's actual text
    #puts "$indent [$self set indent_level] $self "

    foreach kid [$self getKids] {
	$kid Show_hierarchy "$indent    "
    }
}			;# end instproc Show_hierarchy

#
# Procedure:
#	Show_menu_level - given a menu object, displays it's children
#	(i.e. show the items at the next identation level) 
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	a list containing the menu items that are children of the current item
# Notes:
# 	This routine is no longer used (and I think I can delete it soon). 
#	Build_menu now does all the work
#
Menu instproc Show_menu_level { } {
    global SM_menu_main

    # Toggle the display state from collapsed to expanded or vice versa
    $self Toggle_item_state

    set menu_items {}
    foreach kid [$self getKids] {
	lappend menu_items [$kid Get_item_text]
    }

    return $menu_items
}			;# end instproc Show_menu_level

#
# Procedure:
#	Get_item_text - returns the display string for this menu item/object
# Inputs:
#	accel_flag - if "true", then display the accelerator for 
#		     menu item as well. 
#		   - if "title" then fetch the title bar string for 
#		     a menu branch. This is used for captive menus 
#		     (aka pruning). The two usages of this paramter
#		     are mutually exclusive
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#	If a string is present in SysMan Menu's catalog, then 
# 	the convention is that all menu object names are in the form SM_name. 
#	However, the message name is all in lowercase with 
#	_lbl appended. So we can get the desired string by 
#	lowercasing the object name and concenating _lbl
#
Menu instproc Get_item_text {   {lbl_flag {}} } {

    global SM_menu_main
    $self instvar accel catalog

    # If the catalog name for this item is undefined, use our own
    # message catalog
    if { $catalog == "" } {			;# If no catalog specified
	set cat SM_cat				;# then use ours
    } else {					;# Else use the app-specific

	$self Open_msg_catalog
	set cat $catalog.cat
    }
	
    # For a title bar label, use the special form of the message
    # catalog identifier name
    if { $lbl_flag == "title" } {
	set msg_cat_id_extension "_title_lbl"
    } else {
	set msg_cat_id_extension "_lbl"
    }


    # Get the text string for this item
    set label "[string tolower $self]$msg_cat_id_extension"
    set text [$cat catgets $label ]

    if { $lbl_flag == "title" && $text == ""} {	;# If _title_lbl not there
	# Just use standard menu item string
        set label "[string tolower $self]_lbl"
	set text [$cat catgets $label ]
	
    }

    # If the string isn't found, then fall back to object name
    # so at least something is displayed
    if { $text == {} } {
	set text $self
    }
 
    # If we are also displaying accelerators, then format
    # the string as: <menu string> (<menu accelerator>)
    if { $lbl_flag == "true" } {

	# If the accelerator is non-null, then format it
	if { ![cequal $accel ""] } {
	    set text [append text " \[$accel\]" ]
	}
    }

    return $text

}			;# end instproc Get_item_text

#
# Procedure:
#	Build_menu - walks the menu hierarchy and creates the display
#	needed to populate a SUIT list box.
# Inputs:
#	indent - indentation for this level of the menu
#	accel  - boolean indicating whether to display
#		 menu accelerators in addition to the menu text
#	show_all - true if we want to display the entire tree. This 
#		 is used by the -list command line function to 
#		 displayed a fully expanded menu. 
# Outputs:
# 	None
# Returns:
#	The list representing the current state of tree. This is used
#	to setVal our SUIT list box. 
# Notes:
# 	We use this routine to re-setVal the list each time instead
#	of trying to add/delete levels from the list. The api for 
#	add/delete is all search based so it's rather expensive. So far,
#	complete list updates don't seem to flash and are very fast. 
#
#	The root of the menu is level 0. We want to avoid displaying 
#	this item in the list (since it's not a "real" menu item that 
#	the user can select 
#
Menu instproc Build_menu { {indent ""} {accel ""} {show_all {}} } {
    global SM_menu_main

    set cat ${SM_menu_main}.cat
    set menu_indent "    "	;# Indent by 4 spaces for each level
    set result {}		;# Accumulate the displayed items here

    # For each of the sublevels at this level of the tree
    # add them at proper indentation level
    foreach menu_item [$self getKids] {

	set  item ""			;# Clear current item
	# Get proper display string for the current item
	append item $indent [$menu_item Get_tree_item_text $accel $show_all]
	# If we want the menu items additional attributes, append them here
	if { $show_all == "show_all_with_attributes" } {
	    append item [$menu_item Get_attributes $indent]
	}

	lappend result $item		;# Add this item to a list



        if { $show_all != {} } {		;# If we want the entire tree

		set submenu \
		    [$menu_item Build_menu "$menu_indent$indent" \
					$accel $show_all]
		append result " " $submenu	;# And add them to the list

	} else {
	    # Process next level of menu only if it's visible
	    if { [$menu_item Is_item_expanded] == "yes" } {   

		set submenu \
		    [$menu_item Build_menu "$menu_indent$indent" $accel]
		append result " " $submenu	;# And add them to the list
	    }
	}

    } ;# end foreach

    return $result				;# Add this level's items

}			;# end instproc Build_menu


############################################################################
#
# The following collection of proc's implement the "switch processing"
# for the menu object definitions. Each menu definition can include
# the following arguments which are implemented as instprocs:
#     - menu_accel	- the accelerator for this menu item. For example, the
#	                  accelerator for Network Management/Basic Network 
#			  Config is netconfig
#     - menu_target	- the name of a suitlet or a raw mcl. If a 
#			  raw mcl, then it will be sent to autogen 
#			  to create a suitlet. Only menu leaves
#			  have targets so this instproc is defined
#			  in sysman_menu_leaf_class.tcl and not here
#
#     - menu_privs 	- The privileges that are required to run this
#			  menu accelerator.  This is used to construct
#			  initial input to the dop database.
#
#     - menu_keywords	- the name of a message catalog entry
#			  containing the keywords for this menu item
#			  Keywords help a user locate task related to 
#			  a specific aspect of systems management. 
#			  Keywords are specific to menu leaves. 
#			  This instproc is actually in sysman_menu_leaf_class
#			  since it's leaf specific. 
#     - menu_keywords_msg - Same as above but deprecated. Kept 
#			  only for compatbility with exising stuff
#
#     - menu_catalog	- the name of a message catalog containing 
#		          the localized strings to be used to display 
#			  menu items. By default, the base menu definition 
#			  from Digital includes the menu strings in 
#			  sysman menu specific files. 
#			  When outside groups/thrid parties add menu items, 
#			  then the strings live in app specific catalogs. 
#     -menu_help_volume	- name of the help volume for the SUITlet. By default
#			  the help vol. name is the same as the suitlet's
#			  main window name. Typically this is the same
#			  as the genUID filename as well. 
#
#			  In certain cases however, 
#			  N distinct suitlets share a single help volume or
#			  a suitlet decides to have a non-default name. 
#			  In those cases, -menu_help_volume must be 
#			  specified. It's the name of the volume without
#			  the path or extension. 
#
#			  Since a menu branch is not associated with 
#			  an executable target there is nothing to 
#			  derive the help volume name from so all branches
#			  need to specify this if appropriate. 
#
#     -menu_help_locid	- a specific locid within the help volume
#
#     -menu_focus	- for cluster apps, indicates the type of focus
#			  they accept
#
############################################################################

#
# Procedure:
#	menu_accel - the accelerator for this menu item. Accelerators
#	are used when the menu is invoked from the command line
#	to come up at a specific level of the tree or to immediately
#	execute a given task. 
# Inputs:
#	args - the name of the accelerator. 
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	Accelerator names must be unique. If they aren't, when the user
#	goes to use one he'll get the first one when we traverse the 
#	tree using a dfs (depth first search). 
#
Menu instproc menu_accel { a } {

    $self set accel $a
}				 ;# end proc menu_accel


#
# Procedure:
#	menu_privs - the privileges required to launch
#	the associated menu branch.  If null or not present
#	then no special privileges are associated and the
#       branch will run using the current user privileges.
#	
# Inputs:
#	args - the list of privileges
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	This information is only used during system installation
#	to build the privilege database.
#
Menu instproc menu_privs { args } {

    $self set privs $args
}				 ;# end proc menu_privs


#
# Procedure:
#	menu_leaf_default_privs - the default privileges all descendents of
#	this branch. If null or not present
#	then any parents default privileges  will be used.
#	
# Inputs:
#	args - the list of privileges
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	This information is only used during system installation
#	to build the privilege database.
#	[pjw] I think this routine should be moved to _branch_class.tcl
#	This is a branch attr and not a general Menu object attr. 
#
Menu instproc menu_leaf_default_privs { args } {

    $self set leaf_default_privs $args
}				 ;# end proc menu_leaf_default_privs


#
# Procedure:
#	menu_catalog - the name of the message catalog to get the 
#	UI strings and accelerators for this menu item. 
#	This is specified without the trailing ".msg" file extension
#	If null, the menu item's string is assumed to be in the 
#	Sysman Menu's own catalog.  Dynamically added applications
#	must supply thier own catalog
# Inputs:
#	catname - the name of the message catalog
# Outputs:
# 	None
# Returns:
#	None
# Notes:
# 	Accelerator names must be unique. If they aren't, when the user
#	goes to use one he'll get the first one when we traverse the 
#	tree using a dfs (depth first search). 
#
Menu instproc menu_catalog { catname } {

    # The catalog code does not want the .msg extension. If 
    # the user specified it, strip it off
    regsub "\.msg$" $catname "" catname

    $self set catalog $catname
}				 ;# end proc menu_catalog


#
# Procedure:
#	menu_help_volume - the name of the help volume for 
#	this leaf or branch. This is only required when the suitlet
#	uses a non-standard volume name. 
# Inputs:
#	v - the name of the volume. 
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
#
Menu instproc menu_help_volume { v } {

    $self set help_volume $v
}				 ;# end proc menu_help_volume

#
# Procedure:
#	menu_locid - the name of the help volume for 
#	this leaf or branch. This is only required when the suitlet
#	uses a non-standard volume name. 
# Inputs:
#	locid - the name of the volume. 
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
#
Menu instproc menu_help_locid { locid } {

    $self set help_locid $locid
}				 ;# end proc menu_help_locid

#
# Procedure:
#	Map_selection_to_menu_object - the menu items are displayed in a
#	SUIT list box. Given an list box's currently selected item we
#       figure out which menu object that is. 
#
#
# Inputs:
#	index - index of display string in the list box  
# Outputs:
# 	None
# Returns:
#	menu object name for the selected item
#	
# Notes:
#	SUIT list box indices are 0 based. 
#
#	There is a nasty SUIT issue here. Selections cannot return
#	the index of the selected item on the web. Instead they return
#	the string of the selected item. We are back to the classic
#	problem of having non-unique strings in a list box. 
#	This means that there is no guaranteed mapping from a list
#	box item back to our data. There has to be a restriction that list
#	box display strings be unique (at a given level of the menu). 
#	This restriction is Web based - it's not present in the Tk and Ctk
#	domains. If we implement the java client then the problem would go 
#	away. For example assuming the menu is displaying:
#		Printers
#		   Show 
#		Networks
#		   Show 
#	and the user chooses Networks/Show then this would not work in 
#	current SUIT implementation. We'd search for it and find the
#	first one. It WOULD work if Printers is collapsed. That's not 
#	good for the user however. This will have to be a Syman Menu
#	restriction for the cgi-based Web - we have to insist that items 
#	at the same level have unique names. If we get the java web client, 
#	then this restriction should go away
#
#
#
proc Map_selection_to_menu_object { index } {

    global SM_menu_def_root
    global _SM_count 


    # Map the index back to a menu object 
    set _SM_count -1		;# See comment in Find_item_by_index for 
				;# why this is inited to -1

    return [$SM_menu_def_root Find_item_by_index $index]


}	;# end map_selection_to_menu_object 



#
# Procedure:
#	Find_item_by_index - given an index of a selected item 
#	in a SUIT list box (the first item is index 0) map that
#	back to a corresponding menu object. The algorithm needs
#	to take into account which menu items are expanded or collapsed. 
#	For example, if the menu definition looks like:
#		Printers
#		    Show
#		Networks
#		    Show
#	and the currently displayed list box looks like:
#		Printers
#		Networks
#		    Show
#	then the index of 2 means Networks/Show. We can figure this out by
#	realizing that Printers is collapsed. This is hard to do though cause
#	of the Web! We can't keep permanent state in our menu objects (such
#	which menu items are expanded or collapsed) cause that is gone once we
#	display a page on the Web. In order to pull this off we 
#	need to keep this crucial state in a perm global and restore it 
#	each time. 
#
#
# Inputs:
#	_SM_count - this global is used to count the menu objects
#		    we visit. It's inited in the caller to -1 cause
#		    lists are 0 based and we will increment it immediately
#		    (i.e. first list item is index 0 == count of 0). 
#
# Outputs:
# 	None
# Returns:
#	menu object name for the specified index
# Notes:
#
Menu instproc Find_item_by_index  {index} {
    global _SM_count
    $self instvar item_state		;# Expanded/collapsed status

    # Recursively walk the tree using a depth first traversal and count
    # each item until we hit the one that matches the selection index. 

    foreach menu_item [$self getKids] {
	incr _SM_count				;# Count this item

	if { $_SM_count == $index} {		;# Have we found it? 
	    return $menu_item			;# Yes, return the object
	} elseif { [$menu_item Is_item_expanded] == "yes" } {
	    # Else, if the menu item has visible children, count them
	    set match [$menu_item Find_item_by_index $index]
	    if { $match != {} } { return $match}
	} 
    } ;# end foreach menu item child

    # If we fall off the end of the for loop then just return to caller
    # in order to keep searching
    return {}

} ;# end Find_item_by_index

#
# Procedure:
#	Find_accelerator - walks the menu data structure hierachy looking 
#	for the item with the specified accelerator
# Inputs:
#	search_accel  - name of the specificed accelerator to look for
# Outputs:
#	none
# Returns:
#	The accel if found, null string if not
# Notes:
#
Menu instproc Find_accelerator { search_accel } {

    set found_item ""

    # For each of the sublevels at this level of the tree
    foreach menu_item [$self getKids] {

	set accel [$menu_item set accel]
	if { [cequal $accel $search_accel] == 1} {	;# If a match
	    set found_item $menu_item
	    return $found_item				;# Save this menu obj.
	}

	# Recursively process the children of this item
	set found_item [$menu_item Find_accelerator $search_accel]
	if { [cequal $found_item ""] != 1 } {
		return $found_item
	}
    } ;# end foreach


} ;# end instproc Find_accelerator

#
# Procedure:
#	Find_named_item - walks the menu hierachy looking 
#	for the item with the specified display name
# Inputs:
#	name  - display name of the specificed item
# Outputs:
# 	None
# Returns:
#	The accel if found, null string if not
# Notes:
#
Menu instproc Find_named_item { name } {

    set found_item ""

    # For each of the sublevels at this level of the tree
    foreach menu_item [$self getKids] {

	set item_name [$menu_item Get_item_text]
	if { [cequal $item_name $name] == 1} {	;# If a match
	    set found_item $menu_item
	    return $found_item				;# Save this menu obj.
	}

	# Recursively process the children of this item
	set found_item [$menu_item Find_named_item $name]
	if { $found_item != "" } {
		return $found_item
	}
    } ;# end foreach


} ;# end instproc Find_named_item

#
# Procedure:
#	Find_task_by_keyword - searchs the menu defintion for 
#	all suitlets that match a specified keyword
# Inputs:
#	kw - the keyword to search for
#
# Outputs:
# 	None
# Returns:
#	The list representing the matching menu items. This is used
#	to setVal our SUIT list box. 
# Notes:
#	Note that the current policy is that only leaves are searched
#	for keywords. Branches aren't supposed to have them. However, 
#	the getKids method returns both leaves and branches and we 
#	look at items display string (in addition to it's keywords)
#	which means that we can get matches that are branches!!!
#	It would have been nicer (and more OO) to create a subclass of 
#	getKids (getLeafChildren) that returned only leaves. For now, 
#	we just check explicitly and ignore the branches
#	
#
Menu instproc Find_task_by_keyword { kw } {

    global SM_display_accelerators

    set result {}		;# Accumulate the matchingitems here

    # For each of the sublevels at this level of the tree
    # see which match
    foreach menu_item [$self getKids] {



	set kw_list [$menu_item Get_keyword_list]

	# If the message catalog string is malformed (like a mismatched
	# quote), that can cause lsearch to barf so be defensive.
	if { ![ catch { lsearch -glob $kw_list $kw } match ] } {
	    if { $match != -1 } {			;# If a match
	        lappend result $menu_item		;# Save this menu obj.
	    } else {					;# Else no match

		# Only look at he display string for Leaves
	        if { [$menu_item info class] != "Menu_branch" } {

		    # So include the menu item's string name in the 
		    # search as well. If accels are on, we search them too
		    set item_name [$menu_item Get_item_text \
					    $SM_display_accelerators]
		    set item_name [string tolower $item_name]
		    if { ![ catch { lsearch -glob $item_name $kw } match ] } {
			if { $match != -1 } {		;# If a match
			    lappend result $menu_item	;# Save this menu obj.
			}
		    }
		} ;# end if not a branch 

	    } ;# End if matched
	} 

	# Recursively process the children of this item
	set matching_tasks [$menu_item Find_task_by_keyword $kw]
	if { [llength $matching_tasks] != 0} {
	    append result " " $matching_tasks
	}

    } ;# end foreach

    return $result				;# Add this level's items

}			;# end instproc Find_task_by_keyword

#
# Procedure:
#	Get_keyword_list - fetch this items keyword list from
#	the message catalog
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	A tcl list of keywords
# Notes:
#
Menu instproc Get_keyword_list { } {

    global SM_menu_main
    global SM_menu_debug    

    $self instvar catalog		;# Name of this items message catalog
    $self instvar keywords		;# The name of the keyword 
					;# catalog identifier
 
    # If we have already looked up this items keywords in the message
    # catalog, then just return them. This is performance optimization
    # to avoid multiple message catalog lookups
    if { [$self info vars keywords_lc] != "" } {
	return [$self set keywords_lc]
    }

    # If there are no keywords defined for this menu item simply return
    # and empty list
    if { $keywords == "" } {
	$self set keywords_lc {}
	return {}
    }

    # If the catalog name for this item is undefined, use our own
    # message catalog
    if { $catalog == "" } {			;# If no catalog specified
	set cat SM_cat				;# then use ours
    } else {					;# Else use the app-specific
	$self Open_msg_catalog
	set cat $catalog.cat
    }
	

    # Get the text string for this item
    set kw_list [$cat catgets ${keywords}_lbl]

    # Lowercase it for searches
    $self set keywords_lc [string tolower $kw_list]

    if { $SM_menu_debug == "true" } {
	if { $kw_list == "" } {
	    puts "Keywords not found for $self"
	} 
    }

    return [$self set keywords_lc]
}			;# end instproc Get_keyword_list


#
# Procedure:
#	Build_actions - Essentially the same as Build_menu but
#         returns the list of actions and information that can
#         be used to create the privileged action list
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#      Nothing
#	
# Notes:
# 	
#
Menu instproc Build_actions {{parent_privs {}}} {
    global SM_menu_main SM_menu_tasks_dir

    set result {}		;# Accumulate the displayed items here

    # For each of the sublevels at this level of the tree

    set prev_parent_privs $parent_privs
    foreach menu_item [$self getKids] {

	set  item ""			;# Clear current item

	# If this leaf is just a manpage then there is no dop
	# action for it!!
        if { [$menu_item info class] == "Menu_leaf" && \
	     [$menu_item set target] == "manpage" } {
	    continue
        }

	set privs [$menu_item set privs]
	if {$privs == {}} {
	    set privs [join $parent_privs]
	}

	# Get proper display string for the current item

	if {[$menu_item info vars target] != {}} {
	    set target \
		"{path\t{suit:$SM_menu_tasks_dir/[$menu_item set target] *}}"
	} else {
	    set target \
		"{path\t{suit:/usr/sbin/sysman [$menu_item set accel] *}}"

	    # See if this branch has some default privs, if so, make these the
	    # current parent privs, otherwise, just keep using whatever we have
	    # and pass them down to the next level

            set branch_privs [$menu_item set leaf_default_privs]
	    if {$branch_privs != {}} {
	        set parent_privs $branch_privs
	    }
	}
	
        if {[$menu_item set accel] != {}} {
            append item \
	"\t[$menu_item set accel] {\n\t\t{privs\t{$privs}}\n\t\t$target\n}\n"
            lappend result $item            ;# Add this item to a list
        }

        set submenu [$menu_item Build_actions $parent_privs]
        append result " " $submenu      ;# And add them to the list
	set parent_privs $prev_parent_privs
        
    } ;# end foreach

    return $result				;# Add this level's items

}

#
# Procedure:
#	List_actions - When the user specifies -action on the command line
#	we emit a list of the corresponding actions to be created
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#      Nothing
#	
# Notes:
# 	
#
Menu instproc List_actions { } {

   global SM_menu_def_root

   # Use the priv on the top level menu object as the default
   # This is currently SystemManagement
   set default_priv [$SM_menu_def_root set leaf_default_privs]

   set l  [$SM_menu_def_root Build_actions $default_priv]

   foreach i $l {
	puts $i
   }

   return {}
} ;# end List_actions

#
# Procedure:
#	List_menu - When the user specifies -list on the command line
#	we emit a listing the menu hierarchy including the accelator names. 
# Inputs:
#	type - the default is type is show_all which signifies the 
#	       cli -list usage where we display a formatted list of tree
#	       If type is "show_attributes" then we display the menu
#	       item's parameters like -menu_focus, -menu_task etc. 
#	   
# Outputs:
# 	None
# Returns:
#	string containing menu names/accels
#	
# Notes:
# 	
#
Menu instproc List_menu { {type {show_all}} } {

   global SM_menu_def_root


   set l  [$SM_menu_def_root Build_menu "" true $type ]

   foreach i $l {
	puts $i
   }

} ;# end List_menu

#
# Procedure:
#	Make_parents_visible - When the user specifies a accelerator
#	on the command line, if it's a branch, we need to set it selected
#	and scrolled to the top of the window. To do that, we first need to 
#	set all the branch's parents to be visible
#
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	string containing menu names/accels
#	
# Notes:
# 	
#
Menu instproc Make_parents_visible { } {

    $self set item_state "expanded"
    set parent [$self set _parent]	;# Get it's parent

    if {$parent != "" } {
	$parent Make_parents_visible
    }

} ;# end Make_parents_visible

#
# Procedure:
#	Get_item_index - given a menu object, find what it's 
#	selection index is so we can set it selected in a SUIT list box. 
#	We recursivley walk the expanded menu true keeping 
#	a count until we find the target menu item. Note that 
#	the first item in a SUIT list is index 0. 
#
#
# Inputs:
#	search_item - menu item to look for 
#
# Outputs:
# 	None
# Returns:
#	list box selection index for this menu item
# Notes:
#
Menu instproc Get_item_index  { search_item  {count -1} } {

    $self instvar item_state		;# Expanded/collapsed status

    # Recursively walk the tree using a depth first traversal and count
    # each item until we hit the one that matches the selection index. 

    foreach menu_item [$self getKids] {
	incr count				;# Count this item

	if { $menu_item == $search_item} {	;# Have we found it? 
	    return $count			;# Yes, return the object
	} elseif { [$menu_item Is_item_expanded] == "yes" } {
	    # Else, if the menu item has visible children, count them
	    set index [$menu_item Get_item_index $search_item $count]
	    if { $index != {} } { return $index }
	    
	} 
    } ;# end foreach menu item child

    # If we fall off the end of the for loop then just return to caller
    # in order to keep searching
    return {}

} ;# end Get_item_index

#
# Procedure:
#	Prune_empty_branches - the base menu definition has several 
#	placeholder branches that have no childen. These are so 
#	other users know what the intended menu layout is and can 
#	add to it. If there are no children at runtime, we nuke
#	these empty branches cause we don't want them to show up
#	in the display. 
#
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
Menu instproc Prune_empty_branches  { } {


    # Recursively walk the tree using a depth first traversal and
    # nuke empty branches

    foreach menu_item [$self getKids] {

	$menu_item Prune_empty_branches

	# If this item is a menu branch with no leaves then 
	# nuke it from the tree
	set class [$menu_item info class]
	if { $class == "Menu_branch"  && 
	     [$menu_item getKids] == {} } {
	    $menu_item destroy
	}

    } ;# end foreach menu item child

    return {}

} ;# end Prune_empty_branches


#
# Procedure:
#	Prune_menu - many times we want to display just a subset of 
#	the menu, rooted at a specified branch. Just rearrange
#	the tree and orphan all the unwanted objects
#
# Inputs:
#	item - menu branch that is the new root of the tree
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#	We don't reap the unsed objects. They just dangle until we exit. 
#	We assume that the pruned items are children of the root. IOW,
#	it doesn't really make sense for anyone but the root to 
#	call this routine. 
#
Menu instproc Prune_menu  { item } {

    $self instvar _kids

    set _kids [$item getKids]		;# Replace this node's kids

    # Set all the kids to have the new parent    
    foreach menu_item [$item getKids] {

	$menu_item set _parent $self

    } ;# end foreach menu item child

    return {}

} ;# end Prune_menu


#
# Procedure:
#	Open_msg_catalog - Init the message catalog for this object. 
#	This is an application's private message catalog that 
#	contains the menu item strings and menu item keywords. 
# Inputs:
#	None
# Outputs:
# 	None
# Returns:
#	None
# Notes:
#
Menu instproc Open_msg_catalog { } {

    global SM_menu_debug    

    $self instvar catalog		;# Name of this items message catalog
 
    if { $catalog != "" } {		;# If app has specified a catalog

	# Make sure to catopen the app's catalog only once
	# by setting a flag. The catalog code
	# itself has this smarts but this is a less expensive check
	if { [$self info vars sm_catalog_opened] == ""} {
	    
	    _Catalog $catalog.cat $catalog	;# Init a cat obj. and open it
	    $self set sm_catalog_opened "true"
	}

    }

}	;# end instproc Open_msg_catalog
