# 
# @DEC_COPYRIGHT@
#
# HISTORY
# $Log: buttonbox.tcl,v $
# Revision 1.1.1.1  2003/01/23 18:34:40  ajay
# Initial submit to CVS.
#
#
# Revision 1.1.4.4  1997/12/03  22:03:05  Todd_Moyer
# 	Sysman code drop for BL14.
# 	[1997/12/01  21:57:41  Todd_Moyer]
#
# Revision 1.1.2.6  1997/10/24  13:57:43  William_Athanasiou
# 	remove 'option add' statements
# 	[1997/10/24  13:49:44  William_Athanasiou]
# 
# Revision 1.1.2.5  1997/03/28  16:46:32  William_Athanasiou
# 	Postpone UI creation until first display for performance reasons
# 	[1997/03/28  16:27:52  William_Athanasiou]
# 
# Revision 1.1.2.4  1997/02/24  19:17:14  William_Athanasiou
# 	Added infomsg class; and used dialogshell for suit windows
# 	[1997/02/24  19:10:18  William_Athanasiou]
# 
# Revision 1.1.2.3  1997/01/16  14:03:59  William_Athanasiou
# 	Prepend _UIT_ to all classes
# 	[1997/01/15  21:12:01  William_Athanasiou]
# 
# Revision 1.1.2.2  1996/09/24  21:12:14  William_Athanasiou
# 	initial version of files/minor changes to TKutils files
# 	[1996/09/24  21:07:31  William_Athanasiou]
# 
# $EndLog$
# 
# @(#)$RCSfile: buttonbox.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:34:40 $
# 
#
# _UIT_Buttonbox
# ----------------------------------------------------------------------
# Manages a framed area with Motif style buttons.  The button box can 
# be configured either horizontally or vertically.  
#
# ----------------------------------------------------------------------
#  AUTHOR: Mark L. Ulferts               EMAIL: mulferts@spd.dsccc.com
#          Bret A. Schuhmacher           EMAIL: bas@wn.com
#
#  @(#) $Id: buttonbox.tcl,v 1.1.1.1 2003/01/23 18:34:40 ajay Exp $
# ----------------------------------------------------------------------
#            Copyright (c) 1995 DSC Technologies Corporation
# ======================================================================
# Permission to use, copy, modify, distribute and license this software 
# and its documentation for any purpose, and without fee or written 
# agreement with DSC, is hereby granted, provided that the above copyright 
# notice appears in all copies and that both the copyright notice and 
# warranty disclaimer below appear in supporting documentation, and that 
# the names of DSC Technologies Corporation or DSC Communications 
# Corporation not be used in advertising or publicity pertaining to the 
# software without specific, written prior permission.
# 
# DSC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING 
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND NON-
# INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, AND THE
# AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, 
# SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO EVENT SHALL 
# DSC BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
# ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION,
# ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
# SOFTWARE.
# ======================================================================

# ------------------------------------------------------------------
#                            BUTTONBOX
# ------------------------------------------------------------------
Class _UIT_Buttonbox -superclass _UIT_Widget

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------

_UIT_Buttonbox instproc init {args} {
   
   $self _define -pady 0 Pad _padY
   $self _define -padx 0 Pad _padX
   $self _define -orient "horizontal" Orient _orient
   $self _define -window . Window _Window
   
   $self next
   $self instvar _Wdgt _Opt

   $self set _resizeFlag {}         ;# Flag for resize needed.
   $self set _buttonList {}         ;# List of all buttons in box.
   $self set _displayList {}        ;# List of displayed buttons.
   
   # Add Configure bindings for geometry management.  
   #
   #
   # Set up some bindings for map and configure events.
   #
   bind bboxMap$self <Map> "$self _setBoxSize"
   bind bboxConfig$self <Configure> "$self _positionButtons"

   bindtags $_Wdgt(hull) \
	 [linsert [bindtags $_Wdgt(hull)] 0 bboxMap$self]
   bindtags $_Wdgt(hull) \
	 [linsert [bindtags $_Wdgt(hull)] 1 bboxConfig$self]
   
   pack propagate $_Wdgt(hull) no
   
   #
   # Explicitly handle configs that may have been ignored earlier.
   #
   if {![string compare _UIT_Buttonbox [$self info class]]} {
      $self configure -padx 0 -pady 0 -orient horizontal
      
      eval {$self configure} $args
   }
}

#
# Provide a lowercased access method for the _UIT_Buttonbox class.
# 
proc buttonbox {pathName args} {
   uplevel _UIT_Buttonbox $pathName $args
}


# ------------------------------------------------------------------
#                           DESTRUCTOR
# ------------------------------------------------------------------
_UIT_Buttonbox instproc destroy {} {
   $self instvar _resizeFlag
   if {$_resizeFlag != ""} {after cancel $_resizeFlag}
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -window
#
# Note the toplevel id of the Chameleon container.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc _Window {val} {
}


# ------------------------------------------------------------------
# OPTION: -pady
#
# Pad the y space between the button box frame and the hull.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc _padY {val} {
   $self _setBoxSize
}

# ------------------------------------------------------------------
# OPTION: -padx
#
# Pad the x space between the button box frame and the hull.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc _padX {val} {
   $self _setBoxSize
}

# ------------------------------------------------------------------
# OPTION: -orient
#
# Position buttons either horizontally or vertically.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc _orient {val} {
   $self instvar _Opt

   switch $_Opt(-orient) {
      "horizontal" -
      "vertical" {
	 $self _setBoxSize
      }
      
      default {
	 error "bad orientation option \"$_Opt(-orient)\",\
	       should be either horizontal or vertical"
      }
   }
}

# ------------------------------------------------------------------
#                            METHODS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# METHOD: index index
#
# Searches the buttons in the box for the one with the requested tag,
# numerical index, keyword "end" or "default".  Returns the button's 
# tag if found, otherwise error.
# ------------------------------------------------------------------    
_UIT_Buttonbox instproc index {index} {
   $self instvar _buttonList _Opt _Wdgt
   
   if {[llength $_buttonList] > 0} {
      if {[regexp {(^[0-9]+$)} $index]} {
	 if {$index < [llength $_buttonList]} {
	    return $index
	 } else {
	    error "Buttonbox index \"$index\" is out of range"
	 }
	 
      } elseif {$index == "end"} {
	 return [expr [llength $_buttonList] - 1]
	 
      } elseif {$index == "default"} {
	 foreach knownButton $_buttonList {
	    if {[$_Wdgt($knownButton) cget -defaultring]} {
	       return [lsearch -exact $_buttonList $knownButton]
	    }
	 }
	 
	 error "Buttonbox \"$_Wdgt(hull)\" has no default"
	 
      } else {
	 if {[set idx [lsearch $_buttonList $index]] != -1} {
	    return $idx
	 }
	 
	 error "bad _UIT_Buttonbox index \"$index\": must be number, end,\
	       default, or pattern"
      }
      
   } else {
      error "Buttonbox \"$_Wdgt(hull)\" has no buttons"
   }
}

# ------------------------------------------------------------------
# METHOD: add tag ?option value option value ...?
#
# Add the specified button to the button box.  All PushButton options
# are allowed.  New buttons are added to the list of buttons and the 
# list of displayed buttons.  The PushButton path name is returned.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc add {tag args} {
   $self instvar _Wdgt _buttonList _displayList _Opt
   
   set nm [lindex [split $tag .] end]
   if {[info commands $tag] != {}} {
      # button already exists, but UI widget needs to be created.
      $tag _createUIrep $self.p${nm}
      set _Wdgt($tag) $tag
   } else {
      set _Wdgt($tag) [Button $tag -widgetname $self.p${nm}]
   }

   if {$args != ""} {
      uplevel $_Wdgt($tag) configure $args
   }
   
   lappend _buttonList $tag
   lappend _displayList $tag
   
   $self _setBoxSize
}

# ------------------------------------------------------------------
# METHOD: addTkButton tag ?option value option value ...?
#
# Add the specified Tkbutton to the button box.  New buttons are added
# to the list of buttons and the list of displayed buttons. 
# ------------------------------------------------------------------
_UIT_Buttonbox instproc addTkButton {tag args} {
   $self instvar _Wdgt _buttonList _displayList _Opt
   
   set _Wdgt($tag) [_UIT_Pushbutton $self.p${tag}]
   if {$args != ""} {
      uplevel $_Wdgt($tag) configure $args
   }

   lappend _buttonList $tag
   lappend _displayList $tag
   
   $self _setBoxSize
}

# ------------------------------------------------------------------
# METHOD: insert index tag ?option value option value ...?
#
# Insert the specified button in the button box just before the one 
# given by index.  All PushButton options are allowed.  New buttons 
# are added to the list of buttons and the list of displayed buttons.
# The PushButton path name is returned.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc insert {index tag args} {
   $self instvar _Wdgt _Opt _buttonList _displayList
   
   set nm [lindex [split $tag .] end]
   if {[info commands $tag] != {}} {
      #button already exists, but UI widget needs to be created.
      $tag _createUIrep $self.p${nm}
      set _Wdgt($tag) $tag
   } else {
      set _Wdgt($tag) [Button $tag -widgetname $self.p${nm}]
   }

   if {$args != ""} {
      uplevel $_Wdgt($tag) configure $args
   }
    
   set index [$self index $index]
   set _buttonList [linsert $_buttonList $index $tag]
   set _displayList [linsert $_displayList $index $tag]
   
   $self _setBoxSize
}

# ------------------------------------------------------------------
# METHOD: delete index
#
# Delete the specified button from the button box.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc delete {index} {
   $self instvar _Wdgt _Opt _buttonList _displayList
   
   set index [$self index $index]
   set tag [lindex $_buttonList $index]
   
   destroy $_Wdgt($tag)
   
   set _buttonList [lreplace $_buttonList $index $index]
   
   if {[set dind [lsearch $_displayList $tag]] != -1} {
      set _displayList [lreplace $_displayList $dind $dind]
   }
   
   $self _setBoxSize
}

# ------------------------------------------------------------------
# METHOD: default index
#
# Sets the default to the push button given by index.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc default {index} {
   $self instvar _Wdgt _displayList _buttonList
   
   set index [$self index $index]
   
   set defbtn [lindex $_buttonList $index]
   
   foreach knownButton $_displayList {
      if {$knownButton == $defbtn} {
	 $_Wdgt($knownButton) configure -defaultring yes
      } else {
	 $_Wdgt($knownButton) configure -defaultring no
      }
   }

   $self _setBoxSize
}

# ------------------------------------------------------------------
# METHOD: hide index
#
# Hide the push button given by index.  This doesn't remove the button 
# permanently from the display list, just inhibits its display.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc hide {index} {
   $self instvar _Wdgt _displayList _buttonList
   
   set index [$self index $index]
   set tag [lindex $_buttonList $index]
   
   if {[set dind [lsearch $_displayList $tag]] != -1} {
      place forget [$_Wdgt($tag) _widget]
      set _displayList [lreplace $_displayList $dind $dind] 
      
      $self _setBoxSize
   }
}

# ------------------------------------------------------------------
# METHOD: show index
#
# Displays a previously hidden push button given by index.  Check if 
# the button is already in the display list.  If not then add it back 
# at it's original location and redisplay.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc show {index} {
   $self instvar _Wdgt _buttonList _displayList

   set index [$self index $index]
   set tag [lindex $_buttonList $index]
   
   if {[lsearch $_displayList $tag] == -1} {
      set _displayList [linsert $_displayList $index $tag]
      
      $self _setBoxSize
   }
}

# ------------------------------------------------------------------
# METHOD: invoke ?index?
#
# Invoke the command associated with a push button.  If no arguments
# are given then the default button is invoked, otherwise the argument
# is expected to be a button index.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc invoke {args} {
   $self instvar _buttonList _Wdgt
   
   if {[llength $args] == 0} {
      $_Wdgt([lindex $_buttonList [$self index default]]) invoke
      
   } else {
      $_Wdgt([lindex $_buttonList [$self index [lindex $args 0]]]) \
	    invoke
   }
}

# ------------------------------------------------------------------
# METHOD: buttonconfigure index ?option? ?value option value ...?
#
# Configure a push button given by index.  This method allows 
# configuration of pushbuttons from the _UIT_Buttonbox level.  The options
# may have any of the values accepted by the add method.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc buttonconfigure {index args} {
   $self instvar _buttonList _Wdgt
   
   set tag [lindex $_buttonList [$self index $index]]
   
   set retstr [uplevel $_Wdgt($tag) configure $args]
   
   $self _setBoxSize
   
   return $retstr
}

# -----------------------------------------------------------------
# PRIVATE METHOD: _getMaxWidth
#
# Returns the required width of the largest button.
# -----------------------------------------------------------------
_UIT_Buttonbox instproc _getMaxWidth {} {
   $self instvar _displayList _Wdgt

   set max 0   
   foreach tag $_displayList {
      set w [winfo reqwidth [$_Wdgt($tag) _widget]]
      
      if {$w > $max} {
	 set max $w
      }
   }
   
   return $max
}

# -----------------------------------------------------------------
# PRIVATE METHOD: _getMaxHeight
#
# Returns the required height of the largest button.
# -----------------------------------------------------------------
_UIT_Buttonbox instproc _getMaxHeight {} {
   $self instvar _Wdgt _displayList
   
   set max 0
   foreach tag $_displayList {
      set h [winfo reqheight [$_Wdgt($tag) _widget]]

      if {$h > $max} {
	 set max $h
      }
   }
   
   return $max
}
  
# ------------------------------------------------------------------
# METHOD: _setBoxSize ?when?
#
# Sets the proper size of the frame surrounding all the buttons.
# If "when" is "now", the change is applied immediately.  If it is 
# "later" or it is not specified, then the change is applied later, 
# when the application is idle.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc _setBoxSize {{when later}} {
   $self instvar _Wdgt _Opt _displayList _resizeFlag
   
   if {[winfo ismapped $_Wdgt(hull)]} {
      if {$when == "later"} {
	 if {$_resizeFlag == ""} {
	    set _resizeFlag [after idle "$self _setBoxSize now"]
	 }
	 return
      } elseif {$when != "now"} {
	 error "bad option \"$when\": should be now or later"
      }
      
      set _resizeFlag ""
      
      set numBtns [llength $_displayList]
      
      if {$_Opt(-orient) == "horizontal"} {
	 set minw [expr $numBtns * [$self _getMaxWidth] \
	       + ($numBtns+1) * $_Opt(-padx)]
	 set minh [expr [$self _getMaxHeight] + 2 * $_Opt(-pady)]
	 
      } else {
	 set minw [expr [$self _getMaxWidth] + 2 * $_Opt(-padx)]
	 set minh [expr $numBtns * [$self _getMaxHeight] \
	       + ($numBtns+1) * $_Opt(-pady)]
      }
      
      #
      # Remove the configure event bindings on the hull while we adjust the
      # width/height and re-position the buttons.  Once we're through, we'll
      # update and reinstall them.  This prevents double calls to position
      # the buttons.
      #
      set tags [bindtags $_Wdgt(hull)]
      if {[set i [lsearch $tags bbox-config]] != -1} {
	 set tags [lreplace $tags $i $i]
	 bindtags $_Wdgt(hull) $tags
      }
      
      $_Wdgt(hullcmd) configure -width $minw -height $minh

      update idletasks
      
      $self _positionButtons
      
      bindtags $_Wdgt(hull) [linsert $tags 0 bbox-config]
      
      #
      # Remove the binding for the map event if it still exists.
      #
      set tags [bindtags $_Wdgt(hull)]
      
      if {[set i [lsearch $tags bbox-map]] >= 0} {
	 set tags [lreplace $tags $i $i]
	 bindtags $_Wdgt(hull) $tags
      }
   }
}
    
# ------------------------------------------------------------------
# METHOD: _positionButtons
# 
# This method is responsible setting the width/height of all the 
# displayed buttons to the same value and for placing all the buttons
# in equidistant locations.
# ------------------------------------------------------------------
_UIT_Buttonbox instproc _positionButtons {} {
   $self instvar _Wdgt _Opt _displayList 
   
   set bf $_Wdgt(hull)
   set numBtns [llength $_displayList]
   
   # 
   # First, determine the common width and height for all the 
   # displayed buttons.
   #
   if {$numBtns > 0} {
      set bfWidth [winfo width $_Wdgt(hull)]
      set bfHeight [winfo height $_Wdgt(hull)]
      
      if {$bfWidth >= [winfo reqwidth $_Wdgt(hull)]} {
	 set _btnWidth [$self _getMaxWidth] 
	 
      } else {
	 if {$_Opt(-orient) == "horizontal"} {
	    set _btnWidth [expr $bfWidth / $numBtns]
	 } else {
	    set _btnWidth $bfWidth
	 }
      }	    
      
      if {$bfHeight >= [winfo reqheight $_Wdgt(hull)]} {
	 set _btnHeight [$self _getMaxHeight]
	 
      } else {
	 if {$_Opt(-orient) == "vertical"} {
	    set _btnHeight [expr $bfHeight / $numBtns]
	 } else {
	    set _btnHeight $bfHeight
	 }
      }	    
   }
   
   #
   # Place the buttons at the proper locations.
   #
   if {$numBtns > 0} {
      if {$_Opt(-orient) == "horizontal"} {
	 set leftover [expr [winfo width $bf] \
	       - 2 * $_Opt(-padx) - $_btnWidth * $numBtns]
	 
	 if {$numBtns > 0} {
	    set offset [expr $leftover / ($numBtns + 1)]
	 } else {
	    set offset 0
	 }
	 if {$offset < 0} {set offset 0}
	 
	 set xDist [expr $_Opt(-padx) + $offset]
	 set incrAmount [expr $_btnWidth + $offset]
	 
	 foreach button $_displayList {
	    place [$_Wdgt($button) _widget] -anchor w \
		  -x $xDist -relx 0 -y 0\
		  -width $_btnWidth -height $_btnHeight
	    
	    set xDist [expr $xDist + $incrAmount]
	 }
	 
      } else {
	 set leftover [expr [winfo height $bf] \
	       - 2 * $_Opt(-pady) - $_btnHeight * $numBtns]
	 
	 if {$numBtns > 0} {
	    set offset [expr $leftover / ($numBtns + 1)]
	 } else {
	    set offset 0
	 }
	 if {$offset < 0} {set offset 0}
	 
	 set yDist [expr $_Opt(-pady) + $offset]
	 set incrAmount [expr $_btnHeight + $offset]
	 
	 foreach button $_displayList {
	    place [$_Wdgt($button) _widget] -anchor n \
		  -y $yDist -relx .5 -x 0 -rely 0 \
		  -width $_btnWidth -height $_btnHeight
	    
	    set yDist [expr $yDist + $incrAmount]
	 }
      }
   }
}


