#!/usr/bin/tclsh
# 
# @DEC_COPYRIGHT@
#
# HISTORY
# $Log: mclistbox.tcl,v $
# Revision 1.2  2003/05/21 19:40:22  ajay
# Ported the fix for QAR 96137 and 96138 to fix bugs with
# multicolumn list widgets.
#
# Revision 1.1.1.1  2003/01/23 18:34:40  ajay
# Initial submit to CVS.
#
#
# Revision 1.1.6.1  2001/01/11  15:53:37  William_Athanasiou
# 	Fix multicolumn listbox issues for qar 78664
#
# Revision 1.1.2.3  2001/01/08  16:52:20  William_Athanasiou
# 	Update curses mclist implementation
#
# $EndLog$
# 
# @(#)$RCSfile: mclistbox.tcl,v $ $Revision: 1.2 $ (DEC) $Date: 2003/05/21 19:40:22 $
# 


######################################
#
# Start of widget class.
#
# This widget implements a multicolumn listbox by using a canvas on which
# is placed a series of columns each containing a button and a listbox.  The
# vertical scrollbar moves the list entries up and down.  The horizontal 
# scrollbar moves the canvas left and right to expose additional columns.
#
######################################


Class _UIT_Mclistbox -superclass _UIT_Labeledwidget

_UIT_Mclistbox instproc init {args} {

   # Define the options available to the MClistbox widget
   $self _define -vscrollmode none ScrollMode _vscrollMode
   $self _define -hscrollmode none ScrollMode _hscrollMode
   $self _define -dblclickcommand {} Command _dblClickCommand
   $self _define -selectioncommand {} Command _selectionCommand
   $self _define -selectmode single SelectMode _selectMode
   $self _define -items {} Items _items
   $self _define -width 0 Width _width
   $self _define -height 0 Height _height
   $self _define -visibleitems 20x5 VisibleItems _visibleItems
   $self _define -headings {} Headings _headings
   $self _define -sortcommand {} Command _sortCommand

   # Define the instance variables for the MClistbox
   $self instvar _Wdgt _Opt _NextColumn  \
       _selection _Lists _Btns _adjustCanvas _adjustHeight\
       _prevHeight _prevWidth _dontAdjust _afterExpose

   set _NextColumn 0
   set _configFlag {}

   set _selection {}           ;# Last selection indcies.
   set _Lists {}
   set _Btns {}
   set _adjustCanvas {}
   set _adjustHeight {}
   set _prevHeight -9
   set _prevWidth -9
   set _dontAdjust {}
   set _afterExpose {}

   # init widget skeleton creation
   #   This gives us the general framework we need for building
   #   the multicolumn list widget.  The two instance variables
   #   _Wdgt and _Opt contain the Tk/Ctk widget heirarchy and the
   #   multicolumn listboxes options.
   #
   eval $self next

   # create the widget

   $_Wdgt(hullcmd) configure -borderwidth 0
   
   #
   # Create the widget representation.
   #   The interior widget is the childsite of the parent labeledwidget
   #   class.  This will be used to build the list.

   set _Wdgt(listframe) $_Wdgt(interior)
   pack $_Wdgt(listframe) -fill both -expand yes -padx 0 -pady 0

   # Create the canvas to hold the buttons and listboxes.
   #
   set _Wdgt(canvas) [canvas $_Wdgt(listframe).canvas  \
			  -borderwidth 0]
   
   # Create scrollbars
   #
   set _Wdgt(hsb) [scrollbar $_Wdgt(listframe).hsb \
		       -orient horizontal -takefocus 1]
   set _Wdgt(vsb) [scrollbar $_Wdgt(listframe).vsb \
		       -orient vertical -takefocus 1 \
		       -command "$self _scrollList"]
   #
   # Create a border for all of the listboxes and buttons.  The canvas
   # will be placed in this border.
   #
   set _Wdgt(border) [frame $_Wdgt(listframe).border -borderwidth 1]
   bind $_Wdgt(border) <Configure> "$self _configEvent"

   #
   # Create a binding that will be invoked whenever the toplevel
   # of this window is exposed.  This is a hack to get around a problem
   # where the buttons and separator disappears.
   bind [lindex [bindtags [winfo toplevel $_Wdgt(border)]] 1] \
       <Expose> "+$self _restoreHeaders later"
   
   # Associate the canvas and the scrollbars
   #
   $_Wdgt(hsb) configure -command "$_Wdgt(canvas) xview"
   $_Wdgt(canvas) configure -xscrollcommand "$self _canvasScroll"

   # Create a frame to be placed inside the canvas to contain all of the 
   # listboxes, buttons and separator.
   #
   set _Wdgt(innerFrame) [frame $_Wdgt(canvas).sframe]
   set _Wdgt(separator) [frame $_Wdgt(innerFrame).sep \
			     -height 1 -borderwidth 1]
   
   # Put the widget together.  No listboxes or buttons have been created yet.
   #
   pack $_Wdgt(canvas) -in $_Wdgt(border) -expand 1 -fill both

   grid configure $_Wdgt(border) -column 0 -row 0 -sticky nsew
   grid configure $_Wdgt(hsb) -column 0 -row 1 -sticky ew
   grid configure $_Wdgt(vsb) -column 1 -row 0 -sticky ns
   grid columnconfigure $_Wdgt(listframe) 0 -weight 1
   grid rowconfigure $_Wdgt(listframe) 0 -weight 1

   # Create the window in the canvas and place the scrolled frame into it.
   #
   $_Wdgt(canvas) create window 0 0 -anchor nw -window $_Wdgt(innerFrame)

   # Do nothing else until listboxes and buttons are created. So, Explicitly
   # handle configs that may have been ignored earlier.
   #
   if {![string compare _UIT_Mclistbox [$self info class]]} {
      set _Opt(-visibleitems) 20x5
      $self configure -labelpos nw -labeljustify left \
	  -width 0 -height 0 \
	  -hscrollmode none -vscrollmode none -selectmode single
      
      eval {$self configure} $args
   }

   # Create unique listbox bindings:  Bindtags name = Mclistbox$self
   $self _createBindtags
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# Record the command for handling sort requests.

_UIT_Mclistbox instproc _sortCommand {val} {
   $self set _Opt(-sortcommand) $val
}

# The -headings option is used to specify a series of columns
# in a compacted form.

_UIT_Mclistbox instproc _headings {val} {
   # val is a list whose elements are lists of the form:
   # {column name, column label, column width}
   # Note that none of this is optional.
   if [info exists _Opt(-headings)] {
      if ![lempty $_Opt(-headings)] {
	 # remove old lists and buttons
	 # then create new ones
      }
   }

   foreach heading $val {
      lassign $heading name label min
      if [lempty $min] {
	 set min 10
      }
      $self add -min $min -label $label -command "$self _sort $name"
   }
}

# ------------------------------------------------------------------
# OPTION: -width
#
# Specifies the width of the scrolled list box as an entire unit.
# The value may be specified in any of the forms acceptable to 
# Tk_GetPixels.  Any additional space needed to display the other
# components such as labels, margins, and scrollbars force the listbox
# to be compressed.  A value of zero along with the same value for 
# the height causes the value given for the visibleitems option 
# to be applied which administers geometry constraints in a different
# manner.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _width {val} {
   $self instvar _Wdgt _Opt 
 
   set _Opt(-width) $val
   if {$_Opt(-width) != 0} {
      pack propagate $_Wdgt(shell) no
      
      $_Wdgt(listframe) configure -width 1
      $_Wdgt(shell) configure \
	    -width [winfo pixels \
	    $_Wdgt(shell) $_Opt(-width)] 
   } else {
      $self configure -visibleitems $_Opt(-visibleitems)
   }
}

# ------------------------------------------------------------------
# OPTION: -height
#
# Specifies the height of the scrolled list box as an entire unit.
# The value may be specified in any of the forms acceptable to 
# Tk_GetPixels.  Any additional space needed to display the other
# components such as labels, margins, and scrollbars force the listbox
# to be compressed.  A value of zero along with the same value for 
# the width causes the value given for the visibleitems option 
# to be applied which administers geometry constraints in a different
# manner.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _height {val} {
   $self instvar _Wdgt _Opt
   
   set _Opt(-height) $val
   if {$_Opt(-height) != 0} {
      pack propagate $_Wdgt(shell) no
      
      $_Wdgt(listframe) configure -height 1
      ${self}-cmd configure \
	    -height [winfo pixels $_Wdgt(shell) $_Opt(-height)] 
   } else {
      $self configure -visibleitems $_Opt(-visibleitems)
   }
}

# ------------------------------------------------------------------
# OPTION: -visibleitems
#
# Specified the widthxheight in characters and lines for the listbox.
# This option is only administered if the width and height options
# are both set to zero, otherwise they take precedence.  With the
# visibleitems option engaged, geometry constraints are maintained
# only on the listbox.  The size of the other components such as 
# labels, margins, and scroll bars, are additive and independent, 
# effecting the overall size of the scrolled list box.  In contrast,
# should the width and height options have non zero values, they
# are applied to the scrolled list box as a whole.  The listbox 
# is compressed or expanded to maintain the geometry constraints.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _visibleItems {val} {
   $self instvar _Wdgt _Opt

   if {[regexp {^[0-9]+x[0-9]+$} $_Opt(-visibleitems)]} {
      if {($_Opt(-width) == 0) && ($_Opt(-height) == 0)} {
	 set chars [lindex [split $_Opt(-visibleitems) x] 0]
	 set lines [lindex [split $_Opt(-visibleitems) x] 1]
	 
#	 pack propagate $_Wdgt(shell) yes
	 
#	 $_Wdgt(listframe) configure -width $chars -height $lines
	 grid rowconfigure $_Wdgt(listframe) 0 -minsize $lines
	 grid columnconfigure $_Wdgt(listframe) 0 -minsize $chars

      }
      
   } else {
      error "bad visibleitems option\
	    \"$_Opt(-visibleitems)\": should be\
	    widthxheight"
   }
}

# ------------------------------------------------------------------
# OPTION: -selectmode
#
# Set/get the current selection mode of each of the listboxes.
# ------------------------------------------------------------------

_UIT_Mclistbox instproc _selectMode {val} {
   foreach lst [$self set _Lists] {
      $lst configure -selectmode $val
   }
}

# ------------------------------------------------------------------
# OPTION: -items
#
# Set/get the current list of items in the listbox.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _items {val} {
   set index [$self nearest 0]
   foreach lst [$self set _Lists] {
      $lst delete 0 end
   }
   $self set _Opt(-items) {}
   
   if ![lempty $val] {
      eval {$self insert end} $val
      
      if {$index < [$self size]} {
	 $self yview $index
      }
   } else {
      $self _setFocus 0
   }
      
}

# ------------------------------------------------------------------
# OPTION: -vscrollmode
#
# Enable/disable display and mode of veritcal scrollbars.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _vscrollMode {val} {
   $self instvar _Opt
   
   set _Opt(-vscrollmode) $val
   switch $_Opt(-vscrollmode) {
      static {
	 $self _scrollbarDisplay vsb on
      }
      
      dynamic -
      none {
	 $self _scrollbarDisplay vsb off
      }
      
      default {
	 error "bad vscrollmode option\
	       \"$_Opt(-vscrollmode)\": should be\
	       static, dynamic, or none"
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -hscrollmode
#
# Enable/disable display and mode of horizontal scrollbars.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _hscrollMode {val} {
   $self instvar _Opt
   
   switch [set _Opt(-hscrollmode) $val] {
      static {
	 $self _scrollbarDisplay hsb on
      }
      
      dynamic -
      none {
	 $self _scrollbarDisplay hsb off
      }
      
      default {
	 error "bad hscrollmode option\
	       \"$_Opt(-hscrollmode)\": should be\
	       static, dynamic, or none"
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -dblclickcommand
#
# Specify a command to be executed upon double click of a listbox 
# item.  Also, create a couple of bindings used for specific
# selection modes
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _dblClickCommand {val} {
   $self set _Opt(-dblclickcommand) $val
}

# ------------------------------------------------------------------
# OPTION: -selectioncommand
#
# Specifies a command to be executed upon selection of a listbox 
# item.  The command will be called upon each selection regardless 
# of selection mode..
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _selectionCommand {val} {
   $self set _Opt(-selectioncommand) $val
}

# ------------------------------------------------------------------
#                             Methods
# ------------------------------------------------------------------

# -----------------
# Public methods

# ------------------------------------------------------------------
# METHOD: curselection 
#
# Returns a list containing the indices of all the elements in the 
# listbox that are currently selected.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc curselection {} {
   set lst [lindex [$self set _Lists] 0]
   return [$lst curselection]
}

# ------------------------------------------------------------------
# METHOD: activate index
#
# Sets the active element to the one indicated by index.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc activate {index} {
   set ndx [$self index $index]
   foreach lst [$self set _Lists] {
      set return [$lst activate $ndx]
   }
   return $return
}

# ------------------------------------------------------------------
# METHOD: bbox index
#
# Returns four element list describing the bounding box for the list
# item at index
# ------------------------------------------------------------------
_UIT_Mclistbox instproc bbox {index} {
   #return [[$self set _Wdgt(listbox)] bbox [$self index $index]]
   #this is very strange.  We may need to combine the bounding boxes 
   # of all the lists in the mclistbox to come up with a true bbox
   # not necessary now.
}

# ------------------------------------------------------------------
# METHOD clear 
#
# Clear the listbox area of all items.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc clear {} {
   $self delete 0 end
}

# ------------------------------------------------------------------
# METHOD: see index
#
# Adjusts the view such that the element given by index is visible.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc see {index} {
   set ndx [$self index $index]
   foreach lst [$self set _Lists] {
      $lst see $ndx
   }
}

# ------------------------------------------------------------------
# METHOD: index index
#
# Returns the decimal string giving the integer index corresponding 
# to index.  The index value may be a integer number, active,
# anchor, end, @x,y, or a pattern.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc index {index} {
   $self instvar _Opt _Wdgt _Lists
   set lst [lindex $_Lists 0]

   if {[regexp \
	 {(^[0-9]+$)|(active)|(anchor)|(end)|(^@[0-9]+,[0-9]+$)} $index]} {
      return [$lst index $index]
      
   } else {
      # ok, this is another weird one. since item is actually spread 
      # across all the listboxes.
      set indexValue [lsearch -glob $_Opt(-items) $index]
      
      if {$indexValue == -1} {
	 error "bad _UIT_Mclistbox index \"$index\": must be\
	       active, anchor, end, @x,y, number, or a pattern"
      }
      
      return $indexValue
   }
}

# ------------------------------------------------------------------
# METHOD: delete first ?last?
#
# Delete one or more elements from list box based on the first and 
# last index values.  Indexes may be a number, active, anchor, end,
# @x,y, or a pattern.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc delete {first {last {}}} {
   $self instvar _Opt _Lists

   set first [$self index $first]
   
   if {$last != {}} {
      set last [$self index $last]
   } else {
      set last $first
   }
    
   if ![lempty $_Opt(-items)] {
      if {$first <= $last} {
	 # Delete the elements from each of the columns
	 # by iterating throught the lists.
	 foreach lst $_Lists {
	    $lst delete $first $last
	 }
	 # update the -items option to reflect the new list contents.
	 set _Opt(-items) [lreplace $_Opt(-items) $first $last]
      } else {
	 error "first index must not be greater than second"
      }
   }
}

# ------------------------------------------------------------------
# METHOD: get first ?last?
#
# Returns the elements of the listbox indicated by the indexes. 
# Indexes may be a number, active, anchor, end, @x,y, ora pattern.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc get {first {last {}}} {
   $self instvar _Opt
   set first [$self index $first]
   
   if {$last != {}} {
      set last [$self index $last]
   }
    

   # Get the list elements from the -item array as these are already
   # kept in the form that we need to present to the user.

   if {$last == {}} {
      return [lindex $_Opt(-items) $first]
   } else {
      return [lrange $_Opt(-items) $first $last]
   }
}

# ------------------------------------------------------------------
# METHOD: getcurselection 
#
# Returns the contents of the listbox element indicated by the current 
# selection indexes.  Short cut version of get and curselection 
# command combination.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc getcurselection {} {
   $self instvar _Opt
   set rlist {}
   
   if {[$self selecteditemcount] > 0} {
      set cursels [$self curselection]
      
      foreach sel $cursels {
	 lappend rlist [lindex $_Opt(-items) $sel]
      }
   }
   
   return $rlist
}

# ------------------------------------------------------------------
# METHOD: insert index string ?string ...?
#
# Insert zero or more elements in the list just before the element 
# given by index.  Modify the $_Opt(-items) value as well.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc insert {index lst args} {
   $self instvar _Opt _Lists

   set index [$self index $index]
   catch {
      eval {lappend allLines $lst} $args
      # build columns of data so insertion can be done
      # as quickly as possible.
      foreach line $allLines {
	 foreach lst $_Lists {
	    set val [lvarpop line]
	    lappend col($lst) $val
	 }
      }
      foreach lst $_Lists {
	 eval {$lst insert $index} $col($lst)
      }
   } error

   if {$error == 0 || [lempty $error]} {
      set _Opt(-items) [eval {linsert $_Opt(-items) $index} $allLines]
      $self _setFocus 1
   } else {
      return $error
   }
}

# ------------------------------------------------------------------
# METHOD: nearest y
#
# Given a y-coordinate within the listbox, this command returns the 
# index of the visible listbox element nearest to that y-coordinate.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc nearest {y} {
   [lindex [$self set _Lists] 0] nearest $y
}

# ------------------------------------------------------------------
# METHOD: scan option args 
#
# Implements scanning on listboxes.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc scan {option args} {
   # NOT SUPPORTED
#    eval [$self set _Wdgt(listbox)] scan $option $args
}

# ------------------------------------------------------------------
# METHOD: selection option first ?last?
#
# Adjusts the selection within the listbox.  The index value may be 
# a integer number, active, anchor, end, @x,y, or a pattern.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc selection {option first {last {}}} {
   $self instvar _Lists
   
   set first [$self index $first]
   
   if {$last != {}} {
      set last [$self index $last]
   } 
   
   # build a command.

   switch $option {
      anchor {
	 set cmd "selection anchor $first"
      }
      
      clear {
	 if {$last == {}} {
	    set cmd "selection clear $first"
	 } else {
	    set cmd "selection clear $first $last"
	 }
      }
      
      includes {
	 set cmd "selection includes $first"
      }
      
      set {
	 if {$last == {}} {
	    set cmd "selection set $first"
	 } else {
	    set cmd "selection set $first $last"
	 }
      }
   }
   #
   # Duplicate the command for each list.
   #
   foreach lst $_Lists { eval {$lst} $cmd }
}

# ------------------------------------------------------------------
# METHOD: size 
#
# Returns a decimal string indicating the total number of elements 
# in the listbox.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc size {} {
   return [[lindex [$self set _Lists] 0] size]
}

# ------------------------------------------------------------------
# METHOD: selecteditemcount 
#
# Returns a decimal string indicating the total number of selected 
# elements in the listbox.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc selecteditemcount {} {
   return [llength [[lindex [$self set _Lists] 0] curselection]]
}

# ------------------------------------------------------------------
# METHOD: justify direction
#
# Justifies the list scrolled region in one of four directions: top,
# bottom, left, or right.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc justify {direction} {
   $self instvar _Wdgt

   switch $direction {
      left   { set cmd "xview moveto 0" }
      right  { set cmd "xview moveto 1" }
      top    { set cmd "yview moveto 0" }
      bottom { set cmd "yview moveto 1" }
      default {
	 error "bad justify argument \"$direction\": should\
	       be left, right, top, or bottom"
      }
   }
   foreach lst [$self set _Lists] {eval $lst $cmd}
}

# ------------------------------------------------------------------
# METHOD: sort mode
#
# Sort the current list in either "ascending/increasing" or 
# "descending/decreasing" order.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc sort {mode {column {}} } {
   $self instvar _Opt

   if [lempty $column] {
      
      switch $mode {
	 ascending -
	 increasing {
	    $self configure -items [lsort -increasing $_Opt(-items)]
	 }
	 descending -
	 decreasing {
	    $self configure -items [lsort -decreasing $_Opt(-items)]
	 }
	 default {
	    error "bad sort argument \"$mode\": should be\
	       ascending, descending, increasing, or decreasing"
	 }
      }

   } else {
      
   }
}

# ------------------------------------------------------------------
# METHOD: xview args
#
# Change or query the vertical position of the text in the list box.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc xview {args} {
   return [eval [lindex [$self set _Lists] 0] xview $args]
}

# ------------------------------------------------------------------
# METHOD: yview args
#
# Change or query the horizontal position of the text in the list box.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc yview {args} {
   foreach lst [$self set _Lists] {
       set rc [eval $lst yview $args]
   }
   return $rc
}



# ----------------
# Private Methods

# ------------------------------------------------------------------
# METHOD: _sort

_UIT_Mclistbox instproc _sort {column} {
   $self instvar _Opt
   if ![lempty $_Opt(-sortcommand)] {
      uplevel \#0 [$self set _Opt(-sortcommand)] $column
   }
}


# Method: _setFocus 
#
# This method sets and removes focus from all of the multicolumn
# listbox.

_UIT_Mclistbox instproc _setFocus {state} {
   $self instvar _Lists _Btns _Wdgt

   set btns $_Btns
   foreach lst $_Lists {
      [lvarpop btns] configure -takefocus $state
      $lst configure -takefocus $state
   }
   $_Wdgt(vsb) configure -takefocus $state
   $_Wdgt(hsb) configure -takefocus $state
}

# Method: _configEvent
#
#  This method is called whenever a configuration event occurs for the border.
#  It handles the resize and setup of the scrollregion

_UIT_Mclistbox instproc _configEvent {} {
   $self instvar _Wdgt _adjustCanvas _Lists _prevHeight _prevWidth

   #
   # Don't do anything until the widget is mapped and at least one
   # list exists.

   if ![winfo ismapped $_Wdgt(hull)] { return }
   if [lempty $_Lists] {return}

   #
   # Now start the adjustment process.
   
   set width [winfo width $_Wdgt(innerFrame)]
   set height [expr [winfo height $_Wdgt(border)] - 2]
   set cwidth [expr [winfo width $_Wdgt(border)] - 2]

   #
   # border height includes column header and separator
   # so subtract 2 to get height of listbox
   #
   if [expr $_prevHeight != $height] {
      set _prevHeight $height
      $self _adjustListHeight [expr $height - 2] 
   }
   
   if [cequal $_adjustCanvas "pending"] {
      $self _adjustCanvas
   } else {
#      $_Wdgt(canvas) configure -height $height
#      $_Wdgt(canvas) configure -scrollregion "0 0 $width $height"
   }
   set _prevWidth $cwidth
}

#
# The Height of each of the listboxes must be set manually to ensure that
# the listbox is visible from top to bottom within the canvas.

_UIT_Mclistbox instproc _adjustListHeight {h} {
   $self instvar _Lists _Wdgt 

   foreach lst $_Lists {
      $lst configure -height $h
      raise $lst
   }
#   raise $_Wdgt(innerFrame)
}

# Method: _adjustCanvas
#
#     Adjust sizes of listboxes and buttons since we changed the contents
#     of the frame.

_UIT_Mclistbox instproc _adjustCanvas {} {
   $self instvar _Wdgt _adjustCanvas

   if ![winfo ismapped $_Wdgt(hull)] {
      set _adjustCanvas pending
      return
   }

   $self instvar _Wdgt _Opt _Btns _Lists _NextColumn _Tags _dontAdjust
   # Purpose:
   #   This routine adjusts the size of the columns based on their
   #   configuration parameters.
   #
   # Algorithim:
   #   All lists and buttons start out at their default sizes or their -min.  
   #   If those sizes are smaller than the available space, then the columns 
   #   that can grow are expanded equally until either their -max value is 
   #   reached or all the free space is used.

   if [expr $_NextColumn < 1] return

   set fwidth 0
   foreach lst $_Lists {
      set fwidth [expr $fwidth + [$lst cget -width]]
   }

   set cwidth [winfo width $_Wdgt(canvas)]
   set site   [winfo width $_Wdgt(listframe)]

   set b [winfo width $_Wdgt(border)]

   if [expr $site > $fwidth] {
      # frame is larger than canvas so, we can expand the listboxes in the
      # canvas to fill the frame.  If none of the listboxes can be expanded,
      # then should the frame be shrunk to the size of the canvas?
      set chunk [expr $site - $fwidth - $_NextColumn - 1]

      while [expr $chunk > 0] {
	 set found {}

	 # go through each listbox not including one that may have just been
	 # adjusted and if the listbox isn't at its max
	 # increment it by 1
	 foreach lst [lindex [intersect3 $_dontAdjust $_Lists] end] {
	    set size [expr $_Opt($lst,-size) + 1]
	    if [expr $size > $_Opt($lst,-max)] {
	       continue
	    }

	    # update listbox to new size
	    set _Opt($lst,-size) $size
	    $lst configure -width $size

	    # increase new frame width
	    incr fwidth
	    
	    # decrease remaining space by 1; quit if less than 0
	    incr chunk -1
	    if [expr $chunk < 0] break
	    set found 1
	 }
	 if [lempty $found] break
      }

      set cwidth [winfo width $_Wdgt(canvas)]
#      set fwidth [winfo width $_Wdgt(innerFrame)]
      set site   [winfo width $_Wdgt(listframe)]
   }

   set height [expr [winfo height $_Wdgt(border)] - 2]
   if [expr $cwidth > $fwidth] {
      $_Wdgt(canvas) configure -scrollregion "0 0 $cwidth $height"
   } else {
      $_Wdgt(canvas) configure -scrollregion "0 0 $fwidth $height"
   }
}

# Method: add tag ?option value option value?
#
# Add a column to the listbox.  All columns contain a button and a listbox
#

_UIT_Mclistbox instproc add {args} {
   $self instvar _Wdgt _Btns _Lists _NextColumn _Opt

   # set ndx/rownum for new column

   set ndx $_NextColumn; incr _NextColumn

   # Create listbos and assign the multicolumn bindings to it.

   lappend _Lists [set lst \
		[listbox $_Wdgt(innerFrame).lst${ndx} -borderwidth 0 \
		     -height 1 -yscrollcommand "$self _listScroll" \
		    -selectmode $_Opt(-selectmode)]]
   bindtags $lst [lreplace [bindtags $lst] 1 1 MCListbox${self}]

   # Create button
   
   lappend _Btns [set btn [button $_Wdgt(innerFrame).btn${ndx} \
			   -takefocus 0]]
   bind $btn <Key-minus> "$self _resize btn -1 $btn"
   bind $btn <Key-plus>  "$self _resize btn  1 $btn"
   bind $btn <FocusIn>   "$self _makeVisible $lst"

   #
   # set defaults then parse the arguments passed in

   if ![lempty $args] {
      eval {$self _columnConfigure $lst $btn} \
	  -min 0 -max 999 -size 1 \
	  $args
   }
   
   
   
   # Pack button and listbox into canvas
   
   grid configure $lst -row 2 -column $ndx -sticky nsew
   grid configure $btn -row 0 -column $ndx -sticky w

   if [expr $ndx == 0] {
      # for first listbox need to set expandability of row
      grid rowconfigure $_Wdgt(innerFrame) 2 -weight 1
      grid configure $_Wdgt(separator) -row 1 -column 0 \
	  -columnspan 1 -sticky ew
   } else {
      grid configure $_Wdgt(separator) -columnspan $_NextColumn
   }

   $self _adjustCanvas
}

##
# Method to allow the contents of a column to be reconfigured.  The
# min, max, label, and command options can be changed for any column.

_UIT_Mclistbox instproc columnconfigure {ndx args} {
   $self instvar _Lists _Btns _Wdgt

   set btn [lindex $_Btns  $ndx]
   set lst [lindex $_Lists $ndx]
   eval {$self _columnConfigure $lst $btn} $args

   $self _adjustCanvas

   #
   # Now make sure that the changes in the column show up on 
   # the screen.  This is needed due to CTK issues.
   if  [winfo ismapped $_Wdgt(innerFrame)] {
      after idle "raise $_Wdgt(innerFrame)"
   }
}

_UIT_Mclistbox instproc _columnConfigure {lst btn args} {
   if [lempty $args] return
   if [lempty $lst] {
      # error
      return
   }
   $self instvar _Opt

   if [info exists _Opt($lst,-size)] {
      set prevSize $_Opt($lst,-size)
   } else {
      set prevSize -1
   }

   foreach {opt val} $args {
      switch -- $opt {
	 -min -
	 -max -
	 -size {
	    # do nothing at this point.  Wait until all arguments
	    # are parsed before determining actual value of the
	    # size of the listbox.
	 }
	 -label {
	    $btn configure -text " $val"
	    if [expr $_Opt($lst,-min) == 0] {
	       set _Opt($lst,-min) [winfo reqwidth $btn]
	    }
	 }
	 -command {
	    $btn configure -command $val
	 }
      }
      set _Opt($lst,$opt) $val
   }
   
   # now make sure size is bounded by min and max
   set _Opt($lst,-size) [min $_Opt($lst,-max) \
			     [max $_Opt($lst,-size) $_Opt($lst,-min)]]
   if [expr $prevSize != $_Opt($lst,-size)] {
      # if size has changed, adjust the listbox width
      $lst configure -width $_Opt($lst,-size)
   }
}
   
#########################################
# Bindings for MCListboxes
#
#  These bindings are used for all instances of the Mclistbox widget
#  so they only need to be defined once.
#
##########################################
_UIT_Mclistbox instproc _createBindtags {} {

   eval [subst -nocommands {
      
      bind MCListbox${self} <Up> {
	 foreach W [$self set _Lists] {
	    tkListboxUpDown \$W -1
	 }
      }
      bind MCListbox${self} <Down> {
	 foreach W [$self set _Lists] {
	    tkListboxUpDown \$W 1
	 }
      }
      bind MCListbox${self} <Left> {
	 #
	 # tab like functionality
	 focus [tk_focusPrev %W]

	 # tab only to listboxes?
	 
	 # move contents of list left?
	 #%W xview scroll -1 units
      }
      bind MCListbox${self} <Right> {
	 #
	 # Tab like functionality
	 focus [tk_focusNext %W]
	 
	 # tab only to listboxes?
	 
	 # move contents of list right?
	 #%W xview scroll 1 units
      }
      bind MCListbox${self} <Prior> {
	 foreach W [$self set _Lists] {
	    \$W yview scroll -1 pages
	    tkListboxGoto \$W @0,0
	 }
      }
      bind MCListbox${self} <Next> {
	 foreach W [$self set _Lists] {
	    \$W yview scroll 1 pages
	    tkListboxGoto \$W @0,9999999
	 }
      }
      bind MCListbox${self} <Home> {
	 foreach W [$self set _Lists] {
	    tkListboxGoto \$W 0
	 }
      }
      bind MCListbox${self} <End> {
	 foreach W [$self set _Lists] {
	    tkListboxGoto \$W end
	 }
      }
      bind MCListbox${self} <space> {
	 foreach W [$self set _Lists] {
	    tkListboxBeginSelect \$W [\$W index active]
	 }
      }
      bind MCListbox${self} <Select> {
	 foreach W [$self set _Lists] {
	    tkListboxBeginSelect \$W [\$W index active]
	 }
      }
      bind MCListbox${self} <Escape> {
	 foreach W [$self set _Lists] {
	    tkListboxCancel \$W
	 }
      }
      bind MCListbox${self} <Key-space> {
         foreach W [$self set _Lists] {
            tkListboxBeginSelect \$W [\$W index active]
         }
         $self _makeSelection
         break
      }
      bind MCListbox${self} <Return> {
         foreach W [$self set _Lists] {
           tkListboxBeginSelect \$W [\$W index active]
         }
         $self _dblclick
         break
      }
      bind MCListbox${self} <Key-plus> {
	 $self _resize lst 1 %W
      }
      bind MCListbox${self} <Key-minus> {
	 $self _resize lst -1 %W
      }
      bind MCListbox${self} <FocusIn> {
	 $self _makeVisible %W
      }
   }]
}


# ------------------------------------------------------------------
# PROTECTED METHOD: _makeSelection 
#
# Evaluate the selection command.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _makeSelection {} {
   uplevel #0 [$self set _Opt(-selectioncommand)]
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _dblclick 
#
# Evaluate the double click command option if not empty.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _dblclick {} {
   uplevel #0 [$self set _Opt(-dblclickcommand)]
}	

# ------------------------------------------------------------------
# PRIVATE METHOD: _scrollbarDisplay var wdgt mode
#
# Displays the vertical or horizontal scrollbar based on the input mode.
# ------------------------------------------------------------------

_UIT_Mclistbox instproc _scrollbarDisplay {wdgt state} {
   $self instvar _Wdgt
   
   switch $state  {
      on {
	 if [winfo ismapped $_Wdgt($wdgt)] return
	 grid configure $_Wdgt($wdgt)
      }
      
      off {
	 if [winfo ismapped $_Wdgt($wdgt)] {
	    grid remove $_Wdgt($wdgt)
	 }
      }
      
      default {
	 error "invalid argument \"$mode\": should be on or off"
      }
   }
}
   
# ------------------------------------------------------------------
# PRIVATE METHOD: _listScroll first last
#
# Performs scrolling and display of scrollbars based on the first and
# last listbox views as well as the current -vscrollmode settings.
# ------------------------------------------------------------------
_UIT_Mclistbox instproc _listScroll {first last} {
   $self instvar _Wdgt _Opt 
   
   $_Wdgt(vsb) set $first $last
   
   if {$_Opt(-vscrollmode) == "dynamic"} {
      if {($first == 0) && ($last == 1)} {
	 $self _scrollbarDisplay vsb off
      } else {
	 $self _scrollbarDisplay vsb on
      }
   }
}

_UIT_Mclistbox instproc _scrollList {args} {
   foreach lst [$self set _Lists] {
      eval {$lst yview} $args
   }
}

_UIT_Mclistbox instproc _canvasScroll {first last} {
   $self instvar _Wdgt _Opt
   
   $_Wdgt(hsb) set $first $last

   if {$_Opt(-hscrollmode) == "dynamic"} {
      if {($first == 0) && ($last == 1)} {
	 $self _scrollbarDisplay hsb off
      } else {
	 $self _scrollbarDisplay hsb on
      }
   }
}

_UIT_Mclistbox instproc _resize {type increment wdgt} {
   $self instvar _Wdgt _Lists _Btns _Opt _dontAdjust

   if [cequal $type "btn"] {
      set ndx [lsearch -exact $_Btns $wdgt]
      set lst [lindex $_Lists $ndx]
   } else {
      set ndx [lsearch -exact $_Lists $wdgt]
      set lst $wdgt
   }
   
   set newSize [min $_Opt($lst,-max) \
		    [max $_Opt($lst,-min) \
			 [expr $_Opt($lst,-size) + $increment]]]
   if [expr $newSize != $_Opt($lst,-size)] {
      set _dontAdjust $lst
      $self columnconfigure $ndx -size $newSize
      set _dontAdjust {}
   }
}

_UIT_Mclistbox instproc _makeVisible {wdgt} {
   $self instvar _Wdgt
   
   # Calculate what's showing on the screen
   # at the moment
   set canvas $_Wdgt(canvas)
   lassign [$canvas xview] start end
   set size [winfo width $_Wdgt(innerFrame)]

   set left  [winfo x $wdgt]
   set right [expr $left + [winfo width $wdgt]]

   set wstart [expr $left / ${size}.0]
   set wend   [expr $right / ${size}.0]
   if [expr $wstart >= $start && $wend <= $end] {
      return
   } 
   
   $canvas xview moveto $wstart
}

_UIT_Mclistbox instproc constrainSize {args} {
   return
}

_UIT_Mclistbox instproc _restoreHeaders {when} {
   $self instvar _afterExpose _Btns _Wdgt _Lists
   if [cequal $when "later"] {
      if [lempty $_afterExpose] {
	 set _afterExpose [after idle "$self _restoreHeaders now"]
      }
      return
   }

   set _afterExpose {}
#   raise $_Wdgt(separator)
#   foreach b $_Btns {raise $b}
#   foreach l $_Lists {raise $l}
   after idle "raise $_Wdgt(innerFrame)"
}
