# 
# @DEC_COPYRIGHT@
#
# HISTORY
# $Log: scrolledtext.tcl,v $
# Revision 1.1.1.1  2003/01/23 18:34:40  ajay
# Initial submit to CVS.
#
#
# Revision 1.1.4.5  1999/04/23  15:21:46  William_Athanasiou
# 	Fixed problem of missing error messages when task fails to start
# 	[1999/04/22  18:53:35  William_Athanasiou]
#
# Revision 1.1.4.4  1998/04/22  18:46:26  Anthony_Hoffman
# 	code drop for sysman bl 16 part 2
# 	[1998/04/21  15:24:14  Anthony_Hoffman]
# 
# Revision 1.1.2.4  1998/04/02  19:30:08  William_Athanasiou
# 	Fix keyboard traversal
# 	[1998/04/02  19:29:29  William_Athanasiou]
# 
# Revision 1.1.2.3  1998/01/12  19:06:00  William_Athanasiou
# 	Run text through col to eliminate overstrike characters
# 	[1998/01/12  19:05:43  William_Athanasiou]
# 
# Revision 1.1.2.2  1997/04/22  14:50:20  William_Athanasiou
# 	Added scrolledtext.tcl for Curses
# 	[1997/04/14  19:25:57  William_Athanasiou]
# 
# Revision 1.1.2.6  1997/04/11  14:12:10  William_Athanasiou
# 	fixed tab movement/passwd now invisible
# 	[1997/04/11  14:11:18  William_Athanasiou]
# 
# Revision 1.1.2.5  1997/04/09  21:37:38  William_Athanasiou
# 	Fixed dynamic addition of scrollbars
# 	[1997/04/09  18:17:09  William_Athanasiou]
# 
# Revision 1.1.2.4  1997/04/09  17:10:10  William_Athanasiou
# 	Corrected problem with disabled state and focusability
# 	[1997/04/09  17:08:54  William_Athanasiou]
# 
# Revision 1.1.2.3  1997/04/09  16:30:36  William_Athanasiou
# 	Update scrolledtext for use in GUI
# 	[1997/04/09  16:14:35  William_Athanasiou]
# 
# $EndLog$
# 
# @(#)$RCSfile: scrolledtext.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:34:40 $
# 
# Scrolledtext
# ----------------------------------------------------------------------
# Implements a scrolled text widget with additional options to manage
# the vertical scrollbar.  This includes options to control the method
# in which the scrollbar is displayed, i.e. statically or  dynamically.
# Options also exist for adding a label to the scrolled text area and
# controlling its position.  Import/export of methods are provided for 
# file I/O.
#
# ----------------------------------------------------------------------
#  AUTHOR: Mark L. Ulferts               EMAIL: mulferts@spd.dsccc.com
#
#  @(#) $Id: scrolledtext.tcl,v 1.1.1.1 2003/01/23 18:34:40 ajay Exp $
# ----------------------------------------------------------------------
#            Copyright (c) 1995 DSC Technologies Corporation
# ======================================================================
# Permission to use, copy, modify, distribute and license this software 
# and its documentation for any purpose, and without fee or written 
# agreement with DSC, is hereby granted, provided that the above copyright 
# notice appears in all copies and that both the copyright notice and 
# warranty disclaimer below appear in supporting documentation, and that 
# the names of DSC Technologies Corporation or DSC Communications 
# Corporation not be used in advertising or publicity pertaining to the 
# software without specific, written prior permission.
# 
# DSC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING 
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND NON-
# INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, AND THE
# AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, 
# SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO EVENT SHALL 
# DSC BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
# ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION,
# ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
# SOFTWARE.
# ======================================================================


# ------------------------------------------------------------------
#                           SCROLLEDTEXT
# ------------------------------------------------------------------
Class _UIT_Scrolledtext -superclass _UIT_Labeledwidget


#
# Provide a lowercased access method for the Scrolledtext class.
# 
proc scrolledtext {pathName args} {
    uplevel _UIT_Scrolledtext $pathName $args
}

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc init {args} {

   $self _define -scrollmargin 0 Margin _scrollmargin
   $self _define -sbwidth 1 Width _sbwidth
   $self _define -vscrollmode dynamic ScrollMode _vscrollmode
   $self _define -hscrollmode dynamic ScrollMode _hscrollmode
   $self _define -width 0 Width _width
   $self _define -height 0 Height _height
   $self _define -visibleitems 70x20 VisibleItems _visibleitems

   $self next
   $self instvar _Wdgt _Opt _vmode _hmode _reconfigPlanned _initialized

   set _vmode off            ;# Vertical scroll mode
   set _hmode off            ;# Vertical scroll mode
   set _reconfigPlanned ""   ;# non-null, filler update pending
   set _initialized 0        ;# Initialization flag.
   
   $_Wdgt(hullcmd) configure -borderwidth 0
   
   #
   # Create some frames to hold both the top and bottom halfs of the 
   # widget.  The top will contain both the list and vertical scroll 
   # bar.  The bottom houses the horizontal scrollbar and some filler.
   #
   set _Wdgt(textframe) [frame $_Wdgt(interior).textframe]
   pack $_Wdgt(textframe) -fill both -expand yes
    
   set _Wdgt(bottomframe) [frame $_Wdgt(textframe).bot]
   pack $_Wdgt(bottomframe) -fill x -side bottom
    
   set _Wdgt(hmargin) [frame $_Wdgt(textframe).hm]
   pack $_Wdgt(hmargin) -side bottom
    
   set _Wdgt(topframe) [frame $_Wdgt(textframe).top]
   pack $_Wdgt(topframe) -fill both -expand yes
    
   #
   # Frame around text.
   #
   set _Wdgt(borderFrame) [frame $_Wdgt(topframe).borderFrame ]
   $self _keep $_Wdgt(borderFrame) -borderwidth
   
   # 
   # Create the text area.
   #
   set _Wdgt(text) [text $_Wdgt(borderFrame).text \
	 -borderwidth 0 -wrap none\
	 -width 1 -height 1 \
	 -xscrollcommand "$self _scrollText $_Wdgt(bottomframe).horizsb" \
	 -yscrollcommand "$self _scrollText $_Wdgt(topframe).vertsb"]
   $self _keep $_Wdgt(text) -padx -pady \
	 -spacing1 -spacing2 -spacing3 -state -wrap
   $self _rename $_Wdgt(text) -takefocus -texttakefocus
   pack $_Wdgt(text) -fill both -expand yes 
    
   #
   # Create the vertical margin
   #
   set _Wdgt(vmargin) [frame $_Wdgt(topframe).vmargin -width 0]
    
   # 
   # Create the vertical scroll bar.
   #
   set _Wdgt(vertsb) [scrollbar $_Wdgt(topframe).vertsb -orient vertical \
	 -command "$_Wdgt(text) yview" -takefocus 1]
   $self _rename $_Wdgt(vertsb) -takefocus -vsbtakefocus
   bind $_Wdgt(vertsb) <Next> "$self _jumpPage"
   bind $_Wdgt(vertsb) <Prior> "$self _jumpPage -1"

   pack $_Wdgt(vertsb) -side right -fill y
   pack $_Wdgt(vmargin) -side right
   pack $_Wdgt(borderFrame) -fill both -expand yes -side left
    
   #
   # Next the horizontal scrollbar.
   #
   set _Wdgt(horizsb) [scrollbar $_Wdgt(bottomframe).horizsb \
	 -orient horizontal -takefocus 1\
	 -command "$_Wdgt(text) xview"]
   set _Wdgt(hfill) [frame $_Wdgt(bottomframe).hfill -height 1]
   $self _rename $_Wdgt(horizsb) -takefocus -hsbtakefocus

   pack $_Wdgt(horizsb) -side left -fill x -expand yes
   
   #
   # Create the filler frame and compute its width.
   #
   set _Wdgt(filler) [frame $_Wdgt(bottomframe).filler \
	 -width [$self _fillerWidth]]
   pack $_Wdgt(filler) -side right 

   bind $_Wdgt(hmargin) <Configure> "$self _configureFiller"
   bind $_Wdgt(vertsb) <Configure> "$self _configureFiller"
   bind $_Wdgt(vmargin) <Configure> "$self _configureFiller"
   bind $_Wdgt(horizsb) <Configure> "$self _configureFiller"
   
   #
   # Explicitly handle configs that may have been ignored earlier.
   #
   set _initialized 1
   if {![string compare _UIT_Scrolledtext [$self info class]]} {
      $self configure -borderwidth 1 -scrollmargin 0 -width 0 \
	    -height 0 -visibleitems 70x20 -vscrollmode dynamic \
	    -hscrollmode dynamic -labelpos nw  -labeljustify left

      eval {$self configure} $args
   }
   $self configure -scrollmargin $_Opt(-scrollmargin) \
	 -vscrollmode $_Opt(-vscrollmode)
}

# ------------------------------------------------------------------
#                           DESTURCTOR
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc destroy {} {
   $self instvar _reconfigPlanned
   if {$_reconfigPlanned != ""} {after cancel $_reconfigPlanned}
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -scrollmargin
# Set the distance between the scrollbars and the text area.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _scrollmargin val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized

    if {$_initialized} {
	set pixels [winfo pixels $_Wdgt(hmargin) \
		$_Opt(-scrollmargin)]
	
	if {$_hmode == "on"} {
	    $_Wdgt(hmargin) config -width 1 -height $pixels
	}
	
	if {$_vmode == "on"} {
	    $_Wdgt(vmargin) config -width $pixels -height 1
	}
	
	$_Wdgt(filler) config -width [$self _fillerWidth] -height 1
    }
}

# ------------------------------------------------------------------
# OPTION: -sbwidth
#
# Set the width of the scrollbars.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _sbwidth val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized
   
   if {$_initialized} {
      $_Wdgt(vertsb) configure -width $_Opt(-sbwidth)
      $_Wdgt(horizsb) configure -width $_Opt(-sbwidth)
   }
}

# ------------------------------------------------------------------
# OPTION: -vscrollmode
#
# Enable/disable display and mode of veritcal scrollbar.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _vscrollmode val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized
   
   if {$_initialized} {
      switch $_Opt(-vscrollmode) {
	 static {
	    $self _vertScrollbarDisplay on
	 }
	 
	 dynamic -
	 none {
	    $self _vertScrollbarDisplay off
	 }
	 
	 default {
	    error "bad vscrollmode option\
		  \"$_Opt(-vscrollmode)\": should\
		  be static, dynamic, or none"
	 }
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -hscrollmode
#
# Enable/disable display and mode of horizontal scrollbars.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _hscrollmode val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized

    switch $_Opt(-hscrollmode) {
	static {
	   $self _horizScrollbarDisplay on
	}
	
	dynamic -
	none {
	   $self _horizScrollbarDisplay off
	}
	
	default {
	    error "bad hscrollmode option\
		    \"$_Opt(-hscrollmode)\": should\
		    be static, dynamic, or none"
	}
    }
}

# ------------------------------------------------------------------
# OPTION: -width
#
# Specifies the width of the scrolled text as an entire unit.
# The value may be specified in any of the forms acceptable to 
# Tk_GetPixels.  Any additional space needed to display the other
# components such as labels, margins, and scrollbars force the text
# to be compressed.  A value of zero along with the same value for 
# the height causes the value given for the visibleitems option 
# to be applied which administers geometry constraints in a different
# manner.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _width val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized

   if {$_Opt(-width) != 0} {
      pack propagate $_Wdgt(hull) no
      
      $_Wdgt(text) configure -width 1
      $_Wdgt(hullcmd) configure \
	    -width [winfo pixels $_Wdgt(hull) $_Opt(-width)] 
   } else {
      if {$_initialized} {
	 $self configure -visibleitems $_Opt(-visibleitems)
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -height
#
# Specifies the height of the scrolled text as an entire unit.
# The value may be specified in any of the forms acceptable to 
# Tk_GetPixels.  Any additional space needed to display the other
# components such as labels, margins, and scrollbars force the text
# to be compressed.  A value of zero along with the same value for 
# the width causes the value given for the visibleitems option 
# to be applied which administers geometry constraints in a different
# manner.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _height val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized
   
   if {$_Opt(-height) != 0} {
      pack propagate $_Wdgt(hull) no
      
      $_Wdgt(text) configure -height 1
      $_Wdgt(hullcmd) configure \
	    -height [winfo pixels $_Wdgt(hull) $_Opt(-height)] 
   } else {
      if {$_initialized} {
	 $self configure -visibleitems $_Opt(-visibleitems)
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -visibleitems
#
# Specified the widthxheight in characters and lines for the text.
# This option is only administered if the width and height options
# are both set to zero, otherwise they take precedence.  With the
# visibleitems option engaged, geometry constraints are maintained
# only on the text.  The size of the other components such as 
# labels, margins, and scroll bars, are additive and independent, 
# effecting the overall size of the scrolled text.  In contrast,
# should the width and height options have non zero values, they
# are applied to the scrolled text as a whole.  The text is 
# compressed or expanded to maintain the geometry constraints.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _visibleitems val {
   $self instvar _Wdgt _Opt _hmode _vmode _initialized

   if {[regexp {^[0-9]+x[0-9]+$} $_Opt(-visibleitems)]} {
      if {($_Opt(-width) == 0) && \
	    ($_Opt(-height) == 0)} {
	 set chars [lindex [split $_Opt(-visibleitems) x] 0]
	 set lines [lindex [split $_Opt(-visibleitems) x] 1]
	 
	 pack propagate $_Wdgt(hull) yes
	 $_Wdgt(text) configure \
	       -width $chars -height $lines
      }
      
   } else {
      error "bad visibleitems option\
	    \"$_Opt(-visibleitems)\": should be\
	    widthxheight"
   }
}

# ------------------------------------------------------------------
#                            METHODS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# METHOD: bbox index
#
# Returns four element list describing the bounding box for the list
# item at index
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc bbox {index} {
   return [[$self set _Wdgt(text)] bbox $index]
}

# ------------------------------------------------------------------
# METHOD clear 
#
# Clear the text area.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc clear {} {
    [$self set _Wdgt(text)] delete 1.0 end
}

# ------------------------------------------------------------------
# METHOD import filename
#
# Load text from an existing file (import filename)
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc import {filename} {
   if [file executable /usr/bin/col] {
      set f [open "| /usr/bin/col -b < $filename"]
   } else {
      set f [open $filename r]
   }
   $self insert end [read $f]
   close $f
}

# ------------------------------------------------------------------
# METHOD export filename
#
# write text to a file (export filename)
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc export {filename} {
   set f [open $filename w]
   
   set txt [[$self set _Wdgt(text)] get 1.0 end]
   puts $f $txt
    
   flush $f
   close $f
}

# ------------------------------------------------------------------
# METHOD compare index1 op index2
#
# Compare indices according to relational operator.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc compare {index1 op index2} {
    return [[$self set _Wdgt(text)] compare $index1 $op $index2]
}

# ------------------------------------------------------------------
# METHOD debug ?boolean?
#
# Activates consistency checks in B-tree code associated with text
# widgets.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc debug {args} {
    eval [$self set _Wdgt(text)] debug $args
}

# ------------------------------------------------------------------
# METHOD delete first ?last?
#
# Delete a range of characters from the text.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc delete {first {last {}}} {
    [$self set _Wdgt(text)] delete $first $last
}

# ------------------------------------------------------------------
# METHOD dlineinfo index
#
# Returns a five element list describing the area occupied by the
# display line containing index.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc dlineinfo {index} {
    return [[$self set _Wdgt(text)] dlineinfo $index]
}

# ------------------------------------------------------------------
# METHOD get index1 ?index2?
#
# Return text from start index to end index.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc get {index1 {index2 {}}} {
    return [[$self set _Wdgt(text)] get $index1 $index2]
}

# ------------------------------------------------------------------
# METHOD index index
#
# Return position corresponding to index.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc index {index} {
    return [[$self set _Wdgt(text)] index $index]
}

# ------------------------------------------------------------------
# METHOD insert index chars ?tagList?
#
# Insert text at index.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc insert {args} {
    eval [$self set _Wdgt(text)] insert $args
}

# ------------------------------------------------------------------
# METHOD mark option ?arg arg ...?
#
# Manipulate marks dependent on options.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc mark {option args} {
    return [eval [$self set _Wdgt(text)] mark $option $args]
}

# ------------------------------------------------------------------
# METHOD scan option args
#
# Implements scanning on texts.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc scan {option args} {
    eval [$self set _Wdgt(text)] scan $option $args
}

# ------------------------------------------------------------------
# METHOD search ?switches? pattern index ?varName?
#
# Searches the text for characters matching a pattern.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc search {args} {
    return [eval [$self set _Wdgt(text)] search $args]
}

# ------------------------------------------------------------------
# METHOD see index
#
# Adjusts the view in the window so the character at index is 
# visible.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc see {index} {
    [$self set _Wdgt(text)] see $index
}

# ------------------------------------------------------------------
# METHOD tag option ?arg arg ...?
#
# Manipulate tags dependent on options.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc tag {option args} {
    return [eval [$self set _Wdgt(text)] tag $option $args]
}

# ------------------------------------------------------------------
# METHOD window option ?arg arg ...?
#
# Manipulate embedded windows.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc window {option args} {
    return [eval [$self set _Wdgt(text)] window $option $args]
}

# ------------------------------------------------------------------
# METHOD xview
#
# Changes x view in widget's window.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc xview {args} {
    return [eval [$self set _Wdgt(text)] xview $args]
}

# ------------------------------------------------------------------
# METHOD yview
#
# Changes y view in widget's window.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc yview {args} {
    return [eval [$self set _Wdgt(text)] yview $args]
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _fillerWidth 
#
# Compute the width of the filler frame based on the vertical 
# scrollbar width plus the margin.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _fillerWidth {} {
   $self instvar _Wdgt _vmode _Opt
   if {$_vmode == "on"} {
      if {$_Opt(-scrollmargin) > 1} {
	 return [expr [winfo reqwidth $_Wdgt(vertsb)] + \
	       [winfo reqwidth $_Wdgt(vmargin)]]
      } else {
	 return [winfo reqwidth $_Wdgt(vertsb)]
      }
   } else {
      return 0
   }
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _configureFiller 
#
# Set up to do an update of the filler if one is not all ready 
# planned.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _configureFiller {} {
   $self instvar _reconfigPlanned
   if {$_reconfigPlanned == ""} {
      set _reconfigPlanned [after idle "$self _updateFiller"]
   }
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _updateFiller 
#
# Update the width of the filler following a configure event on 
# a scrollbar or margin.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _updateFiller {} {
   $self instvar _Wdgt _reconfigPlanned
   $_Wdgt(filler) config -width [$self _fillerWidth] -height 1
   set _reconfigPlanned ""
}

# ------------------------------------------------------------------
# PRIVATE METHOD: $self _vertScrollbarDisplay mode
#
# Displays the vertical scrollbar based on the input mode.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _vertScrollbarDisplay {mode} {
   $self instvar _vmode _Wdgt _Opt

   switch $mode  {
      on {
	 set _vmode on
	 
	 pack forget $_Wdgt(text)
	 pack $_Wdgt(vertsb) -side right -fill y -padx 0

	 if {($_Opt(-scrollmargin)) > 1} {
	    $_Wdgt(vmargin) config -height 1 -width \
		  [winfo pixels $_Wdgt(vmargin) \
		  $_Opt(-scrollmargin)]
	 
	    pack $_Wdgt(vmargin) -side right -padx 0
	 }

	 pack $_Wdgt(text) -fill both -expand yes -side left -padx 0
	 
	 update idletasks
	 
	 $_Wdgt(filler) config -width [$self _fillerWidth] -height 1
      }
      
      off {
	 set _vmode off
	 
	 pack forget $_Wdgt(vertsb)
	 pack forget $_Wdgt(vmargin)
	 
      }
      
      default {
	 error "invalid argument \"$mode\": should be on or off"
      }
   }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _horizScrollbarDisplay mode
#
# Displays the horizontal scrollbar based on the input mode.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _horizScrollbarDisplay {mode} {
   $self instvar _hmode _Wdgt _Opt
   switch $mode  {
      on {
	 set _hmode on
	 
	 if {($_Opt(-scrollmargin)) > 0} {
	    $_Wdgt(hmargin) config -width 1 -height \
		  [winfo pixels $_Wdgt(hmargin) \
		  $_Opt(-scrollmargin)]
	    
	    pack $_Wdgt(hmargin) -fill x -side bottom -padx 0 -pady 0
	 }
	 pack $_Wdgt(horizsb) -side left -fill x -expand yes
      }
      
      off {
	 set _hmode off
	 
	 pack forget $_Wdgt(hmargin)
	 pack forget $_Wdgt(horizsb)
      }
      
      default {
	 error "invalid argument \"$mode\": should be on or off"
      }
   }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _jumpPage -1|1
#
# Move one page forward or backward.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _jumpPage { {direction {1}} } {
   [$self set _Wdgt(text)] yview scroll $direction page
}


# ------------------------------------------------------------------
# PRIVATE METHOD: _scrollText wid first last
#
# Performs scrolling and display of scrollbars based on the first and
# last text views as well as the current -vscrollmode and -hscrollmode
# settings.
# ------------------------------------------------------------------
_UIT_Scrolledtext instproc _scrollText {wid first last} {
   $self instvar _Wdgt _Opt _vmode _hmode
   
   $wid set $first $last
   
   if {$wid == $_Wdgt(vertsb)} {
      if {$_Opt(-vscrollmode) == "dynamic"} {
	 if {($first == 0) && ($last == 1)} {
	    if {$_vmode != "off"} {
	       $self _vertScrollbarDisplay off
	    }
	    
	 } else {
	    if {$_vmode != "on"} {
	       $self _vertScrollbarDisplay on
	    }
	 }
      }
      
   } elseif {$wid == $_Wdgt(horizsb)} {
      if {$_Opt(-hscrollmode) == "dynamic"} {
	 if {($first == 0) && ($last == 1)} {
	    if {$_hmode != "off"} {
	       $self _horizScrollbarDisplay off
	    }
	    
	 } else {
	    if {$_hmode != "on"} {
	       $self _horizScrollbarDisplay on
	    }
	 }
      }
   }
}
