# 
# @DEC_COPYRIGHT@
#
# HISTORY
# $Log: scrolledlistbox.tcl,v $
# Revision 1.1.1.1  2003/01/23 18:34:41  ajay
# Initial submit to CVS.
#
#
# $EndLog$
# 
# @(#)$RCSfile: scrolledlistbox.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:34:41 $
# 
#
# _UIT_Scrolledlistbox
# ----------------------------------------------------------------------
# Implements a scrolled listbox with additional options to manage
# horizontal and vertical scrollbars.  This includes options to control
# which scrollbars are displayed and the method, i.e. statically,
# dynamically, or none at all.  
#
# WISH LIST:
#   This section lists possible future enhancements.  
#
#   1) Option to position scrollbars, i.e. vertical: left/right and
#      horizontal: top/bottom, taking the filler into account.
#
# ----------------------------------------------------------------------
#  AUTHOR: Mark L. Ulferts               EMAIL: mulferts@spd.dsccc.com
#
#  @(#) $Id: scrolledlistbox.tcl,v 1.1.1.1 2003/01/23 18:34:41 ajay Exp $
# ----------------------------------------------------------------------
#            Copyright (c) 1995 DSC Technologies Corporation
# ======================================================================
# Permission to use, copy, modify, distribute and license this software 
# and its documentation for any purpose, and without fee or written 
# agreement with DSC, is hereby granted, provided that the above copyright 
# notice appears in all copies and that both the copyright notice and 
# warranty disclaimer below appear in supporting documentation, and that 
# the names of DSC Technologies Corporation or DSC Communications 
# Corporation not be used in advertising or publicity pertaining to the 
# software without specific, written prior permission.
# 
# DSC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING 
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND NON-
# INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, AND THE
# AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, 
# SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO EVENT SHALL 
# DSC BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
# ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION,
# ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
# SOFTWARE.
# ======================================================================

#
# Default resources.
#
option add *Scrolledlistbox.borderWidth 2 widgetDefault
option add *Scrolledlistbox.relief sunken widgetDefault
option add *Scrolledlistbox.scrollMargin 1 widgetDefault
option add *Scrolledlistbox.vscrollMode static widgetDefault
option add *Scrolledlistbox.hscrollMode static widgetDefault
option add *Scrolledlistbox.selectmode browse widgetDefault
option add *Scrolledlistbox.width 0 widgetDefault
option add *Scrolledlistbox.height 0 widgetDefault
option add *Scrolledlistbox.visibleItems 20x10 widgetDefault
option add *Scrolledlistbox.labelPos n widgetDefault

#
# Usual options.
#
#itk::usual _UIT_Scrolledlistbox {
#    keep -activebackground -activerelief -background -borderwidth -cursor \
#	 -elementborderwidth -foreground -highlightcolor -highlightthickness \
#	 -jump -labelfont -selectbackground -selectborderwidth \
#	 -selectforeground -textbackground -textfont -troughcolor 
#}

# ------------------------------------------------------------------
#                          SCROLLEDLISTBOX
# ------------------------------------------------------------------
Class _UIT_Scrolledlistbox -superclass _UIT_Labeledwidget

_UIT_Scrolledlistbox instproc init {args} {

   $self _define -scrollmargin 1 Margin _scrollMargin
   $self _define -sbwidth 10 Width _sbWidth
   $self _define -vscrollmode static ScrollMode _vscrollMode
   $self _define -hscrollmode static ScrollMode _hscrollMode
   $self _define -items {} Items _items
   $self _define -dblclickcommand {} Command _dblClickCommand
   $self _define -selectioncommand {} Command _selectionCommand
   $self _define -width 0 Width _width
   $self _define -height 0 Height _height
   $self _define -visibleitems 20x10 VisibleItems _visibleItems

   $self set _vmode off              ;# Vertical scroll mode
   $self set _hmode off              ;# Vertical scroll mode
   $self set _selection {}           ;# Last selection indcies.

   $self set _initialized 0            ;# Initialization flag.
   
   eval {$self next} $args
   $self instvar _Wdgt _Opt

   $_Wdgt(hullcmd) configure -borderwidth 0
    
   #
   # Create some frames to hold both the top and bottom halfs of the 
   # widget.  The top will contain both the list and vertical scroll 
   # bar.  The bottom houses the horizontal scrollbar and some filler.
   #
   set _Wdgt(listframe) [frame $_Wdgt(interior).listframe]
   $self _keep $_Wdgt(listframe) -background -cursor

   pack $_Wdgt(listframe) -fill both -expand yes 
    
   $self set _Wdgt(bottomframe) [frame $_Wdgt(listframe).botttomframe]
   $self _keep $_Wdgt(listframe) -background -cursor

   pack $_Wdgt(bottomframe) -fill x -side bottom
    
   set _Wdgt(hmargin) [frame $_Wdgt(listframe).hmargin]
   $self _keep $_Wdgt(hmargin) -background -cursor
   
   pack $_Wdgt(hmargin) -side bottom
    
   set _Wdgt(topframe) [frame $_Wdgt(listframe).topframe]
   $self _keep $_Wdgt(topframe) -background -cursor
   
   pack $_Wdgt(topframe) -fill both -expand yes
    
   # 
   # Create the listbox.
   #
   set _Wdgt(listbox) [listbox $_Wdgt(topframe).listbox \
	 -width 1 -height 1 \
	 -xscrollcommand "$self _scrollList $_Wdgt(bottomframe).horizsb" \
	 -yscrollcommand "$self _scrollList $_Wdgt(topframe).vertsb"]
   $self _keep $_Wdgt(listbox) -borderwidth -cursor -exportselection \
	 -foreground -highlightcolor \
	 -highlightthickness -relief -selectbackground \
	 -selectborderwidth -selectforeground -selectmode
   
   $self _rename $_Wdgt(listbox) -font -textfont 
   $self _rename $_Wdgt(listbox) -background -textbackground 
   $self _rename $_Wdgt(listbox) -highlightbackground -background 
    
   #
   # Create the vertical margin
   #
   set _Wdgt(vmargin) [frame $_Wdgt(topframe).vmargin -width 0]
   $self _keep $_Wdgt(vmargin) -background -cursor
   
   # 
   # Create the vertical scroll bar.
   #
   set _Wdgt(vertsb) [scrollbar $_Wdgt(topframe).vertsb \
	 -orient vertical -command "$_Wdgt(listbox) yview"]
   $self _keep $_Wdgt(vertsb) -activebackground -activerelief -background \
	 -borderwidth -cursor -elementborderwidth \
	 -highlightcolor -jump -highlightthickness -relief \
	 -repeatdelay -repeatinterval -troughcolor
   $self _rename $_Wdgt(vertsb) -highlightbackground -background
   
   pack $_Wdgt(vertsb) -side right -fill y
   pack $_Wdgt(vmargin) -side right
   pack $_Wdgt(listbox) -fill both -expand yes -side left
    
   #
   # Next the horizontal scrollbar.
   #
   set _Wdgt(horizsb) [scrollbar $_Wdgt(bottomframe).horizsb \
	 -orient horizontal -command "$_Wdgt(listbox) xview"]
   $self _keep $_Wdgt(horizsb) -activebackground -activerelief -background \
	 -borderwidth -cursor -elementborderwidth \
	 -highlightcolor -jump -highlightthickness -relief \
	 -repeatdelay -repeatinterval -troughcolor
   $self _rename $_Wdgt(horizsb) -highlightbackground -background

   pack $_Wdgt(horizsb) -side left -fill x -expand yes
   
   #
   # Create the filler frame and compute its width.
   #
   $self set _Wdgt(filler) [frame $_Wdgt(bottomframe).filler \
	 -width [$self _fillerWidth]]
   $self _keep $_Wdgt(filler) -background -cursor

   pack $_Wdgt(filler) -side right 
    
   # 
   # Create a set of bindings for monitoring the selection.
   #
   
   bind SLBSelect$self <Control-Key-backslash> "$self _makeSelection"
   bind SLBSelect$self <Control-Key-slash> "$self _makeSelection"
   bind SLBSelect$self <Key-Escape> "$self _makeSelection"
   bind SLBSelect$self <Shift-Key-Select> "$self _makeSelection"
   bind SLBSelect$self <Control-Shift-Key-space> "$self _makeSelection"
   bind SLBSelect$self <Key-Select> "$self _makeSelection"
   bind SLBSelect$self <Key-space> "$self _makeSelection"
   bind SLBSelect$self <Control-Shift-Key-End> "$self _makeSelection"
   bind SLBSelect$self <Control-Key-End> "$self _makeSelection"
   bind SLBSelect$self <Control-Shift-Key-Home> "$self _makeSelection"
   bind SLBSelect$self <Control-Key-Home> "$self _makeSelection"
   bind SLBSelect$self <Shift-Key-Down> "$self _makeSelection"
   bind SLBSelect$self <Shift-Key-Up> "$self _makeSelection"
   bind SLBSelect$self <Control-Button-1> "$self _makeSelection"
   bind SLBSelect$self <Shift-Button-1> "$self _makeSelection"
   bind SLBSelect$self <ButtonRelease-1> "$self _makeSelection"
   bind SLBSelect$self <Double-1> "$self _dblclick"
   
   bindtags $_Wdgt(listbox) \
	 [linsert [bindtags $_Wdgt(listbox)] end SLBSelect$self]
   
   #
   # Explicitly handle configs that may have been ignored earlier.
   #
   if {![string compare _UIT_Scrolledlistbox [$self info class]]} {
      $self configure -borderwidth 2 -relief sunken -scrollmargin 1 \
	    -vscrollmode static -hscrollmode static -selectmode browse \
	    -width 0 -height 0 -visibleitems 20x10 -labelpos n \
	    -highlightthickness 2 -labeljustify left
      
      eval {$self configure} $args
   }
   
   $self set _initialized 1
   $self configure -scrollmargin $_Opt(-scrollmargin)
}

#
# Provide a lowercased access method for the _UIT_Scrolledlistbox class.
# 
proc scrolledlistbox {pathName args} {
   uplevel _UIT_Scrolledlistbox $pathName $args
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -scrollmargin
#
# Set the distance between the scrollbars and the list box.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _scrollMargin {val} {
   $self instvar _initialized _hmode _vmode _Wdgt _Opt
   
   if {$_initialized} {
      set _Opt(-scrollmargin) $val
      set pixels [winfo pixels $_Wdgt(hmargin) $val]
      
      if {$_hmode == "on"} {
	 $_Wdgt(hmargin) configure -width 1 -height $pixels
      }
      
      if {$_vmode == "on"} {
	 $_Wdgt(vmargin) configure -width $pixels -height 1
      }
      
      $_Wdgt(filler) configure -width [$self _fillerWidth] -height 1
   }
}

# ------------------------------------------------------------------
# OPTION: -sbwidth
#
# Set the width of the scrollbars.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _sbWidth {val} {
   $self instvar _Wdgt _initialized _Opt
   
   if {$_initialized} {
      set _Opt(-sbwidth) $val
      $_Wdgt(vertsb) configure -width $_Opt(-sbwidth)
      $_Wdgt(horizsb) configure -width $_Opt(-sbwidth)
      
      update idletasks
      
      $_Wdgt(filler) configure -width [$self _fillerWidth] -height 1
   }
}

# ------------------------------------------------------------------
# OPTION: -vscrollmode
#
# Enable/disable display and mode of veritcal scrollbars.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _vscrollMode {val} {
   $self instvar _Opt
   
   set _Opt(-vscrollmode) $val
   switch $_Opt(-vscrollmode) {
      static {
	 $self _vertScrollbarDisplay on
      }
      
      dynamic -
      none {
	 $self _vertScrollbarDisplay off
      }
      
      default {
	 error "bad vscrollmode option\
	       \"$_Opt(-vscrollmode)\": should be\
	       static, dynamic, or none"
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -hscrollmode
#
# Enable/disable display and mode of horizontal scrollbars.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _hscrollMode {val} {
   $self instvar _Opt
   
   switch [set _Opt(-hscrollmode) $val] {
      static {
	 $self _horizScrollbarDisplay on
      }
      
      dynamic -
      none {
	 $self _horizScrollbarDisplay off
      }
      
      default {
	 error "bad hscrollmode option\
	       \"$_Opt(-hscrollmode)\": should be\
	       static, dynamic, or none"
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -items
#
# Set/get the current list of items in the listbox.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _items {val} {
   $self instvar _Wdgt _Opt

   set index [$_Wdgt(listbox) nearest 0]

   $_Wdgt(listbox) delete 0 end

   eval {$_Wdgt(listbox) insert end} $val
#   foreach ent $_Opt(-items) {
#      $_Wdgt(listbox) insert end $ent
#   }
   
   if {$index < [$_Wdgt(listbox) size]} {
      $_Wdgt(listbox) yview $index
   }
}

# ------------------------------------------------------------------
# OPTION: -dblclickcommand
#
# Specify a command to be executed upon double click of a listbox 
# item.  Also, create a couple of bindings used for specific
# selection modes
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _dblClickCommand {val} {
   $self set _Opt(-dblclickcommand) $val
}

# ------------------------------------------------------------------
# OPTION: -selectioncommand
#
# Specifies a command to be executed upon selection of a listbox 
# item.  The command will be called upon each selection regardless 
# of selection mode..
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _selectionCommand {val} {
   $self set _Opt(-selectioncommand) $val
}

# ------------------------------------------------------------------
# OPTION: -width
#
# Specifies the width of the scrolled list box as an entire unit.
# The value may be specified in any of the forms acceptable to 
# Tk_GetPixels.  Any additional space needed to display the other
# components such as labels, margins, and scrollbars force the listbox
# to be compressed.  A value of zero along with the same value for 
# the height causes the value given for the visibleitems option 
# to be applied which administers geometry constraints in a different
# manner.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _width {val} {
   $self instvar _Wdgt _Opt _initialized
   
   if {$_Opt(-width) != 0} {
      pack propagate $_Wdgt(hull) no
      
      $_Wdgt(listbox) configure -width 1
      $_Wdgt(hullcmd) configure \
	    -width [winfo pixels $_Wdgt(hull) $_Opt(-width)] 
   } else {
      if {$_initialized} {
	 $self configure -visibleitems $_Opt(-visibleitems)
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -height
#
# Specifies the height of the scrolled list box as an entire unit.
# The value may be specified in any of the forms acceptable to 
# Tk_GetPixels.  Any additional space needed to display the other
# components such as labels, margins, and scrollbars force the listbox
# to be compressed.  A value of zero along with the same value for 
# the width causes the value given for the visibleitems option 
# to be applied which administers geometry constraints in a different
# manner.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _height {val} {
   $self instvar _Wdgt _Opt _initialized
   
   set _Opt(-height) $val
   if {$_Opt(-height) != 0} {
      pack propagate $_Wdgt(hull) no
      
      $_Wdgt(listbox) configure -height 1
      $_Wdgt(hullcmd) configure \
	    -height [winfo pixels $_Wdgt(hull) $_Opt(-height)] 
   } else {
      if {$_initialized} {
	 $self configure -visibleitems $_Opt(-visibleitems)
      }
   }
}

# ------------------------------------------------------------------
# OPTION: -visibleitems
#
# Specified the widthxheight in characters and lines for the listbox.
# This option is only administered if the width and height options
# are both set to zero, otherwise they take precedence.  With the
# visibleitems option engaged, geometry constraints are maintained
# only on the listbox.  The size of the other components such as 
# labels, margins, and scroll bars, are additive and independent, 
# effecting the overall size of the scrolled list box.  In contrast,
# should the width and height options have non zero values, they
# are applied to the scrolled list box as a whole.  The listbox 
# is compressed or expanded to maintain the geometry constraints.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _visibleItems {val} {
   $self instvar _Wdgt _Opt

   if {[regexp {^[0-9]+x[0-9]+$} $_Opt(-visibleitems)]} {
      if {($_Opt(-width) == 0) && ($_Opt(-height) == 0)} {
	 set chars [lindex [split $_Opt(-visibleitems) x] 0]
	 set lines [lindex [split $_Opt(-visibleitems) x] 1]
	 
	 pack propagate $_Wdgt(hull) yes
	 
	 $_Wdgt(listbox) configure \
	       -width $chars -height $lines
      }
      
   } else {
      error "bad visibleitems option\
	    \"$_Opt(-visibleitems)\": should be\
	    widthxheight"
   }
}

# ------------------------------------------------------------------
#                            METHODS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# METHOD: curselection 
#
# Returns a list containing the indices of all the elements in the 
# listbox that are currently selected.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc curselection {} {
   return [[$self set _Wdgt(listbox)] curselection]
}

# ------------------------------------------------------------------
# METHOD: activate index
#
# Sets the active element to the one indicated by index.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc activate {index} {
    return [[$self set _Wdgt(listbox)] activate [$self index $index]]
}

# ------------------------------------------------------------------
# METHOD: bbox index
#
# Returns four element list describing the bounding box for the list
# item at index
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc bbox {index} {
    return [[$self set _Wdgt(listbox)] bbox [$self index $index]]
}

# ------------------------------------------------------------------
# METHOD clear 
#
# Clear the listbox area of all items.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc clear {} {
   $self delete 0 end
}

# ------------------------------------------------------------------
# METHOD: see index
#
# Adjusts the view such that the element given by index is visible.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc see {index} {
    [$self set _Wdgt(listbox)] see [$self index $index]
}

# ------------------------------------------------------------------
# METHOD: index index
#
# Returns the decimal string giving the integer index corresponding 
# to index.  The index value may be a integer number, active,
# anchor, end, @x,y, or a pattern.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc index {index} {
   $self instvar _Opt _Wdgt
   
   if {[regexp \
	 {(^[0-9]+$)|(active)|(anchor)|(end)|(^@[0-9]+,[0-9]+$)} $index]} {
      return [$_Wdgt(listbox) index $index]
      
   } else {
      set indexValue [lsearch -glob $_Opt(-items) $index]
      
      if {$indexValue == -1} {
	 error "bad _UIT_Scrolledlistbox index \"$index\": must be\
	       active, anchor, end, @x,y, number, or a pattern"
      }
      
      return $indexValue
   }
}

# ------------------------------------------------------------------
# METHOD: delete first ?last?
#
# Delete one or more elements from list box based on the first and 
# last index values.  Indexes may be a number, active, anchor, end,
# @x,y, or a pattern.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc delete {first {last {}}} {
   $self instvar _Opt

   set first [$self index $first]
   
   if {$last != {}} {
      set last [$self index $last]
   } else {
      set last $first
   }
    
   if {[llength $_Opt(-items)] > 0} {
      if {$first <= $last} {
	 $self configure -items [lreplace $_Opt(-items) $first $last]
      } else {
	 error "first index must not be greater than second"
      }
   }
}

# ------------------------------------------------------------------
# METHOD: get first ?last?
#
# Returns the elements of the listbox indicated by the indexes. 
# Indexes may be a number, active, anchor, end, @x,y, ora pattern.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc get {first {last {}}} {
   $self instvar _Wdgt
   set first [$self index $first]
   
   if {$last != {}} {
      set last [$self index $last]
   }
    
   if {$last == {}} {
      return [$_Wdgt(listbox) get $first]
   } else {
      return [$_Wdgt(listbox) get $first $last]
   }
}

# ------------------------------------------------------------------
# METHOD: getcurselection 
#
# Returns the contents of the listbox element indicated by the current 
# selection indexes.  Short cut version of get and curselection 
# command combination.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc getcurselection {} {
   $self instvar _Wdgt
   set rlist {}
   
   if {[$self selecteditemcount] > 0} {
      set cursels [$_Wdgt(listbox) curselection]
 
      foreach sel $cursels {
	 lappend rlist [$_Wdgt(listbox) get $sel]
      }
   }
   return $rlist
}

# ------------------------------------------------------------------
# METHOD: insert index string ?string ...?
#
# Insert zero or more elements in the list just before the element 
# given by index.  Modify the $_Opt(-items) value as well.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc insert {index string args} {
   $self instvar _Opt _Wdgt

   set index [$self index $index]
   
   if {[catch {eval {$_Wdgt(listbox) insert $index $string} $args} \
	 result] == 0} {
      
      set _Opt(-items) [eval {linsert $_Opt(-items) $index $string} $args]
      
   } else {
      return $result
   }
}

# ------------------------------------------------------------------
# METHOD: nearest y
#
# Given a y-coordinate within the listbox, this command returns the 
# index of the visible listbox element nearest to that y-coordinate.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc nearest {y} {
    [$self set _Wdgt(listbox)] nearest $y
}

# ------------------------------------------------------------------
# METHOD: scan option args 
#
# Implements scanning on listboxes.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc scan {option args} {
    eval [$self set _Wdgt(listbox)] scan $option $args
}

# ------------------------------------------------------------------
# METHOD: selection option first ?last?
#
# Adjusts the selection within the listbox.  The index value may be 
# a integer number, active, anchor, end, @x,y, or a pattern.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc selection {option first {last {}}} {
   $self instvar _Wdgt

   set first [$self index $first]
   
   if {$last != {}} {
      set last [$self index $last]
   } 
   
   switch $option {
      anchor {
	 $_Wdgt(listbox) selection anchor $first
      }
      
      clear {
	 if {$last == {}} {
	    $_Wdgt(listbox) selection clear $first 
	 } else {
	    $_Wdgt(listbox) selection clear $first $last
	 }
      }
      
      includes {
	 $_Wdgt(listbox) selection includes $first
      }
      
      set {
	 if {$last == {}} {
	    $_Wdgt(listbox) selection set $first 
	 } else {
	    $_Wdgt(listbox) selection set $first $last
	 }
      }
   }
}

# ------------------------------------------------------------------
# METHOD: size 
#
# Returns a decimal string indicating the total number of elements 
# in the listbox.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc size {} {
   return [[$self set _Wdgt(listbox)] size]
}

# ------------------------------------------------------------------
# METHOD: selecteditemcount 
#
# Returns a decimal string indicating the total number of selected 
# elements in the listbox.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc selecteditemcount {} {
   return [llength [[$self set _Wdgt(listbox)] curselection]]
}

# ------------------------------------------------------------------
# METHOD: justify direction
#
# Justifies the list scrolled region in one of four directions: top,
# bottom, left, or right.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc justify {direction} {
   $self instvar _Wdgt

   switch $direction {
      left { 
	 $_Wdgt(listbox) xview moveto 0
      }
      right {
	 $_Wdgt(listbox) xview moveto 1
      }
      top {
	 $_Wdgt(listbox) yview moveto 0
      }
      bottom {
	 $_Wdgt(listbox) yview moveto 1
      }
      default {
	 error "bad justify argument \"$direction\": should\
	       be left, right, top, or bottom"
      }
   }
}

# ------------------------------------------------------------------
# METHOD: sort mode
#
# Sort the current list in either "ascending/increasing" or 
# "descending/decreasing" order.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc sort {mode} {
   $self instvar _Opt
   
   switch $mode {
      ascending -
      increasing {
	 $self configure -items [lsort -increasing $_Opt(-items)]
      }
      descending -
      decreasing {
	 $self configure -items [lsort -decreasing $_Opt(-items)]
      }
      default {
	 error "bad sort argument \"$mode\": should be\
	       ascending, descending, increasing, or decreasing"
      }
   }
}

# ------------------------------------------------------------------
# METHOD: xview args
#
# Change or query the vertical position of the text in the list box.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc xview {args} {
   return [eval [$self set _Wdgt(listbox)] xview $args]
}

# ------------------------------------------------------------------
# METHOD: yview args
#
# Change or query the horizontal position of the text in the list box.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc yview {args} {
   return [eval [$self set _Wdgt(listbox)] yview $args]
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _fillerWidth 
#
# Compute the width of the filler frame based on the vertical 
# scrollbar width plus the margin.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _fillerWidth {} {
   $self instvar _Wdgt _vmode
   if {$_vmode == "on"} {
      return [expr [winfo reqwidth $_Wdgt(vertsb)] + \
	    [winfo reqwidth $_Wdgt(vmargin)]]
   } else {
      return 1
   }
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _makeSelection 
#
# Evaluate the selection command.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _makeSelection {} {
   uplevel #0 [$self set _Opt(-selectioncommand)]
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _dblclick 
#
# Evaluate the double click command option if not empty.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _dblclick {} {
   uplevel #0 [$self set _Opt(-dblclickcommand)]
}	

# ------------------------------------------------------------------
# PRIVATE METHOD: _vertScrollbarDisplay mode
#
# Displays the vertical scrollbar based on the input mode.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _vertScrollbarDisplay {mode} {
   $self instvar _Wdgt _Opt _vmode
   
   switch $mode  {
      on {
	 set _vmode on
	 
	 $_Wdgt(vmargin) config -height 1 -width \
	       [winfo pixels $_Wdgt(vmargin) \
	       $_Opt(-scrollmargin)]
	 
	 pack forget $_Wdgt(listbox)
	 pack $_Wdgt(vertsb) -side right -fill y
	 pack $_Wdgt(vmargin) -side right
	 pack $_Wdgt(listbox) -fill both -expand yes -side left 
	 
	 update idletasks
	 
	 $_Wdgt(filler) config -width [$self _fillerWidth] -height 1
      }
      
      off {
	 set _vmode off
	 
	 pack forget $_Wdgt(vertsb)
	 
	 $_Wdgt(vmargin) config -width 1 -height 1
	 $_Wdgt(filler) config -width 1 -height 1
      }
      
      default {
	 error "invalid argument \"$mode\": should be on or off"
      }
   }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _horizScrollbarDisplay mode
#
# Displays the horizontal scrollbar based on the input mode.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _horizScrollbarDisplay {mode} {
   $self instvar _Wdgt _hmode _Opt
   
   switch $mode  {
      on {
	 set _hmode on
	 
	 $_Wdgt(hmargin) config -width 1 -height \
	       [winfo pixels $_Wdgt(hmargin) \
	       $_Opt(-scrollmargin)]
	 
	 pack $_Wdgt(horizsb) -side left -fill x -expand yes
	 
	 update idletasks
	 
	 $_Wdgt(filler) config -width [$self _fillerWidth] -height 1
      }
      
      off {
	 set _hmode off
	 
	 pack forget $_Wdgt(horizsb)
	 
	 $_Wdgt(hmargin) config -width 1 -height 1
	 $_Wdgt(filler) config -width 1 -height 1
      }
      
      default {
	 error "invalid argument \"$mode\": should be on or off"
      }
   }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _scrollList wid first last
#
# Performs scrolling and display of scrollbars based on the first and
# last listbox views as well as the current -vscrollmode and -hscrollmode
# settings.
# ------------------------------------------------------------------
_UIT_Scrolledlistbox instproc _scrollList {wid first last} {
   $self instvar _Wdgt _Opt _hmode _vmode
   
   $wid set $first $last
   
   if {$wid == $_Wdgt(vertsb)} {
      if {$_Opt(-vscrollmode) == "dynamic"} {
	 if {($first == 0) && ($last == 1)} {
	    if {$_vmode != "off"} {
	       $self _vertScrollbarDisplay off
	    }
	    
	 } else {
	    if {$_vmode != "on"} {
	       $self _vertScrollbarDisplay on
	    }
	 }
      }
      
   } elseif {$wid == $_Wdgt(horizsb)} {
      if {$_Opt(-hscrollmode) == "dynamic"} {
	 if {($first == 0) && ($last == 1)} {
	    if {$_hmode != "off"} {
	       $self _horizScrollbarDisplay off
	    }
	    
	 } else {
	    if {$_hmode != "on"} {
	       $self _horizScrollbarDisplay on
	    }
	 }
      }
   }
}

_UIT_Scrolledlistbox instproc constrainSize {{true {1}}} {
   $self instvar _initialized _Opt _Wdgt

   if {$true} {
      pack propagate $_Wdgt(hull) no
      
      $_Wdgt(listbox) configure -width 1 -height 1
      $_Wdgt(hullcmd) configure \
	    -height [winfo pixels $_Wdgt(hull) $_Opt(-height)] \
	    -width [winfo pixels $_Wdgt(hull) $_Opt(-width)] 
   } else {
      if {$_initialized} {
	 $self configure -visibleitems $_Opt(-visibleitems)
      }
   }
}
