# 
# @DEC_COPYRIGHT@
#
# HISTORY
# $Log: table.tcl,v $
# Revision 1.1.1.1  2003/01/23 18:34:39  ajay
# Initial submit to CVS.
#
#
# Revision 1.1.11.2  2000/11/08  18:05:47  Richard_Taft
# 	Sometimes id is non-numeric on delete.
#
# Revision 1.1.2.27  2000/11/07  18:36:15  Richard_Taft
# 	Sometimes id is non-numeric on delete.
#
# Revision 1.1.2.26  2000/04/14  13:16:58  Richard_Taft
# 	Fixed delete proc so recID gets updated
#
# Revision 1.1.2.25  1999/12/15  14:33:47  Todd_Moyer
# 	Added optional 'filter' arg to 'read' method.
#
# Revision 1.1.2.24  1999/11/02  19:11:56  Omar_Ortiz
# 	Turned on the wait flag for the InfoMsg in case of errors
#
# Revision 1.1.2.23  1999/07/23  19:11:22  Todd_Moyer
# 	Replaced comparison operator with cequal.
# 	  It had problems with "if {e1 != e2} ..."
# 	  because e# interpreted as 0 * 10 ^ #,
# 	  which equals zero for any #.
# 	[1999/07/23  19:10:08  Todd_Moyer]
#
# Revision 1.1.2.22  1998/11/30  21:52:11  Todd_Moyer
# 	Allow for manual override on error reporting.
# 	[1998/11/30  21:51:14  Todd_Moyer]
# 
# Revision 1.1.2.21  1998/07/14  22:01:51  Todd_Moyer
# 	Have buffer method accept 'reset' to discard uncommitted changes.
# 	[1998/07/14  21:43:48  Todd_Moyer]
# 
# Revision 1.1.2.20  1998/06/26  21:41:10  Todd_Moyer
# 	_dump method now dumps rec's values too.
# 	[1998/06/26  20:56:14  Todd_Moyer]
# 
# Revision 1.1.2.19  1998/03/12  19:37:10  Todd_Moyer
# 	Fixed problems in addAt
# 	[1998/03/12  19:34:13  Todd_Moyer]
# 
# Revision 1.1.2.18  1998/03/12  15:59:50  Todd_Moyer
# 	Needed to declare some instvar for addAt
# 	[1998/03/12  15:56:56  Todd_Moyer]
# 
# Revision 1.1.2.17  1998/01/22  21:38:09  Todd_Moyer
# 	Validate method now takes operation as an
# 	argument instead of trying to guess.
# 	Added comments that design doc can parse.
# 	[1998/01/22  21:37:06  Todd_Moyer]
# 
# Revision 1.1.2.16  1998/01/21  19:48:06  Todd_Moyer
# 	Fixed validate method for adding a new rec.
# 	[1998/01/21  19:46:21  Todd_Moyer]
# 
# Revision 1.1.2.15  1997/12/01  21:23:45  Todd_Moyer
# 	Fixed typo in delete ($msg -> $stat).
# 	[1997/12/01  21:22:39  Todd_Moyer]
# 
# Revision 1.1.2.14  1997/08/19  18:33:26  Todd_Moyer
# 	Don't report read, add, modify or delete as system errors.
# 	Added validate method.
# 	[1997/08/19  18:15:59  Todd_Moyer]
# 
# Revision 1.1.2.13  1997/04/03  13:22:25  Todd_Moyer
# 	Put out error as a string, not a list.
# 	[1997/04/03  13:20:39  Todd_Moyer]
# 
# Revision 1.1.2.12  1997/03/31  15:50:03  Todd_Moyer
# 	_dump now puts records in ascii order.
# 	[1997/03/28  19:38:44  Todd_Moyer]
# 
# 	Added indent arg to dump and _dump methods.
# 	[1997/03/28  18:45:34  Todd_Moyer]
# 
# Revision 1.1.2.11  1997/03/26  19:56:14  Todd_Moyer
# 	Added validation to add, addAt, modify and delete methods.
# 	Return a set of error codes or "" if OK.
# 	Added a parameter for add and addAt to receive new record ID.
# 	Added getRecID method.
# 	Deleted setRec method.  Now use buffer, setVal and modify.
# 	[1997/03/26  19:25:44  Todd_Moyer]
# 
# Revision 1.1.2.10  1997/02/07  16:44:23  Deepa_Bachu
# 	Added missin comment leader "#" in delete method.
# 	[1997/02/07  16:43:36  Deepa_Bachu]
# 
# Revision 1.1.2.9  1997/02/07  16:18:20  Todd_Moyer
# 	Clear out the rec buf when doing a delete.
# 	[1997/02/07  16:17:05  Todd_Moyer]
# 
# Revision 1.1.2.8  1997/02/04  15:19:01  Todd_Moyer
# 	Added _dump method to show detailed debug info.
# 	[1997/02/04  15:17:41  Todd_Moyer]
# 
# Revision 1.1.2.7  1997/01/31  16:42:30  Todd_Moyer
# 	Rec ID in getSaveRecID needs to be a list.
# 	  Also, changed _UIT_Table in init to Table.
# 	[1997/01/31  16:40:17  Todd_Moyer]
# 
# Revision 1.1.2.6  1997/01/30  22:48:11  Todd_Moyer
# 	Encapsulated getNumRecID and getDataRecID within getSaveRecID and
# 	  getRestoreRecID.
# 	[1997/01/30  22:46:28  Todd_Moyer]
# 
# Revision 1.1.2.5  1997/01/30  21:18:03  Todd_Moyer
# 	Add getTempID and getPermID from super class (now called getNumRecID
# 	  and getDataRecID).
# 	[1997/01/30  20:38:27  Todd_Moyer]
# 
# Revision 1.1.2.4  1997/01/16  14:15:52  William_Athanasiou
# 	Prepend _UIT_ to all classes and globals
# 	[1997/01/15  21:14:52  William_Athanasiou]
# 
# Revision 1.1.2.3  1997/01/06  22:14:30  Todd_Moyer
# 	Store attribute names once instead of for each record.
# 	[1997/01/06  22:10:30  Todd_Moyer]
# 
# Revision 1.1.2.2  1996/11/27  23:22:55  Todd_Moyer
# 	Created.
# 	[1996/11/27  23:02:29  Todd_Moyer]
# 
# $EndLog$
# 
# @(#)$RCSfile: table.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:34:39 $
# 



#@ A table holds a record of default values (indexed by "default") and
#@   zero or more records in array-get format indexed by a unique ID.  
#@   It also contains an ordered list of the IDs and
#@   a buffer for the one record that is being worked on at any given time.


Class _UIT_Table -superclass _UIT_Rsrc

# ---------------------------------------------------

_UIT_Table instproc init {dataID} {
    eval $self next Table $dataID
    $self set _nextId 1
    $self set _numRecs 0
    $self set _order {}
}


# ===================================================
# public methods.

# ---------------------------------------------------
#@ Add the new record before element "targetId" or at the end if no target
#@ given.  Put the new ID in newIdPtr.
#@ Return "" if OK, or a list of error codes otherwise.

_UIT_Table instproc add {newIdPtr  {targetId end} } {
    upvar $newIdPtr newId
    $self instvar _data _order

    if {[cequal $targetId end]} {
	set orderIdx   $targetId
	set nextRec    {}
    } else {
	set orderIdx   [lsearch -exact $_order $targetId]
	if {$orderIdx < 0} {
	    error "_UIT_Table::add  couldn't find targetID $targetId in add."
	}
	set nextRec    $_data($targetId)
    }

    return [$self _add newId nextRec $orderIdx]
}



# ---------------------------------------------------
#@ Add the new record before index "targetIdx" (starting at zero).
#@ Put the new ID in newIdPtr.
#@ Return "" if OK, or a list of error codes otherwise.


_UIT_Table instproc addAt {newIdPtr targetIdx} {
    upvar $newIdPtr newId
    $self instvar _data _order

    if {[cequal $targetIdx end]} {
	set nextRec    {}
    } else {
	set nextRecID  [lindex $_order $targetIdx]
	set nextRec    $_data($nextRecID)
    }

    return [$self _add newId nextRec $targetIdx]
}



# ---------------------------------------------------
#@ Set the record buffer to the data from the specified record.
#@ If the record is not specified, it sets the default record values.
#@ If it is specified, it must be a valid ID, "default" or "reset".
#@ "reset" re-reads the current record, effectively discarding
#@ uncommitted changes.

_UIT_Table instproc buffer { {id "default"} } {
    $self instvar _recBuf

    if {[cequal $id "reset"]} {
	set id [$_recBuf getRecID]
    }
    $_recBuf read $id
}


# ---------------------------------------------------
#@ Delete the specified record from this copy of the data.
#@ Return the status: "" if OK or a list of error codes if not.

_UIT_Table instproc delete {delID} {
    $self instvar _data _dispError _order _numRecs _recBuf

    # validate that deleting record is OK
    set newRec ""
    set stat [$self _validateDL newRec $delID]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
	return $stat
    }

    # look for the named record
    set idx [lsearch -exact $_order $delID]
    if {$idx < 0} {
	set    msg "_UIT_Table::delete couldn't find targetID $delID.  "
	append msg "Record not deleted."
	error $msg
    }

    # delete record in data layer
    set stat [$self _deleteDL $delID]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
	return $stat
    }

    # reset the rec buffer to the default incase the current rec is
    # the one being deleted.  This is needed because in the web the
    # rec buf is being saved and restored.
    # $self buffer

    # delete the record from this copy
    unset _data($delID)
    set _order [lreplace $_order $idx $idx]
    incr _numRecs -1

    # make sure that we no longer point to the deleted record
    set id [$_recBuf getRecID]
    if { [ctype digit $id] } {
	incr id -1
	if { $id < 0 } {
	    set id 0
	}
	$_recBuf setRecID $id
    }
    return ""
}


# ---------------------------------------------------
#@ Print the table contents on stdout for debugging.
 
_UIT_Table instproc dump {{indent ""}} {
    puts "\n$indent$self"
    foreach id [$self set _order] {
	puts "$indent  $id [$self set _data($id)]"
    }
    puts ""
    $self next
}


# ---------------------------------------------------
#@ Find the first record with attribute values matching those in the target and
#@   return its ID.  "tags" indicates which attributes are to be matched and
#@   their order to make the search faster.
#@   Return the ID of the first record that matches, or -1 if none.
# @@@ may be inefficient!

_UIT_Table instproc findRec {tags targetArrayName} {
    upvar 1 $targetArrayName target

    foreach id [$self set _order] {
	$self getRec $id data
	set found 1
	foreach tag $tags {
	    if { ! [cequal $data($tag) $target($tag)]} {
		set found 0
		break
	    }
	}
	if {$found} {
	    return $id
	}
    }
    return -1
}


# ---------------------------------------------------
#@ Get the ordered list of record IDs.

_UIT_Table instproc getIDs {listName} {
    upvar 1 $listName dest
    set dest [$self set _order]
    set listName
}


# ---------------------------------------------------
#@ Load the specified record into the specified array.
#@ Used by _UIT_Rec to read its values and lists to format
#@ display strings.

_UIT_Table instproc getRec {id arrName} {
    upvar 1 $arrName arr
   
   set vals [$self set _data($id)]
   foreach att [$self set _atts] {
      set arr($att) [lvarpop vals]
   }
}


# ---------------------------------------------------
#@ Get the ID of the current record.

_UIT_Table instproc getRecID {} {
    [$self set _recBuf] getRecID
}


# ---------------------------------------------------
#@ Given a "data" record ID (handle), return that record's "numeric" ID.
#@ See getSaveRecID for an explaination of data and numeric record IDs.

_UIT_Table instproc getRestoreRecID {savedLst} {
    $self instvar _data
    
    set convert [lindex $savedLst 0]
    set dataID      [lindex $savedLst 1]

    # If ID is a data one, convert it to a numeric one.  (numerics are
    # more efficient to use.)
    if {$convert} {
	foreach id [$self set _order] {
	    if {[cequal $dataID $_data($id)]} {
		return $id
	    }
	}
	$self dump
	error "_UIT_Table::getRestoreRecID cannt find ID |$dataID|"
    }

    # Otherwise, assume ID is a literal.
    set dataID
}


# ---------------------------------------------------
#@ Given a "numeric" record ID (handle), return that record's "data" ID.

#@ A numeric record ID is a short handle that will not change during the
#@ execution of a process even if the record content changes (as long as the
#@ record isn't deleted).  This is currently just an integer assigned at
#@ run-time.
#@ A data record ID will be the same no matter which process asks for it
#@ as long as the data doesn't change.  This is currently the whole record.
#@ Numeric IDs are useful because they're short and fast and don't change
#@ even if the record data changes.  Data IDs are useful for CGI scripts
#@ that have to pass data between two processes and for data
#@ layers that use their own indexing methods.

_UIT_Table instproc getSaveRecID {numID} {
    $self instvar _data

    # If the record ID is numeric it is assumed to be a numeric ID that
    # must be converted into a data one.
    # This is needed because when a restore is done it may not have
    # the same numeric ID.  (They may be on different calls to a CGI script.)
    if {[ctype digit $numID]} {
	# note that ID needs to be converted and convert it
	set     lst 1
	if {! [info exists _data($numID)]} {
	    $self dump
	    error "_UIT_Table::getSaveRecID cannt find ID |$numID|"
	}
	lappend lst $_data($numID)

    # Otherwise, assume ID is a literal.
    } else {
	set     lst 0
	lappend lst $numID
    }

    list $lst
}


# ---------------------------------------------------
#@ Modify the record.
#@   Validate the recBuf values, and write them into this object.
#@   If validation fails, display an error msg.
#@ Return the status: "" if OK or a list of error codes if not.

_UIT_Table instproc modify {} {
    $self instvar _data _dispError _recBuf
    
    # get record data into list format
    array set data [$_recBuf getVals]
    $self _cvtArr2Lst data newRec
    set recID [$_recBuf getRecID]

    # validate record
    set stat [$self _validateDL newRec $recID]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
	return $stat
    }

    # modify record in data layer
    set stat [$self _modifyDL newRec $recID]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
	return $stat
    }

    # if OK, overwrite resource record
    set _data($recID) $newRec
    
    return ""
}



# ---------------------------------------------------
#@ Get the number of records.

_UIT_Table instproc numRecs {} {
    $self set _numRecs
}


# ---------------------------------------------------
#@ Validate the whether doing the specified operation to the current record
#@ is OK, and if it not, display an error msg.
#@ "oper" must be "add" "mod" or "del"
#@ Return the status: "" if OK or an error msg if not.

_UIT_Table instproc validate {oper} {
    $self instvar _data _dispError _recBuf

    # get record data into list format
    if {[cequal $oper del]} {
	set newRec ""
    } else {
	array set data [$_recBuf getVals]
	$self _cvtArr2Lst data newRec
    }

    # get recID
    set recID [$_recBuf getRecID]
    if {[cequal $oper add]} {
	set recID ""
    }

    # validate record in data layer
    set stat [$self _validateDL newRec $recID]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
    }

    return $stat
}


# ===================================================
# private methods.

#@ Validate and add the record from the record buffer.
#@ Put the ID for the new record in the recBuf and the newRecPtr.
#@ Return a list of error codes from the validation, or "" if OK.

_UIT_Table instproc _add {newIdRef nextRecRef newOrderIdx} {
    upvar $newIdRef    newId
    upvar $nextRecRef  nextRec
    $self instvar _data _dispError _nextId _numRecs _order _recBuf
    
    # get record data into list format
    array set data [$_recBuf getVals]
    $self _cvtArr2Lst data newRec

    # validate record
    set stat [$self _validateDL newRec {}]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
	return $stat
    }

    set stat [$self _addDL newRec nextRec $newOrderIdx]
    if {! [cequal $stat ""]} {    
	if {$_dispError} {
	    InfoMsg error $stat "" 1
	}
	return $stat
    }

    # get new ID, increment number of recs, add new record
    set      newId         $_nextId
    incr     _nextId
    incr     _numRecs
    $_recBuf setRecID      $newId
    set      _data($newId) $newRec
    set      _order        [linsert $_order $newOrderIdx $newId]
    return ""
}


# ---------------------------------------------------
#@ print the details of the table for debugging

_UIT_Table instproc _dump {{indent ""}} {
    $self instvar _data

    puts "\n$indent$self"
    puts "$indent  _nextId  [$self set _nextId]"
    puts "$indent  _numRecs [$self set _numRecs]"
    puts "$indent  _order   [$self set _order]"
    puts "$indent  _atts    [$self set _atts]"
    foreach id [lsort [array names _data]] {
	puts "$indent  $id $_data($id)"
    }
    puts "$indent  _recBuf  [$self set _recBuf]"
    [$self set _recBuf] _dump $indent
}



# ====================== pure virtual methods =============================
# These methods are not defined in this abstract class, but must be in the
# subclasses that will actually be instantiated.

# ---------------------------------------------------
#@ Read the data from the data layer and expand the record into the buffer.
#@ Assumes there is exactly one record.

_UIT_Table instproc read { {filter ""} } {
    pureVirtual _UIT_Table read
}


# ---------------------------------------------------
#@ Does the data layer-specific work for adding.

_UIT_Table instproc _addDL {newRecRef nextRecRef newOrderIdx} {
    pureVirtual _UIT_Table _addDL
}


# ---------------------------------------------------
#@ Does the data layer-specific work for deleting.

_UIT_Table instproc _deleteDL {recID} {
    pureVirtual _UIT_Table _deleteDL
}


# ---------------------------------------------------
#@ Does the data layer-specific work for modify.

_UIT_Table instproc _modifyDL {newRecRef recID} {
    pureVirtual _UIT_Table _modifyDL
}


# ---------------------------------------------------
#@ Does the data layer-specific work for validation.
#@ For an add the recID is ""  For a delete the newRec contains ""
#@ For a modify both contain data.

_UIT_Table instproc _validateDL {newRecRef recID} {
    pureVirtual _UIT_Table _validateDL
}


# ===================================================
