#!/usr/share/sysman/bin/sysmansh
# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: catgets.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:30 $
# 

#---------------------------------------------------------------#
#
#  Manage raw message catalogs.  
#
#  Overview:
#
#  _Catalog:  Defines a class for referencing catalog objects
#  _Catfile:  Defines a class for containing catalog information
#	
#---------------------------------------------------------------#

Class _Catalog

#---------------------------------------------------------------#
#
#  _Catalog:
#
#  init - Initialize the object and optionally open a catalog
#          file if supplied when the object is created
#  catopen - Open a catalog.  If a full path is given for a
#            catalog file that file is used directly, otherwise
#            an NLSPATH search is performed according to the
#            rules defined by catopen(3).
#  catgets - Retrieve a message using symbolic identifier using
#            the catalog
#
#  exists - Check for the existance of a symbol in the associated
#            message catalog
#
#---------------------------------------------------------------#


_Catalog instproc init {{catname {}} args} {
    $self instvar catid catid_alt dstringflag

    set catid "_Catalog_default"
    set catid_alt {}
    set dstringflag $args
    
    if {$catname != {}} {
        if {[$self catopen $catname] == {}} {
#            error "couldn't open $catname"
        }
    }
}

_Catalog instproc catopen {catalog} {
    $self instvar catid catname dstringflag

    set catname $catalog
    
    if {[_Catfile info instances _msgcat_$catname] ==  "_msgcat_$catname"} {
   	set catid "_msgcat_$catname"
        return $catid
    }

    set catid [_Catfile _msgcat_$catname $catname $dstringflag]
    return $catid
}


#---------------------------------------------------------------#
#
#  catgets symbol args
#
#  Lookup symbol in the currently open message catalog.  If
#   for some reason the symbol doesn't exist in the open
#   catalog then open the english message catalog and lookup
#   the message there.  This situation can occur if the
#   english catalog is newer than a translated catalog and
#   it contains messages not present in the translated catalog.
#
#   Args are the optional arguments which are substituted into parameters
#   in the message string (i.e. $1%s $2%s etc). 
#
#   If we can't find the symbol in the standard english catalog then
#    just return the symbol value.
#
#---------------------------------------------------------------#

_Catalog instproc catgets {symbol args} {
    $self instvar catid catname catid_alt
   
    if {[$catid exists $symbol]} {		;# If msg. id is in the catalog
	set msg [$catid catgets_nocheck $symbol]

	if { [cequal $args {}] == 0 } {	;# If there are message parameters

	   # Then format them into the message
	   set msg [eval {format "$msg"}  $args]
	}

      return $msg
    }

    if {$catid_alt == {}} {
       set catid_alt [_Catfile _msgcat_default_$catname \
	    $catname en_US.ISO8859-1]
    }

    set msg [$catid_alt catgets $symbol]
    if { $args != "" } {	;# If there are message parameters

       # Then format them into the message
       set msg [eval {format "$msg"}  $args]
    }



    return [$catid_alt catgets $symbol]
}


#---------------------------------------------------------------#
#
#   Passthru function to get the catfile numerical id for
#    use with Helpfiles.
#
#---------------------------------------------------------------#

_Catalog instproc catgetid {symbol} {
   [$self set catid] catgetid $symbol
}

proc _Catalog_default {operation args} {
	return $args
}

#---------------------------------------------------------------#
#
#   Check if a given symbol exists in the message catalog.
#   Returns {} if the symbol doesn't exist, otherwise it returns the
#   symbol.  This is used to detect the difference between a
#   null message and a non-existant symbol
#
#---------------------------------------------------------------#

_Catalog instproc exists {symbol} {
    return [[$self set catid] exists $symbol]
}

_Catalog instproc catsets {symbol text} {
    [$self set catid] catsets $symbol $text
}

#---------------------------------------------------------------#
#	
#  The _Catfile Class defines an instance of a catalog.  Each
#   instance holds all of the messages and default strings
#   associated with each symbolic identifier.  This class is
#   private and should only be referenced by _Catalog objects.
#
#  init - Initialize the object and optionally load the
#          catalog if the object is created/called with arguments
#
#  _catopenfile - Private routine to open a catalog file using
#             the catopen(3) rules.
#
#  loadcatalog - Open the catalog and read through the raw
#                message file.  Each symbolic name and associated
#                string is read into an internal array for later
#                access.
#  catgets - Retrieve the message based on the symbolic identifier
#
#  exists  - Test if a symbolic identifier was defined.
#
#---------------------------------------------------------------#

Class _Catfile

_Catfile instproc init {catname {lang {}} args} {
    $self instvar _msgNum _catname 

    set _catname $catname

    if {$catname != {}} {
        set catfile [$self _catopenfile $catname $lang]
        if {$catfile != {}} {
            if {[$self loadcatalog $catfile] == {}} {
                error "'$catfile' not found"
            }

#---------------------------------------------------------------#
#
#   Ok, now we need to get the dstrings loaded in case we need
#    them.  Default strings are needed in case we have a skew
#    between the english version of a message catalog and the
#    internationalized versions, this way, if a message doesn't
#    exist in the internationalized catalog we will simply print
#    the default english string.  This can happen when we ship
#    new english catalogs but don't have the translations complete.
#
#    If there are no dstr tcl files then this is a no-op.
#
#---------------------------------------------------------------#

	    if {$args != {}} {
	        catch { ${_catname}_dstrInit } junk
	    }
        }
    }
}


#---------------------------------------------------------------#
#
#  Open a catalog file using NLSPATH if necessary
#  returns file name if successful, {} if not.  This follows
#  the same conventions as catopen(3) C library call.
#
#---------------------------------------------------------------#

_Catfile instproc _catopenfile {catname {lc_message {}}} {
    global env SysmanDir
  
    if {[cindex $catname 0] == "/"} {
        return $catname
    }

    if {$lc_message == {}} {
        if {[info exists env(LANG)] == 1} {
            set lc_message $env(LANG)
        } else {
            set lc_message {en_US.ISO8859-1}
        }
    }
    
    set setuid [expr [id effective userid] != [id userid]]

    if {$setuid == 0} {
        if {[info exists env(NLSPATH)] == 1} {
            set nlspath $env(NLSPATH)
        } else {
	    if {[info exists env(PLATV4_0)] == 1} {
               set nlspath ${SysmanDir}/usr/lib/nls/msg/%L/%N
	    } else {
		set nlspath /usr/lib/nls/msg/%L/%N
	    }
        }
    } else {

#---------------------------------------------------------------#
#
#  For security reasons only use the default NLSPATH if running
#   setuid
#
#---------------------------------------------------------------#

    if {[info exists env(PLATV4_0)] == 1} {
        set nlspath ${SysmanDir}/usr/lib/nls/msg/%L/%N
    } else {
        set nlspath {/usr/lib/nls/msg/%L/%N}
    }
    }

    set lang $lc_message

#---------------------------------------------------------------#
#
#  Parse off the territory, codeset, and language
#
#---------------------------------------------------------------#

    set territory [ctoken lang "._"]
    set codeset [ctoken lang "._"]
    set lang [ctoken lang "._"]

#---------------------------------------------------------------#
#
#   Search for the file using the NLSPATH.  Substitute each of
#    the NLSPATH modifiers for the appropriate values.
#
#   If we are running as a setuid program (effective uid != uid)
#    then don't automatically look for the catalog
#    in the current directory to mimic the behavior of catopen(3).
#
#   If we cannot find the specific language variant of a given
#    catalog using the NLSPATH we attempt to open the english
#    message catalog as a last resort before giving up.
#
#---------------------------------------------------------------#

    if {$setuid == 0} {
        set localnls ":./%N"
    } else {
    	set localnls {}
    }

    set nlslist [split $nlspath$localnls :]
    foreach np "$nlslist" {
    	set newpath $np
	regsub -all "%L" $newpath $lc_message newpath
	regsub -all "%N" $newpath $catname newpath
	regsub -all "%l" $newpath $lang newpath
	regsub -all "%t" $newpath $territory newpath
	regsub -all "%c" $newpath $codeset newpath
	if {[file exists ${newpath}.msg] == 1} {
	    return ${newpath}.msg
	}
    }

#---------------------------------------------------------------#
#
#   Ok, if we've gotten to here we've failed to find the
#     catalog, so we now need to look for the first en_US.ISO8859-1
#     catalog we can find on the search path, as a last resort we
#     add the standard /usr/lib/nls/ path 
#
#---------------------------------------------------------------#

    if {[info exists env(PLATV4_0)] == 1} {
    	set default_catalog ":$SysmanDir/usr/lib/nls/msg/en_US.ISO8859-1/%N"
    } else {
    	set default_catalog ":/usr/lib/nls/msg/en_US.ISO8859-1/%N"
    }
    
    set nlslist [split $nlspath$localnls$default_catalog :]
    foreach np "$nlslist" {
    	set newpath $np
	regsub -all "%L" $newpath en_US.ISO8859-1 newpath
	regsub -all "%N" $newpath $catname newpath
	regsub -all "%l" $newpath $lang newpath
	regsub -all "%t" $newpath $territory newpath
	regsub -all "%c" $newpath $codeset newpath
	if {[file exists ${newpath}.msg] == 1} {
	    return ${newpath}.msg
	}
    }

    return {}
}

#---------------------------------------------------------------#
#
#   Load the catalog file into the object directly from the
#    .msg file.  This should work for both symbol and numeric
#    .msg files.
#
#---------------------------------------------------------------#

_Catfile instproc loadcatalog {catfile} {
   $self instvar _catname _msgNum _msgID
   
   if {$catfile != {}} {
      set nextid 1
      set lline {}
      set line {}
      
      set catfilefd [open $catfile r]
      set context [scancontext create]
      
      scanmatch $context {^\$} { set lline {} }
      
      scanmatch $context {\\$} {
	 append lline [string trimright $matchInfo(line) "\\"]
	 continue
      }
      
      scanmatch $context {^$} {
	 set lline {}
      }
      
      # Default match
      scanmatch $context {
	 
	 if ![lempty $lline] {
	    append lline $matchInfo(line)
	 } else {
	    set lline $matchInfo(line)
	 }
	 
	 set lline [string trimleft $lline]
	 set symbol [ctoken lline " \t"]
	 set lline [subst -novariables -nocommands [string trim $lline]]
	 set _msgNum($symbol) [string trim "$lline" {"}]
         set _msgID($symbol) $nextid
         incr nextid
         set lline {}
      }

      scanfile $context $catfilefd
    }

    return $catfile
}

_Catfile instproc catgets {symbol} {
    $self instvar _msgNum
    if {[info exists _msgNum($symbol)]} {
       return $_msgNum($symbol)
    } else {
       return {}
    }
}


#---------------------------------------------------------------#
#
#   Check if a given symbol exists in the message catalog.
#   Returns {} if the symbol doesn't exist, else it returns the
#   symbol.  This is used to detect the difference between a
#   null message and a non-existant symbol
#
#---------------------------------------------------------------#

_Catfile instproc exists {symbol} {
   $self instvar _msgNum
   info exists _msgNum($symbol)
}

_Catfile instproc catgets_nocheck {symbol} {
   return [$self set _msgNum($symbol)]
}

_Catfile instproc catsets {symbol text} {
    $self set _msgNum($symbol) $text
}

_Catfile instproc catgetid {symbol} {
    if {[$self exists $symbol]} {
	$self set _msgID($symbol)
    } else {
    	return 0
    }
}


#---------------------------------------------------------------#
#
#   Overrides to catopen & catgets.  We'll be defining these
#    to use the above classes so we don't want anyone to access
#    the builtins.  If the renamed ones already exists we're probably
#    just re-sourcing this file so skip it.
#
#---------------------------------------------------------------#

if {[info commands catopen.tclx] == {}} {
    rename catopen catopen.tclx
    rename catgets catgets.tclx
    rename catclose catclose.tclx
}

#---------------------------------------------------------------#
#
#  Reimplement catopen to use our objects
#
#---------------------------------------------------------------#

proc catopen {arg1 args} {

    set arg2 [lassign "$arg1 $args" catname]
    if {$arg2 != {}} {
        set catname $arg2
    }
    set basename [file tail [file rootname $catname]]

    _Catalog _msgCat_$basename [file rootname $catname]
}

proc catgets {catHandle setnum msgnum {defaultstr {}}} {

    $catHandle catgets $msgnum
}

proc catclose {arg1 args} {

    set arg2 [lassign "$arg1 $args" cathdl]
    if {$arg2 != {}} {
        set cathdl $arg2
    }

    $cathdl destroy
}

