#!/usr/share/sysman/bin/sysmansh
# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: install_procs.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:30 $
# 

##---------------------------------------------------------------------------
 # 
 # MODULE:
 #      install_procs.tcl
 #
 # ABSTRACT:
 #	This library contains standard routines which are used by 
 #	the installation applications.  Other developers may use
 #	these routines, but any modifications should be discussed
 #	with the Installation team.
 #
 # NOTES:
 #
 #	The routines in this file are layed out in the following manner:
 #
 #	o General Routines
 #		Update_Initialize
 #		List_ReadFile
 #		INST_OutputErrorMessage
 #		INST_CallShellRoutine
 #		INST_SecureTmpDir
 #
 #	o New Hardware Delivery (NHD) Routines
 #		HW_AddDatabaseEntry
 #		HW_CompareEntries
 #		HW_FindDifferences
 #		HW_FindItem
 #		HW_ReadDatabase
 #		HW_RemoveDatabaseEntry
 #		HW_WriteDatabase
 #		HW_WriteDatabaseEntry  (Internal use only)
 #
 #	o Menu Routines
 #		MENU_AskQuestion
 #		MENU_Get_Answer
 #		MENU_Write
 #
 #	o Graphical Routines
 #		GUI_UpdateLogMsg
 #		GUI_Dialog
 #		GUI_TextWidget_Create
 #		GUI_Dialog_display
 #		GUI_Dialog_log
 #		GUI_Calc_Location
 #		GUI_Center
 #		AddTicks
 #		SetTicks
 #		TickEvery
 #		GUICheck
 #		GUITick
 #		ReadCommandPipe 
 #		GUI_Status_Create
 #		GUI_Status_BindPFH
 #		HelpPointerMessage
 #		GUI_SetEnglishFonts
 #		GUI_SetInitialInstallFont
 #
##---------------------------------------------------------------------------



##############################################################################
#			  General Routines                                   #
##############################################################################


##---------------------------------------------------------------------------
 #
 # Proc:  Update_Initialize
 #
 #	A routine to initialize some settings for the Update Install
 #	applications.  
 #
 # Inputs: 
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc Update_Initialize {} {

	global env 
	global ui 
	global status_Pipe 
	global logfile
	global tmpdir
	global catalogPath
	global hwdir
	global FSlist		; # list of possible file systems
	global FStype		; # array of file system information

	#
	# Default the logfile to /var/adm/smlogs/update.log,
	# but allow the update code to override this using
	# the UPDLOG environment variable.
	#
	set env(LOGFILENAME) "/var/adm/smlogs/update.log"
	catch {set env(LOGFILENAME) $env(UPDLOG)}

	#
	# Relative directory path to NHD database files
	#
        set hwdir "usr/sys/hardware"

	#
	# Define the temporary directory so that all of the files
	# created by update are in the same place.
	#
	if [catch {set tmpdir "$env(TMPDIR)"}] {
		set tmpdir "/var/tmp/update"
	}

	#
	# status_Pipe is the pathname to a named pipe used to control
	# the status indicator
	#
	if [catch {set status_Pipe "$env(STATUSPIPE)"}] {
		set status_Pipe "/tmp/updpipe"
	}

	#
	# Set the path for the catalog files 
	#
	if [catch {set locmnt "$env(LOCMNT)"}] {
		set catalogPath "/usr/lib/nls/msg/en_US.ISO8859-1"
	} else {
		set catalogPath "$locmnt/usr/lib/nls/msg/en_US.ISO8859-1"
	}

	#
	# Initialize the file system variables
	#
	set FSlist {root usr var i18n}
	#
	# Set the string used by the 'df' and 'mount' commands
	#
	set FStype(dfStr.root) "/"
	set FStype(dfStr.usr)  "/usr"
	set FStype(dfStr.var)  "/var"
	set FStype(dfStr.i18n) "/usr/i18n"

	#
	# Set $ui to be either gui or menu
	#
	catch {set ui [string toupper "$env(SYSMANUI)"]}
	if {! [cequal $ui "GUI"]} {
		set ui MENU
		return
	}
	
	#############################
	# GUI ONLY after this point #
	#############################

	#
	# Set fonts based on screen resolution
	#
	if {[winfo screenwidth .] <= 800 } {
		option add {*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Label*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Message*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Button*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Radiobutton*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Checkbutton*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Listbox*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Entry*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Text*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Menu*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Menuentry*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
		option add {*Menubutton*font} \
			{-Adobe-Helvetica-Medium-R-Normal--10-100-*}
	} else {
		option add {*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Label*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Message*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Button*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Radiobutton*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Checkbutton*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Listbox*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Entry*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Text*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Menu*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Menuentry*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
		option add {*Menubutton*font} \
			{-Adobe-Helvetica-Medium-R-Normal--17-120-*}
	}
}



##---------------------------------------------------------------------------
 #
 # Proc:  List_ReadFile
 #
 #	Reads the specified file and returns the contents of the file in
 #	a list format.  Each element of the list represents a line in the
 #	file.  If the file was not found or could not be opened a value 
 #	of -1 is returned.
 #
 # Inputs: 
 #
 #	inputFileName - The name of the file which is to be read
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #
 #	On success -> a list with each element in the list representing
 #		      a line in the file.
 #
 #	On failure -> -1
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc List_ReadFile {inputFileName} {

	set ret -1
	catch {set ret [split [exec cat $inputFileName] \n]}
	return $ret
}				; ## end routine List_ReadFile


##---------------------------------------------------------------------------
 #
 # Proc: INST_OutputErrorMessage
 #
 #	Display an error message in GUI or CCi format. 
 #
 # Inputs: 
 #	msg - the error message to display
 #
 # Outputs: 
 #      None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #	o Error messages that require the user to make a decision
 #	  (i.e: two or more possible paths) should not use this
 #	  routine.
 #
 #	o If an update Requires the messages Inst_Error_Title and Inst_Ok be set
 #	  in the GUI section of the message catalog
 #	
##---------------------------------------------------------------------------
proc INST_OutputErrorMessage {msg} {

	global ui			; # user interface type (GUI or MENU)
	global env

	if {! [info exists ui]} {       ; # if we are not in update
		if {[catch { winfo exists . } err]} {
			set ui "MENU"   ; # didn't work, not GUI
		} else {
			set ui "GUI"    ; # set ui so we can use it next time
		}
	}
	if {[cequal $ui "GUI"]} {	; # if update, use the update msg cat
		if { [info exists env(UPDFLAG)] } {
			wm withdraw .
			set ans [GUI_Dialog .errorW \
                        [sm_catgets Inst_Error_Title GUI] \
                        $msg {error} 0 \
                        [sm_catgets Inst_Ok GUI]]
		} else {
			wm withdraw .
			set ans [GUI_Dialog .errorW \
			[sm_catgets I18N_catobj i18n_error_title_txt Install] \
			$msg {error} 0 \
			[sm_catgets I18N_catobj i18n_ok_txt]]
		}
	} else {
		MENU_Write "\n$msg\n"
	}
}



##---------------------------------------------------------------------------
 #
 # Proc: INST_CallShellRoutine
 #
 #	Executes the specified shell routine and returns the output
 #	and status code.
 #
 # Inputs: 
 #	library - the name of a library to source
 #	
 #	routine - the name of the routine within the library that is 
 #		  to be executed
 #
 #	routineArgs - the arguments to the routine
 #
 # Outputs: 
 #      None
 #
 # Returns: 
 #	A list with two elements.  The first element is the status
 #	code returned by the routine and the second element is the
 #	output from the routine. 
 #
 # Notes:
 #	None
 #	
##---------------------------------------------------------------------------
proc INST_CallShellRoutine {lib routine {args {}}} {

	#
	# Call INST_SecureTmpDir() to check to see if we are root,
	# and if so place our tmp file in a secure dir.
	#

	set tmpFile [INST_SecureTmpDir .INST_CallShellRoutine.[pid] /tmp CALLSHELL]
	#
	# Check to see if anything was actually returned
	# from INST_SecureTmpDir(), if not then exit.
	# The user will have gotten the error message from
	# INST_SecureTmpDir(), so there is no need to report
	# it now.
	#

	if {[cequal $tmpFile ""]} {
		exit 1
	}

	set retVal [ system ". $lib ; $routine [join $args] > $tmpFile 2>&1" ]

	catch {set outVal [exec cat $tmpFile]}
	unlink -nocomplain $tmpFile

	return [list $retVal $outVal]
}


##---------------------------------------------------------------------------
 #
 # Proc:  INST_SecureTmpDir
 #
 #    Checks to see if the user is root, and if so places the tmp file
 #    in a secure dir.
 #
 # Inputs:
 #    TmpName - Name of the temporary file.
 #    Path - Where you wish to place this file.
 #    DirName - If root, the name of the secure directory you wish to use.
 #              If you do not supply one, then it defaults to installtmp.
 #
 # Outputs:
 #    None
 #
 # Returns:
 #    Where the file has been placed, with it's name.
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc INST_SecureTmpDir {TmpName Path {DirName installtmp}} {
 
	set TmpDirName ""
 
	if {[catch {set my_uid [exec id -u]} err]} {
		INST_OutputErrorMessage $err
	} else {
		if {[cequal "$my_uid" 0]} {
			set my_umask [umask]
			umask 077
			if {[catch {exec rm -Rf $Path/$DirName} err]} {
				INST_OutputErrorMessage $err
			} else {
				mkdir -path $Path/$DirName
				set TmpDirName $Path/$DirName/$TmpName
			}
			umask $my_umask
		} else {
			set TmpDirName $Path/$TmpName
		}
	}
 
	return $TmpDirName
 
}
 
 
##############################################################################
#			    NHD Routines                                     #
##############################################################################

##---------------------------------------------------------------------------
 #
 # Proc:  HW_AddDatabaseEntry
 #
 #	After installing hardware subsets it is necessary to leave a
 #	record on the system that says what hardware support was
 #	installed.  This record is used by the update install process
 #	to determine what to do with the installed hardware products.
 #	When hardware support for one piece of hardware is installed,
 #	it is possible that support for other pieces of hardware were
 #	also installed, since the other hardware support may be in the
 #	same subsets as the original hardware support.  Therefore, the
 #	database from the hardware media needs to be searched and a
 #	record left on the installed system for each piece of hardware
 #	support that exactly matches the subsets that were installed.
 #
 # Inputs: 
 #	filename   - the name of the file to add the entry to
 #
 #	moduleList - a list of the entries read from the HW database
 #		     on the hardware media.
 #
 #	subsetList - the list of subsets that were loaded
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	0 - Failure
 #	1 - Success
 #
 # Notes:
 #	None
 #
##---------------------------------------------------------------------------
proc HW_AddDatabaseEntry {filename moduleList subsetList} {

	#
	# Setup so that any errors can be passed back to the 
	# calling routine.
	#
	upvar err err

	#
	# Read in the database to be updated.
	#
	set existingList [HW_ReadDatabase $filename]

	#
	# Loop through each module and determine if support was loaded.
	#
	foreach entry $moduleList {
		set tmpList [intersect3 $subsetList [keylget entry subsets]]

		#
		# Create the record only if the entry has the exact
		# same subsets as the ones in the subsetList.  If the
		# entry has more or less subsets, full support was not
		# loaded and a record should not be left on the system.
		#
		if {[lempty [lindex $tmpList 0]] && \
			[lempty [lindex $tmpList 2]]} {

			#
			# Strip any entries from the existing list that
			# match the newly installed hardware support.  
			# This will help eliminate duplicate records.
			#
			set tmpList [lindex [HW_FindDifferences \
				$existingList [list $entry]] 1]

			#
			# Add the newly created record to the 
			# existing list.
			#
			set existingList [union $tmpList [list $entry]]
		}
	}

	#
	# Write out all of the entries which need to remain
	# in the database.
	#
	set status [HW_WriteDatabase $filename $existingList]
	return $status
}				; # end of HW_AddDatabaseEntry



##---------------------------------------------------------------------------
 #
 # Proc:  HW_CompareEntries
 #
 #	Compare two aysnchronous hardware support entries to see if 
 #	they are the same.  Two entries are the same if the vendor and
 #	type strings are the same.  
 #
 # Inputs: 
 #	entry1 - the first hardware support entry
 #	entry2 - the second hardware support entry
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	0 - the entries are different
 #	1 - the entries are the same
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc HW_CompareEntries {entry1 entry2} {

	#
	# Two hardware support entries are the same if the vendor and
	# type match.
	#
	if {[cequal [keylget entry1 vendor] [keylget entry2 vendor]] &&
	    [cequal [keylget entry1 type] [keylget entry2 type]]} {
		return 1
	} else {
		return 0
	}
}				; # end of HW_CompareEntries



##---------------------------------------------------------------------------
 #
 # Proc:  HW_FindDifferences
 #
 #	Compares two lists of hardware support entries and returns all
 #	of the elements in list1 that do not exist in list2.  The
 #	comparison is done using the unique key formed by the vendor
 #	and type fields. 
 #
 # Inputs: 
 #	list1 - the first list of hardware support entries
 #	list2 - the second list of hardware support entries
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	A list with two elements.  The first element is the list of
 #	elements that exist in both list1 and list2 and the second
 #	element is the list of elements that are in list1 but not in
 #	list2. 
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc HW_FindDifferences {list1 list2 {common {1}}} {

	#
	# If either list is empty, there is nothing in common between
	# the two lists, so return an empty list for the common entries
	# and the entire list1 for the unique items in list1.
	#
	if {[lempty $list1] || [lempty $list2]} {
		return [list {} $list1]
	}

	set foundList {}
	set notFoundList {}
	foreach hw_entry $list1 {
		set found 0
		foreach entry $list2 {
			if {[HW_CompareEntries $hw_entry $entry]} {
				set found 1
				break
			}
		}

		if {$found} {
			if {[cequal $common 1]} {
				lappend foundList $hw_entry
			} else {
				lappend foundList $entry
			}
		} else {
			lappend notFoundList $hw_entry 
		}
	}

	return [list $foundList $notFoundList]
}				; # end of HW_FindDifferences



##---------------------------------------------------------------------------
 #
 # Proc:  HW_FindItem
 #
 #	Searches the list specified by listInfo for the string 
 #	specified by searchString.
 #
 # Inputs: 
 #	listInfo - the list to search
 #
 #	searchString - the string to search for
 #
 #	keyName - the named element that identifies which field to search
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	A list with two elements.  The first element is the list of
 #	elements that contained the searchString and the second
 #	element is the list of elements that did not.
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc HW_FindItem {listInfo searchString keyName} {

	set notFoundList {}
	set foundList {}

	#
	# Loop through the list searching for the named element that
	# matches the search string.
	#
	foreach kit $listInfo {
		set found [lmatch -exact [keylget kit $keyName] $searchString]

		if {[lempty $found]} {
			lappend notFoundList $kit
		} else {
			lappend foundList $kit
		}
	}

	return [list $foundList $notFoundList]
}				; # end of HW_FindItem



##---------------------------------------------------------------------------
 #
 # Proc: HW_ReadDatabase
 #
 #	Reads a set of hardware support files and returns the 
 #	information to the calling routine in the form of a 
 #	named list.
 #
 # Inputs: 
 #	location - the location to search for the hardware support files
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	o A -1 if there was an error within the routine
 #	o An empty list if no files are found
 #	o A named list of the entries found
 #
 # Notes:
 #	o The file format is a set of entries with each entry 
 #	  having the format:
 #
 #		{ 
 #			{vendor name string} 
 #			{hardware name string} 
 #			{list of subsets} 
 #			{list of supported releases} 
 #			{kernel build flag}
 #			{path/filename of kernel module}
 #			{path of kit directory}
 #			{path/filename of kit file}
 #		}
 #
##---------------------------------------------------------------------------
proc HW_ReadDatabase {location} {

	#
	# If the location is a directory, use all of the files in that
	# directory.  If it is a file, use the specific file.  Otherwise
	# return an empty list, since it is neither a file nor a directory
	#
	if {[file isdirectory $location]} {
		set fnameList [glob -nocomplain $location/*.nhd $location/*.hw]
	} elseif {[file exists $location]} {
		set fnameList $location
	} else {
		return {}
	}
 
	#
	# For every file found read in the contents of the file.  
	# Comment lines (start with #) and blank lines are stripped.
	#
	set tmpList {}
	set defaultEntry "CONVERTED FROM 4.0F"
	foreach fname $fnameList {
	        if {[catch {set no_comm [exec sed -n \
	                -e {s/^ *#.*//} \
	                -e {/[^ ]/p} \
		        $fname]} err]} {
			return -1
		}
		
		if {! [lempty $no_comm]} {
			#
			# Check if the first element is actually a
			# hardware entry, if not, file is corrupt,
			# and returns an error
			#
			set listcheck [lindex $no_comm 0]
			set listcheckL [llength $listcheck]
			if {$listcheckL != 8 && $listcheckL != 4} {
				return -1
			}

			#
			# For each element create a named list and
			# append it to the existing list.  The file
			# format is described in the NOTES section of
			# the header for this routine. 
			#
			foreach elemt $no_comm {
				#
				# Ignore malformed entries
				#
				set elemtL [llength $elemt]
				if {$elemtL != 8 && $elemtL != 4} {
					continue
				}
				switch -exact -- $elemtL {
					4 {
					keylset localVar \
						vendor [lindex $elemt 0] \
						type [lindex $elemt 1] \
						subsets [lindex $elemt 2] \
						releases [lindex $elemt 3] \
						kernel_build 1 \
						module_name $defaultEntry \
						kit_dir $defaultEntry \
						kit_file $defaultEntry
					lappend tmpList $localVar
					}
					8 {
					keylset localVar \
						vendor [lindex $elemt 0] \
						type [lindex $elemt 1] \
						subsets [lindex $elemt 2] \
						releases [lindex $elemt 3] \
						kernel_build [lindex $elemt 4] \
						module_name [lindex $elemt 5] \
						kit_dir [lindex $elemt 6] \
						kit_file [lindex $elemt 7]
					lappend tmpList $localVar
					}
				}
			}
		}
	}

	return $tmpList
}				; # end of HW_ReadDatabase




##---------------------------------------------------------------------------
 #
 # Proc:  HW_RemoveDatabaseEntry
 #
 #	After removing hardware subsets it is also necessary to remove
 #	the record of the hardware support so that the update
 #	installation does not think the subset is still installed.
 #	This routine removes any entry from the database that contains
 #	the subset that was removed.  If one or more subsets has been
 #	removed, the associated hardware support no longer works, so
 #	the entry is removed from the database as well.
 #
 # Inputs: 
 #	filename - the name of the hardware database file
 #
 #	subsetList - the list of subsets that were removed
 #
 # Outputs: 
 #      None
 #
 # Returns: 
 #	0 - Failure
 #	1 - Success
 #
 # Notes:
 #	None
 #
##---------------------------------------------------------------------------
proc HW_RemoveDatabaseEntry {filename subsetList} {

	#
	# Read in the database to be updated.
	#
	set existingList [HW_ReadDatabase $filename]

	#
	# Loop through each module and determine if support was loaded.
	#
	set tmpList {}
	foreach entry $existingList {
		#
		# Only remove those entries that contain one or more 
		# of the subsets that were removed.  Removing any of
		# the subsets causes the hardware support to be 
		# incomplete, so the records need to be removed.
		#
		if {[lempty [intersect [keylget entry subsets] $subsetList]]} {
			#
			# Since there is no intersection between the
			# two lists it means the existing entry should
			# remain.
			#
			lappend tmpList $entry
		}
	}

	#
	# Write out all of the entries which need to remain
	# in the database.
	#
	set failure [HW_WriteDatabase $filename $tmpList]
	return $failure
}				; # end of HW_RemoveDatabaseEntry



##---------------------------------------------------------------------------
 #
 # Proc:  HW_WriteDatabase
 #
 #	Given a file and the entries to put in the file, this routine
 #	will create the database file.
 #
 # Inputs: 
 #	filename - the name of the file to be written
 #
 #	entryList - the list of entries to write into the file
 #
 # Outputs: 
 #      None
 #
 # Returns: 
 #	0 - Failure
 #	1 - Success
 #
 # Notes:
 #	None
 #
##---------------------------------------------------------------------------
proc HW_WriteDatabase {filename entryList} {


	#
	# Setup so that any errors can be passed back to the 
	# calling routine.
	#
	upvar err err
	set failure 1

	#
	# Backup the previous file by moving it to .bak
	#
	if {[file exists $filename]} {
		if {[file exists $filename.bak]} {
			unlink $filename.bak
		}
		frename $filename $filename.bak
	}

	#
	# Create an empty file and then write all of the remaining
	# entries back to the file.
	#
	close [open $filename w]
	foreach entry $entryList {
		#
		# Write the entry
		#
		if {! [HW_WriteDatabaseEntry $filename $entry]} {
			set failure 0
		}
	}

	#
	# If any failure occurred during the write of the entry,
	# put back the original file.
	#
	if {[cequal $failure "0"] && [file exists $filename.bak]} {
		unlink $filename
		frename $filename.bak $filename
	}

	return $failure
}				; # end of HW_WriteDatabase



##---------------------------------------------------------------------------
 #
 # Proc: HW_WriteDatabaseEntry
 #
 #	Internal use only.  Append the specified entry to the
 #	specified file.
 #
 # Inputs: 
 #	filename - the name of the file to which the entry will be written
 #
 #	entry - the data to be written.  Must be a valid keyed list with
 #		all of the fields defined.
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	0 - Failure
 #	1 - Success
 #
 # Notes:
 #	o This routine is for use by the routines HW_WriteDatabase and 
 #	  HW_AddDatabaseEntry only.  Other routines should use
 #	  HW_AddDatabaseEntry to add entries to a file.  
 #
 #	o The file format is a set of entries with each entry 
 #	  having the format:
 #
 #		{ 
 #			{vendor name string} 
 #			{hardware name string} 
 #			{list of subsets} 
 #			{list of supported releases} 
 #			{kernel build flag}
 #			{path/filename of kernel module}
 #			{path of kit directory}
 #			{path/filename of kit file}
 #		}
 #
##---------------------------------------------------------------------------
proc HW_WriteDatabaseEntry {filename entry} {

	#
	# Setup so that any errors can be passed back to the 
	# calling routine.
	#
	upvar err err

	#
	# Open the file so that the record can be appended
	#
	if {[catch {set fileId [open $filename a]} err]} {
		return 0
	}

	#
	# Write the entry
	# 
	puts $fileId "{"
	puts $fileId "\t{[keylget entry vendor]}"
	puts $fileId "\t{[keylget entry type]}"
	puts $fileId "\t{[keylget entry subsets]}"
	puts $fileId "\t{[keylget entry releases]}"
	puts $fileId "\t{[keylget entry kernel_build]}"
	puts $fileId "\t{[keylget entry module_name]}"
	puts $fileId "\t{[keylget entry kit_dir]}"
	puts $fileId "\t{[keylget entry kit_file]}"
	puts $fileId "}"

	close $fileId
	return 1
}				; # end of HW_WriteDatabaseEntry



##############################################################################
#			    MENU Routines                                    #
##############################################################################


##---------------------------------------------------------------------------
 #
 # Proc:  MENU_AskQuestion
 #
 #	This procedure asks a question (character-cell) and retrieves
 #	the user's input.  The question is repeated until the user
 #	enters a valid choice.
 #
 # Inputs: 
 #	text	- The question text
 #	choices	- A list of possible answers
 #	default	- The default answer
 #
 # Outputs: 
 #	A question string to the user
 #
 # Returns: 
 #	The valid user answer
 #	
 # Notes:
 #
##---------------------------------------------------------------------------
proc MENU_AskQuestion {text choices default} {

	#
	# Build the choices string, for example: (a/b/c)
	#
	set choiceStr "("
	foreach value $choices {
		set choiceStr "${choiceStr}${value}/"
	}
	regsub ".$" $choiceStr ")" choiceStr

	#
	# Loop until a valid answer is found
	#
	while 1 {
		#
		# Output the question, for example:
		#      Continue (y/n) [n]?
		#
		puts -nonewline stdout [cexpand "$text \
			$choiceStr \[$default\]: "]
		flush stdout

		#
		# Get the user's input
		#
		gets stdin answer
		set answer [string tolower $answer]

		#
		# Allow the user to select the default unless the
		# default was not specified.
		#
		if {! [lempty $default] && $answer == ""} {
			return $default
		} 

		#
		# Check the choices to see if the input is valid
		#
		set value [lmatch -exact $choices $answer]
		if {! [lempty $value]} {
			return $value
		}
	}
}				; # end routine MENU_AskQuestion



##---------------------------------------------------------------------------
 #
 # Proc:  MENU_Get_Answer
 #
 #	This routine will ask the specified question and validate the 
 #	user's response.  The response must be listed in the specified 
 #	answer_list, otherwise the question is repeated.  This is a 
 #	non-graphical routine.
 #
 # Inputs: 
 #
 #	introText  - Any introductory text.
 #
 #	question   - The question text to be displayed to the user
 #
 #	answerList - The list of acceptable answers
 #
 #	returnList - An optional argument which if supplied is a list
 #		     of values to return.  Each element in the return
 #		     list must map to one of the possible answers.
 # Outputs: 
 #
 #	Outputs the specified question and gets the user's response
 #
 # Returns: 
 #
 #	The index of the user's selection in the valid answer array, if 
 #	the returnList argument is not provided.  Otherwise, the value
 #	returned will be the element in the returnList which corresponds
 #	to the user's selection from the answerList.
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc MENU_Get_Answer {introText question answerList {returnList {}}} {

	#
	# Loop until a valid answer is found
	#
	set index -1
	while {$index < 0} {
		#
		# Output the question text and wait for a response
		#
		if {! [lempty $introText]} {
			MENU_Write $introText
		}
		puts -nonewline stdout "$question: "
		flush stdout
		gets stdin answer

		#
		# Make the string lower case and then ensure an exact match
		# with one of the elements in the answer list.
		#
		set answer [string tolower $answer]
		set answerList [string tolower $answerList]
		set index [lsearch -exact $answerList $answer]
	}

	#
	# Return the correct value
	#
	if {! [lempty $returnList]} {
		return [lindex $returnList $index]
	} else {
		return $index
	}
}				; ## end routine MENU_Get_Answer


##---------------------------------------------------------------------------
 #
 # Proc:  MENU_Write
 #
 #	Outputs the specified text to the user, formatting for an 80
 #	column screen.
 #
 # Inputs: 
 #
 #	output - The text to output to the user
 #
 # Outputs: 
 #
 #	The specified text formated for an 80 column screen
 #
 # Returns: 
 #	None
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc MENU_Write {output} {

	#
	# Open the tmpFile. The tmpfile is used because some 
	# error messages are too long for the "echo" command.
	#

	#
	# Call INST_SecureTmpDir() to check to see if we are root,
	# and if so place our tmp file in a secure dir.
	#

	set tmpFile [INST_SecureTmpDir tmpfile.install /tmp MENU]

	#
	# Check to see if anything was actually returned
	# from INST_SecureTmpDir(), if not then exit.
	# The user will have gotten the error message from
	# INST_SecureTmpDir(), so there is no need to report
	# it now.
	#

	if {[cequal $tmpFile ""]} {
		exit 1
	}

	if {[catch {set fileid [open $tmpFile w]} err]} {
		puts stderr $err
		return
	}

	#
	# Write the data
	#
	puts $fileid $output
	close $fileid

	#
	# Format the output
	#
	catch {system "/usr/bin/fmt -70 < $tmpFile"}

	#
	# Remove the temporary file
	#
	catch {set status [unlink $tmpFile]}
}				; ## end of routine MENU_Write




##############################################################################
#			    GUI Routines                                     #
##############################################################################



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_UpdateLogMsg
 #
 #	A routine to output the specified mesasge to the log file.
 #
 # Inputs: 
 #	msg - the text message to be added to the log file
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #	Assumes Update_Initialize has been run to define the logfile
 #	or that the environment variable LOGFILENAME has been exported.
 #
##---------------------------------------------------------------------------
proc GUI_UpdateLogMsg {msg} {

	global env 

	catch {set fd [open $env(LOGFILENAME) a]}
	catch {[puts $fd $msg]}
	catch {[close $fd]}
}				; ## end of routine GUI_UpdateLogMsg




##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Dialog
 #
 #	Configurable dialog box.  Unlike the tk_dialog widget
 #	it does not force the user to use specific fonts.
 #
 # Inputs: 
 #
 #	args  - A set of arguments in the following order: 
 #		o widget name
 #		o dialog title
 #		o text to be displayed
 #		o number of the button which is to receive the focus 
 #		  (-1 for no focus)
 #		o a list of button labels, one per button (empty if 
 #		  no buttons to be displayed)
 #
 # Outputs: 
 #
 #	A dialog box with the specified text and buttons
 #
 # Returns: 
 #
 #	The number corresponding to the button selected by the user.
 #	Buttons are numbered starting at 0 and incrementing by 1 for
 #	each button.
 #
 # Notes:
 #
 #	The widget name supplied will be used for the new window,
 #	destroying any other widget by that name.
 #
 #	The answerDialog global can be set by the caller, causing
 #	the dialog window to exit.
 #
##---------------------------------------------------------------------------
proc GUI_Dialog {args} {

	global env
	global answerDialog

	#
	# Assign the parameters
	#
	lassign $args widget title text bitmap default 
	set buttonList [lrange $args 5 end]
	if {[cequal $buttonList "{}"]} {set buttonList {}}

	#
	# Set locale-specific font during initial installation.  Does
	# nothing in environments other than initial install.
	#
	GUI_SetInitialInstallFont

	#
	# Create and display the dialog box
	#
	GUI_Dialog_display $widget $title $text $bitmap \
		$default "" $buttonList

	#
	# Only wait for user input if a button was created
	#
	if {! [lempty $buttonList]} {
		tkwait variable answerDialog
		destroy $widget
	}

	#
	# Log the results
	#
	GUI_Dialog_log $text $buttonList

	update
	return $answerDialog
}				; ## end of routine GUI_Dialog



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_TextWidget_Create
 #
 #	Creates a text widget with a specific default configuration.
 #	This way all messages will have the same bahvior without
 #	having to duplicate the code.
 #
 #	The proc accepts a frame and packs the new widget within this
 #	frame. If the frame doesn't exist, then it will be created.
 #
 # Inputs: 
 #
 #	frame - the frame that the new text widget will be packed in
 #
 #	text - the text to display
 #
 # Outputs: 
 #
 #	None
 #
 # Returns: 
 #
 #	None
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUI_TextWidget_Create {frame text} {

	global env

	#
	# If the frame doesn't exist, then create it here
	#
	if {![winfo exists $frame]} {
		frame $frame -relief raised -borderwidth 1 
		pack $frame -anchor n -fill both -expand 1
	}

	#
	# Destroy any existing text widget with the same name
	#
	catch {destroy $frame.messageM}

	#
	# Determine the default wrap length
	#
	set textLength [clength $text]
	if {$textLength > 70} {
		set textWidgetWidth 70
	} else {
		set textWidgetWidth [expr $textLength + 2]
	}

	#
	# Count the extra number of characters generated by \n and \t
	#
	set extraChars [expr [regsub -all "\t" $text "" tmpVar] * 8]
	set extraLines [regsub -all "\n" $text "" tmpVar]

	#
	# Calculate the height of the text area
	#
	set textWidgetHeight [expr ($textLength + $extraChars) /  \
		$textWidgetWidth + 1 + $extraLines]

	#
	# Determine the maximum text widget height based upon the screen size
	#
	if {[winfo screenwidth .] <= 800} {
		set textWidgetHeightMax 30
	} else {
		set textWidgetHeightMax 45
	}

	#
	# If the text widget height exceeds the maximum height that will
	# fit on the screen, then crop the text widget and create a scroll bar
	#
	# Note that this processing is only performed during full install
	#
	if { $textWidgetHeight > $textWidgetHeightMax && \
			! [info exists env(UPDFLAG)] } {

		#
		# Crop the text widget height
		#
		set textWidgetHeight $textWidgetHeightMax

		#
		# Destroy any existing scroll widget with the same name
		#
		catch {destroy $frame.scrollyS}

		#
		# Create a vertical scrollbar on the right-hand side of the
		# text widget
		#
		scrollbar $frame.scrollyS -relief sunken -width 13 \
			-takefocus 0 -orient vertical \
			-command "$frame.messageM yview"
		pack $frame.scrollyS -side right -fill y 

		#
		# Define the text widget with a scroll command
		#
		text $frame.messageM -takefocus 0 -borderwidth 0 -wrap word \
			-cursor {} -width $textWidgetWidth \
			-height $textWidgetHeight \
			-yscrollcommand "$frame.scrollyS set"

	} else {

		#
		# Define the text widget without a scroll command
		#
		text $frame.messageM -takefocus 0 -borderwidth 0 -wrap word \
			-cursor {} -width $textWidgetWidth \
			-height $textWidgetHeight
	}

	#
	# Pack the text widget and insert the text
	#
	pack $frame.messageM -fill both \
		-expand 1 -padx 0.25i -pady 0.25i
	$frame.messageM insert end $text

	#
	# Make sure the user cannot enter text into the text field
	#
	regsub Text [bindtags $frame.messageM] "" newTags
	bindtags $frame.messageM $newTags
}



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Dialog_display
 #
 #	Displays a dialog box.
 #
 # Inputs: 
 #
 #	widget - the widget name of the dialog box
 #	title - the title of the dialog box
 #	text - the message to display
 #	bitmap - icon to show (e.g: error, info, question etc)
 #	default - number of the button which is to receive the focus
 #		  (-1 for no focus)
 #	helpId - the LOCID of the on-line help for the dialog box
 #	buttonList - list of buttons 
 #
 # Outputs: 
 #
 #	A dialog box with the specified text and buttons
 #
 # Returns: 
 #
 #	Nothing
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUI_Dialog_display {widget title text bitmap default helpId buttonList} {

	global env
	global answerDialog


	#################
	# Create window #
	#################
	catch {destroy $widget}
	toplevel $widget -class Dialog
	wm title $widget $title
	pack append $widget

	#
	# If there are no buttons defined, the double-click in the
	# upper left corner closes the window and returns -1.
	# Otherwise, disable the close function so that the user is
	# forced to select a button.  
	#
	if {[lempty $buttonList]} {
		wm protocol $widget WM_DELETE_WINDOW {set answerDialog -1}
	} else {
		wm protocol $widget WM_DELETE_WINDOW {set DO_NOTHING {}}
	}


	##############
	# Text Frame #
	##############
	frame $widget.textF -relief raised -borderwidth 1 
	pack $widget.textF -anchor n -fill both -expand 1

	#
	# Put the informational, error, question etc bitmaps in 
	# the dialog.
	#
	if {! [lempty $bitmap]} {
		label $widget.textF.symbolL -bitmap $bitmap
		pack $widget.textF.symbolL -padx 0.125i -pady 0.125i \
			-side left
	}

	################
	# Text Message #
	################
	#
	# Setup the text message environment
	#
	GUI_TextWidget_Create $widget.textF $text

	######################
	# Listbox Definition #
	######################
	#
	# Some text messages has an associated list of items.  This is
	# a convientent method of displaying them.
	#
	if {[info exists env(LISTBOX_ITEMS)]} {
		#
		# Listbox/scrollbars for available items
		#
		frame $widget.itemsF -relief raised -borderwidth 1
		frame $widget.itemsF.buttonsF
		frame $widget.itemsF.scrollF
		scrollbar $widget.itemsF.scrollxS -relief sunken -width 13 \
			-takefocus 0 -orient horizontal \
				-command "$widget.itemsF.componentsL xview"
		scrollbar $widget.itemsF.scrollyS -relief sunken -width 13 \
			-takefocus 0 -orient vertical \
			-command "$widget.itemsF.componentsL yview"
		listbox $widget.itemsF.componentsL \
			-width 40 -height 7 -relief sunken -takefocus 0 \
			-yscrollcommand "$widget.itemsF.scrollyS set" \
			-xscrollcommand "$widget.itemsF.scrollxS set"

		#
		# Determine the size of the padding frame which will prevent
		# the horizontal scrollbar from exceeding the length of the 
		# listbox.
		#
		set xpad [expr [$widget.itemsF.scrollxS cget -width] + 2* \
			([$widget.itemsF.scrollxS cget -bd] + \
			 [$widget.itemsF.scrollxS cget -highlightthickness])]
		set ypad [expr [$widget.itemsF.scrollyS cget -width] + 2* \
			([$widget.itemsF.scrollyS cget -bd] + \
			 [$widget.itemsF.scrollyS cget -highlightthickness])]
		frame $widget.itemsF.padF -width $ypad -height $xpad

		#
		# Pack the listbox widgets
		#	
		pack $widget.itemsF -anchor n -fill both -expand 1
		pack $widget.itemsF.scrollF -in $widget.itemsF \
			-side bottom -fill x 
		pack $widget.itemsF.padF -in $widget.itemsF.scrollF \
			-side right
		pack $widget.itemsF.scrollxS -in $widget.itemsF.scrollF \
			-side bottom -fill x
		pack $widget.itemsF.scrollyS -in $widget.itemsF \
			-side right -fill y 
		pack $widget.itemsF.componentsL -in $widget.itemsF \
			-side left -fill both -expand 1 

		#
		# Add the items	
		#
		foreach item "$env(LISTBOX_ITEMS)" {
			$widget.itemsF.componentsL insert end $item
		}
	}


	######################
	# Button Definitions #
	######################
	set answerDialog 255 
	set focusButton {}
	if {! [lempty $buttonList]} {
		frame $widget.buttonsF -relief raised -borderwidth 1
		pack $widget.buttonsF -fill x

		set buttonNumber 0
		foreach label $buttonList {
			#
			# Do not create a button if the label is empty
			#
			if {[lempty $label]} {
				continue
			}

			#
			# Create the button
			#
			button $widget.buttonsF.$buttonNumber -text $label \
				-command "set answerDialog $buttonNumber"
			pack $widget.buttonsF.$buttonNumber -side left \
				-expand 1 -pady 0.1i -padx 0.25i

			#
			# Set the focus to the requested button
			#
			if {$default == $buttonNumber} {
				focus $widget.buttonsF.$buttonNumber
			}
			incr buttonNumber 1
		}
	}

	#
	# Create a help button only if the helpId parameter was 
	# assigned and this is an update installation.
	#
	if {! [lempty $helpId] && [info exists env(UPDFLAG)] } {
		#
		# Button: Help
	        #
	        button $widget.buttonsF.helpB \
	                -text [sm_catgets Update_HelpBtn_Text GUI] \
	                -command {
	                        upvar #0 msg_${appName}_LocIds locId
	                        HelpOnline $appName $locId(LOCID) \
	                                $locId(${helpId}_LOCID)}
	        pack $widget.buttonsF.helpB -side left -expand 1 \
			-pady 0.1i -padx 0.25i
	}

	#
	# Make sure the window is visible and has the focus.
	# Centering the window must be done after all of the widgets
	# have been packed, but before it is visible, so that the
	# correct dimensions are used.
	#
	GUI_Center $widget [winfo parent $widget]
	catch {grab $widget}
	raise $widget
}				; ## end routine GUI_Dialog_display



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Dialog_log
 #
 #	Outputs the dialog message and the selected button to the log
 #	file.
 #
 # Inputs: 
 #
 #	text - the message to display
 #	buttonList - list of buttons 
 #
 # Outputs: 
 #
 #	A message to the specified log file (LOGFILENAME)
 #
 # Returns: 
 #
 #	Nothing
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUI_Dialog_log {text buttonList} {

	global env
	global answerDialog

	#
	# Get the log file name from the environment variable
	#
	set dialogFile {}
        catch {set dialogFile "$env(LOGFILENAME)"}

	#
	# Only output a log message if the dialogFile variable is set.
	#
	if { ! [lempty $dialogFile] } {
		#
		# Output the text message
		#
		set tmpText "\nDIALOG MESSAGE:\n$text"
	        catch {exec echo $tmpText >> $dialogFile}

		#
		# Output the listbox items list as well
		#
		if {[info exists env(LISTBOX_ITEMS)]} {
			set tmpText "LISTBOX ITEMS:\n$env(LISTBOX_ITEMS)"
		        catch {exec echo $tmpText >> $dialogFile}
		}

		#
		# Determine which button was selected
		#
	        set buttonNumber 0
	        foreach label $buttonList {
	                if {[cequal $answerDialog $buttonNumber]} {
				set tmpText "USER SELECTION: $label\n"
	                        catch {exec echo $tmpText >> $dialogFile}
			}
			incr buttonNumber 1
		}
	}
}				; ## end routine GUI_Dialog_log



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Calc_Location
 #
 #	The routine will calculate the location of given the pecentages 
 #	and the parent window.  A few examples are provided below:
 #
 #	e.g. GUI_Calc_Location 10 15 .   
 #		This will return a location which is 10% of the total screen
 #		in the x direction and 15% of the total screen in the y
 #		direction.
 #
 #	e.g. GUI_Calc_Location 10 15 .window1   
 #		This will return a location which is offset from the position
 #		of the parent window by 10% of the total screen in the x
 #		direction and 15% of the total screen in the y direction.
 #
 # Inputs: 
 #
 #	percentX - The percentage to offset from the parent on the x-axis,
 #		   which must be in the range 0 - 100.
 #
 #	percentY - The percentage to offset from the parent on the y-axis,
 #		   which must be in the range 0 - 100.
 #
 #	parent   - The parent window (defaults to the root window if the
 #		   parameter is {} or was not specified)
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #
 #	A valid geometry string.  If either of the percentages are 
 #	out of range then the geometry string will be +0+0 (upper 
 #	left corner).
 #
 # Notes:
 #
 #	When using the geometry string some window managers may override
 #	the location of the window, if the window would not have fit on
 #	the screen using the original geometry string.
 #
##---------------------------------------------------------------------------
proc GUI_Calc_Location {percentX percentY {parent {}}} {

	#
	# Default the parent to the root window if not defined
	#
        if [cequal $parent ""] {set parent .}
     
	#
	# Make sure the range of the input parameters is within 0-100%
	# If it is not then return "+0+0" as the default
	#
	if { $percentX < 0 || $percentX > 100 || $percentX < 0 || \
		$percentX > 100 } {
		return "+0+0"
	}

	#
	# Get the parent's x-axis and y-axis position
	#
        set parentX [winfo x $parent]
        set parentY [winfo y $parent]

	#
	# Get the total width/height of the display
	#
        set screenX [expr [winfo screenwidth .] / 100]
        set screenY [expr [winfo screenheight .] / 100]

	#
	# Determine the new x-axis and y-axis positions
	#
        set x [expr $parentX + [expr $screenX * $percentX]]
        set y [expr $parentY + [expr $screenY * $percentY]]
     
        return "+$x+$y"
}				; ## end routine GUI_Calc_Location

##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Center
 #
 #	The routine will center a window on the screen regardless of
 #	display type
 #
 # Inputs: 
 #	aWindow - The window to be centered
 #
 #	centerOnWidget - The widget which determines where the 
 #			 center is.  By default it is the full 
 #			 screen.
 #
 # Outputs: 
 #	Centers the window given by window name
 #
 # Returns: 
 #	None
 #	
 # Notes:
 #
 #    o If you call this routine after the window is mapped to
 #	the screen the window will "jump" from its current 
 #	position to the center point.
 #	
 #    o Do NOT call 'tkwait visibility' after calling GUI_Center.
 #      The VisibilityNotify event is processed when GUI_Center
 #	invokes 'update' and therefore the event no longer exists
 #	upon the return from this routine.
 #
##---------------------------------------------------------------------------
proc GUI_Center {aWindow {centerOnWidget {}}} {

	if { ![winfo exists $aWindow] } {
		return
	}

	#
	# Make sure we are up to date.
	#
	wm withdraw $aWindow
	update

	#
	# Find the center point for the aWindow.  The center point
	# will be the same as the center point of centerOnWidget, or
	# if centerOnWidget is not defined then the center point 
	# will be the center of the screen.
	#
	if {[catch {set cow_ismapped [winfo ismapped $centerOnWidget]}]} {
		set cow_ismapped 0
	}
	if {$cow_ismapped} {
		#
		# Get the center of the centerOnWidget so we know
		# what point should be the center for the aWindow.
		#
		set centerPointX [expr [winfo rootx $centerOnWidget] + \
			([winfo width $centerOnWidget] / 2)]
		set centerPointY [expr [winfo rooty $centerOnWidget] + \
			([winfo height $centerOnWidget] / 2)]
	} else {
		#
		# Get the center of the entire display
		#
	        set centerPointX [expr [winfo screenwidth .] / 2]
	        set centerPointY [expr [winfo screenheight .] / 2]
	}

	#
	# Get the width and height of the window
	#
	set winW [winfo reqwidth $aWindow]
	set winH [winfo reqheight $aWindow]

	#
	# Determine the x and y coordinates
	#
	set x [expr $centerPointX - ($winW / 2)]
	set y [expr $centerPointY - ($winH / 2)]

	#
	# Place the window in the center and display it
	#
	wm geometry $aWindow +$x+$y
	wm deiconify $aWindow
}				; ## end routine GUI_Center



##---------------------------------------------------------------------------
 #
 # Proc:  AddTicks
 #
 #	Adds ticks to the expected number of ticks for the indicator.
 #
 #
 # Inputs: 
 #	n - number of ticks to add 
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #	Operates on globals maxTicks and curTick
 #
##---------------------------------------------------------------------------
proc AddTicks {n} {    
	global maxTicks;	# Total number of ticks expected 
	global curTick;		# index of last tick  

	set ticksLeft [expr $maxTicks - $curTick]
		
	if {$ticksLeft == 0} {
		return
	}
	
	#
	# Add the tick.  We now want to have $ticksLeft + $n ticks left
	# over, but we want our current % done to stay the same.  (otherwise
	# the indicator would move backwards when we added the ticks.)
	# We need to make sure our new # of ticks left represents the same
	# percentage of the indicator as before. 
	# 
	set maxTicks [expr ceil((double($ticksLeft + $n) / \
			double ($ticksLeft)) * $maxTicks)]
	set curTick [expr $maxTicks - ($ticksLeft + $n)] 
}				; ## end routine AddTicks
##---------------------------------------------------------------------------
 #
 # Proc:  SetTicks
 #
 #	Resets the # of total ticks expected by the indicator to the
 #	requested amount while keeping the percentage done the same.
 #
 #
 # Inputs: 
 #	n - number of ticks to expect
 #
 # Outputs: 
 #	nothing
 #
 # Returns: 
 #	nothing
 #
 # Notes:
 #	Operates on globals maxTicks and curTick
 #
##---------------------------------------------------------------------------
proc SetTicks {n} {    
	global maxTicks;	# Total number of ticks expected 
	global curTick;		# index of last tick  

	set ticksLeft [expr $maxTicks - $curTick]
		
	if {$ticksLeft == 0} {
		return
	}
	
	#
	# Set the ticks.  We now want to have $n ticks left
	# over, but we want our current % done to stay the same.  (otherwise
	# the indicator would move backwards when we added the ticks.)
	# We need to make sure our new # of ticks left represents the same
	# percentage of the indicator as before. 
	# 
	set maxTicks [expr ceil((double($n) / \
			double ($ticksLeft)) * $maxTicks)]
	set curTick [expr $maxTicks - $n] 
}

##---------------------------------------------------------------------------
 #
 # Proc:  TickEvery
 #
 #	Set up indicator to tick every n seconds.  Automatically adds a tick
 #	to the total accepted by the indicator every time a tick occurs.
 #
 # Inputs: 
 #	n - number of seconds to tick 
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc TickEvery {n} {    

	global autotick
	
	#
	# if autotick is false then the indicator has received
	# a "tickevery end" command
	#
	if {$autotick} {
		AddTicks 1 
		GUITick
		after [expr $n * 1000] "TickEvery $n"
	}
}				; ## end routine TickEvery


##---------------------------------------------------------------------------
 #
 # Proc:  GUICheck
 #
 #	This forces the progress indicator temporarily to 100%, places a 
 #	check mark next to the current item, then highlights
 #	the next checklist item and sets the progress bar back to 0%.
 #
 # Inputs: 
 #	None 
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUICheck {} {

	global fgcolor;			# foreground of inactive checklist item
	global bgcolor;			# background of inactive check item
	global curTick;			# current tick number 
	global curCheckPoint;		# current check point
	global bar;			# id of progress bar rectangle
	global checkCount;		# total number of checkpoints 
	global maxBarLength;		# maximum progress bar length
	global locmnt;			# mount point of distribution
	global Label;			# label array
	global maxTicks;		# maximum num of ticks for this bar
	global checkList;		# array of checklist items
	global widget;			# parent widget name
	global checkmark;		# filename of check mark image
	global origTicks;		# starting maxTicks for each bar
					# before any ticks are added 
					# dynamically (post-initialization).

	#
	# Return success if this status indicator doesn't include a
	# checklist
	#
	if { [llength $checkList] == 0 } {
		return 
	}
		

	#
	# Return success if we have already checked off all of the 
	# checklist items
	#
	if {$curCheckPoint >= $checkCount} {
		return 
	}

	# 
	# Temporarily set the progress bar to 100% to show that the current
	# action completed
	#
	# Finish ticks for current item
	#
	set ticksLeft [expr $maxTicks - $curTick]
	for {set i 0} {$i < $ticksLeft} {incr i} {
		GUITick
	}
	
	# 
	# Without some delay here, we'll never see the progress bar
	# finish 
	#
	after 500 {
		if { [info exists bar] } { 
				${widget}.statusBarC delete $bar
				${widget}.statusPercentL \
					configure -text "0%"
		}    
	}
	#
	# Check off current check point
	#
	${widget}.checkBoxB${curCheckPoint} configure \
		 -image [image create photo -file $checkmark]
	#
	# Change foreground and background back to normal so that
        # this item no longer appears highlighted
	#
	${widget}.checkLabelB${curCheckPoint} configure -fg $fgcolor \
		-bg $bgcolor
	
	# 
	# if we aren't at the last check point, highlight the 
	# next check point by reversing the foreground and background
	#
	if { [expr $curCheckPoint + 1] < $checkCount   } {
		${widget}.checkLabelB[expr $curCheckPoint + 1] configure \
			-bg $fgcolor -fg $bgcolor
	}

	incr curCheckPoint
	set curTick 0
	set maxTicks $origTicks

	return 1
}				; ## end routine GUICheck



##---------------------------------------------------------------------------
 #
 # Proc:  GUITick
 #
 #	For GUI, this moves the progress bar by one tick. Does nothing
 #	if the progress bar is already complete.
 #
 # Inputs: 
 #	None  
 #
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	1 on success
 #	0 on failure
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUITick {} {
 
	global curCheckPoint;	# current check point
	global curTick;		# current tick number 
	global maxTicks;	# maximum number of ticks
	global maxBarLength;	# maximum length of progress bar
	global barHeight;	# progress bar height
	global bar;		# id of progress bar rectangle
	global checkCount;	# total number of check points
	global widget;		# name of status indicator frame
	global checkList;	# list of check point labels
	global tickLock;	# lock indicator
	global pendingTicks;	# the number of ticks that have yet to 
				# be processed

	#
	# Lock the tick process and force all ticks to wait until this
	# one is complete
	#
	set tickLock 1


	#
	# Make sure we aren't past the last check point
	#
	if { (${curCheckPoint} < $checkCount) || ([llength $checkList] == 0)} {
		#
		# make sure were not already at 100%
		#
		if { ${curTick} < $maxTicks} {
			#
			# Since pendingTicks can be updated asynchronously
			# and we want to use the same value throughout the
			# routine, set a local variable to the current value.
			#
			set newTicks $pendingTicks
			if { $newTicks > 0 } {
				set curTick [expr $curTick + 1 + $newTicks]
			} else {
				set curTick [expr $curTick + 1]
			}

			#
			# Make sure the current ticks does not
			# exceed the maximum ticks.
			#
			if { $curTick > $maxTicks } {
				set curTick $maxTicks
			}

			#
			# calculate percentage done.. add 0.0 to get 
			# floating point result
			#
			set percentDone [expr ($curTick + 0.0) / $maxTicks]

			set barLength [expr ${percentDone} * ${maxBarLength}]
			  
			set x1 0c
			set y1 0c
			#
			# padding to ensure bar fills canvas area
			#
			set x2 [expr $barLength + .3]c
			set y2 [expr $barHeight + .1]c 	

			if { [info exists bar] } { 
				${widget}.statusBarC delete $bar
			}    
			set bar [${widget}.statusBarC create rectangle \
				$x1 $y1 $x2 $y2 -width .5m \
				-outline black -fill DarkRed ]
			${widget}.statusPercentL \
			configure -text [expr int ($percentDone * 100)]%

			#
			# Decrement the pendingTicks value to reflect the
			# number of ticks we have processed
			#
			set pendingTicks [expr $pendingTicks - $newTicks]

			#
			# If the ticks are coming in too fast the 
			# bar is not drawn and we do not see the
			# progress, so force a redraw every time.
			#
			update
		}
	}  

	#
	# Unlock the tick process
	#
	set tickLock 0
}				; ## end routine GUITick



##---------------------------------------------------------------------------
 #
 # Proc:  ReadCommandPipe 
 #	Read from a pipe. Calls the appropriate status
 #	control routine based on the command string read from the pipe.  
 #	The following are legal commands:
 #		tick <number of ticks | clear > 
 #				calls StatusTick the appropriate # of times
 #				"tick clear" clears the progress bar
 #
 #		check - Calls Status Check to check off the current checklist
 #			item
 #
 #		tickevery <n|end> - tick automatically every n seconds         
 #				    or end auto ticking.
 #	
 # 		addticks <n> - add n ticks to the total expected by the
 #			       indicator.  
 #		                         
 #		message <message string> - Displays the message string
 #
 #		exit - exits 
 #
 # Inputs: 
 #	None
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 # 	None
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc ReadCommandPipe {} {

	global maxTicks;		# total ticks expected
	global origTicks;		# original total ticks before addticks
	global curTick;			# current tick 
	global pipeID;			# file ID of pipe to read commands from
	global widget;			# parent widget for status indicator
	global autotick;		# flag to indicate when auto ticking
                                        # is on 
	global bar;			# id of progress bar object on canvas
	global tickLock;		# lock indicator
	global pendingTicks;		# the number of ticks that have yet
					# to be processed

	gets $pipeID commandStr

	#
	# set the first word in the string to command, and the
	# rest to args 
	#
	set arg [lassign $commandStr command]
	set command [string tolower $command]
	
	switch -- $command {
		tick {
			if [cequal $arg "clear"] {
				if { [info exists bar] } {
                                	${widget}.statusBarC delete $bar
					${widget}.statusPercentL configure -text "0%"
                        	}
				set maxTicks $origTicks
				set curTick 0
			} elseif [string match {[0-9]*} $arg] {
				for {set i 0} {$i < $arg} {incr i} {
					if { $tickLock } {
						incr pendingTicks
					} else {
						GUITick
					}
				}
			} else {
				puts stderr "Status pipe: bad args \
					to ticks command - usage: \
					tick <number of ticks | clear>"
				flush stderr
			}
		}
		check {
			GUICheck
		}
		message {
			${widget}.statusMessageF.statusMessageM \
					configure -text $arg 
		}
		addticks {
			if {[string match {[0-9]*} $arg]} {
				AddTicks $arg
			}
		}
		setticks {
			if {[string match {[0-9]*} $arg]} {
				SetTicks $arg
			}
		}		
		tickevery {
			if {[string match "end" $arg]} {
				set autotick 0
			} elseif {[string match {[0-9]*} $arg]} {
				set autotick 1
				TickEvery $arg 
			}
		}
		exit {
			exit 1
		}
		default {
			puts stderr \
				"Status pipe: unrecognizable command: $command"
			flush stderr
		}
	}
}				; ## end routine ReadCommandPipe


##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Status_Create
 #
 #	Displays a status dialog with checklist items and a
 #	progress indicator
 #
 #	The procedures StatusTick, StatusCheck, and Status_Message
 #	are used to control the progress output.
 #
 # Inputs: 
 #
 #	totalTicks - number of ticks expected by indicator
 #	aWidget - a valid widget name
 #	labelList - a list of check point labels. 
 #	length - length of the status bar in centimeters	
 #	height - height of indicator
 #	pipeName - file name of pipe to use to control the indicator
 # 	 	
 # Outputs: 
 #	Creates the status indicator
 #
 # Returns: 
 #	None
 #
 # Notes:
 #	This routine sets the following globals:
 #	  checkList, maxBarLength, barHeight, curTick, CurCheckPoint, 
 #	  maxTicks, checkCount, widget, pipeID, fgcolor, bgcolor, origTicks
 #
##---------------------------------------------------------------------------
proc GUI_Status_Create {{totalTicks} {aWidget} {labelList} {length} {height}\
			{pipeName}} {

	global auto_path;		
	global checkList;	# list of labels to display in the checklist
	global maxBarLength;	# length of progress bar in centimeters  
	global barHeight;	# height of progress bar
	global curTick;		# current progress bar tick number
	global curCheckPoint;   # currently highlighted checkPoint
	global maxTicks;	# number of ticks to take progress bar to 100%
	global checkCount;	# total number of check points in checklist
	global widget;		# name of status indicator frame
	global pipeID;		# fileID of fifo for indicator control	
	global locmnt;		# distribution mount point
	global checkbox;	# empty check box bitmap
	global checkmark;	# checkmark bitmap
	global fgcolor;		# inactive checklist item foreground color
	global bgcolor;		# inactive checklist item background color
	global origTicks;	# since maxTicks can change if a tick is added
				# dynamically, we need to keep the original
				# number around to reset maxTicks for the
				# next checklist item
	global tickLock;	# lock indicator
	global pendingTicks;	# the number of ticks that have yet
				# to be processed


	if {![info exists locmnt]} {
		set locmnt ""
	} 
	
	set checkmark $locmnt/usr/dt/appconfig/icons/C/installckmk.m.pm
	set checkbox $locmnt/usr/dt/appconfig/icons/C/installnockmk.m.pm

	set auto_path "$auto_path ."
	
	set tickLock 0
	set pendingTicks 0

	if {$totalTicks < 1} {
		puts stderr "totalTicks value invalid: $totalTicks"
		flush stderr
		return 0
	}

	if {$height <= 0} {
		puts stderr "height value invalid: $height"
		flush stderr
		return 0
	}

	if {$length <= 0} {
		puts stderr "length value invalid: $length"
		flush stderr
		return 0
	}

	if {[cequal $pipeName ""]} {
		puts stderr "pipe name not set."
		flush stderr
		return 0
	}
		
	set checkList $labelList
	set maxTicks $totalTicks
	set origTicks $totalTicks
	set widget $aWidget
	set maxBarLength $length
	set barHeight $height
	set curCheckPoint 0
	set curTick 0
	set checkCount 0

	#
	# Create a frame for the entire status indicator
	#
	frame $widget -width [expr $maxBarLength + 0.2]c 

	#
	# Create a frame to hold the status message area and the progress
	# bar. This is necessary to line up the percentage done with the
	# right side of the progress bar.  
	#
	frame ${widget}.statusF

	#
	# Create a frame for the status message area.  This area will
	# include a label for status messages on the left, and a
	# percentage done on the right
	#    
	frame ${widget}.statusMessageF

	# 
	# Create the percentage done label
	#  
	label ${widget}.statusPercentL -text "0%"

	#
	# Create Status Bar Area     
	#
	canvas ${widget}.statusBarC -relief sunken \
		-borderwidth 2 \
		-width [expr $maxBarLength + 0.2]c \
		-height ${barHeight}c
	
	#
	# Status Message Area 
	#
	message ${widget}.statusMessageF.statusMessageM \
		-width [expr $maxBarLength - 2]c
	
	#
	# Do this if the caller specified checkpoint labels
	#	
	if {[llength $checkList] > 0 } {
		#
		# Create CheckPoint Labels
		#
		frame ${widget}.checklistF
		pack ${widget}.checklistF -pady .05i
	
		foreach label $checkList {
			#
			# Create a frame for each item to hold both 
			# the check box and the label
			#
			frame ${widget}.checklistF.checkF${checkCount} 
			pack ${widget}.checklistF.checkF$checkCount -anchor w
	
			#
			# Both the check box and the label will be buttons
			# so the label could have the ability to be greyed
                        # out.
			#
			button ${widget}.checkBoxB${checkCount} \
				-image [image create photo -file $checkbox] \
				-relief flat -borderwidth 0 -takefocus 0
			button ${widget}.checkLabelB${checkCount} \
				-text $label -takefocus 0 \
				-borderwidth 0 
			pack ${widget}.checkBoxB${checkCount} \
				${widget}.checkLabelB${checkCount} \
				-side left \
				-in ${widget}.checklistF.checkF${checkCount}

			incr checkCount
		}
		set fgcolor [option get ${widget}.checkLabelB0 \
			foreground Foreground]
 		set bgcolor [option get ${widget}.checkLabelB0 \
                        background Background]
		#
		# highlight first label
		#
		${widget}.checkLabelB0 configure -bg $fgcolor -fg $bgcolor

	} 

	pack $widget 
	pack ${widget}.statusF
	pack ${widget}.statusPercentL
	pack ${widget}.statusMessageF.statusMessageM -side left -fill x
	pack ${widget}.statusMessageF -fill x -anchor e -in ${widget}.statusF
	pack ${widget}.statusBarC -anchor e -in ${widget}.statusF

	#	
	# Open the pipe to process commands from
	#
	if {[cequal $pipeName ""]} {
		return 0
	}

	if ![file exists $pipeName] {
		exec mknod $pipeName p
 	}
	
	set pipeID [open $pipeName r+]

	#
	#	Start processing commands read from the pipe
	#	to control the dialog
	#
	fileevent $pipeID readable {ReadCommandPipe}

	return 1
}				; ## end routine GUI_Status_Create



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Status_BindPFH 
 #
 #	This routine binds the appropriate PFH message catalog tags to
 #	the status indicator widgets
 #
 # Inputs: 
 #
 #	progressPFHTag - tag to bind to the progress bar
 #	checkPFHTagList - list of tags to bind to the checklist items
 #		   	if the status indicator was initialized with no 
 #			checklist items, this list can be left empty
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	0 on failure
 #	1 on success
 #
 # Notes:
 #
 #	There must be a pointer messages area created by the 
 #	sm_buildHelpText routine in existance prior to the call 
 #	to this routine.  It also assumes that the message
 #	catalogs have been created using sm_messageCatalogInitialization.
 #
##---------------------------------------------------------------------------
proc GUI_Status_BindPFH {progressPFHTag {checkPFHTagList {} } } {

	if {![winfo exists .statusF.status.statusBarC]} {
		puts stderr \
		   "GUI_Status_Create must be called before GUI_Status_BindPFH"
		flush stderr
		return 0
	}

	HelpPointerMessage .statusF.status.statusBarC $progressPFHTag

	set i 0

	foreach tag $checkPFHTagList {
		HelpPointerMessage .statusF.status.checkLabelB${i} $tag 
		incr i
	}

	return 1
}				; ## end routine GUI_Status_BindPFH



##---------------------------------------------------------------------------
 #
 # Proc:  HelpPointerMessage
 #
 #	This routine creates a pointer message with the text specified
 #      in "message" and assigns it to the widget specified in "widget".
 #
 # Inputs: 
 #
 #	widget - The widget to which the pointer message will be bound
 #
 #	tag    - The message tag from the message catalog
 #
 #	args   - The arguments used to format the message string (optional)
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #
 #	There must be a pointer messages area created by the 
 #	sm_buildHelpText routine in existance prior to the call 
 #	to this routine.  It also assumes that the message
 #	catalogs have been created using sm_messageCatalogInitialization.
 #
##---------------------------------------------------------------------------
proc HelpPointerMessage {widget tag args} {

	global symbolicName


	if {! [winfo exists $widget]} {return}

	#
	# Get the message string and format if needed
	#
	set msgStr [sm_catgets $tag PFH]
	if {[info exists args]} {
		#
		# Extra formatting arguments will be ignored, but 
		# not enough arguments causes a stack dump.  If there
		# are not enough arguments the msgStr will remain as
		# it was in the message catalog, which is better than
		# nothing.
		#
		catch {set msgStr [format $msgStr $args]}
	}

        #
        # Associate the pointer message to the given widget
        #
        bind $widget <Enter> " \
		HelpEntered_widget {$msgStr} \
			$symbolicName([sm_getToplevelName $widget]Help)"
        bind $widget <Leave> " \
		HelpLeaving_widget"
}				; ## end routine HelpPointerMessage


##---------------------------------------------------------------------------
 #
 # Proc:  GUI_SetInitialInstallFont
 #
 #	This routine sets the default font during an initial install.
 #	The purpose is to get the right font in a foreign locale.
 #
 # Inputs: 
 #
 #	font_tag - index into Fonts() array.
 #
 # Outputs: 
 #	None
 #
 # Returns: 
 #	None
 #
 # Notes:
 #
 #	The code assumes that the /isl/languages/$LANG/ directory
 #	contains a tclIndex file that identifies a locale-specific
 #	guii_init_fonts{}.  This routine creates a global array
 #	called Fonts() naming various fonts appropriate to the
 #	current locale.
 #
##---------------------------------------------------------------------------
proc GUI_SetInitialInstallFont {{font_tag MESSAGE}} {

	global auto_path \
		env \
		Fonts \
		ROOT_DIR

	#
	# Ignore repeat calls to this routine.
	#
	if {[array exists Fonts]} {
		return
	}

	#
	# Import current locale name.
	#
	set lang "C"
	if {[info exists env(LANG)]} {
		set lang $env(LANG)
	}

	#
	# Allow isl/ directory to be elsewhere than the root
	# file system, such as in the test harness.
	#
	set initial_install_root ""
	if {[info exists ROOT_DIR]} {
		if {[file isdirectory $ROOT_DIR]} {
			set initial_install_root $ROOT_DIR
		}
	}

	#
	# Do nothing if the locale-specific directory
	# does not exist.
	#
	set loc_dir $initial_install_root/isl/languages/$lang
	if {! [file isdirectory $loc_dir]} {
		return
	}

	#
	# Call the locale-specific font initialization routine.
	#
	set auto_path "$loc_dir $auto_path"
	set loc_font_cmd guii_init_fonts
	if {[catch {$loc_font_cmd} err]} {

		#
		# Don't report any error if the command does not
		# exist -- that's the normal English (C locale)
		# case. Instead call GUI_SetEnglishFonts 
		#
		if {! [lempty [info procs $loc_font_cmd]]} {
			puts stderr "$loc_font_cmd{}:\n  $err"
		} elseif {[catch {GUI_SetEnglishFonts} err]} { 
			puts stderr "GUI_SetEnglishFonts:\n $err"
		}
	}

	#
	# Set the default font.
	#
	if {[info exists Fonts($font_tag)]} {
		option add {*font} $Fonts($font_tag)
	}

}				; ## end routine GUI_SetInitialInstallFont

## --------------------------------------------------------------------------
 # Proc: GUI_SetEnglishFonts
 #
 # Inputs:
 #	None
 #
 # Outputs:
 #	None
 #
 # Returns:
 #	Nothing
 #
 # Notes:
 #	Initialize the fonts used to display English text.
## --------------------------------------------------------------------------
proc GUI_SetEnglishFonts {} {

	global Fonts

	# Define the default fonts based on screen width.  Note that the
	# TABLE and HEADING fonts must be fixed width, and must be the
	# same width, because they are used together to display tables
	# of data.
	#
	if {[winfo screenwidth .] <= 800} {
		set Fonts(DATA)	"-Adobe-Helvetica-Medium-R-Normal--8-80-*"	
		set Fonts(DBUTTON) "-Adobe-Helvetica-Medium-R-Normal--10-100-*"
		set Fonts(ENTRY)   "-Adobe-Helvetica-Medium-R-Normal--8-80-*"
		set Fonts(LABEL)  "-Adobe-Helvetica-Medium-R-Normal--8-80-*"
		set Fonts(TABLE)   "5x8"
		set Fonts(HEADING) "5x8"
		set Fonts(MESSAGE) "-Adobe-Helvetica-Medium-R-Normal--10-100-*"
	} else {
		set Fonts(DATA)	"-Adobe-Helvetica-Medium-R-Normal--14-140-*" 
		set Fonts(DBUTTON) "-Adobe-Helvetica-Bold-R-Normal--14-140-*"
		set Fonts(ENTRY) "-Adobe-Helvetica-Medium-R-Normal--14-140-*"
		set Fonts(LABEL) "-Adobe-Helvetica-Bold-R-Normal--14-140-*" 
		set Fonts(TABLE) "8x13" 
		set Fonts(HEADING) "8x13bold"
		set Fonts(MESSAGE) "-Adobe-Times-Medium-R-Normal-*-180-*"
	}

} ; ## End GUI_SetEnglishFonts

##
