#!/usr/share/sysman/bin/sysmansh
# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: ismprocs.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:30 $
# 
# Procs:
#  proc ism_GetItems   { IsmHandle } 
#    given an initialized IsmHandle, it will return all the items in the ISM
#
#  proc ism_GetAttrs   { IsmHandle ItemName } 
#    given an initialized IsmHandle and an Item Name, it will return all the
#    attributes
#
#  proc ism_GetIndex { IsmHandle ItemName }
#    given an initialized IsmHandle and an Item Name, it will return the
#    index key
#
#  proc ism_GetLiteral { IsmHandle ItemName AttrName }
#    given an initialized IsmHandle, an ItemName and an Attribute, it will
#
#      - return the value of the accepted literals if available
#      - return a {} if no literal was specified in the attribute validiation clause
#
#  proc ism_GetAttrVal { IsmHandle ItemName AttrName }
#    given an initialized IsmHandle, an ItemName and an Attribute, it will
#
#      - return the value of the Attribute, if its order 1
#      - return a list of values based on the records if its order n
#
#  proc ism_GetOrder   { IsmHandle ItemName }
#    given an initialized IsmHandle and Item Name it will return the order of
#    the item
#
#  proc IsmInit        { AppName } 
#    given an appname (as in the directory /usr/share/sysman/*config), it will
#    source in the ISM file, global a variable called IsmHdl and do
#    initialization for that ISM.
#
#  proc Display        { IsmHandle }
#    Sample proc to show the usage of all the above procs.  


##
## ismprocs.tcl
##
proc _TestEmpty {field errMsg} {

  if { $field == 0 } {
     puts stderr "$errMsg"
     return 0;
  }
  return 1;

}

################################################################

proc ism_GetItems { {IsmHandle {0}} } {
##
## Returns a list of items available in the IsmHandle
##

  
  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  global ISMDEFs
  return [keylkeys ISMDEFs($IsmHandle)]

}

proc ism_GetAttrs { {IsmHandle {0}} {ItemName {0}} } {
##
## Returns a list of attributes available in the 
## ItemName for a given IsmHandle
##

  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $ItemName "Supply an Item Name"] == 0 } {
     return 0;
  }

  if { [lsearch -exact [ism_GetItems $IsmHandle] $ItemName] == -1 } {
     puts stderr "$ItemName does not exist in $IsmHandle ISM"
     return 0;
  }

  set attrList ""
  global ISMDEFs

  keylget ISMDEFs($IsmHandle)  ${ItemName}.attributes aL
  set attrList [keylkeys aL]
  unset aL
  return $attrList

}

proc ism_GetIndex { {IsmHandle {0}} {ItemName {0}} } {

##
## Returns a list of indexes available in the 
## ItemName for a given IsmHandle
##

  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $ItemName "Supply an Item Name"] == 0 } {
     return 0;
  }

  if { [lsearch -exact [ism_GetItems $IsmHandle] $ItemName] == -1 } {
     puts stderr "$ItemName does not exist in $IsmHandle ISM"
     return 0;
  }

  set attrList ""
  global ISMDEFs

#
#  If the index isn't there then return the first attribute name in its place
#
  
  if {[keylget ISMDEFs($IsmHandle)  ${ItemName}.index aL] == 0} {
      set attrList [lindex [ism_GetAttrs $IsmHandle $ItemName] 0]
  } else {
      set attrList [lindex $aL 1]
  }
  return [join $attrList]

}

proc ism_GetLiteral { {IsmHandle {0}} {ItemName {0}} {AttrName {0}} } {
##
## Returns the value of an attribute literal declaration for a given
## ItemName and IsmHandle
##

  global ISMDEFs

  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $ItemName "Supply an Item Name"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $AttrName "Supply an Attribute Name"] == 0 } {
     return 0;
  }

  if { [lsearch -exact [ism_GetItems $IsmHandle] $ItemName] == -1 } {
     puts stderr "$ItemName does not exist in $IsmHandle ISM"
     return 0;
  }

  if { [lsearch -exact [ism_GetAttrs $IsmHandle $ItemName] $AttrName] == -1 } {
     puts stderr "$AttrName does not exist in $ItemName item"
     return 0;
  }

  set ltrlList {}
  if {[keylget ISMDEFs($IsmHandle)  ${ItemName}.attributes.${AttrName}.valid aL] != 0} {
      if {[lindex $aL 0] == "ltrl"} {
      	set ltrlList [lindex $aL 1]
      }
  }

  return [join $ltrlList]
}

proc ism_GetAttrVal { {IsmHandle {0}} {ItemName {0}} {AttrName {0}} } {
##
## Returns the value of an attribute, AttrName, for a given
## ItemName and IsmHandle
##

  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $ItemName "Supply an Item Name"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $AttrName "Supply an Attribute Name"] == 0 } {
     return 0;
  }

  if { [lsearch -exact [ism_GetItems $IsmHandle] $ItemName] == -1 } {
     puts stderr "$ItemName does not exist in $IsmHandle ISM"
     return 0;
  }

  if { [lsearch -exact [ism_GetAttrs $IsmHandle $ItemName] $AttrName] == -1 } {
     puts stderr "$AttrName does not exist in $ItemName item"
     return 0;
  }

  if {[ism_GetOrder $IsmHandle $ItemName] == 1} {
    return [_ism_GetOrder1Value $IsmHandle $ItemName $AttrName]
  }
  return [_ism_GetOrdernValue $IsmHandle $ItemName $AttrName]
}

proc _ism_GetOrder1Value { IsmHandle ItemName AttrName } {
##
## Internal Proc:
##   Returns the value of an attribute, AttrName, for a given
##   ItemName and IsmHandle in an Order 1 Resource
##

  if { [info level] <= 1 } {
    puts "[info level 0]: is an internal proc - cannot be called!!"
    return 0;
  }

  global $IsmHandle SCode

  set ItmHdl [ ItmMgr $IsmHandle $ItemName new ]

  ItmMgr $ItmHdl set _action load

  set RetCode [ ItmMgr $ItmHdl validate ]
  if { $RetCode != $SCode(OK) } {
    return "";
  }
  set RetCode [ ItmMgr $ItmHdl apply ]
  if { $RetCode != $SCode(OK) } {
    return "";
  }

  set Val [ ItmMgr $ItmHdl get $AttrName]

  ItmMgr $ItmHdl free

  return $Val
}

proc _ism_GetOrdernValue { IsmHandle ItemName AttrName } {
##
## Internal Proc:
##   Returns a keyedlist of values based on record number for an attribute,
##   AttrName, for a given ItemName and IsmHandle in an Order n Resource
##

  if { [info level] <= 1 } {
    puts "[info level 0]: is an internal proc - cannot be called!!"
    return 0;
  }

  global $IsmHandle SCode

  set ItmHdl [ ItmMgr $IsmHandle $ItemName new ]

  ItmMgr $ItmHdl set _action loadList

  set RetCode [ ItmMgr $ItmHdl validate ]
  if { $RetCode != $SCode(OK) } {
    return "";
  }
  set RetCode [ ItmMgr $ItmHdl apply ]
  if { $RetCode != $SCode(OK) } {
    return "";
  }

  set ValList {}
  foreach obj [set objList [ItmMgr $ItmHdl get __list]] {
    set ValPair {}
    if {[ItmMgr $obj get _type] != $ItemName} {
      continue
    }
    lappend ValPair $obj
    lappend ValPair [ItmMgr $obj get $AttrName]
    lappend ValList $ValPair
  }

  ItmMgr $ItmHdl free

  return $ValList
}

proc ism_GetOrder { {IsmHandle {0}} {ItemName {0}} } {
##
## Returns the order of the item - either 1 for order 1 or "n" for order n.
##

  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  if { [_TestEmpty $ItemName "Supply an Item Name"] == 0 } {
     return 0;
  }
  
  if { [lsearch -exact [ism_GetItems $IsmHandle] $ItemName] == -1 } {
     puts stderr "$ItemName does not exist in $IsmHandle ISM"
     return 0;
  }
  
  global ISMDEFs

  if {[keylget ISMDEFs($IsmHandle) ${ItemName}.resource resource]} {
    if {![keylget resource order order]} {
      set order n
    }
  } else {
    set order 1
  }

  unset resource 
  return $order
}

proc IsmInit { {AppName {0}} } {
##
## Searches for the appname in /usr/share/sysman tree and 
##   - sources the Associated ISM File
##   - performs a InitISM step
##   - exports the Handle in the variable IsmHdl to the sysmansh level 0.
##

  if { [_TestEmpty $AppName "Supply AppName as in /usr/share/sysman/"] == 0 } {
     return 0;
  }

  set AppName [string tolower $AppName]

  global env

  if { [lsearch [array names env] SYSMANDIR ] == -1 } {
    set sysmandir "/usr/share/sysman"
  } else {
    set sysmandir $env(SYSMANDIR)
  }

  set dirname "${sysmandir}/${AppName}"

  if {[file exists $dirname] == 0} {
    puts stderr "$dirname directory does not exist. Please check"
    return 0; 
  }

  ## Make sure that the path is a directory that we have access to
  if {[string compare [file type $dirname] directory] != 0} {
    puts stderr "$dirname is not a directory. Please check"
    return 0; 
  }

  # get a list of files contained in the directory
  set file {}
  set file [glob -nocomplain -- "${dirname}/*.ism"]
  
  if { $file == "" } {
    puts stderr "$dirname does not have an ISM file. Please check"
    return 0; 
  }
  
  if { [llength $file] != 1} {
    puts stderr "$dirname has more than one ISM file. Please check"
    return 0; 
  }

  global ISMDEFs
  set ismlist [array names ISMDEFs]
  set idx [lsearch $ismlist $AppName*]
  # If ISM def is not in ISMDEFs, source ISM file
  if { $idx == -1 } {
  	uplevel #0 source ${file}
	}

  set ismlist [array names ISMDEFs]
  set idx [lsearch $ismlist $AppName*]
  # set Ismhdl [lindex [lsort [array names ISMDEFs] ] 1]
  set Ismhdl [lindex [array names ISMDEFs] $idx]
  uplevel #0 global IsmHdl
  uplevel #0 set IsmHdl $Ismhdl

  set result [ItmMgr $Ismhdl InitISM]
  if {$result == 0} {
      return $Ismhdl
  } else {
      return $result
  }
}

###########################################
###########################################
###### Proc To Display Everything #########
###########################################
###########################################

proc Display { {IsmHandle {0}} } {

  if { [_TestEmpty $IsmHandle "Supply an initialized ISM Handle"] == 0 } {
     return 0;
  }

  set Spacer "                            "

  foreach Item [ism_GetItems $IsmHandle] {

    puts "ItemName: $Item"
    puts "----------------------------------------------"
      
    foreach Attr [ism_GetAttrs $IsmHandle $Item] {
      	
      set AttrName [string range $Attr$Spacer 0 20]
      set putStr [format "\t%s - %s" $AttrName [ism_GetAttrVal \
                                              $IsmHandle $Item $Attr] ]
      puts $putStr
    
    }
    puts "\n\n"
  }

}

