# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: ksmutils.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:30 $
# 
set KSMCONFIG ${SysmanDir}/bin/ksmconfig

# response is an output parameter; it will be set to the name of the 
#  that category that contains target member or {} if not found
# node is the hostname to search on or "" for local host
# set_name is name of the targeted KSM set. 
# target_member is the name of the member sought
# 
# Search all categories of the targeted set for the target_member.
# Put the category name in response or {} if not found.
proc ksmFindMem { response node set_name target_member } {
    upvar $response rsp
    set rsp {}

    # get all category names
    ksmReadCatNames r1 $node $set_name

    # Search each category until the member is found
    set r {}
    foreach cat $r1 {
	if {[_ksmFind r $node name current $set_name/$cat $target_member]} {
	    set rsp $cat
	    return
	}
    }
    return
}


# response is an output parameter; it will be set to the name of the
#  member that contains the target attribute or {} if not found
# node is the hostname to search on or "" for local host
# set_name is name of the targeted KSM set. 
# category is the name of the category in which to search
# target_attribute_name is the name of the sought after attribute

# Search all members of the targeted category under the targeted set for
# the target_attribute_name
# Put the member name in response or {} if not found.
proc ksmFindAtt { response node set_name category target_attribute_name } {
    upvar $response rsp
    set rsp {}

    # get category's members
    ksmReadMemNames r2 $node $set_name $category
    foreach memb $r2 {

	# for each member, look for the attribute
	_ksmReadAtt r3 $node both current $set_name/$category/$memb \
	    $target_attribute_name

	set name [lindex $r3 0]
	if {[cequal $name $target_attribute_name]} {

	    set rsp $memb
	    return
	}
    }

    return
}

# PRIVATE FUNCTION
# response is an output parameter; it will contain the string sought or {}
# node is the hostname to search on or "" for local host
# gtype is what to get: a "value" or "name"
# atype is the attributes' "current", "saved" or "default" values
# thing is component or component/member or component/member/attr
# find is the member or attribute sought
proc _ksmFind { response node gtype atype thing find } {
    upvar $response rsp
    set rsp {}

    # If reading info on local host, it's quicker not to use -h option
    if { $node != "" && ![cequal $node localhost] } {
	set node_opt "-h $node"
    } else {
	set node_opt ""
    }

    global KSMCONFIG
    set command "$KSMCONFIG $node_opt -g $gtype -A $atype"

    # if thing is empty, we are looking for the set
    # otherwise, we are looking for component or attribute
    if { $thing != {} } {
	set command "$command $thing"
    }

    # run the command
    if {[catch {set ksm [open |$command r ]} msg]} {
	error "Could not execute command \"$command\"\nReason: $msg"
    }


    while {[gets $ksm line] >= 0} {
	set name [string trim $line]

	# ksmconfig returns "Disconnecting from" message
	# when run on remote nodes and "Subsystem error:"
	# message when you give a bad set or category name
	if {![cequal "Disconnecting from $node" $name] &&
	    ![string match "Subsystem error:*" $name] } {

	    # if we've found the name, return it in rsp
	    if {[cequal $find $name]} {
		set rsp $name
		return 1
	    }
	}
    }

    # name not found
    return 0
}

# PRIVATE FUNCTION
# response is an output parameter; it will contain the thing requested
# node is the hostname to search or "" for local host
# gtype is what to get: a "value" or "name" 
# atype is the attributes' "current", "saved" or "default" values
# thing is the set/component or set/component/member; if not present or {},
#  it means to return the list of sets
# args contains the optional remaining parameters; if present they should
#  be attribute names
proc _ksmRead { response node gtype atype {thing {}} args} {
    upvar $response rsp
    set rsp {}
    
    # If reading info on local host, it's quicker not to use -h option
    if { $node != "" && ![cequal $node localhost] } {
	set node_opt "-h $node"
    } else {
	set node_opt ""
    }

    global KSMCONFIG
    set command "$KSMCONFIG $node_opt -g $gtype -A $atype"

    # if thing is empty, we are looking for the set
    # otherwise, we are looking for component or attribute
    if { $thing != {} } {
	set command "$command $thing"
    }

    # args are specific attributes to read
    # empty args means read them all
    if { $args != {} } {
	foreach a $args {
	    set command "$command -a $a"
	}
    }

    # run the command
    if {[catch {set ksm [open |$command r ]} msg]} {
	error "Could not execute command \"$command\"\nReason: $msg"
    }

    while {[gets $ksm line] >= 0} {
	set name [string trim $line]

	# ksmconfig returns "Disconnecting from" message
	# when run on remote nodes and "Subsystem error:"
	# message when you give a bad set or category name
	if {![cequal "Disconnecting from $node" $name] &&
	    ![string match "Subsystem error:*" $name] } {
	    lappend rsp $name
	}
    }
    return
}

# response is an output parameter; it will be set to a list of
#  KSM sets available
# node is the hostname to run on or "" for local host
# 
# Set response to the (list of) KSM sets
proc ksmReadSetNames { response node } {
    upvar $response rsp

    return [_ksmRead rsp $node name current]
}

# response is an output parameter; it will be set to a list of
#  categories available under the given KSM set
# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which to look for categories
# 
# Set response to the (list of) categories under the given set
proc ksmReadCatNames { response node set_name } {
    upvar $response rsp

    return [_ksmRead rsp $node name current $set_name]
}

# response is an output parameter; it will be set to a list of
#  members under the given category in the given set.
# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which to look
# category is the category under the given set in which to look
# 
# Set response to the (list of) members under set_name/category
proc ksmReadMemNames { response node set_name category } {
    upvar $response rsp

    return [_ksmRead rsp $node name current $set_name/$category]
}

# PRIVATE FUNCTION
# response will contain a list of names, values or alternating
#   names and values, depending on the value of nameOrValue
# node is the hostname to search or "" for local host
# nameOrValue is what to return: "name", "value" or "both"
# attrType tells whether to return the attribute's "current", "default" 
#  or "saved" value
# memb is the set/category/member path
# optional args are attributes to look for
proc _ksmReadAtt { response node nameOrValue attrType memb args} {
    upvar $response rsp
    set rsp {}


    # nameOrValue will be used to index into the 
    # returned name/value pairs
    switch $nameOrValue {
	name  { set idx 0 }
	value { set idx 1 }
	both  { }	;# no op
    }

    # read the attributes (eval keeps args as individual arguments)
    eval _ksmRead r \$node value \$attrType \$memb $args

    # r will contain a header line and 
    # lists of the form "name = value"
    foreach pair $r {
	set lst [split $pair =]

	# The first line is a header that tells the category.
	# It doesn't have the form "name = value"
	if {[llength $lst] == 2} {

	    if { [cequal both $nameOrValue] } {
		# We want both name and values
		# alternating in one long list
		set name [string trim [lindex $lst 0]]
		set val [string trim [lindex $lst 1]]

		lappend rsp $name $val

	    } else {
		# we want only names or values
		lappend rsp [string trim [lindex $lst $idx]] 
	    }
	}
    }
}

# response is an output parameter; it will be set to a list of
#  attribute names for the given member of the given category in
#  the given set
# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which to look
# category is the category under the given set in which to look
# member is the member name in the given category under the given set
# 
# Set response to the (list of) attributes under set_name/category/member
proc ksmReadAttNames { response node set_name category member } {
    upvar $response rsp

    return [_ksmReadAtt rsp $node name current $set_name/$category/$member]
}

# response is an output parameter and an array; it will contain elements
#  whose names are the names of attributes and whose values are the current
#  values for those attributes in the given member of the given category in 
#  the given set
# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which to look
# category is the category under the given set in which to look
# member is the member name in the given category under the given set
# args are optional attribute names whose values are sought
# 
# Set response to an array containing attribute values under 
#   set_name/category/member.  The array is indexed by the attribute
#   name.  If args is null, the current values of all attributes are returned;
#   otherwise, only the given attributes' current values are returned.
proc ksmReadAttVals { response node set_name category member args} {
    upvar $response rsp

    # Make sure rsp is empty
    catch "unset rsp"

    # eval keeps args as individual arguments
    eval _ksmReadAtt r \$node both current \$set_name/\$category/\$member $args

    # put returned values in an array
    array set rsp $r

    return 
}

# response is an output parameter and an array; it will contain elements
#  whose names are the names of attributes and whose values are the saved
#  values for those attributes in the given member of the given category in 
#  the given set
# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which to look
# category is the category under the given set in which to look
# member is the member name in the given category under the given set
# args are optional attribute names whose values are sought
# 
# Set response to an array containing saved attribute values under 
#   set_name/category/member.  The array is indexed by the attribute
#   name.  If args is null, all attributes are returned; otherwise,
#   only the given attributes' saved values are returned.
proc ksmReadAttBoots { response node set_name category member args } {
    upvar $response rsp

    # Make sure rsp is empty
    catch "unset rsp"

    # eval keeps args as individual arguments
    eval _ksmReadAtt r \$node both saved \$set_name/\$category/\$member $args

    # put returned values in an array
    array set rsp $r

    return
}

# response is an output parameter and an array; it will contain elements
#  whose names are the names of attributes and whose values are the default
#  values for those attributes in the given member of the given category in 
#  the given set
# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which to look
# category is the category under the given set in which to look
# member is the member name in the given category under the given set
# args are optional attribute names whose values are sought
# 
# Set response to an array containing default attribute values under 
#   set_name/category/member.  The array is indexed by the attribute
#   name.  If args is null, all attributes are returned; otherwise,
#   only the given attributes' default values are returned.
proc ksmReadAttDefaults { response node set_name category member args } {
    upvar $response rsp

    # Make sure rsp is empty
    catch "unset rsp"

    # eval keeps args as individual arguments
    eval _ksmReadAtt r \$node both default \$set_name/\$category/\$member $args

    # put returned values in an array
    array set rsp $r

    return
}

# PRIVATE FUNCTION
# node is the hostname to run on, or "" for local host
# saveType is "current" or "saved"
# memb is the set/category/member path
# attribute_value_list is a list of alternating attribute names and values
# Command returns the output of the ksmconfig command
proc _ksmWriteAtt { node saveType memb attribute_value_list } {
    # If setting vals on local host, it's quicker not to use -h option
    if { $node != "" && ![cequal $node localhost] } {
	set node_opt "-h $node"
    } else {
	set node_opt ""
    }

    global KSMCONFIG
    set command "$KSMCONFIG $node_opt -s -A $saveType $memb"

    foreach {att val} $attribute_value_list {
	set command "$command -a $att=\"$val\""
    }

    return [eval exec $command]
}


# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which find the given category
# category is the category under the given set in which to find the 
#   given member
# member is the member name in the given category under the given set
#   in which to find the given attributes
# attribute_value_list is a list of alternating attribute names and values
#
# Change the current values of each given attribute in set/category/member
proc ksmWriteAttVals {node set_name category member attribute_value_list } {
    return [_ksmWriteAtt $node current $set_name/$category/$member \
		$attribute_value_list]
}

# node is the hostname to run on or "" for local host
# set_name is the KSM set name in which find the given category
# category is the category under the given set in which to find the 
#   given member
# member is the member name in the given category under the given set
#   in which to find the given attributes
# attribute_value_list is a list of alternating attribute names and values
#
# Change the saved values of each given attribute in set/category/member
proc ksmWriteAttBoots {node set_name category member attribute_value_list } {
    return [_ksmWriteAtt $node saved $set_name/$category/$member \
		$attribute_value_list]
}

#### DEBUGGING PROCEDURES

proc printAllByType { TYPE {node {}}} {
    ksmReadSetNames allSets $node
    foreach set $allSets {
	ksmReadCatNames allCats $node $set
	foreach cat $allCats {
	    ksmReadMemNames allMembs $node $set $cat
	    foreach memb $allMembs {
		catch {unset attrArray}
		ksmReadAtt${TYPE} attrArray $node $set $cat $memb
		echo "\n$set/$cat/$memb"
		if { [ catch {parray attrArray} ] }  {
		    catch {
			foreach attr $attrArray { echo $attr }
		    }
		}
	    }
	}
    }
}
