#!/usr/share/sysman/bin/sysmansh
# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: mclUtils.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:30 $
# 


# ----------------------------------------------------------------------  
# Make sure I18N message catalog is available.

# This check determines if the object has been created yet cause it's
#   implemented as a command.  Not pretty, but it works.
if {[cequal [info command I18N_catobj] ""]} {
    _Catalog I18N_catobj
    I18N_catobj catopen i18n_motif_shared_text
}
# Make sure mcl message catalog is available.
if {[cequal [info command MclUtils_catobj] ""]} {
    _Catalog MclUtils_catobj
    MclUtils_catobj catopen mclUtils
}



###
##  Common Util Procs
###
#----------------------------------------------------------------
##
## This internal helper proc will determine if a given instance of
## an object does exist in the sysmansh env.
##
#----------------------------------------------------------------
###
###  Mcl Utils 
###

proc MCL_ReadAll {component group arrayPtr orderPtr numPtr {filter ""}} {

    upvar $arrayPtr array
    upvar $orderPtr order
    upvar $numPtr   num

    set instance "/${component}/${group}"

    if {[$instance MIIsLoaded] != 1} {
	# just do this to load the Mcl
	set anyAtt [lindex [$instance children] 0]
	$anyAtt GET {} stat
	if { ! [cequal $stat ""]} {
	    return $stat
	}
        $instance ResetNext
        $instance NextIndex
    }

    $instance LOADALL array order num $filter
}

#----------------------------------------------------------------
##
##  This proc will read data from a group or rows of data from a
##  table using a user supplied key.  Any error or warning message
##  will be returned and the resultant data will be returned via
##  the resultVar variable.
##
##  If the instance is a TABLE then keycount will be > 0 and
##   we need to use GETNEXT to iterate through, else, we just
##   return the single row via GET and never increment to the
##   next entry.
##
#----------------------------------------------------------------
proc MCL_ReadRowByKey {component group resultVar key} {

    set instance "/${component}/${group}"
    upvar $resultVar result

    set result {}
    set err {}

    set result [$instance GET $key err]
    return $err
}

#----------------------------------------------------------------
##
##  This proc will read data from a group or rows of data from a
##  table.  Any error or warning message will be returned and the 
##  resultant data will be returned via the resultVar variable.
##
##  Calling this proc with the optional argument "seed" set to 0
##  will result in resetting the Management Interface (MI) pointer
##  to the top of the table.  Every subsequent call with "seed" set
##  to 1 will return rows of data in "resultVar".  When there is no
##  more  data to be returned, a Tcl NULL string will be returned in
##  "resultVar".
##
##  If the instance is a TABLE then keycount will be > 0 and
##   we need to use GETNEXT to iterate through, else, we just
##   return the single row via GET and never increment to the
##   next entry.
##
#----------------------------------------------------------------
proc MCL_ReadRow {component group resultVar {seed 1}} {

    set instance "/${component}/${group}"
    upvar $resultVar result

    set result {}

    if {$seed == 0} {
        $instance ResetNext
    }

    set err {}
    if {[$instance set keycount] == 0} {
    	 set result [$instance GET {} err]
    } else {
       	 set result [$instance GETNEXT {} err]
    }

    return $err
}

#----------------------------------------------------------------
##
## Add a row to the group/table through MCL interface.
##  
## Input:  name of the component, 
##         name of the group, 
##         a variable to return results ,
##         row to be added (Tcl list) 
## 
##         optionally the nextrow (Tcl list which makes ordered table 
##         entries) In Mcl its straight forward - In DMI we have to 
##         cleanTable and then re-write the table in this order.
##
##         optionally, a reference to re-read the rec into
##
## Output: return a Tcl String for error; {} if no errors
##
#----------------------------------------------------------------
proc MCL_AddRow {component group resultVar row {nextrow {}} {newRecRef {}}} {
  upvar $resultVar result

  # Need to reset the key map and read the key after the FLUSH and LOAD
  #   cause those procs may change key values.  Without the key map it
  #   would be impossible to find a record if the key values were changed.
  set instance "/${component}/${group}"
  set result   {}
  $instance ResetKeyMap
  $instance MIAddRow $row $nextrow
  if {[set ret [$instance FLUSH]] != {} } {
    return $ret
  }
  if {[set ret [$instance LOAD]] != {} } {
    return $ret
  }
  set keyidx [$instance MakeKeyCheckingChanges $row]

  # If pointer passed in, pass back rec just added.
  #   Picks up data added/changed by mcl.
  if { ! [cequal $newRecRef ""]} {
      upvar $newRecRef newRec
      set newRec [$instance GET $keyidx]
  }

  # Set the MI Rowpointer to the keyindex of the added row.
  $instance MISetNext $keyidx
  return {}
}


#----------------------------------------------------------------
##
## Modify an entire row through MCL or DMI interface.
##
## Input:  name of the component, 
##         name of the group, 
##         a variable to return results ,
##         row to be modified (Tcl list) - oldrow,
##         modified row (Tcl list)  - newrow
## 
##         optionally, a reference to re-read the rec into
##
## Output: return a Tcl String for error; {} if no errors
##
#----------------------------------------------------------------
proc MCL_ModRow {component group resultVar oldrow newrow {newRecRef {}}} {
  upvar $resultVar result

  # Need to reset the key map and read the key after the FLUSH and LOAD
  #   cause those procs may change key values.  Without the key map it
  #   would be impossible to find a record if the key values were changed.
  set instance "/${component}/${group}"
  set result   {}
  $instance ResetKeyMap
  $instance MIModRow $oldrow $newrow
  if {[set ret [$instance FLUSH]] != {} } {
    return $ret
  }
  if {[set ret [$instance LOAD]] != {} } {
    return $ret
  }
  set keyidx [$instance MakeKeyCheckingChanges $newrow]

  # If pointer passed in, pass back rec just modified.
  #   Picks up data added/changed by mcl.
  if { ! [cequal $newRecRef ""]} {
      upvar $newRecRef newRec
      set newRec [$instance GET $keyidx]
  }

  # Set the MI Rowpointer to the keyindex of the modified row.
  $instance MISetNext $keyidx
  return {}
}


#----------------------------------------------------------------
##
## Delete a row of data from either MCL or DMI interface.
##
## Input:  name of the component, 
##         name of the group, 
##         a variable to return results ,
##         row to be deleted (Tcl list) 
## 
## Output: return a Tcl String for error; {} if no errors
##
#----------------------------------------------------------------

  ##
  # Logic:  . Using MakeKEY to obtain the keyindex for the row to be deleted
  #         
  #         . Set the MI rowpointer to the keyindex 
  #         
  #         . Read the row of data using miReadRow 
  #           - This will give us the current row which is equal to the
  #             row to be deleted.
  #         
  #         . Read and save the row of data using miReadRow again
  #           - This will give is the current row which is the next row
  #             after the row that we are deleting (move2row).
  #         
  #         . Using MIDelRow, delete the row of data in the internal tables
  #      
  #         . Perform a FLUSH operation -> will take care of SETs in FLUSH
  #
  #         . Perform a LOAD operation - will take care of GETs in LOAD
  #
  #         . Set the MI Rowpointer to the keyindex of the move2row.
  #
  # At any point if there was an error, then return with the error
  #

proc MCL_DelRow {component group resultVar row} {

  set instance "/${component}/${group}"
  upvar $resultVar result

  set result {}
  set keyidx [$instance MakeKEY $row]
  set result [ $instance MIDelRow $row ]
  if {[set ret [$instance FLUSH]] != {} } {
    return $ret
  }
  if {[set ret [$instance LOAD]] != {} } {
    return $ret
  }
  return {}
}

#----------------------------------------------------------------
##
## Perform all kinds of validation based on the oldrow and newrow.
##
## We could be called for one of the following operations:
##     add, mod or del
##
## Logic to determine the operation:
##   
##     if both newrow and oldrow are empty - then error
##
##     if oldrow == NULL and newrow != NULL - then oper = add
## 
##     if newrow == NULL and oldrow != NULL - then oper = del
##    
##     if newrow != NULL and oldrow != NULL - then oper = mod
##
##
## What kind of validation to call?
##
##   if oper == "add" , then
##
##       forevery child of the group that newrow belongs to:
##          perform Validate of Attribute from newrow
## 
##       if group type = TABLE
##         perform ValidateRow of group on newrow
##
##       perform Validate of group 
##
##   if oper == "mod" , then
##
##       forevery child of the group that old/newrow belongs to:
##          if attribute values are different
##            perform Validate of Attribute 
## 
##       if group type = TABLE
##         perform ValidateRow of group 
##
##       perform Validate of group 
##         
##   if oper == "del" , then
##
##       forevery child of the group that oldrow belongs to:
##            perform Validate of Attribute for delete
## 
##       if group type = TABLE
##         perform ValidateRow of group 
##
##       perform Validate of group 
##         
## What CI procs could the user have to writen?
##
##    $attr  proc Validate      operation oldval  newval  
##    $group proc Validate      operation oldrow  newrow  
##    $table proc ValidateRow   operation oldrow  newrow  
##
## MCL engine will just do a return {} for the above instprocs.
##
#----------------------------------------------------------------
proc MCL_Validate {comp group resultVar oldrow newrow} {
   set instance "/${comp}/${group}"
   upvar $resultVar result

   set result {}
   set retStat {}              ;# status to return at the end
   set curStat {}              ;# return status from calls to MCL

   ##
   # Decide on the operation based on newrow and oldrow.
   # Check string len because comparing large integers 
   #   to {} will cause a stack trace.
   # 
   set newlen [ string length $newrow ]
   set oldlen [ string length $oldrow ]
   if { $newlen == 0 && $oldlen == 0 } {
     return {Error: validate options incorrect : Internal Error}
   } else {
     set operation "mod"
     if { $oldlen == 0} {
       set operation "add"
     } else {
       if { $newlen == 0} {
         set operation "del"
       } else {
         set operation "mod"
       }
     }
   }

   ##
   # need to validate each child
   #
   foreach attr [ /$comp/$group children ] {
     ##
     # get the location of the attribute in the row using GetLoc
     #
     set iloc [/$comp/$group GetLoc [ $attr ID ] ]
     set oldVal "[join [list [lindex $oldrow $iloc]]]"
     set newVal "[join [list [lindex $newrow $iloc]]]"

     ##
     #  check to see if the attribute value is valid for TCL
     #
     set curStat [_rangeValidation $attr [$attr TYPE] $newVal]

     if { $curStat == {} } {
       ##
       # if range validation passed, then perform the
       # attribute-level validation (defined by the MCL author) for
       # the type of operation that the MI wanted us to validate
       #
       if { $operation == "mod" } {
	 if { [cequal $oldVal $newVal] != 1 } {
	   set curStat [ $attr Validate "mod" $oldVal $newVal ]
	 }
       } else {
	 ##
	 # del or add
	 #
	 if { $operation == "del" } {
	   set curStat [ $attr Validate $operation $oldVal {} ]
	 } else {
	   set curStat [ $attr Validate $operation {} $newVal ]
	 }
       }
     }
     ##
     # check the curstat and append the result to retstat
     #
     if {$curStat != {}} {
       lappend retStat $curStat
       set curStat {}
     }
   }

   ##
   # call ValidateRow and check the curstat and append the result to retstat
   #
   set curStat [ /$comp/$group ValidateRow $operation $oldrow $newrow ]
   if {$curStat != {}} {
     lappend retStat $curStat
     set curStat {}
   }

   ##
   # If there is a proc called Validate then call it
   #
   set curStat [ /$comp/$group Validate $operation $oldrow $newrow ]
   if {$curStat != {}} {
     lappend retStat $curStat
   }

   ##
   # set the result string using retStat and separate each error
   # with a carriage return.
   #
   set result [ join $retStat "\n" ]
}

#----------------------------------------------------------------
##
## Obtain a default row from the group definition.  This characteristic *may*
## *not* be available through the DMI interface, we may need to fake it.
##
## Input:  name of the component, 
##         name of the group, 
##         a variable to return results ,
## 
## Output: return a Tcl String for error; {} if no errors
##
#----------------------------------------------------------------
proc MCL_GetDefaults {component group resultVar} {

  set instance "/$component/$group"
  upvar $resultVar result

  set result {}
  set ret {}
  ##
  # for every child in the group that we are interested in
  #
  foreach child [ $instance children ] {
     
      ##
      # see if the MCL writer provided a VALUE definition, which will 
      # contain a default VALUE.
      #
      catch { set val [ $child VALUE ] }
      # echo "$child $val"

      ##
      # if there is no VALUE, then we need make up one
      #
      if {$val == {}} {
        ##
	# obtain the TYPE of the attribute
	#
        switch -regexp -- [string toupper [$child TYPE]] {
        
	  DATE {
	    ##
	    # return the current date in MifDate format
	    #
	    MifDate $instance/dt
	    set val [$instance/dt getmif]
	    destroy $instance/dt
	  }
	  COUNTER -
	  COUNTER64 -
	  GAUGE -
	  INTEGER -
	  INT -
	  INT64 -
	  INTEGER64 {
	    ##
	    # return default as 0
	    #
	    set val 0
	  }
	  OCTETSTRING(.*) {
	    set val 00
	  }
	  DISPLAYSTRING(.*) -
	  STRING  -
	  STRING(.*) {
	    ##
	    # return a NULL string 
	    #
	    set val {}
	  }
       }
     }
     lappend ret $val
  }

  set result $ret
  return {}
}


#----------------------------------------------------------------
##
## Obtain the list of children for the parent that is passed to us.
##
## Input:  name of the parent 
##         a variable to return results ,
##
##   Note:  example of parent is "/component" or "/component/group"
##          i.e., "/psinfo" or "/psinfo/psopt"
##
##             if parent == component  output is a tcl list of groups
##             if parent == group/table output is a tcl list of attributes
## 
## Output: return a Tcl String for error; {} if no errors
##
#----------------------------------------------------------------
proc MCL_GetKids {parent resultVar} {

  set instance "${parent}"
  upvar $resultVar result

  set result {}

  foreach child [$parent children] {
    lappend result [file tail $child]
  }
  return {}
}

#----------------------------------------------------------------
##
## Look for the env variable called MCLDIR, if thats set use that as the
## standard location to obtain all the mcls, else make one up using
## sysmandir/ci
##
#----------------------------------------------------------------
proc GetMCLDIR {} {

  global env STD_MCLDIR
  global SysmanDir

  set STD_MCLDIR "${SysmanDir}/mcl"

  if { [lsearch [array names env] MCLDIR ] != -1 } {
    set STD_MCLDIR $env(MCLDIR)
  }

  if {[file exists $STD_MCLDIR] == 0} {
    puts stderr "$STD_MCLDIR directory does not exist. Please check"
    return 1;
  }
  
  ## Make sure that the path is a directory that we have access to
  if { [string compare [file type $STD_MCLDIR] link] != 0} {
    if { [string compare [file type $STD_MCLDIR] directory] != 0 } {
      puts stderr "$STD_MCLDIR is not a directory. Please check"
      return 1;
    }
  }

  return 0;
}
#----------------------------------------------------------------
##
## Source the MCL files that represent the MCL components (comps).
##
#----------------------------------------------------------------
proc MclInit {{comps {}}} {

  set return_msg ""
  foreach mcl [GetMclFiles $comps] {
    set return_msg [uplevel #0 source $mcl]
  }
  return $return_msg
}

#----------------------------------------------------------------
##
## Grep for MCLs and return a list of Mcl files in which they appear.
##
#----------------------------------------------------------------
proc GetMclFiles {{components {}}} {

  global STD_MCLDIR
  if { [ GetMCLDIR ] != 0 } {
    return
  }

  global mclInfo 
  global mclNames 

  if {[info exists mclInfo] != 1} {
  
      ##
      # get a list of files contained in the directory
      #
      set files [glob -nocomplain -- "${STD_MCLDIR}/*.mcl"]
     
      if { $files == "" } {
        ##puts stderr "warning: $dirname does not have any MCL files!!"
        return {};
      }

      set mclInfo(filelist) $files
      if {[file exists ${STD_MCLDIR}/mclIndex] == 1} {
          source ${STD_MCLDIR}/mclIndex
      }
  }

  set flist {}

  foreach c $components {
      if {[info exists mclNames($c)] == 1} {
          lappend flist $mclNames($c)
      } else {

          global fname

	  if {[info exists context] != 1} {
              set context [scancontext create]
              set pattern {[^#]*COMPONENT[    ]+([a-zA-Z_0-9\.\-]+)[       ]+}
              scanmatch $context $pattern {
                  global mclNames fname 
                  set mclNames($matchInfo(submatch0)) $fname
                  return 0
              }
          }
      
          foreach fname $mclInfo(filelist) {

            if { [file exists $fname] == 1 } { 

              if {[catch {set fd [open $fname r]} fname_error]} {
                    puts stderr "ERROR: $fname_error"
                    exit 1
              }

              scanfile $context $fd
              close $fd
              lvarpop mclInfo(filelist)

              if {[info exists mclNames($c)] == 1} {
                  lappend flist $mclNames($c)
                  break
              }
            }
          }
      }
  }

  if {[info exists context] == 1} {
      scancontext delete $context
  }

  return $flist
}


#----------------------------------------------------------------
##
## Returns the list of MCL components
##
#----------------------------------------------------------------
proc MCL_GetComponents {resultVar} {

  upvar $resultVar result

  set result [ GetMclComponents ]

  return $result
}


#---------------------------------------------------------------#
#
#  create a "mclIndex" file in current directory
#
#---------------------------------------------------------------#

proc mkmclindex {{mcldir {}}} {
global mclInfo
global mclNames

# attempt a get on a non-existent component to force a full
#  read of all mcl files

set flist [GetMclFiles all-of-the-components]
write_file "mclIndex" "array set mclNames \{[array get mclNames]\}"
return {}
}

#----------------------------------------------------------------
##
## When a list of files is provided, ignore it ...
## Force a read of all the mclfiles in the system and get
##  the components from those files.
##
#----------------------------------------------------------------
proc GetMclComponents { {files {}} } {
  global mclInfo
  
  GetMclFiles ----all-of-them----

  global mclNames 
  set MclList [lsort [array names mclNames]]

  return $MclList
}

#----------------------------------------------------------------
##
## Initializes the state of the Mcl engine / DMI SL:  This call will
## prepare the internal data structures so that the next MI call be
## performed.
##
## Input:  variable number of arguments.  Each argument is the name of a
##         component.  ex, miInit psinfo crontab lmf - this will initialize
##         the Mcl engine to source in the associated Mcls whose components 
##         are psinfo, crontab and lmf.  If no argument is given then this
##         call will initialize all the Mcls on the system. 
##
## Output: no return code NOW (*reserved for future*)
##
#----------------------------------------------------------------
proc MCL_Init {args} {

  if { [ GetMCLDIR ] != 0 } {
    return
  }

  set return_msg ""
  foreach arg $args {
    if { [info command "/$arg"] != "/$arg" } {
      ##
      ## source the MCL file if and only if its the first time
      ##
      set return_msg [MclInit $arg]
    }
  }
  return $return_msg
}

#----------------------------------------------------------------
##
## Clean the internal data structures and the MCL definitions
##
#----------------------------------------------------------------
proc MCL_Done {args} {
  
  set args [lindex $args 0]
  if { $args != {} } {
    set components $args
  } else {
    set components [Component info instances]
  }
  foreach comp $components {
    foreach group [ $comp children ] {
      foreach attr [ $group children ] {
	$attr destroy
      }
      $group _ClearTable
      $group destroy
    }
    $comp destroy
  }
}

#----------------------------------------------------------------
##
## This is a very temp. routine from the MI perspective. This
## proc will dump the contents of a component in a very crude
## form to a file (if specified optionally) or to file called
## /tmp/${component}.PID or /tmp/Sm_SysMan_Tmp/${component}.PID
## depending if it is called by a non-root user, or root.
##
#----------------------------------------------------------------
proc MCL_Dump {component {file {stdout}}} {

  if {$file == ""} {
    set file [lindex [sm_sectmpopen "$component"] 0]
  }

  set instance "/${component}"

  $instance DumpTable $file

}



# ======================================================
# useful utilities

# ----------------------------------------------------------------------
# Given a list of tags and a list of values, load the values into the specified
#   array indexed by the tags.  The values in valList are assumed to be in
#   tagList order.

proc _list2TagValues {tagList valList arrRef} {
    upvar $arrRef  arr

    set i 0
    foreach tag $tagList {
	set arr($tag) [lindex $valList $i]
	incr i
    }
}



# ----------------------------------------------------------------------
# Given an array and list of tags, create a list from the array values
#   in tag order.  The array is assumed to have a value for each tag.

proc _tagValues2List {arrRef tagList valListRef} {
    upvar $arrRef      arr
    upvar $valListRef  valList

    foreach tag $tagList {
	lappend valList $arr($tag)
    }
}



# ----------------------------------------------------------------------  
# Return primary if condition true, and secondary if false.

proc conditional {cond primary secondary} {
    if {$cond} {
	return $primary
    }
    return $secondary
}


# ----------------------------------------------------------------------  
# Return "true" or "false" given various variations on that.

proc regularBool {irregular} {

    set lower [string tolower $irregular]
    set idx   [lsearch -exact {false no 0 {} down} $lower]
    if {$idx >= 0} {
	return "false"
    }
    return "true"
}


# ---------------------------------------------------
# Set the specified variable to the specified value, unless the value
# is null.  If it is null, leave the variable as it is.

proc setUnlessNull {varRef val} {
    upvar $varRef var

    if { ! [cequal $val ""]} {
	    set var  $val
    }
}



# ---------------------------------------------------
# Return just the system name from the fully qualified host name.
# For example, given zeus.unx.dec.com, return zeus

proc NetMcl_extractHostNameOnly {fullHostName} {
    set full [split [$self getVal host] .]
    return [lindex $full 0]
}


# ---------------------------------------------------
# Return just the network domain from the fully qualified host name.
# For example, given zeus.unx.dec.com, return unx.dec.com

proc NetMcl_extractNetDomain {fullHostName} {
    set full [split [$self getVal host] .]
    return [join [lrange $full 1 end] .]
}

# ---------------------------------------------------
# Returns the name associated with an IP address in the /etc/host file,
#   or "" if not found.

proc NetMcl_getHostName {ipAddr} {
    global _netmcl_hostInfo SysmanFocusHost SysmanOnCluster

    # Decided to create dependancy directly on the /etc/host file instead
    #   of a dependancy on the hostMappings mcl because I think it's less
    #   messy.  Also, should be quicker on repeated calls.
    # Do this only once for speed.
    if { ! [info exists _netmcl_hostInfo]} {
	if { ! [catch "set fd [open /etc/hosts r]"]} {
	    while {[gets $fd line] >= 0} {
		set ip   [lindex $line 0]
		set name [lindex $line 1]
		if {[cequal [NetMcl_isIpFormat $ip] ""]} {
		    set _netmcl_hostInfo($ip) $name
		}
	    }
	}
    }

    if { [info exists _netmcl_hostInfo($ipAddr)] } {
	return $_netmcl_hostInfo($ipAddr)
    }
    return ""
}


# ---------------------------------------------------
# Looks for interface in rc.config entries.
#   If found return RC Mgr ID number; else return first unused ID.
#   If all IDs used, return "FULL".

proc NetMcl_getIFNum {iface} {
    global NetMcl_reservedNetDev
    global SysmanFocusHost SysmanOnCluster

    set RCMGR "rcmgr"
    
    # check to see if focused to a node on a cluster
    if { $SysmanOnCluster } {
	if { ! [cequal $SysmanFocusHost ""] } {
	    set node_id [cluster member node $SysmanFocusHost]
	    set RCMGR "rcmgr -h $node_id"
	}
    }
    
    set max [eval $RCMGR get MAX_NETDEVS]
    if { ! [regexp {^[0-9]+$} $max]} {
	set max 24
    }

    # Check all NETDEV_# entries for this device or at least an unused slot.
    #   If device found, return it.
    set unused ""
    for {set i 0} {$i < $max} {incr i} {
	set netDev [eval $RCMGR get NETDEV_$i]
	if {[cequal $netDev $iface]} {
	    return $i
	}
	if {[cequal $unused ""] && \
		! [info exists NetMcl_reservedNetDev($i)] && \
		[cequal $netDev ""]} {
	    set unused $i
	}
    }

    # Couldn't find the specified net interface, return the first unused slot.
    #   Note that this interface is intending to use this ID, so another
    #   doesn't think it's available.
    if { ! [cequal $unused ""]} {
	set NetMcl_reservedNetDev($unused) $iface
	return $unused
    }

    return "FULL"
}

# ---------------------------------------------------
# Determine type from device name.  Returns one of
#   UNKNOWN ATM ETHERNET FDDI TOKEN MEMORYCHANNEL
#   Lifted from /usr/share/sysman/netconfig/NCM.ism::NetMcl_IFType

proc NetMcl_getType {deviceName} {

    regsub {[0-9]+$} $deviceName "" type
    switch -exact -- $type {
	el -
	elan -
	le -
	ln -
	te -
	tu -
	xna {
	    return ETHERNET
	}

	lis {
	    return ATM
	}

	faa -
	fta -
	fza -
	mfa -
	mfs {
	    return FDDI
	}

	tra {
	    return TOKEN
	}

	mc {
	    return MEMORYCHANNEL
	}

	# Removed support for slip in Platinum BL9
    }

    return UNKNOWN
}



# ----------------------------------------------------------------------
# Return "" if a valid system name.  Return error string if not.

proc NetMcl_isHostName {str {nullOk 1}} {

    set str [string trim $str]
    if {$nullOk && [cequal $str ""]} {
	return ""
    }

    if { ! [sm_IsDomainName $str] } {
	return [MclUtils_catobj catgets mcl_validSysName_txt]
    }
    return ""
}


# ----------------------------------------------------------------------
# Return "" if 4 or more numbers between 0 and 255 inclusive and 
# dot "." separated.  Return error string if not.

proc NetMcl_isIpFormat {networkAddress {minNumOctets 4}} {

    set octets [split $networkAddress "."]
    if {[llength $octets] < $minNumOctets} {
	return [MclUtils_catobj catgets mcl_validNetAddr_txt]
    }

    foreach octet $octets {
	if { ! [ctype digit $octet] || \
		$octet < 0 || \
		$octet > 255} {
	    return [MclUtils_catobj catgets mcl_validNetAddr_txt]
	}
    }

    return ""
}


# ----------------------------------------------------------------------
# Return "" if valid IP address.  Return error string if not.
# This checks for valid IPv4 address.  See also NetMcl_isIpV6Addr, below.

proc NetMcl_isIpAddr {ipAddr {minNumOctets 4}} {
    if {[cequal $ipAddr "DYNAMIC"]} {
        return ""
    }

    set stat [NetMcl_isIpFormat $ipAddr $minNumOctets]
    if { ! [cequal $stat ""]} {
	return $stat
    }

    # Beware that some of these can be "" when checking networks (<4 octets)
    lassign [split $ipAddr "."] first second third

    # Class A addresses
    #   1.*.*.* to
    # 127.*.*.* OK
    if {$first <= 0} {
	return [MclUtils_catobj catgets mcl_ipFirst_txt]
    }
    if {$first <= 127} {
	return ""
    }

    # Class B addresses
    # 128.1.*.* to
    # 191.254.*.* OK
    if {$first <= 191} {
	if {$second >= 0 && $second <= 254} {
	    return ""
	} else {
	    return [MclUtils_catobj catgets mcl_ipSecond_txt]
	}
    }

    # Class C addresses
    # 192.0.1.* to
    # 223.255.254.* OK
    if {$first <= 223} {
	if {$third >= 0 && $third <= 254} {
	    return ""
	} else {
	    return [MclUtils_catobj catgets mcl_ipThird_txt]
	}
    }

    return [MclUtils_catobj catgets mcl_ipFirst_txt]
}

#--------------------------------------------------
# NetMcl_isIpV6Addr - check syntax of IPv6 address
# 
# Synopsis:
#   NetMcl_isIpV6Addr ?-allowIPv4? addr
#
# Inputs:
#   -allowIPv4 
#	(optional) if present, will return {} for a 
#	syntactically valid IPv4 address
#   address
#	the address whose syntax we are checking
#	
#
# Output:
#   returns {} if addr is a sytactically valid IPv6 address
#    (or a syntactically valid IPv4 address, if -allowIPv4
#    is present)
#   returns an error string if addr is not syntactically correct
#
# See Network Admin Guide, chapter 3, for details
# on IPv6 syntax.
proc NetMcl_isIpV6Addr {args} {

    # The library where we find ip6_testaddr defined
    set library /usr/share/sysman/lib/ipv6.so

    # The default is to consider IPv4 addresses as 
    # invalid IPv6 addresses
    set allowIPv4 0

    # Parse the arguments
    switch [llength $args] {
	1 {				;# one argument
	    set addr $args		;# it's the address to validate
	}

	2 {				;# two arguments

	    set opt [lindex $args 0]	;# first arg
	    if { ![cequal $opt "-allowIPv4"] } {
		# If it's not this option, print a usage message
		return [MclUtils_catobj catgets mcl_ipv6Usage_txt]
	    } else {
		# User wishes to consider IPv4 addresses valid
		set allowIPv4 1
	    }

	    set addr [lindex $args 1]	;# second arg
	}

	default {
	    # Too many or too few args
	    return [MclUtils_catobj catgets mcl_ipv6Usage_txt]
	}
    }

    # Load the command from the shared library if
    # it hasn't already been loaded
    if { [info commands ip6_testaddr] == {} &&
	 [ catch {
	     load $library ip6_testaddr
	 } msg ] } {

	# Couldn't load the library for some reason
	set err [MclUtils_catobj catgets mcl_ipv6LoadErr_txt]
	return [format $err $library]
    }

    if { [cequal [ip6_testaddr $addr] {}] } {

	# Address was not a valid IPv6 or IPv4 address
	return [MclUtils_catobj catgets mcl_ipv6GeneralErr_txt]

    } else {

	# Address was a valid IPv6 address.

	# Currently, ip6_testaddr allows IPv4 addresses
	# (I don't know if this is a bug or not), so we have 
	# to perform this additional check

	if { !$allowIPv4 && [NetMcl_isIpAddr $addr 4] == {} } {

	    # Address was a valid IPv4 address, 
	    # but we don't want those.
	    return [MclUtils_catobj catgets mcl_ipv6GeneralErr_txt]
	}

	return {}
    }
}


# ----------------------------------------------------------------------  
# Given the name and network domain, return the fully qualified host name.
# For example, given zeus and unx.dec.com, return zeus.unx.dec.com

proc NetMcl_joinFullHost  {name domain} {
    return [join "$name $domain" .]
}

# ===================================================

#----------------------------------------------------------------
##
##Verifies if this is the only node or the focus of the cluster?
##Return 1 if not in cluster or if cluster is focused on this node. 
##Otherwise, return 0.
##
#----------------------------------------------------------------
proc NetMcl_IfHost {} {
    global SysmanFocusHost SysmanOnCluster

    set host_name [exec /sbin/hostname]

    # return a 1 if we are running it for the host or cluster and the
    # target is not set.

    if { !$SysmanOnCluster || \
            ( [string compare $SysmanFocusHost $host_name] == 0 ) }  {
        return 1
    } else {
        return 0
    }
}


#--------------------------------------------------
# Make sure the value is one of the possible enum values.

proc mclVerifyEnum {val posVals msgCatLbl} {

    set val [string trim $val]
    if {[lsearch -exact $posVals $val] <= -1} {
	set msg [MclUtils_catobj catgets $msgCatLbl]
	set format  [MclUtils_catobj catgets mcl_enumErr_txt]
	set message [format $format $msg $val $posVals ""]
	return [mclFormatErr $message]
    }
    return ""
}


#--------------------------------------------------
# Make sure the value is between the specified integers.

proc mclVerifyNumRange {val lo hi msgCatLbl} {

    set val [string trim $val]
    if { ! [ctype digit $val] || \
	$val < $lo || $val > $hi} {
        set tag [MclUtils_catobj catgets $msgCatLbl]
        return  [mclFormatNumRangeErr $tag $val $lo $hi]
    }
    return ""
}


#--------------------------------------------------
# Make sure the value is either a valid network or system address or name.
#   If value can be a network and/or a system, that's the same except
#   don't need four octets; 1 is enough.  In that case, will need to lower
#   default min octets.

proc mclVerifyNetNameOrAddr {val {minOctets 4}} {

    # If the first token is all numeric, assume it's suppose to be an address.
    set val [string trim $val]
    set first [lindex [split $val .] 0]
    if {[ctype digit $first]} {

	set stat [NetMcl_isIpAddr $val $minOctets]
	set tag  [MclUtils_catobj catgets mcl_netAddr_txt]

    } else {

	# The zero indicates that null is not a valid value.
	set stat [NetMcl_isHostName $val 0]
	set tag  [MclUtils_catobj catgets mcl_sysOrNetName_txt]

    }

    if { ! [cequal $stat ""]} {
	 return  [mclFormatInvalidValue $tag $val $stat]
    }
    return ""
}


#--------------------------------------------------
# "Cannot add a $type record.  Need to add hardware and system will
#    add the software record at the next reboot."

proc mclFormatCantAddErr {type} {

    set format  [MclUtils_catobj catgets mcl_cantAddErr_txt]
    set str     [format $format $type]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "Cannot change $rec."

proc mclFormatCantChangeErr {rec} {

    set format  [MclUtils_catobj catgets mcl_cantChangeErr_txt]
    set str     [format $format $rec]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "The record \"$rec\" already exists in $tbl.  Cannot have a duplicate."

proc mclFormatDupErr {rec tbl} {

    set format  [MclUtils_catobj catgets mcl_dupErr_txt]
    set str     [format $format $rec $tbl]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# Prepend "Error: " in i18n format.

proc mclFormatErr {msg} {

    return "[MclUtils_catobj catgets mcl_err_txt] $msg"
}


#--------------------------------------------------
# "The $tag value \"$val\" must be $x, $y or $z."

proc mclFormatEnumErr {tag val valList} {

    set last    [lindex $valList end]
    set choices [join [lreplace $valList end end] ", "]
    set format  [MclUtils_catobj catgets mcl_enumErr_txt]
    set str     [format $format $tag $val $choices $last]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "Cannot run $serv1 because $serv2 already running or configured."

proc mclFormatExcluErr {serv1 serv2} {

    set format  [MclUtils_catobj catgets mcl_excluErr_txt]
    set str     [format $format $serv1 $serv2]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "Cannot parse file \"$file\".  $reason"

proc mclFormatFileSyntaxErr {file {reason ""}} {

    set format  [MclUtils_catobj catgets mcl_fileSyntaxErr_txt]
    set str     [format $format $file $reason]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "The $tag value \"$val\" is invalid.  $reason."

proc mclFormatInvalidValue {tag val {reason ""}} {

    set format  [MclUtils_catobj catgets mcl_invalidValue_txt]
    set str     [format $format $tag $val $reason]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "The $tag value \"$val\" must be an integer between $lo and $hi inclusive."

proc mclFormatNumRangeErr {tag val lo hi} {

    set format  [MclUtils_catobj catgets mcl_numRangeErr_txt]
    set str     [format $format $tag $val $lo $hi]
    return [mclFormatErr $str]
}


#--------------------------------------------------
# "File \"$file\" is missing or does not have read permission."

proc mclFormatReadErr {file} {

    set format  [MclUtils_catobj catgets mcl_readErr_txt]
    set str     [format $format $file]
    return [mclFormatErr $str]
}


#--------------------------------------------------
##
## Verifies that the value is in range for TCL so that
##  an error is not generated when the MCL first uses it.
##
##  Right now, we only check integers, since that is
##  the only type that we've had problems with.  When
##  a MCL tried to compare or compute with a very large
##  integer, an out of range error and stack trace was
##  triggered.  Here, we catch the error and return
##  the message.
##
##  o  type - the attribute's type
##  o  value - value entered by the user.
##
##  Returns an error message
#----------------------------------------------------------------
proc _rangeValidation {att type value } {

  switch -regexp -- [string toupper $type] {
    COUNTER -
    COUNTER64 -
    GAUGE -
    INTEGER -
    INT -
    INT64 -
    INTEGER64 {

      if {[lempty $value]} {
	  return ""
      }

      if { ! [regexp {^[-+]*[0-9]+$} $value] } {
	  set format  [MclUtils_catobj catgets mcl_invalidIntValue_txt]
	  set str     [format $format $value]
	  return      [mclFormatErr $str]
      }

      # TCL will give an out of range error if you
      # try to compare a large value with another int
      if { [catch { expr $value > 0 } curStat] } {
	return $curStat
      } else {
	return {}
      }
    }
  }

  return {}
}

# ===================================================
