#!/usr/share/sysman/bin/sysmansh
# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: selectionBox.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:31 $
# 

##---------------------------------------------------------------------------
 # 
 # MODULE:
 #      selectionBox.tcl
 #
 # ABSTRACT:
 #
 #	The following routines are available for creation/manipulation of
 #	the generic "selection" widget.  The selection "widget" created is
 #	not a true widget since it can not be manipulated using any of the 
 #	widget manipulation commands (e.g., pack etc); however the term 
 #	widget best describes what is created.  The selection "widget"
 #	consists of two listboxes with associated scrollbars, labels,
 #	buttons, and bindings that enable the user to move data
 #	between the boxes.  The top box will contain a list of items,
 #	while the bottom box will contain those items which the user
 #	has selected, hence the name "selection widget".  All of the
 #	routines use the prefix "GUI_Selection".
 #
 #
 # NOTES:
 #
##---------------------------------------------------------------------------




##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_Create
 #
 #	This routine is used to create a "selection" object, which is 
 #	essentially two lists.  The top list is the list of available
 #	items which have not been selected by the user.  The second list
 #	is a list of items which have been selected by the user.  This
 #	object pre-defines a set of methods by which the user can move
 #	items from one list to the other.  Each list contains the 
 #	following widgets:
 #
 #	       1 text label
 #		      Describes the contents of the listbox
 #
 #	       1 listbox
 #		      Contains the items which make up the list
 #
 #	       2 scrollbars
 #		      One vertical and one horizontal to scroll through
 #		      the selections.
 #
 #	       2 buttons
 #		      The leftmost button allows the user to move the 
 #		      currently selected items from this list to the 
 #		      other list.  The rightmost button allows the 
 #		      user to move all of the items in this list to 
 #		      the other list.
 #
 # Inputs: 
 #
 #	widget        - The name of the selection "widget" to be created.
 #
 #	availableList - The list of items which are available for 
 #		        selection by the user, but are not yet 
 #			selected.
 #
 #	selectedList  - The list of items which are already selected
 #		        for the user.
 #
 #	label1        - The text label for the available items list
 #
 #	button1       - The text label for the leftmost button which moves
 #		        the currently user-selected items from the available 
 #		        list to the selected list.
 #
 #	button1all    - The text label for the rightmost button which moves
 #		        all remaining available items to the selected list.
 #
 #	label2        - The text label for the selected items list
 #
 #	button2       - The text label for the leftmost button which moves
 #		        the currently user-selected items from the selected 
 #		        list to the available list.
 #
 #	button2all    - The text label for the rightmost button which moves
 #		        all remaining selected items to the available list.
 #
 # Outputs: 
 #
 # Returns: 
 #
 #	The name for the selection widget which is needed to reference
 #	a particular instance when calling some of the manipulation
 #	routines that are available.
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUI_Selection_Create {args} {

	global symbolicName	; # global array of widgets
	global enableList	; # list of widgets to be configured to 
				  # a normal state when items are selected
	global disableList	; # list of widgets to be configured to 
				  # a disabled state when items are selected


	lassign $args widget availableList selectedList \
		label1 button1 button1all \
		label2 button2 button2all 

	#
	# Initialize the global lists
	#
	set enableList($widget) {}
	set disableList($widget) {}


	#
	# Need to define the widget names so that they can be accessed 
	# by the manipulation routines.
	#
	set symbolicName(${widget}) $widget
	set symbolicName(${widget}Available) $widget.availF.componentsL
	set symbolicName(${widget}AvailMsg) $widget.availF.messsageM
	set symbolicName(${widget}Select) $widget.availF.buttonsF.selectB
	set symbolicName(${widget}SelectAll) $widget.availF.buttonsF.selectallB
	set symbolicName(${widget}Selected) $widget.selectF.componentsL
	set symbolicName(${widget}SelectMsg) $widget.selectF.messageM
	set symbolicName(${widget}Deselect) $widget.selectF.buttonsF.selectB
	set symbolicName(${widget}DeselectAll) $widget.selectF.buttonsF.selectallB

	#
	# For low resolution monitors use a height of 3 on the listboxes.
	#
	if {[winfo screenwidth .] <= 800} {
		set listboxHeight 3
	} else {
		set listboxHeight 7
	}


	#
	# Create the selection "widget"
	#
	frame $widget
	pack $widget -side top -expand 1 -fill both


	###################
	# Available Items #
	###################
	frame $widget.availF
	pack $widget.availF -side top -expand 1 -in $widget -fill both \
		-padx 0.25i -pady 0.1i

	#
	# Display the available text label
	#
	message $symbolicName(${widget}AvailMsg) \
		-width 4i -takefocus 0 -justify left -anchor w \
		-text "$label1" 
	pack $symbolicName(${widget}AvailMsg) -anchor w -fill x 
     
	#
	# Listbox/scrollbars for available items
	#
	frame $widget.availF.buttonsF
	frame $widget.availF.scrollF
	scrollbar $widget.availF.scrollxS -relief sunken -width 13 \
		-takefocus 0 -orient horizontal \
		-command "$symbolicName(${widget}Available) xview"
	scrollbar $widget.availF.scrollyS -relief sunken -width 13 \
		-takefocus 0 -orient vertical \
		-command "$symbolicName(${widget}Available) yview"
	listbox $widget.availF.componentsL \
		-width 40 -height $listboxHeight -relief sunken -takefocus 0 \
		-yscrollcommand "$widget.availF.scrollyS set" \
		-xscrollcommand "$widget.availF.scrollxS set"
	#
	# Determine the size of the padding frame which will prevent
	# the horizontal scrollbar from exceeding the length of the 
	# listbox.
	#
	set xpad [expr [$widget.availF.scrollxS cget -width] + 2* \
		([$widget.availF.scrollxS cget -bd] + \
		 [$widget.availF.scrollxS cget -highlightthickness])]
	set ypad [expr [$widget.availF.scrollyS cget -width] + 2* \
		([$widget.availF.scrollyS cget -bd] + \
		 [$widget.availF.scrollyS cget -highlightthickness])]
	frame $widget.availF.padF -width $ypad -height $xpad

	#
	# Buttons: select/select all
	#
	button $symbolicName(${widget}Select) -text "$button1" -state disabled
	button $symbolicName(${widget}SelectAll) -text "$button1all" 

	#
	# Pack the available widgets
	#
	pack $widget.availF.buttonsF -in $widget.availF \
		-side bottom -fill x -pady 0.1i
	pack $widget.availF.scrollF -in $widget.availF \
		-side bottom -fill x 
	pack $widget.availF.padF -in $widget.availF.scrollF \
		-side right
	pack $widget.availF.scrollxS -in $widget.availF.scrollF \
		-side bottom -fill x
	pack $widget.availF.scrollyS -in $widget.availF \
		-side right -fill y 
	pack $symbolicName(${widget}Available) -in $widget.availF \
		-side left -fill both -expand 1 
	pack $symbolicName(${widget}Select) -in $widget.availF.buttonsF \
		-side left -expand 1 
	pack $symbolicName(${widget}SelectAll) -in $widget.availF.buttonsF \
		-side left -expand 1 

	#
	# Populate the list in the available listbox
	#
	eval {$symbolicName(${widget}Available) insert end} $availableList
     
	#
	# Bindings to move item(s) from the available list to the 
	# selected list.
	#
	bind $symbolicName(${widget}Available) <ButtonPress-1> \
		"GUI_Selection_SelectItem %W %y
		 GUI_Selection_UpdateState $widget"
	bind $symbolicName(${widget}Available) <B1-Motion> \
		{%W select set anchor [%W nearest %y]}
	bind $symbolicName(${widget}Available) <ButtonRelease-1> \
		{%W select set anchor [%W nearest %y]}
	bind $symbolicName(${widget}Available) <Double-Button-1> \
		"GUISelectionMoveItems \
			$symbolicName(${widget}Available) \
			$symbolicName(${widget}Selected) \
			%y
		 GUI_Selection_UpdateState $widget"

	$symbolicName(${widget}Select) configure -command \
		"GUISelectionMoveItems $symbolicName(${widget}Available) \
			$symbolicName(${widget}Selected)
		 GUI_Selection_UpdateState $widget"
	$symbolicName(${widget}SelectAll) configure -command \
		"$symbolicName(${widget}Available) selection set 0 end
		 GUISelectionMoveItems $symbolicName(${widget}Available) \
			$symbolicName(${widget}Selected)
		 GUI_Selection_UpdateState $widget"


	##################
	# Selected Items #
	##################
	frame $widget.selectF 
	pack $widget.selectF -side top -expand 1 -in $widget -fill both \
		-padx 0.25i -pady 0.1i 

	#
	# Display the selected text label
	#
	message $symbolicName(${widget}SelectMsg) \
		-width 4i -takefocus 0 -justify left -anchor w \
		-text "$label2" 
	pack $symbolicName(${widget}SelectMsg) -anchor w -fill x 
     
	#
	# Listbox/scrollbars for selected items
	#
	frame $widget.selectF.buttonsF
	frame $widget.selectF.scrollF
	scrollbar $widget.selectF.scrollxS -relief sunken -width 13 \
		-takefocus 0 -orient horizontal \
		-command "$symbolicName(${widget}Selected) xview"
	scrollbar $widget.selectF.scrollyS -relief sunken -width 13 \
		-takefocus 0 -orient vertical \
		-command "$symbolicName(${widget}Selected) yview"
	listbox $symbolicName(${widget}Selected) \
		-width 40 -height $listboxHeight -relief sunken -takefocus 0 \
		-yscrollcommand "$widget.selectF.scrollyS set" \
		-xscrollcommand "$widget.selectF.scrollxS set"
	#
	# Determine the size of the padding frame which will prevent
	# the horizontal scrollbar from exceeding the length of the 
	# listbox.
	#
	set xpad [expr [$widget.selectF.scrollxS cget -width] + 2* \
		([$widget.selectF.scrollxS cget -bd] + \
		 [$widget.selectF.scrollxS cget -highlightthickness])]
	set ypad [expr [$widget.selectF.scrollyS cget -width] + 2* \
		([$widget.selectF.scrollyS cget -bd] + \
		 [$widget.selectF.scrollyS cget -highlightthickness])]
	frame $widget.selectF.padF -width $ypad -height $xpad

	#
	# Buttons: deselect/deselect all
	#
	button $symbolicName(${widget}Deselect) -text "$button2" \
		-state disabled
	button $symbolicName(${widget}DeselectAll) -text "$button2all" \
		-state disabled

	#
	# Pack selected items
	#
	pack $widget.selectF.buttonsF -in $widget.selectF \
		-side bottom -fill x -pady 0.1i
	pack $widget.selectF.scrollF -in $widget.selectF \
		-side bottom -fill x 
	pack $widget.selectF.padF -in $widget.selectF.scrollF \
		-side right
	pack $widget.selectF.scrollxS -in $widget.selectF.scrollF \
		-side bottom -fill x
	pack $widget.selectF.scrollyS -in $widget.selectF \
		-side right -fill y 
	pack $symbolicName(${widget}Selected) -in $widget.selectF \
		-side left -fill both -expand 1 
	pack $symbolicName(${widget}Deselect) -in $widget.selectF.buttonsF \
		-side left -expand 1 
	pack $symbolicName(${widget}DeselectAll) -in $widget.selectF.buttonsF \
		-side left -expand 1

	#
	# Populate the list in the selected listbox
	#
	eval {$symbolicName(${widget}Selected) insert end} $selectedList
     
	#
	# Bindings to move item(s) from the selected list to the 
	# available list.
	#
	bind $symbolicName(${widget}Selected) <Double-Button-1> \
		"GUISelectionMoveItems \
			$symbolicName(${widget}Selected) \
			$symbolicName(${widget}Available) \
			%y
		 GUI_Selection_UpdateState $widget"
	bind $symbolicName(${widget}Selected) <ButtonPress-1> \
		"GUI_Selection_SelectItem %W %y
		 GUI_Selection_UpdateState $widget"
	bind $symbolicName(${widget}Selected) <B1-Motion> \
		{%W select set anchor [%W nearest %y]}
	bind $symbolicName(${widget}Selected) <ButtonRelease-1> \
		{%W select set anchor [%W nearest %y]}
     
	$symbolicName(${widget}Deselect) configure -command \
		"GUISelectionMoveItems \
			$symbolicName(${widget}Selected) \
			$symbolicName(${widget}Available)
		 GUI_Selection_UpdateState $widget"
	$symbolicName(${widget}DeselectAll) configure -command \
		"$symbolicName(${widget}Selected) selection set 0 end
		 GUISelectionMoveItems $symbolicName(${widget}Selected) \
			$symbolicName(${widget}Available)
		 GUI_Selection_UpdateState $widget"

	return $widget
}				; ## end routine GUI_Selection_Create

     

##---------------------------------------------------------------------------
 #
 # Proc:  GUISelectionMoveItems
 #
 #	This routine SHOULD NOT BE CALLED by any routine except for
 #	GUI_Selection_Create.  It is a routine which moves
 #	all of the selected items from the "src" listbox to the "target"
 #	listbox.  If the optional parameter "index" is supplied then the
 #	item with the specified index will be selected before the move
 #	occurs.
 #
 # Inputs: 
 #
 #	src    - The listbox where the selected items are currently 
 #	         located and from which they will be removed.
 #
 #	target - The listbox to which the items will be added.
 #
 #	index  - The index of the item to be added to the selection list
 #		 prior to the movement (optional).
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.  It is not intended for use 
 #	by any routine except GUI_Selection_Create
 #
##---------------------------------------------------------------------------
proc GUISelectionMoveItems {src target {index {}}} {

	#
	# Select the component located at index if appropriate.
	#
	if {! [lempty $index]} {$src select set [$src nearest $index]}

	#
	# Sort the items in decreasing order so that the removal operation
	# will not cause a numbering change of the selected elements.  Each
	# item is moved from the $src listbox to the $target listbox.
	#
	foreach val [lsort -decreasing -integer [$src curselect]] {
		$target insert end [$src get $val]
		$src delete $val
	}

	#
	# Sort the new list
	#
	set sortedList [lsort [$target get 0 end]]
	$target delete 0 end
	eval {$target insert 0} $sortedList
}				; ## end routine GUISelectionMoveItems



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_BindPFH
 #
 #	This routine is called to bind pointer messages to the selection
 #	"widget" created by the GUI_Selection_Create call.
 #
 # Inputs: 
 #
 #	widget        - The "widget" returned by GUI_Selection_Create.
 #
 #	pfhLbox1      - The pointer message for the available listbox
 #
 #	pfhButton1    - The pointer message for button1
 #
 #	pfhButton1all - The pointer message for button1all
 #
 #	pfhLbox2      - The pointer message for the selected listbox
 #
 #	pfhButton2    - The pointer message for button2
 #
 #	pfhButton2all - The pointer message for button2all
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_BindPFH {args} {

	global symbolicName	; # global array of widgets


	lassign $args widget pfhLbox1 pfhButton1 pfhButton1all \
		pfhLbox2 pfhButton2 pfhButton2all

	if {! [winfo exists $widget] } {return 0}

	#
	# Bind the messages to the Available Listbox
	#
        bind $symbolicName(${widget}Available) <Enter> " \
                HelpEntered_widget {$pfhLbox1} \
                        $symbolicName([sm_getToplevelName $widget]Help)"
        bind $symbolicName(${widget}Available) <Leave> " \
                HelpLeaving_widget"

	#
	# Bind the messages to the Selected Listbox
	#
        bind $symbolicName(${widget}Selected) <Enter> " \
                HelpEntered_widget {$pfhLbox2} \
                        $symbolicName([sm_getToplevelName $widget]Help)"
        bind $symbolicName(${widget}Selected) <Leave> " \
                HelpLeaving_widget"

	#
	# Bind the messages to the Select button
	#
        bind $symbolicName(${widget}Select) <Enter> " \
                HelpEntered_widget {$pfhButton1} \
                        $symbolicName([sm_getToplevelName $widget]Help)"
        bind $symbolicName(${widget}Select) <Leave> " \
                HelpLeaving_widget"

	#
	# Bind the messages to the SelectAll button
	#
        bind $symbolicName(${widget}SelectAll) <Enter> " \
                HelpEntered_widget {$pfhButton1all} \
                        $symbolicName([sm_getToplevelName $widget]Help)"
        bind $symbolicName(${widget}SelectAll) <Leave> " \
                HelpLeaving_widget"

	#
	# Bind the messages to the Deselect button
	#
        bind $symbolicName(${widget}Deselect) <Enter> " \
                HelpEntered_widget {$pfhButton2} \
                        $symbolicName([sm_getToplevelName $widget]Help)"
        bind $symbolicName(${widget}Deselect) <Leave> " \
                HelpLeaving_widget"

	#
	# Bind the messages to the DeselectAll button
	#
        bind $symbolicName(${widget}DeselectAll) <Enter> " \
                HelpEntered_widget {$pfhButton2all} \
                        $symbolicName([sm_getToplevelName $widget]Help)"
        bind $symbolicName(${widget}DeselectAll) <Leave> " \
                HelpLeaving_widget"
}				; ## end routine GUI_Selection_BindPFH


##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_InitHelp
 #
 #	This routine is called to initialize the on-line help for 
 #	the selection "widget".  This is done by registering the 
 #	on item help for the widgets inside of the selection "widget".
 #
 # Inputs: 
 #
 #	widget - The "widget" returned by GUI_Selection_Create.
 #
 #	idlist - The list of catalog identifiers which will be used to 
 #		 register the widgets for on-line help.  The list should 
 #		 contain 6 entries.  These entries correspond to the 
 #		 identifiers for:
 #			available listbox, 
 #			select button,
 #			selectall button,
 #			selected listbox, 
 #			deselect button,
 #			deselectall button
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked and that the message catalogs
 #	be initialized.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_InitHelp {widget idlist} {

	global symbolicName	; # global array of widgets
	global appName		; # application name


	if {! [winfo exists $widget] } {return 0}
	
	lassign $idlist availableL selectB selectallB \
		selectedL deselectB deselectallB

	#
	# Register the widgets for on-item help
	#
	upvar #0 msg_${appName}_LocIds locId
	HelpOnItemRegister $symbolicName(${widget}Available) $appName \
		$locId(LOCID) $locId(${availableL}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}AvailMsg) $appName \
		$locId(LOCID) $locId(${availableL}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}Select) $appName \
		$locId(LOCID) $locId(${selectB}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}SelectAll) $appName \
		$locId(LOCID) $locId(${selectallB}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}Selected) $appName \
		$locId(LOCID) $locId(${selectedL}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}SelectMsg) $appName \
		$locId(LOCID) $locId(${selectedL}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}Deselect) $appName \
		$locId(LOCID) $locId(${deselectB}_LOCID)
	HelpOnItemRegister $symbolicName(${widget}DeselectAll) $appName \
		$locId(LOCID) $locId(${deselectallB}_LOCID)
}				; ## end routine GUI_Selection_InitHelp


##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_ClearSelected
 #
 #	This routine will delete all of the items selected by the 
 #	user.  It will only work in conjunction with the 
 #	GUI_Selection_Create routine.
 #
 # Inputs: 
 #
 #	widget - The "widget" returned by GUI_Selection_Create.
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_ClearSelected {widget} {

	global symbolicName	; # global array of widgets

	if {! [winfo exists $widget] } {return 0}

	#
	# Clear the selections
	#
	$symbolicName(${widget}Selected) delete 0 end

	#
	# Update the state of the buttons
	#
	GUI_Selection_UpdateState $widget
}				; ## end routine GUI_Selection_ClearSelected




##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_GetSelected
 #
 #	The GUI_Selection_Get_Selected routine will return the list of objects
 #	which have been selected by the user.  This routine should only
 #	be called when the user is finished selecting as the list which
 #	is returned contains only the items which were in the "selected"
 #	listbox.  This should only ever be used when expecting data from 
 #	a "selection" object created by GUI_Selection_Create.
 #
 # Inputs: 
 #
 #	widget - The "widget" returned by GUI_Selection_Create.
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_GetSelected {widget} {
	
	global symbolicName	; # global array of widgets

	if {! [winfo exists $widget] } {return 0}

	return "[$symbolicName(${widget}Selected) get 0 end]"
}				; ## end routine GUI_Selection_GetSelected



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_SelectItem
 #
 #	Selects the item at location $index within widget $widget
 #
 # Inputs: 
 #
 #	widget - The widget where the item can be found
 #
 #	index - The y-coordinate at which the event occurred
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
##---------------------------------------------------------------------------
proc GUI_Selection_SelectItem {widget index} {

	$widget select set anchor [$widget nearest $index]
}				; ## end routine GUI_Selection_SelectItem



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_UpdateState
 #
 #	This routine updates the states for the select, selectall, 
 #	deselect, and deselectall buttons so that they reflect the 
 #	current choices.  Any operation which acts on the contents 
 #	of either listbox should call the routine to update the 
 #	state of the buttons.
 #
 # Inputs: 
 #
 #	widget - The "widget" returned by GUI_Selection_Create.
 #			
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_UpdateState {widget} {

	global symbolicName	; # global array of widgets
	global enableList	; # list of widgets to be configured to 
				  # a normal state when items are selected
	global disableList	; # list of widgets to be configured to 
				  # a disabled state when items are selected


	if {! [winfo exists $widget] } {return 0}
	
	#
	# Update the state of the select button
	#
	if {[lempty [$symbolicName(${widget}Available) curselection]]} {
		$symbolicName(${widget}Select) configure -state disabled
	} else {
		$symbolicName(${widget}Select) configure -state normal
	}

	#
	# Update the state of the deselect button
	#
	if {[lempty [$symbolicName(${widget}Selected) curselection]]} {
		$symbolicName(${widget}Deselect) configure -state disabled
	} else {
		$symbolicName(${widget}Deselect) configure -state normal
	}

	#
	# Update the state of the selectall button
	#
	if {[lempty [$symbolicName(${widget}Available) get 0]]} {
		$symbolicName(${widget}SelectAll) configure -state disabled
	} else {
		$symbolicName(${widget}SelectAll) configure -state normal
	}

	#
	# Set a local variable to determine if any selections have
	# been made.
	#
	set noSelection [lempty [$symbolicName(${widget}Selected) get 0]]

	#
	# Update the state of the deselectall button
	#
	if { $noSelection } {
		$symbolicName(${widget}DeselectAll) configure -state disabled
	} else {
		$symbolicName(${widget}DeselectAll) configure -state normal
	}

	#
	# Update the state of the widgets in the enableList.  These
	# widgets should be configured to {normal} when there are 
	# selections in the "Selected" listbox.
	# 
	foreach element $enableList($widget) {
		if { [winfo exists $element] } {
			if { $noSelection } {
				$element configure -state disabled
			} else {
				$element configure -state normal
			}
		}
	}

	#
	# Update the state of the widgets in the disableList.  These
	# widgets should be configured to {disabled} when there are 
	# selections in the "Selected" listbox.
	# 
	foreach element $disableList($widget) {
		if { [winfo exists $element] } {
			if { $noSelection } {
				$element configure -state normal
			} else {
				$element configure -state disabled
			}
		}
	}
}				; ## GUI_Selection_UpdateState



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_SelectState
 #
 #	This routine will cause the state of the widgets in the 
 #	enableList and disableList to be set to normal or disabled
 #	whenever a change to the list of selected items occurs.	
 #	The actual configuring of these widgets is done in the
 #	routine GUI_Selection_UpdateState.
 #
 # Inputs: 
 #
 #	widget      - The "widget" returned by GUI_Selection_Create.
 #			
 #	enable  - list of widgets which should be set to normal
 #		  when a selection has been made.
 #
 #	disable - list of widgets which should be set to disbaled
 #		  when a selection has been made.
 #
 # Outputs: 
 #
 #	enableList  - list of widgets which should be set to 
 #		      normal when a selection has been made.
 #
 #	disableList - list of widgets which should be set to 
 #		      disabled when a selection has been made.
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_SelectState {widget enable disable} {

	global enableList	; # list of widgets to be configured to 
				  # a normal state when items are selected
	global disableList	; # list of widgets to be configured to 
				  # a disabled state when items are selected
 

	if {! [winfo exists $widget] } {return 0}

	#
	# Set the lists to be used by the GUI_Selection_UpdateState 
	# routine.  Upon a selection/unselection the state of the 
	# widgets will be updated.  The widgets in the enableList 
	# will be set to {normal} and the widgets in the 
	# disableList will be set to {disabled}.
	#
	set enableList($widget) $enable
	set disableList($widget) $disable

	#
	# Update the states now
	#
	GUI_Selection_UpdateState $widget
}				; ## end routine GUI_Selection_SelectState



##---------------------------------------------------------------------------
 #
 # Proc:  GUI_Selection_Filter
 #
 #	This routine will add/remove the specified list of items from/to
 #	the selection object created by GUI_Selection_Create.  
 #
 # Inputs: 
 #
 #	widget      - The "widget" returned by GUI_Selection_Create.
 #
 #	operation   - Either "mask" to remove from the list or "unmask" to
 #		      add to the list.
 #
 #	list        - The list of items to add/remove
 #
 # Outputs: 
 #
 # Returns: 
 #
 # Notes:
 #
 #	Requires that the function GUI_Selection_Create be called
 #	before this routine is invoked.
 #
##---------------------------------------------------------------------------
proc GUI_Selection_Filter {widget operation list} {

	global symbolicName	; # global array of widgets

	if {! [winfo exists $widget] } {return 0}

	if [cequal $operation "mask"] {
		#
		# Mask out the entries in the available list
		#
		set newlist [$symbolicName(${widget}Available) get 0 end]
		set newlist [lindex [intersect3 $newlist $list] 0]
		$symbolicName(${widget}Available) delete 0 end
		eval {$symbolicName(${widget}Available) insert end} $newlist

		#
		# Mask out the entries in the selected list
		#
		set newlist [$symbolicName(${widget}Selected) get 0 end]
		set newlist [lindex [intersect3 $newlist $list] 0]
		$symbolicName(${widget}Selected) delete 0 end
		eval {$symbolicName(${widget}Selected) insert end} $newlist
	} else {
		#
		# Add the entries back into the available list.
		#
		set newlist [lsort [concat $list \
			[$symbolicName(${widget}Available) get 0 end]]]
		$symbolicName(${widget}Available) delete 0 end
		eval {$symbolicName(${widget}Available) insert end} $newlist
	}
}				; ## end routine GUI_Selection_Filter


#########---------------------------------------------------------#########
#########		End Selection Widget Code		  #########
#########---------------------------------------------------------#########

