# 
# *****************************************************************
# *                                                               *
# *   Copyright 2002 Compaq Information Technologies Group, L.P.  *
# *                                                               *
# *   The software contained on this media  is  proprietary  to   *
# *   and  embodies  the  confidential  technology  of  Compaq    *
# *   Computer Corporation.  Possession, use,  duplication  or    *
# *   dissemination of the software and media is authorized only  *
# *   pursuant to a valid written license from Compaq Computer    *
# *   Corporation.                                                *
# *                                                               *
# *   RESTRICTED RIGHTS LEGEND   Use, duplication, or disclosure  *
# *   by the U.S. Government is subject to restrictions  as  set  *
# *   forth in Subparagraph (c)(1)(ii)  of  DFARS  252.227-7013,  *
# *   or  in  FAR 52.227-19, as applicable.                       *
# *                                                               *
# *****************************************************************
#
# HISTORY
# 
# @(#)$RCSfile: sm_tools.tcl,v $ $Revision: 1.1.1.1 $ (DEC) $Date: 2003/01/23 18:36:31 $
# 

# Using [A-Z] or [a-z] in regexps in some locales
# causes bad results. Luckily, for our purposes the
# 26 letters of English are all we need.
set UC_LETTER "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
set LC_LETTER "abcdefghijklmnopqrstuvwxyz"

proc sm_applicationToolsInitialization {} {

	sm_widgetInitialization
	sm_cursorInitialization
	sm_navigationInitialization
}


proc sm_messageCatalogInitialization { appName {defaultSection {TXT}} } {

	global catfd
	global ${appName}_catfd
	global i18nfd
	global [translit a-z A-Z $appName]_CATFILE
	global I18N_MOTIF_SHARED_TEXT_CATFILE
        global SysmanDir
        global env

        # SysMan message catalog support code. It currently can't be used for 
        # update installs.  The updates install sets the UPDFLAG environment
	# variable.  It will also not be used if an application sets the 
	# LEGACY_CATINIT environment variable.
	#
        if { ![info exists env(UPDFLAG)] && \
	     ![info exists env(LEGACY_CATINIT)]} {
            source $SysmanDir/utils/catgets.tcl
	}


	${appName}MsgcatInit
	${appName}_dstrInit
	${appName}_LocIdsMsgcatInit

        set catname [subst $[translit a-z A-Z $appName]_CATFILE]
	set catfd [catopen $catname]
	set ${appName}_catfd $catfd

	i18n_motif_shared_textMsgcatInit
	i18n_motif_shared_text_dstrInit

	set i18nfd [catopen $I18N_MOTIF_SHARED_TEXT_CATFILE]

	##-------------------------------------------------------------------
	 #
	 # Proc:  sm_catgets
	 #
	 #      Routine to fetch a message from the message catalog.  If the 
	 #      message cannot be found then a default string containing the 
	 #      message tag is returned.
	 #
	 # Inputs: 
	 #
	 #      msgName      - The message tag of the message to be retrieved
	 #
	 #      sectionName  - An optional argument which determines which 
	 #                     section of the catalog is to be searched for
	 #                     the message tag
	 # Outputs: 
	 #
	 # Returns: 
	 #
	 #      A string containing the requested message.  In the case of 
	 #      a failure a default string containing the message tag is 
	 #      returned.
	 #
	 # Notes:
	 #
	 #	Any variables which should not be evaluated when the 
	 #	sm_messageCatalogInitialization routine is invoked should
	 #	be preceeded by a '\'.
	 #
	##-------------------------------------------------------------------
	eval [subst -nocommands {proc sm_catgets { msgName {sectionName {}} } {

		global catfd appName

		#
		# Create local references for the message catalog globals
		#
		upvar #0 msg_$appName msg 
		upvar #0 ${appName}_dstr dstr

		#
		# If the section is not defined then use the default
		# from the call to sm_messageCatalogInitialization.
		#
		if { [lempty \$sectionName] } {
			set section \$msg($defaultSection)
		} else {
			set section \$msg(\$sectionName)
		}
	
		#
		# Retrieve the message from the catalog.  If a message is
		# not found then try to use the default message.  If the 
		# default message can not be found then return the message
		# tag enclosed in '?'.
		#
		if {[catch {set ret [catgets \$catfd \$section \
			\$msg(\$msgName) [cexpand \$dstr(\$msgName)]]} err]} {
				 set ret "?\$msgName?"
		 }

		 return \$ret
	 }}]			; ## end routine sm_catgets

}			; ## end routine sm_messageCatalogInitialization


proc sm_widgetInitialization {} {

	uplevel #0 rename listbox sm_originalListbox
	uplevel #0 rename _sm_listbox listbox
}


proc sm_cursorInitialization {} {
	global env
	global _smv_isWaiting
	global _smv_waitCursor
	global _smv_grabStack
	global _smv_noAccessCursor

	set _smv_grabStack ""
	set _smv_isWaiting 0
	set _smv_waitCursor watch

	if { [info exists env(SYSMANDIR)] } {
		set cursorFile "$env(SYSMANDIR)/mailconfig/noentry.xbm"
		set maskFile "$env(SYSMANDIR)/mailconfig/noentry.mask"
	} else {
		set cursorFile "/usr/share/sysman/mailconfig/noentry.xbm"
		set maskFile "/usr/share/sysman/mailconfig/noentry.mask"
	}

	if { ![file exists $cursorFile] } {
		set _smv_noAccessCursor draped_box
	} else {
		set _smv_noAccessCursor "@$cursorFile $maskFile black white"
	}

	uplevel #0 rename destroy sm_originalDestroy
	uplevel #0 rename _sm_destroy destroy
	uplevel #0 rename grab sm_originalGrab
	uplevel #0 rename _sm_grab grab
#	uplevel #0 rename toplevel sm_originalToplevel
#	uplevel #0 rename _sm_toplevel toplevel

	if { [lsearch -exact [info procs tkerror] tkerror] != -1 } {
		uplevel #0 rename tkerror sm_originalTtkerror
	}
	uplevel #0 rename _sm_tkerror tkerror

}


proc sm_navigationInitialization {} {

	auto_load tkFocusOK

	uplevel #0 {rename tk_focusPrev ""}
	uplevel #0 rename _sm_tk_focusPrev tk_focusPrev
	uplevel #0 {rename tk_focusNext ""}
	uplevel #0 rename _sm_tk_focusNext tk_focusNext
}


proc _sm_destroy args {
	global _smv_grabStack

	set localArgs $args
	set currentGrabs [grab current]
	foreach grabWindow $currentGrabs {
		if { [lmatch -exact $localArgs $grabWindow] != "" } {
			if { [lmatch -exact $_smv_grabStack $grabWindow] == "" } {
				sm_originalDestroy $localArgs
				_sm_resetCursor

				if { $_smv_grabStack != "" } {
					foreach window $_smv_grabStack {
						set modalWindow [lvarpop _smv_grabStack]
						if { [winfo exists $modalWindow] } {
							sm_originalGrab set $modalWindow
							_sm_setModalCursor
							break
						}
					}
				}
				return

			} else {
				_sm_resetCursor
				sm_originalGrab $grabWindow
			}
		}
	}
	sm_originalDestroy $localArgs
}


proc _sm_grab args {
	global _smv_grabStack

	set localArgs $args
	set globalFlag 0
	switch -exact -- [lindex $localArgs 0] {
		release {
			sm_originalGrab release [lindex $localArgs 1]
			_sm_resetCursor

			if { $_smv_grabStack != "" } {
				foreach window $_smv_grabStack {
					set modalWindow [lvarpop _smv_grabStack]
					if { [winfo exists $modalWindow] } {
						sm_originalGrab set $modalWindow
						_sm_setModalCursor
						break
					}
				}
			}
			return ""
		}
		status -
		-global -
		current {
			return [eval sm_originalGrab $localArgs]
		}
		set {
			lvarpop localArgs
		}
	}

	set currentGrab [sm_originalGrab current]
	if { $currentGrab != $localArgs } {
		if { $currentGrab != "" } {
			lvarpush _smv_grabStack $currentGrab
		}
		sm_originalGrab set $localArgs
	}
	_sm_setModalCursor
}


proc _sm_toplevel { args } {

	set command "sm_originalToplevel $args"
	set toplevelWidget [eval $command]

#	tkwait visibility $toplevelWidget
	wm protocol $toplevelWidget WM_DELETE_WINDOW "destroy $toplevelWidget"

	return $toplevelWidget
}


proc _sm_listbox args {

	set command "sm_originalListbox $args"
	set commandName [eval $command]

	set widgetName [lvarpop args]

	set parent [crange $widgetName 0 [expr [sm_strrchr $widgetName .] - 1]]
	set widgetName [crange $widgetName [expr [sm_strrchr $widgetName .] + 1] end]

	frame $parent.${widgetName}Frame -borderwidth 0 -highlightthickness 0
	frame $parent.${widgetName}Frame.verticalFrame -borderwidth 0 -highlightthickness 0
	scrollbar $parent.${widgetName}Frame.verticalFrame.scrollbar -orient vertical -command [list $commandName yview]
	frame $parent.${widgetName}Frame.horizontalFrame -borderwidth 0 -highlightthickness 0
	scrollbar $parent.${widgetName}Frame.horizontalFrame.scrollbar -orient horizontal -command [list $commandName xview]
	set padSize [expr [$parent.${widgetName}Frame.horizontalFrame.scrollbar cget -width] + 2 * \
		([$parent.${widgetName}Frame.horizontalFrame.scrollbar cget -borderwidth] + \
		[$parent.${widgetName}Frame.horizontalFrame.scrollbar cget -highlightthickness]) ]

	frame $parent.${widgetName}Frame.horizontalFrame.horizontalPadFrame \
		-width $padSize \
		-height $padSize \
		-borderwidth 0 \
		-highlightthickness 0

	bind ${commandName} <Configure> +[subst	{
							after cancel "_sm_checkListbox $commandName"
							after idle "_sm_checkListbox $commandName"
						}]
	bind ${commandName} <ButtonRelease-2> +[subst	{
							after cancel "_sm_checkListbox $commandName"
							after idle "_sm_checkListbox $commandName"
						}]
	bind $parent.${widgetName}Frame.verticalFrame.scrollbar <ButtonRelease-1> "_sm_checkListbox $commandName"
	bind $parent.${widgetName}Frame.horizontalFrame.scrollbar <ButtonRelease-1> "_sm_checkListbox $commandName"

	rename $commandName sm_original$commandName
	eval {
		proc $commandName { args } {
			set commandLine [info level [info level]]
			set commandName [lvarpop commandLine]

			set command "sm_original$commandName $args"
			set returnValue [eval $command]
			if { [_sm_listboxPacked $commandName] } {
				switch -exact -- [lvarpop commandLine] {
					insert -
					delete {
						_sm_checkVerticalScrollbar $commandName
						_sm_checkHorizontalScrollbar $commandName
					}
				}
			}
			after cancel "_sm_checkListbox $commandName"
			after idle "_sm_checkListbox $commandName"
			return $returnValue
		}
	}
	return $commandName
}


proc _sm_checkListbox widget {

	if { [_sm_listboxPacked $widget] } {
		_sm_checkVerticalScrollbar $widget
		_sm_checkHorizontalScrollbar $widget
	}
}


proc _sm_listboxPacked widget {

	if { [catch "pack info $widget"] } {
		return 0
	}
	if { [catch "pack info ${widget}Frame"] } {
		set parent [winfo parent $widget]
		set packSettings [pack info $widget]
		set master [lindex $packSettings [expr [lsearch -exact $packSettings "-in"] + 1]]
		set slaves [pack slaves $master]
		set nextWidgetIndex [expr [lsearch -exact $slaves $widget] + 1]
		pack forget $widget

		pack $widget -in ${widget}Frame.verticalFrame -side left -expand 1 -fill both

		pack ${widget}Frame.verticalFrame -expand 1 -fill both
		pack ${widget}Frame.horizontalFrame -side bottom -fill x

		raise $widget

		if { $nextWidgetIndex >= [llength $slaves] } {
			eval pack ${widget}Frame $packSettings
		} else {
			set nextWidget [lindex $slaves $nextWidgetIndex]
			eval pack ${widget}Frame $packSettings -before $nextWidget
		}
	}
	return 1
}


proc _sm_checkVerticalScrollbar widget {

	scan [winfo geometry $widget] "%dx%d+%d+%d" listboxWidth listboxHeight listboxX listboxY

	set firstVisible [sm_original$widget nearest 0]
	set lastVisible [sm_original$widget nearest $listboxHeight]
	set listSize [sm_original$widget size]

	# If there's nothing in the list return...
	if { $firstVisible < 0 } {
		return
	}

	set numConversions [scan \
		[sm_original$widget bbox $firstVisible] "%d %d %d %d" listboxEntryX listboxEntryY listboxEntryWidth listboxEntryHeight]
	if { $numConversions != 4 } {
		return
	}
	set secondVisible [sm_original$widget nearest [expr $listboxEntryY + $listboxEntryHeight + ($listboxEntryHeight/2)]]

	if { $secondVisible <= 0 } {
		set bboxHeight $listboxEntryHeight
	} else {
		scan [sm_original$widget bbox $secondVisible] "%d %d %d %d" junk secondY junk junk
		# If there's only one item visible set bboxHeight so following division will work...
		if { [set bboxHeight [expr $secondY - $listboxEntryY]] <= 0 } {
			set bboxHeight $listboxEntryHeight
		}
	}

	scan [sm_original$widget bbox $lastVisible] "%d %d %d %d" junk lastY junk lastHeight

	set parent [winfo parent $widget]
	if { $parent == "" } {
		set parent .
	}

	set listboxFrame $parent.[winfo name $widget]Frame
	set listboxVerticalScrollbarFrame $parent.[winfo name $widget]Frame.verticalFrame
	set listboxVerticalScrollbar $parent.[winfo name $widget]Frame.verticalFrame.scrollbar

	set numberOfVisableListboxEntries [expr $listboxHeight / $bboxHeight]
	set calculatedHeight [expr $lastY + $lastHeight + $listboxEntryY]
	if {	$firstVisible != 0 ||
		$numberOfVisableListboxEntries < $listSize ||
		$calculatedHeight > $listboxHeight } {

		# If it already has a scrollbar return...
		if { [catch "pack info $listboxVerticalScrollbar"] == 0 } {
			return
		}
		pack ${widget}Frame.verticalFrame.scrollbar -side right -fill y
		sm_original$widget configure -yscrollcommand [list ${widget}Frame.verticalFrame.scrollbar set]
		if { [catch "pack info ${widget}Frame.horizontalFrame.scrollbar"] == 0 } {
			pack ${widget}Frame.horizontalFrame.horizontalPadFrame -side right
		}
	} else {
		# It doesn't need scrollbar and it doesn't have one so return...
		if { [catch "pack info $listboxVerticalScrollbar"] } {
			return
		}
		sm_original$widget yview 0
		pack forget $listboxVerticalScrollbar
		sm_original$widget configure -yscrollcommand ""
		if { [catch "pack info ${widget}Frame.horizontalFrame.horizontalPadFrame"] == 0 } {
			pack forget ${widget}Frame.horizontalFrame.horizontalPadFrame
		}
	}
}


proc _sm_checkHorizontalScrollbar widget {

	set listboxFrame ${widget}Frame
	set listboxHorizontalScrollbarFrame ${widget}Frame.horizontalFrame
	set listboxHorizontalScrollbar ${widget}Frame.horizontalFrame.scrollbar
	set listboxHorizontalPadFrame ${widget}.Frame.horizontalFrame.horizontalPadFrame

	scan [winfo geometry $widget] "%dx%d+%d+%d" listboxWidth height x y
	set listboxLength [sm_original$widget size]
	for {set i 0} {$i < $listboxLength} {incr i} {
		if { [set indexBoundingBox [sm_original$widget bbox $i]] != "" } {
			scan [sm_original$widget bbox $i] "%d %d %d %d" listboxEntryX listboxEntryY listboxEntryWidth listboxEntryHeight
			if { $listboxEntryWidth > $listboxWidth } {
				if { [catch "pack info $listboxHorizontalScrollbar"] == 1 } {
					pack ${widget}Frame.horizontalFrame.scrollbar -side left -expand 1 -fill x
					if { [catch "pack info ${widget}Frame.verticalFrame.scrollbar"] == 0 } {
						pack ${widget}Frame.horizontalFrame.horizontalPadFrame -side right
					}
					sm_original$widget configure -xscrollcommand [list ${widget}Frame.horizontalFrame.scrollbar set]
				}
				return
			}
		}
	}
	if { [catch "pack info $listboxHorizontalScrollbar"] == 0 } {
		sm_original$widget xview 0
		pack forget ${widget}Frame.horizontalFrame.scrollbar
		pack forget ${widget}Frame.horizontalFrame.horizontalPadFrame
		sm_original$widget configure -xscrollcommand ""
	}
	return
}


proc sm_strchr { string character } {

	set length [clength $string]
	for { set i 0 } { $i < $length } {  incr i } {
	   if [cequal $character [cindex $string $i]] {
	      return $i
	   }
	}
	return -1
}


proc sm_strrchr { string character } {

	set length [expr [clength $string] - 1]
	for { set i $length } { $i >= 0 } {  incr i -1 } {
	   if [cequal $character [cindex $string $i]] {
	      return $i
	   }
	}
	return -1
}


proc sm_isDescendant { parentWidget potentialChildWidget } {

	set stringLength [clength $parentWidget]
	if { $stringLength == 0 } {
		puts "sm_isDescendant: Null value given for parentWidget."
		return -1
	}
	set rc [cequal $parentWidget [crange $potentialChildWidget 0 [expr $stringLength - 1]]]
	return $rc
}


proc _sm_resetCursor {} {

	global _smv_isWaiting

	if { $_smv_isWaiting == 1 } {
		sm_setWaitCursor
	} else {
		sm_setNormalCursor
	}
}


proc _sm_setModalCursor {} {
	global _smv_noAccessCursor
	global _smv_cursorArray

	if { [set modalWidget [grab current]] == "" } {
		return
	}
	set allToplevels ""
	widgetsByClass . Toplevel allToplevels
	lappend allToplevels .
	foreach tl $allToplevels {
		if { ![sm_isDescendant $modalWidget $tl] } {
			if { ![info exists _smv_cursorArray($tl)] } {
				set _smv_cursorArray($tl) [$tl cget -cursor]
			}
			$tl configure -cursor $_smv_noAccessCursor

			set children ""
			windowWidgetTree $tl children
			foreach child $children {
				if { [$child cget -cursor] != "" } {
					if { ![info exists _smv_cursorArray($child)] } {
						set _smv_cursorArray($child) [$child cget -cursor]
					}
					$child configure -cursor $_smv_noAccessCursor
				}
			}
		} else {
			set children ""
			windowWidgetTree $tl children
			lappend children $tl
			foreach child $children {
				if { [info exists _smv_cursorArray($child)] } {
					$child configure -cursor $_smv_cursorArray($child)
					unset _smv_cursorArray($child)
				}
			}
		}
	}
}


proc sm_setWaitCursor {} {
	global _smv_waitCursor
	global _smv_cursorArray
	global _smv_isWaiting

	set _smv_isWaiting 1

	set allToplevels ""
	widgetsByClass . Toplevel allToplevels
	lappend allToplevels .
	foreach tl $allToplevels {
		if { ![info exists _smv_cursorArray($tl)] } {
			set _smv_cursorArray($tl) [$tl cget -cursor]
			$tl configure -cursor $_smv_waitCursor
		}

		set children ""
		windowWidgetTree $tl children
		foreach child $children {
			if { [$child cget -cursor] != "" } {
				if { ![info exists _smv_cursorArray($child)] } {
					set _smv_cursorArray($child) [$child cget -cursor]
					$child configure -cursor $_smv_waitCursor
				}
			}
		}
	}
	update idletasks
}


proc sm_setNormalCursor {} {
	global _smv_waitCursor
	global _smv_cursorArray
	global _smv_isWaiting

	set _smv_isWaiting 0

	foreach tl [array names _smv_cursorArray] {
		if { [winfo exists $tl] } {
			$tl configure -cursor $_smv_cursorArray($tl)
		}
		unset _smv_cursorArray($tl)
	}

	_sm_setModalCursor
	update idletasks
}


proc sm_isWaitCursor { cursor } {
	global _smv_waitCursor

	return [cequal $cursor $_smv_waitCursor]
}


proc sm_getSavedCursor { widget } {
	global _smv_cursorArray

	if { [info exists _smv_cursorArray($widget)] } {
		return $_smv_cursorArray($widget)
	} else {
		return ""
	}
}


proc sm_getStackTrace {} {

	set stackTrace ""
	set levelNumber [expr [info level] - 1]
	while { $levelNumber > 0 } {
		set stackTrace [linsert $stackTrace 0 [lindex [info level $levelNumber] 0]]
		incr levelNumber -1
	}
	return $stackTrace
}


proc sm_hasChildToplevel { parentWidget } {

	foreach child [winfo children $parentWidget] {
		if { [winfo class $child] == "Toplevel" } {
			return 1
		}
	}
	return 0
}


proc HelpPointerFocusRegister { appName widgetName helpTag defaultString } {
	global ${appName}_catfd
	global msg_$appName
	global symbolicName

	set helpWidget [sm_getToplevelName $widgetName]Help
	if { [info exists $symbolicName($helpWidget)] } {
		bind $widgetName <Enter> " \
			HelpEntered_widget [list [catgets $catfd $msg_$appName(PFH) $msg_$appName($helpTag) $defaultString]] $symbolicName($helpWidget)"
		bind $widgetName <Leave> " \
			HelpLeaving_widget"
	}

}


proc sm_getToplevelName { widget } {
	global symbolicName

	if { ![winfo exists $widget] } {
		if { [info exists symbolicName($widget)] == 1 && [winfo exists $symbolicName($widget)] == 1 } {
			set widget $symbolicName($widget)
		} else {
			return ""
		}
	}
	set toplevelWidget [winfo toplevel $widget]

	if [cequal $toplevelWidget "."] {
		return mainWindow
	} else {
		return [string range $toplevelWidget [expr [string last "." $toplevelWidget] + 1] end]
	}
}


proc sm_buildHelpText { parentWidget } {
	global symbolicName

	# build widget $parentWidget.helpFrame.helpText
	set textWidget [text $parentWidget.helpText \
		-width 40 \
		-height {2} \
		-cursor {} \
		-relief flatline \
		-takefocus 0 \
		-wrap word]
	bindtags $textWidget [setSubtract [bindtags $textWidget] Text]

	pack $textWidget -fill x -padx 10

	set symbolicName([sm_getToplevelName $parentWidget]Help) $textWidget
}


proc _sm_createSymbolicIndex { toplevelName name class } {

    global UC_LETTER LC_LETTER

	set newString ""
	for { set i 0 } { $i < [clength $name] } { incr i } {
	    if { [regexp \[${UC_LETTER}${LC_LETTER}_\] [cindex $name $i]] } {
			append newString [cindex $name $i]
		}
	}
	return  "$toplevelName[translit a-z A-Z [crange $newString 0 0]][crange $newString 1 end]$class"
}


proc _sm_createLabelText startString {

	set newString [sm_nameToString $startString]
	if { [sm_endsWith $startString "..."] } {
		append $newString "..."
	}
	return $newString
}


proc sm_endsWith { compareString suffix } {

	if { [cequal [crange $compareString [expr [clength $compareString] - [clength $suffix]] end] $suffix] } {
		return 1
	}
	return 0
}


proc sm_removeSuffix { compareString suffix } {

	if { [sm_endsWith $compareString $suffix] } {
		return [crange $compareString 0 [expr [clength $compareString] - [clength $suffix] - 1 ]]
	}
	return $compareString
}


proc sm_getApplicationPrefix applicationName {

    global UC_LETTER

	set newString ""
	for { set i 0 } { $i < [clength $applicationName] } { incr i } {
		if { [regexp \[$UC_LETTER\] [cindex $applicationName $i]] } {
			append newString [cindex $applicationName $i]
		}
	}
	return [translit A-Z a-z $newString]
}


proc sm_buildSubdialog {	applicationName \
				toplevelWidgetName \
				{windowType {secondary}}\
				{parent {.}}\
				{parentWindow {.}}\
				{modality {nonmodal}} \
				{contentsProcArgs {}} \
				{buttons {ok apply cancel help}} \
				{defaultButton ok} } {
	global symbolicName

	if [cequal $parent "."] {
		set newToplevelName .$toplevelWidgetName
	} else {
		set newToplevelName $parent.$toplevelWidgetName
	}
	if { [winfo exists $newToplevelName] } {
		raise $newToplevelName
		focus -force $newToplevelName
		return 0
	}
	set toplevelWidget [toplevel $newToplevelName]
	set symbolicName($toplevelWidgetName) $toplevelWidget

	if { $parentWindow != "" } {
		wm transient $toplevelWidget $parentWindow
	} else {
		set parentWindow .
	}

	set x [winfo x $parentWindow]
	set y [winfo y $parentWindow]

	# Window manager configurations
	wm title $toplevelWidget "[sm_nameToString $applicationName]: [sm_nameToString $toplevelWidgetName]"
	wm maxsize . [expr [winfo screenwidth .] - 35 ] [expr [winfo screenheight .] - 50 ]
	wm geometry $toplevelWidget +[expr $x + 50]+[expr $y + 50]

	# build the frame to hold the body of the dialog...
	frame $toplevelWidget.contentsFrame \
		-borderwidth 0 \
		-relief flat

	# Call the appropriate build contents proc to fill the body...
	set commandName [sm_getApplicationPrefix $applicationName]_build${toplevelWidgetName}Contents
	#am_registerFileContents $applicationName buildContentsProc $commandName
	eval $commandName $toplevelWidget.contentsFrame $contentsProcArgs

	# build the frame to hold the help text...
	frame $toplevelWidget.helpFrame \
		-borderwidth 0 \
		-relief flat

	sm_buildHelpText $toplevelWidget.helpFrame

	# build the frame to hold the dialog buttons...
	frame $toplevelWidget.dialogButtonFrame \
		-borderwidth 2 

	set commandPrefix "[sm_getApplicationPrefix $applicationName]g_"
	eval sm_buildDialogButtons $applicationName $toplevelWidget.dialogButtonFrame $buttons
	if {	[info exists symbolicName(${toplevelWidgetName}CancelButton)] &&
		[winfo exists $symbolicName(${toplevelWidgetName}CancelButton)] &&
		[lsearch -exact [info commands "${commandPrefix}${toplevelWidgetName}CancelButton"] \
			"${commandPrefix}${toplevelWidgetName}CancelButton"] != -1 } {

		wm protocol $toplevelWidget WM_DELETE_WINDOW \
			"${commandPrefix}${toplevelWidgetName}CancelButton $symbolicName(${toplevelWidgetName}CancelButton)"
	} elseif { [info exists symbolicName(${toplevelWidgetName}OkButton)] &&
		[winfo exists $symbolicName(${toplevelWidgetName}OkButton)] &&
		[lsearch -exact [info commands "${commandPrefix}${toplevelWidgetName}OkButton"] \
			"${commandPrefix}${toplevelWidgetName}OkButton"] != -1 } {

		wm protocol $toplevelWidget WM_DELETE_WINDOW \
			"${commandPrefix}${toplevelWidgetName}OkButton $symbolicName(${toplevelWidgetName}OkButton)"
	} else {
		wm protocol $toplevelWidget WM_DELETE_WINDOW "destroy $toplevelWidget"
	}

	sm_checkColors $toplevelWidget

	if { $defaultButton != "" } {
		set defaultButtonName [translit a-z A-Z [crange $defaultButton 0 0]][crange $defaultButton 1 end]
		if { [info exists symbolicName(${toplevelWidgetName}${defaultButtonName}Button)] } {
			sm_setDefaultButton $symbolicName(${toplevelWidgetName}${defaultButtonName}Button)
		}
	}

	# pack the three frames...
	pack $toplevelWidget.contentsFrame -side top -fill both -expand 1
	pack $toplevelWidget.dialogButtonFrame -side bottom -anchor s -fill both -pady 4
	pack $toplevelWidget.helpFrame -side bottom -fill x

	if { $modality == "modal" } {
		tkwait visibility .$toplevelWidgetName
		grab set .$toplevelWidgetName
	}
	focus -force $toplevelWidget
	return 1
}


proc sm_checkColors toplevelWidget {

	windowWidgetTree $toplevelWidget children
	lappend children $toplevelWidget
	foreach child $children {
		set backgroundColor [$child cget -background]
#		catch {$child configure -disabledforeground ""}
		catch {$child configure -highlightcolor black}
		if { [catch {set highlightBackgroundColor [$child cget -highlightbackground]}] != 0 } {
			continue
		}
		if { $backgroundColor != $highlightBackgroundColor } {
			$child configure -highlightbackground $backgroundColor -highlightcolor black
		}
		catch {$child configure -highlightcolor black}
		if { [winfo class $child] == "Frame" } {
			$child configure -highlightthickness 0
		}
	}
}


proc sm_buildDialogButtons { appName parentWidget args } {

	eval sm_buildButtons $appName $parentWidget horizontal $args
}


proc sm_buildButtons { appName parentWidget orientation args } {
	global i18nfd
	global msg_i18n_motif_shared_text
	global i18n_motif_shared_text_dstr
	global ${appName}_catfd
	global msg_$appName
	global ${appName}_dstr
	global symbolicName

	set toplevelWidgetName [sm_getToplevelName $parentWidget]

	if { [cequal [translit A-Z a-z $orientation] vertical] } {
		set packSide ""
	} else {
		set packSide "-side left"
	}

	set commandPrefix "[sm_getApplicationPrefix $appName]g_"
	set buttonLabelWidth 8
	foreach dispositionButton $args {
		set symbolicButtonName [_sm_createSymbolicIndex $toplevelWidgetName $dispositionButton Button]
		set defaultButtonText [_sm_createLabelText $dispositionButton]
		set buttonType [sm_removeSuffix [translit " " "_" $dispositionButton] "..."]

		set buttonText $defaultButtonText
		switch -exact -- $buttonType {
			commit -
			ok -
			apply -
			close -
			cancel -
			help -
			yes -
			no {
				if { [info exists msg_i18n_motif_shared_text(i18n_${buttonType}_txt)] } {
					set buttonText [catgets $i18nfd $msg_i18n_motif_shared_text(i18n_shared_strings) $msg_i18n_motif_shared_text(i18n_${buttonType}_txt) \
						$i18n_motif_shared_text_dstr(i18n_${buttonType}_txt)]
				}
			}
			default {
				set messageTag [translit a-z A-Z "[sm_getApplicationPrefix \
						$appName]_${toplevelWidgetName}_${buttonType}_Button"]
				if { [info exists [subst msg_${appName}]($messageTag)] } {
					set messageIndex [set [subst msg_${appName}($messageTag)]]
					set textIndex [set [subst msg_${appName}(TEXT)]]
					set buttonText [catgets [set ${appName}_catfd] $textIndex $messageIndex \
						[set ${appName}_dstr($messageTag)]]
				}
			}
		}
		if { [expr [clength $buttonText] + 2] > $buttonLabelWidth } {
			set buttonLabelWidth [expr [clength $buttonText] + 2]
		}
		# BUILD THE BUTTON...
		set symbolicName($symbolicButtonName) $parentWidget.${buttonType}Button
		button $symbolicName($symbolicButtonName) \
			-command "${commandPrefix}$symbolicButtonName $symbolicName($symbolicButtonName)" \
			-text $buttonText
	}
	foreach dispositionButton $args {
		set symbolicButtonName [_sm_createSymbolicIndex $toplevelWidgetName $dispositionButton Button]
		$symbolicName($symbolicButtonName) configure -width $buttonLabelWidth
		eval pack $symbolicName($symbolicButtonName) $packSide -expand 1
	}
}


proc sm_reverseClassInstanceBinding widget {

	set originalTags [bindtags $widget]
	set widgetClass [winfo class $widget]
	set classIndex [lsearch -exact $originalTags $widgetClass]
	if { $classIndex <= 0} {
		return
	}
	bindtags $widget [linsert [lreplace $originalTags $classIndex $classIndex ] [incr classIndex -1] $widgetClass]
}


proc sm_removeClassBinding widget {

	set originalTags [bindtags $widget]
	set widgetClass [winfo class $widget]
	set classIndex [lsearch -exact $originalTags $widgetClass]
	if { $classIndex <= 0} {
		return
	}
	bindtags $widget [setSubtract [bindtags $widget] $widgetClass]
}


proc sm_enqueueStatusMessage { widget message } {
	global _smv_statusMessage
	
	set toplevelName [sm_getToplevelName $widget]
	lappend _smv_statusMessage($toplevelName,Queue) $message
	if { [info exists _smv_statusMessage($toplevelName,Status)] == 0 ||
		$_smv_statusMessage($toplevelName,Status) == "idle" ||
		$_smv_statusMessage($toplevelName,Status) == "override" } {
		set _smv_statusMessage($toplevelName,Status) ready
		_sm_activateStatusMessageMonitor $toplevelName
	}
}


proc _sm_activateStatusMessageMonitor { toplevelName } {
	global _smv_statusMessage
	global symbolicName

	set helpWidget $symbolicName(${toplevelName}Help)
	if { ![winfo exists $helpWidget] } {
		set _smv_statusMessage($toplevelName,Queue) ""
		set _smv_statusMessage($toplevelName,Status) idle
		return
	}
	switch -exact -- $_smv_statusMessage($toplevelName,Status) {
		idle {
			# Window must have been destroyed in the mean time.
		}
		ready {
			set _smv_statusMessage($toplevelName,Status) preamble
			$helpWidget delete 1.0 end
			after 200 "_sm_activateStatusMessageMonitor $toplevelName"
		}
		preamble {
			set _smv_statusMessage($toplevelName,Status) active
			$helpWidget insert end [lvarpop _smv_statusMessage($toplevelName,Queue)]
			update idletasks
			after 7000 "_sm_activateStatusMessageMonitor $toplevelName"
		}
		active {
			set _smv_statusMessage($toplevelName,Status) postamble
			$helpWidget delete 1.0 end
			after 2000 "_sm_activateStatusMessageMonitor $toplevelName"
		}
		override -
		postamble {
			if { $_smv_statusMessage($toplevelName,Queue) == "" } {
				set _smv_statusMessage($toplevelName,Status) idle
			} else {
				set _smv_statusMessage($toplevelName,Status) ready
				after 0 "_sm_activateStatusMessageMonitor $toplevelName"
			}
		}
		default {
			puts "We must have mispelled a status keyword: $_smv_statusMessage($toplevelName,Status)"
		}
	}
}


proc sm_WriteStatusMessage { widget message } {
	global _smv_statusMessage
	global symbolicName

	set toplevelName [sm_getToplevelName $widget]

	set _smv_statusMessage($toplevelName,Status) override

	set helpWidget ${toplevelName}Help
	$symbolicName($helpWidget) delete 1.0 end
	$symbolicName($helpWidget) insert end $message
	update idletasks
}


proc sm_setDefaultButton widget {

	sm_removeDefaultButton $widget

	set packSettings [pack info $widget]
	set master [lindex $packSettings [expr [lsearch -exact $packSettings "-in"] + 1]]
	set slaves [pack slaves $master]

	set parent [winfo parent $widget]
	if { $parent == "" } {
		set parent .
	}

	pack forget $widget

	set defaultButtonFrame [frame $parent.defaultButtonFrame \
		-borderwidth 1 \
		-highlightcolor black \
		-highlightthickness [$widget cget -highlightthickness] \
		-relief sunken \
		-takefocus 1]

	$widget configure -highlightthickness 0 -takefocus 0
	pack $widget -in $defaultButtonFrame -padx 3 -pady 3

	set nextWidgetIndex [expr [lsearch -exact $slaves $widget] + 1]
	if { $nextWidgetIndex >= [llength $slaves] } {
		eval pack $defaultButtonFrame $packSettings
	} else {
		set nextWidget [lindex $slaves $nextWidgetIndex]
		eval pack $defaultButtonFrame $packSettings -before $nextWidget
	}
	raise $widget

	bind $defaultButtonFrame <space> "tkButtonInvoke $widget"

	set parentToplevel [winfo toplevel $widget]
	set children ""
	windowWidgetTree $parentToplevel children
	lappend children $parentToplevel
	foreach child $children {
		bind $child <Return> "tkButtonInvoke $widget; break"
	}
}


proc sm_removeDefaultButton widget {

	set parent [winfo parent $widget]
	if { $parent == "" } {
		set parent .
	}

	set defaultButtonFrame ""
	set parentToplevel [winfo toplevel $widget]
	set children ""
	windowWidgetTree $parentToplevel children
	foreach child $children {
		if { [winfo name $child] == "defaultButtonFrame" } {
			set defaultButtonFrame $child
			set widget [pack slaves $defaultButtonFrame]
		}
	}

	if { $defaultButtonFrame == "" } {
		return
	}

	set packSettings [pack info $defaultButtonFrame]
	set master [lindex $packSettings [expr [lsearch -exact $packSettings "-in"] + 1]]
	set slaves [pack slaves $master]

	pack forget $widget
	pack forget $defaultButtonFrame

	$widget configure -highlightthickness [$defaultButtonFrame cget -highlightthickness] -takefocus 1

	destroy $defaultButtonFrame

	set nextWidgetIndex [expr [lsearch -exact $slaves $defaultButtonFrame] + 1]
	if { $nextWidgetIndex >= [llength $slaves] } {
		eval pack $widget $packSettings
	} else {
		set nextWidget [lindex $slaves $nextWidgetIndex]
		eval pack $widget $packSettings -before $nextWidget
	}
	raise $widget

	set children ""
	windowWidgetTree $parentToplevel children
	lappend children $parentToplevel
	foreach child $children {
		bind $child <Return> {}
	}
}


proc sm_getMessageTag { applicationName widget } {

	if { [cequal [winfo class $widget] Menu] } {
		set toplevelName [sm_getToplevelName [winfo parent $widget]]
	} else {
		set toplevelName [sm_getToplevelName $widget]
	}
	set parent $widget
	while { $parent != "" } {
		set class [winfo class $parent]
		if { [cequal [csubstr $class 0 [clength smComposite]] smComposite] } {
			return [translit a-z A-Z \
				"${applicationName}_[sm_getToplevelName $widget]_[winfo name $parent]_[winfo name $widget]"]
		}
		set parent [winfo parent $parent]
	}

	return [translit a-z A-Z "${applicationName}_${toplevelName}_[winfo name $widget]"]
}


proc sm_locidToMessageTag locid {

    global UC_LETTER

	set newString ""
	append newString [cindex $locid 0]
	for { set i 1 } { $i < [clength $locid] } { incr i } {
	    if { [regexp \[${UC_LETTER}\] [cindex $locid $i]] } {
			append newString _[cindex $locid $i]
		} else {
			append newString [cindex $locid $i]
		}
	}
	return  [translit a-z A-Z $newString]_LOCID
}


proc sm_getLocid { applicationName widget } {

	set compositWidgetName ""

	set parent $widget
	while { $parent != "" } {
		set class [winfo class $parent]
		if { [cequal [csubstr $class 0 [clength smComposite]] smComposite] } {
			set compositWidgetName [winfo name $parent]
			set compositWidgetName [translit a-z A-Z [crange $compositWidgetName 0 0]][crange $compositWidgetName 1 end]
			break
		}
		set parent [winfo parent $parent]
	}

	set toplevelName [sm_getToplevelName $widget]
	set widgetName [winfo name $widget]

	set applicationName [translit a-z A-Z [crange $applicationName 0 0]][crange $applicationName 1 end]
	set toplevelName [translit a-z A-Z [crange $toplevelName 0 0]][crange $toplevelName 1 end]
	set widgetName [translit a-z A-Z [crange $widgetName 0 0]][crange $widgetName 1 end]

	append locid $applicationName $toplevelName $compositWidgetName $widgetName

	return $locid
}


proc _sm_HelpOnPointerCheck { applicationName widget } {
	global _smv_statusMessage
	global smv_pointerHelpEnabled
	global ${applicationName}_dstr
	global ${applicationName}_catfd
	global msg_${applicationName}
	global symbolicName

	set toplevelName [sm_getToplevelName $widget]
	set helpWidget ${toplevelName}Help

	if { !([info exists symbolicName($helpWidget)] && [winfo exists $symbolicName($helpWidget)]) } {
		return
	}

	if { [info exists _smv_statusMessage($toplevelName,Status)] && $_smv_statusMessage($toplevelName,Status) != "idle" } {
		return
	}
	if { !$smv_pointerHelpEnabled } {
		return
	}

	set helpTag [sm_getMessageTag $applicationName $widget]
	if { ![info exists ${applicationName}_dstr($helpTag)] } {
		return
	}

	$symbolicName($helpWidget) delete 1.0 end
	$symbolicName($helpWidget) insert end [catgets [set ${applicationName}_catfd] [set msg_${applicationName}(PFH)] \
			[set msg_${applicationName}($helpTag)] [set ${applicationName}_dstr($helpTag)]]
}


proc _sm_HelpOnPointerClear { widget } {
	global _smv_statusMessage
	global smv_pointerHelpEnabled
	global symbolicName

	set toplevelName [sm_getToplevelName $widget]
	set helpWidget ${toplevelName}Help

	if { !([info exists symbolicName($helpWidget)] && [winfo exists $symbolicName($helpWidget)]) } {
		return
	}

	if { [info exists _smv_statusMessage($toplevelName,Status)] && $_smv_statusMessage($toplevelName,Status) != "idle" } {
		return
	}
	if { !$smv_pointerHelpEnabled } {
		return
	}

	if { [info exists symbolicName($helpWidget)] } {
		$symbolicName($helpWidget) delete 1.0 end
	}
}


proc sm_getI18nString { messageTag defaultString } {
	global i18nfd
	global msg_i18n_motif_shared_text
	global i18n_motif_shared_text_dstr

	return [catgets $i18nfd $msg_i18n_motif_shared_text(i18n_shared_strings) $msg_i18n_motif_shared_text($messageTag) $defaultString] \
}


proc sm_getAlignmentDistance { movableWidget movableWidgetSide alignmentWidget alignmentWidgetSide adjacentWidget adjacentWidgetSide } {

	scan [winfo geometry $movableWidget] "%dx%d+%d+%d" movable(width) movable(height) movable(x) movable(y)
	scan [winfo geometry $alignmentWidget] "%dx%d+%d+%d" alignment(width) alignment(height) alignment(x) alignment(y)
	scan [winfo geometry $adjacentWidget] "%dx%d+%d+%d" adjacent(width) adjacent(height) adjacent(x) adjacent(y)

	set rootx [winfo rootx $alignmentWidget]
	set rooty [winfo rooty $alignmentWidget]
	if { $alignmentWidgetSide == "top" } {
		set alignmentOffset $rooty
	} else {
		set alignmentOffset [expr $rooty + $alignment(height)]
	}

	set rootx [winfo rootx $adjacentWidget]
	set rooty [winfo rooty $adjacentWidget]
	if { $adjacentWidgetSide == "top" } {
		set adjacentOffset $rooty
	} else {
		set adjacentOffset [expr $rooty + $adjacent(height)]
	}

	if { $movableWidgetSide == "top" } {
		set padValue [expr $alignmentOffset - $adjacentOffset]
	} else {
		set padValue [expr $alignmentOffset - $adjacentOffset - $movable(height)]
	}

	if { $padValue < 0 } {
		return 0
	} else {
		return $padValue
	}
}


proc sm_nameToString toplevelName {

    global UC_LETTER LC_LETTER

	set capitalize 1
	for { set i 0 } { $i < [clength $toplevelName] } {} {
	    if { ![regexp \[${UC_LETTER}${LC_LETTER}\] \
		       [cindex $toplevelName $i]] } {
			set capitalize 1
			incr i
			continue
		}
		if { $capitalize } {
			append newString [translit a-z A-Z [cindex $toplevelName $i]]
			set capitalize 0
		} else {
			append newString [cindex $toplevelName $i]
		}
		incr i
		if { [regexp \[${UC_LETTER}\] [cindex $toplevelName $i]] } {
			set capitalize 1
			append newString " "
		} elseif { [cindex $toplevelName $i] != "" && \
			   ![regexp \[${LC_LETTER}\] [cindex $toplevelName $i]] } {
			set capitalize 1
			append newString " "
			incr i
		}
	}

	return $newString
}


proc sm_setDispositionButtonStates { toplevelWidgetName status } {
	global symbolicName

	if { $status == "modified" } {
		set state1 normal
		set state2 disabled
	} else {
		set state1 disabled
		set state2 normal
	}
	foreach dispositionButton [array names symbolicName "${toplevelWidgetName}*Button"] {
		regsub -all -- "($toplevelWidgetName)|(Button)" $dispositionButton "" buttonType
		switch -exact -- $buttonType {
			Apply {
				$symbolicName($dispositionButton) configure -state $state1
			}
			Cancel {
				$symbolicName($dispositionButton) configure -state $state1
			}
			Close {
				$symbolicName($dispositionButton) configure -state $state2
			}
			Commit {
				$symbolicName($dispositionButton) configure -state $state1
			}
			Ok {
				$symbolicName($dispositionButton) configure -state $state1
			}
		}
	}
	if { [info exists symbolicName(${toplevelWidgetName}OkButton)] } {
		if { [info exists symbolicName(${toplevelWidgetName}CancelButton)] } {
			$symbolicName(${toplevelWidgetName}CancelButton) configure -state normal
			$symbolicName(${toplevelWidgetName}OkButton) configure -state $state1
		} else {
			$symbolicName(${toplevelWidgetName}OkButton) configure -state normal
		}
	}
}


proc sm_pendingModifications {} {
	global symbolicName

	set allToplevels ""
	widgetsByClass . Toplevel allToplevels
	foreach toplevelWindow $allToplevels {
		set toplevelName [sm_getToplevelName $toplevelWindow]
		if {	[info exists symbolicName(${toplevelName}CommitButton)] &&
			[$symbolicName(${toplevelName}CommitButton) cget -state] == "normal" } {
			return 1
		}
		if {	[info exists symbolicName(${toplevelName}OkButton)] &&
			[$symbolicName(${toplevelName}OkButton) cget -state] == "normal" } {
			return 1
		}
	}
	return 0
}


proc sm_listboxInsert { listboxWidget newListboxElement position } {

	$listboxWidget insert $position $newListboxElement
	$listboxWidget selection clear 0 end
	$listboxWidget selection set $position
	$listboxWidget see $position
}


proc sm_listboxDelete { listboxWidget position } {

	$listboxWidget delete $position

	set current_size [$listboxWidget size]

	if { $current_size > 0 } {
		if { $current_size > $position } {
			$listboxWidget selection set $position
			$listboxWidget see $position
		} else {
			$listboxWidget selection set $current_size
			$listboxWidget see $current_size
		}
	}
}


proc sm_parseArguments { argumentNames argumentValues } {

	foreach name $argumentNames {
		upvar $name local_${name}
	}

	set excessArgs ""
	while { $argumentValues != "" && $argumentNames != "" } {
		set argument [lindex $argumentValues 0]
		if { [llength $argument] == 2 && [lsearch -exact $argumentNames [lindex $argument 0]] != -1 } {
			set local_[lindex $argument 0] [lindex $argument 1]
			sm_lvarremove argumentNames [lindex $argument 0]
			lvarpop argumentValues
		} elseif { [crange $argument 0 0] == "-" } {
			set name [crange $argument 1 end]
			if { [set index [lsearch -exact $argumentNames $name]] != -1 } {
				lvarpop argumentValues
				set local_$name [lvarpop argumentValues]
				sm_lvarremove argumentNames $name
			} else {
				lappend excessArgs [lvarpop argumentValues]
				lappend excessArgs [lvarpop argumentValues]
			}
		} else {
			set local_[lvarpop argumentNames] $argument
			lvarpop argumentValues
		}
	}

	return [concat $excessArgs $argumentValues]
}


proc sm_lvarremove { listName element } {

	upvar $listName setList
        set index [lsearch -exact $setList $element]
        if { $index != -1 } {
                set setList [lreplace $setList $index $index]
        }

        return $setList
}


proc _sm_tkerror err {
	global errorInfo

	set info $errorInfo
	set button [tk_dialog .tkerrorDialog "Error in Tcl Script" \
		"Error: $err" error 0 OK "Skip Messages" "Stack Trace"]
	if {$button == 0} {
		return
	} elseif {$button == 1} {
		return -code break
	}

	set w .tkerrorTrace
	catch {destroy $w}
	toplevel $w
	wm minsize $w 1 1
	wm title $w "Stack Trace for Error"
	wm iconname $w "Stack Trace"
	frame $w.f
	button $w.f.ok -text OK -width 20 -command "destroy $w"
	button $w.f.grabReleaseButton \
		-text "Release all grabs" \
		-width 20 \
		-command {set _smv_grabStack ""; grab release .tkerrorTrace; %W configure -state disabled}
	text $w.text -relief sunken -bd 2 -yscrollcommand "$w.scroll set" \
		-setgrid true -width 60 -height 20
	scrollbar $w.scroll -relief sunken -command "$w.text yview"
	pack $w.f.ok -side left -expand 1
	pack $w.f.grabReleaseButton -side left -expand 1
	pack $w.f -side bottom -fill x -padx 3m -pady 2m
	pack $w.scroll -side right -fill y
	pack $w.text -side left -expand yes -fill both
	$w.text insert 0.0 $info
	$w.text mark set insert 0.0

	# Center the window on the screen.

	wm withdraw $w
	update idletasks
	set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
		- [winfo vrootx [winfo parent $w]]]
	set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
		- [winfo vrooty [winfo parent $w]]]
	wm geom $w +$x+$y
	sm_setDefaultButton $w.f.ok
	grab set .tkerrorTrace
	wm deiconify $w
}


proc sm_getMaster w {
	set packSettings [pack info $w]
	return [lindex $packSettings [expr [lsearch -exact $packSettings "-in"] + 1]]
}


proc _sm_tk_focusNext w {
	set cur $w
	while 1 {

		# Descend to just before the first child of the current widget.

		set parent $cur
		set children [pack slaves $cur]
		set i -1

		# Look for the next sibling that isn't a top-level.

		while 1 {
			incr i
			if {$i < [llength $children]} {
				set cur [lindex $children $i]
				if {[winfo toplevel $cur] == $cur} {
					continue
				} else {
					break
				}
			}

			# No more siblings, so go to the current widget's parent.
			# If it's a top-level, break out of the loop, otherwise
			# look for its next sibling.

			set cur $parent
			if {[winfo toplevel $cur] == $cur} {
				break
			}
			set parent [sm_getMaster $parent]
			set children [pack slaves $parent]
			set i [lsearch -exact $children $cur]
		}
		if {($cur == $w) || [tkFocusOK $cur]} {
			return $cur
		}
	}
}


proc _sm_tk_focusPrev w {
	set cur $w
	while 1 {

		# Collect information about the current window's position
		# among its siblings.  Also, if the window is a top-level,
		# then reposition to just after the last child of the window.

		if {[winfo toplevel $cur] == $cur}  {
			set parent $cur
			set children [pack slaves $cur]
			set i [llength $children]
		} else {
			set parent [sm_getMaster $cur]
			set children [pack slaves $parent]
			set i [lsearch -exact $children $cur]
		}

		# Go to the previous sibling, then descend to its last descendant
		# (highest in stacking order.  While doing this, ignore top-levels
		# and their descendants.  When we run out of descendants, go up
		# one level to the parent.

		while {$i > 0} {
			incr i -1
			set cur [lindex $children $i]
			if {[winfo toplevel $cur] == $cur} {
				continue
			}
			set parent $cur
			set children [pack slaves $parent]
			set i [llength $children]
		}
		set cur $parent
		if {($cur == $w) || [tkFocusOK $cur]} {
			return $cur
		}
	}
}

